package com.suncode.dbexplorer.database.internal.oracle;

import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.dbexplorer.database.internal.support.AbstractDatabaseImplementor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.HibernateException;
import org.hibernate.type.StandardBasicTypes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Slf4j
@Component
public class OracleDatabase
    extends AbstractDatabaseImplementor
{
    public @Autowired OracleDatabase( OracleSchemaLoader schemaLoader, OracleConnectionUrlResolver urlResolver,
                                      OracleDataTypeRegistry typeRegistry )
    {
        super( DatabaseType.ORACLE, schemaLoader, urlResolver, typeRegistry );
    }

    @Override
    protected String getDriverClassName()
    {
        return "oracle.jdbc.driver.OracleDriver";
    }

    @Override
    public String getCurrentSchemaName( DatabaseSession session )
    {
        return (String) session.hibernateSession()
            .createSQLQuery( "select sys_context( 'userenv', 'current_schema' ) as schema from dual" )
            .addScalar( "schema", StandardBasicTypes.STRING )
            .uniqueResult();
    }

    @Override
    public List<String> getSchemasNames( DatabaseSession session )
    {
        List<String> result = getSchemaNamesFromDbaUsers( session );
        if ( result.isEmpty() )
        {
            log.debug( "User have no access to DBA views. Searching schemas in user's tablespace" );
            result = getSchemaNamesFromAllTables( session );
            if ( result.isEmpty() )
            {
                log.debug( "Could not find any schemas. Returning deafult schema name" );
                return Arrays.asList( session.getDatabase().getDefaultSchemaName() );
            }
        }
        return result;
    }

    @SuppressWarnings( "unchecked" )
    private List<String> getSchemaNamesFromDbaUsers( DatabaseSession session )
    {
        try
        {
            return (List<String>) session.hibernateSession()
                .createSQLQuery( "SELECT username as schema FROM dba_users WHERE default_tablespace not in ('SYSTEM','SYSAUX')" )
                .addScalar( "schema", StandardBasicTypes.STRING )
                .list();
        }
        catch ( HibernateException he )
        {
            return new ArrayList<>();
        }
    }

    @SuppressWarnings( "unchecked" )
    private List<String> getSchemaNamesFromAllTables( DatabaseSession session )
    {
        try
        {
            return (List<String>) session.hibernateSession()
                .createSQLQuery( "SELECT OWNER as schema FROM sys.all_tables WHERE tablespace_name NOT IN ('SYSTEM','SYSAUX') GROUP BY OWNER "
                                     + "UNION SELECT GRANTOR as schema FROM USER_TAB_PRIVS GROUP BY GRANTOR" )
                .addScalar( "schema", StandardBasicTypes.STRING )
                .list();
        }
        catch ( HibernateException he )
        {
            return new ArrayList<>();
        }
    }
}
