package com.suncode.dbexplorer.database.internal;

import com.google.common.base.Suppliers;
import com.suncode.dbexplorer.database.*;
import com.suncode.dbexplorer.database.exception.DatabaseNotAvailableException;
import com.suncode.dbexplorer.database.internal.type.DataTypeRegistry;
import com.suncode.dbexplorer.database.schema.DatabaseSchema;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;
import org.hibernate.boot.model.TypeContributor;
import org.hibernate.cfg.Environment;
import org.springframework.orm.hibernate5.LocalSessionFactoryBuilder;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.List;
import java.util.function.Supplier;

@Slf4j
public class DatabaseImpl
                implements Database, DatabaseImplementor
{
    private static final int MAX_SUPPORTED_ORACLE_VERSION = 11;

    private static final String MAX_SUPPORTED_ORACLE_DIALECT = "org.hibernate.dialect.Oracle10gDialect";

    @Getter
    private final DatabaseImplementor implementor;

    @Getter
    private final DataSource dataSource;

    private Boolean initialized = false;

    private Boolean initializing = false;

    // Pola poniżej mogą być nullami, jeżeli nie ma połączenia z bazą
    private String catalog;

    private String defaultSchemaName;

    private List<String> schemasNames;

    private SessionFactory sessionFactory;

    private Supplier<List<DatabaseSchema>> schemas;

    public DatabaseImpl( DataSource dataSource, DatabaseImplementor implementor, String schema, Boolean available )
    {
        Assert.notNull( dataSource, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( implementor, "[Assertion failed] - this argument is required; it must not be null" );

        this.implementor = implementor;
        this.dataSource = dataSource;

        startDatabaseConnection(available);
    }

    private void startDatabaseConnection( Boolean available )
    {
        if ( available )
        {
            try
            {
                initDatabaseConnection();
            }
            catch ( Exception ex )
            {
                log.error( ex.getMessage(), ex );
                clearDatabaseConnection();
            }
        }
        else
        {
            clearDatabaseConnection();
        }
    }

    private void initDatabaseConnection()
                    throws Exception
    {
        if ( !initializing )
        {
            this.initializing = true;

            try
            {
                LocalSessionFactoryBuilder builder = new LocalSessionFactoryBuilder( dataSource );
                if ( this.implementor instanceof TypeContributor )
                {
                    builder.registerTypeContributor( (TypeContributor) this.implementor );
                }

                if ( isOracleDatabase() && !isSupportedOracleVersion() )
                {
                    builder.setProperty( Environment.DIALECT, MAX_SUPPORTED_ORACLE_DIALECT );
                }

                setSessionFactoryProperties( builder );

                sessionFactory = builder.buildSessionFactory();
                DatabaseSession session = openSession();

                catalog = implementor.getCatalog( session );
                defaultSchemaName = implementor.getCurrentSchemaName( session );
                schemasNames = implementor.getSchemasNames( session );
                session.commit();

                schemas = Suppliers.memoize( () -> readSchemas( schemasNames ) )::get;

                initialized = true;
            }
            finally
            {
                initializing = false;
            }
        }
    }

    private List<DatabaseSchema> readSchemas( final List<String> schemasNames )
    {
        return withinSession( session -> implementor.readSchemas( session, schemasNames ) );
    }

    private void clearDatabaseConnection()
    {
        close();

        catalog = null;
        defaultSchemaName = null;
        schemasNames = null;
        sessionFactory = null;
        schemas = null;
        initialized = false;
    }

    @Override
    public String getSchemaName()
    {
        // Zostawione dla kompatybilności wstecznej
        return getDefaultSchemaName();
    }

    @Override
    public String getDefaultSchemaName()
    {
        return reinitIfNeededAndReturn( () -> defaultSchemaName );
    }

    @Override
    public List<String> getSchemasNames()
    {
        return reinitIfNeededAndReturn( () -> schemasNames );
    }

    @Override
    public List<DatabaseSchema> getSchemas()
    {
        return reinitIfNeededAndReturn( () -> schemas.get() );
    }

    public SessionFactory getSessionFactory()
    {
        return reinitIfNeededAndReturn( () -> sessionFactory );
    }

    @Override
    public String getCatalog()
    {
        return reinitIfNeededAndReturn( () -> catalog );
    }

    private <T> T reinitIfNeededAndReturn( Supplier<T> supplier )
    {
        if ( initialized )
        {
            return supplier.get();
        }
        else
        {
            try
            {
                initDatabaseConnection();
                return supplier.get();
            }
            catch ( Exception ex )
            {
                clearDatabaseConnection();
                throw new DatabaseNotAvailableException( ex );
            }
        }
    }

    @Override
    public DatabaseSchema getSchema()
    {
        return getSchema( defaultSchemaName );
    }

    @Override
    public DatabaseSchema getSchema( String name )
    {
        for ( DatabaseSchema schema : getSchemas() )
        {
            if ( schema.getName().equals( name ) )
            {
                return schema;
            }
        }

        throw new IllegalArgumentException( "There is no database schema with name: " + name );
    }

    @Override
    public void updateSchema()
    {
        schemaUpdated();
    }

    @Override
    public void schemaUpdated()
    {
        DatabaseSession session = openSession();
        this.schemasNames = implementor.getSchemasNames( session );
        this.schemas = Suppliers.memoize( () -> readSchemas( schemasNames ) )::get;
        session.commit();
    }

    @Override
    public DatabaseSession openSession()
    {
        Connection connection = null;
        try
        {
            connection = dataSource.getConnection();
            connection.setAutoCommit( false );

            DatabaseSession session = new DatabaseSessionImpl( connection, this );

            this.initialized = true;
            return session;
        }
        catch ( SQLException ex )
        {
            if ( connection != null )
            {
                try
                {
                    connection.close();
                }
                catch ( SQLException e1 )
                {
                    // TOOD:log
                }
            }

            clearDatabaseConnection();
            throw new DatabaseNotAvailableException( ex );
        }
    }

    @Override
    public <T> T withinSession( SessionUnit<T> unit )
    {
        Assert.notNull( unit, "[Assertion failed] - this argument is required; it must not be null" );

        DatabaseSession session = openSession();
        T value;
        try
        {
            value = unit.doWork( session );
        }
        catch ( RuntimeException | Error ex )
        {
            session.rollback();
            throw ex;
        }
        catch ( Exception ex )
        {
            session.rollback();
            // TODO: exception
            throw new RuntimeException( ex );
        }
        session.commit();

        return value;
    }

    public void close()
    {
        try
        {
            if ( sessionFactory != null )
            {
                sessionFactory.close();
            }
        }
        catch ( HibernateException e )
        {
            // ignore
        }
    }

    // ---------------------------------------------------------------------
    // DatabaseImplementor interface methods
    // ---------------------------------------------------------------------

    @Override
    public boolean handles( DatabaseType type )
    {
        return implementor.handles( type );
    }

    @Override
    public Class<? extends Driver> getDriverClass()
    {
        return implementor.getDriverClass();
    }

    @Override
    public DataTypeRegistry getTypeRegistry()
    {
        return implementor.getTypeRegistry();
    }

    @Override
    public String buildConnectionUrl( ConnectionString connectionString )
    {
        return implementor.buildConnectionUrl( connectionString );
    }

    @Override
    public String getCatalog( DatabaseSession session )
    {
        return implementor.getCatalog( session );
    }

    @Override
    public String getCurrentSchemaName( DatabaseSession session )
    {
        return implementor.getCurrentSchemaName( session );
    }

    @Override
    public List<String> getSchemasNames( DatabaseSession session )
    {
        return implementor.getSchemasNames( session );
    }

    @Override
    public List<DatabaseSchema> readSchemas( DatabaseSession session, List<String> schemasNames )
    {
        return implementor.readSchemas( session, schemasNames );
    }

    @Override
    public String escapeColumnName( String columnName )
    {
        return implementor.escapeColumnName( columnName );
    }

    @Override
    public String escapeTableName( String tableName )
    {
        return implementor.escapeTableName( tableName );
    }

    private boolean isOracleDatabase()
                    throws SQLException
    {
        DatabaseMetaData metaData = dataSource.getConnection().getMetaData();
        return metaData.getDatabaseProductName().toLowerCase().contains( "oracle" );
    }

    private boolean isSupportedOracleVersion()
                    throws SQLException
    {
        DatabaseMetaData metaData = dataSource.getConnection().getMetaData();
        return metaData.getDatabaseMajorVersion() <= MAX_SUPPORTED_ORACLE_VERSION;
    }

    private void setSessionFactoryProperties(LocalSessionFactoryBuilder builder)
    {
        builder.setProperty( "hibernate.native_exception_handling_51_compliance", "true" );
    }

}
