package com.suncode.dbexplorer.database;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.suncode.dbexplorer.database.RecordId.Composite;
import com.suncode.dbexplorer.database.schema.TableSchema;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Rekord reprezentuje jeden wiersz tabeli konkretnej bazy danych.
 * 
 * @author Cezary Kozar 25 wrz 2015
 */
public class Record
{
    private final TableSchema table;

    private final Map<String, Object> data = new HashMap<String, Object>();

    private RecordId id;

    public Record( String schema, String table, Database database )
    {
        Assert.hasText( schema, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.notNull( database, "[Assertion failed] - this argument is required; it must not be null" );

        this.table = database.getSchema(schema).getTable(table);
    }

    /**
     * Zwraca informację czy ten rekord posiada ID to znaczy, że tabela tego rekordu ma klucz główny
     * co umożliwia aktualizację rekordu.
     */
    @JsonIgnore
    public boolean hasId()
    {
        return table.hasPrimaryKey();
    }

    /**
     * Odczytuje id {@linkplain RecordId} tego rekordu w oparciu o jego aktualne dane.
     * 
     * @return aktualny identyfikator rekordu
     */
    @JsonIgnore
    public RecordId getId()
    {
        if ( id != null )
        {
            return id;
        }

        if ( table.hasPrimaryKey() )
        {
            Composite id = RecordId.buildComposite();
            for ( String pkCol : table.getPrimaryKey().getColumns() )
            {
                id.add( pkCol, get( pkCol ) );
            }
            return id.get();
        }
        throw new IllegalStateException( "Record [" + toString() + "] does not have any id in " + table.getName() + " table");
    }

    public void setId( RecordId id )
    {
        // TODO: rekord powinien byc mądrzejszy i trzymac stan -> oryginalne wartości, zmodyfikowane, commit, reject
        this.id = id;
    }

    /**
     * Zwraca tabelę tego rekordu
     */
    @JsonIgnore
    public TableSchema getTable()
    {
        return table;
    }

    /**
     * Zwraca nazwę tabeli tego rekordu
     */
    public String getTableName()
    {
        return table.getName();
    }

    /**
     * Ustawia wartośc podanej kolumny. Jeżeli kolumna nie istnieje w mapowanej tabeli rzucany jest
     * błąd.
     * 
     * @param column nazwa kolumny
     * @param value nowa wartośc
     */
    public void set( String column, Object value )
    {
        if ( !table.hasColumn( column ) )
        {
            throw new IllegalStateException( "Record [" + toString() + "] mapped table does not contain given column ["
                + column + "]" );
        }
        data.put( column, value );
    }

    /**
     * Zwraca aktualną wartośc kolumny rekordu
     * 
     * @param column nazwa kolumny
     * @return aktualna wartośc kolumny
     */
    public Object get( String column )
    {
        return data.get( column );
    }

    /**
     * Zwraca reprezentację rekordu w postaci mapy. Kluczem jest nazwa kolumny.
     * 
     * @return rekord w postaci mapy
     */
    public Map<String, Object> getData()
    {
        return Collections.unmodifiableMap( data );
    }

    @Override
    public String toString()
    {
        return "Record(" + table + ")";
    }
}
