package com.suncode.dbexplorer.database;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.plusmpm.security.TextCipher;

import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.SneakyThrows;

/**
 * @author Cezary Kozar 8 wrz 2015
 */
@Builder
@Getter
@Setter
@NoArgsConstructor
public class ConnectionString
{

    private DatabaseType type;

    private String catalog;

    private String schema;

    private String host;

    private Integer port;

    private String user;

    private String password;

    private String jdbcUrl;

    public ConnectionString( DatabaseType type, String catalog, String schema, String host, Integer port, String user,
                             String password, String jdbcUrl )
    {
        if ( !notEmpty( jdbcUrl ) )
        {
            Assert.hasText( host, "Database host cannot be empty if JDBC URL is not specified" );
            Assert.notNull( type, "[Assertion failed] - this argument is required; it must not be null" );
        }

        this.type = type;
        this.catalog = catalog;
        this.schema = schema;
        this.host = host;
        this.port = port;
        this.user = user;
        this.password = password;
        this.jdbcUrl = jdbcUrl;
    }

    public boolean catalogSpecified()
    {
        return notEmpty( catalog );
    }

    public boolean schemaSpecified()
    {
        return notEmpty( schema );
    }

    private boolean notEmpty( String string )
    {
        return StringUtils.hasText( string );
    }

    @SneakyThrows
    public static ConnectionString fromDomain( ConnectionStringDomain domain )
    {
        return builder()
            .catalog( domain.getCatalog() )
            .host( domain.getHost() )
            .jdbcUrl( domain.getJdbcUrl() )
            .password( TextCipher.decrypt( domain.getPassword() ) )
            .port( domain.getPort() )
            .schema( domain.getSchema() )
            .type( domain.getType() )
            .user( domain.getUser() )
            .build();

    }

    @SneakyThrows
    public static ConnectionString fromJdbcUrl( ConnectionStringDomain domain )
    {
        return builder()
            .jdbcUrl( domain.getJdbcUrl() )
            .password( TextCipher.decrypt( domain.getPassword() ) )
            .schema( domain.getSchema() )
            .user( domain.getUser() )
            .build();
    }

    @SneakyThrows
    public static ConnectionStringDomain toDomain( ConnectionString connectionString )
    {
        return ConnectionStringDomain.builder()
            .catalog( connectionString.getCatalog() )
            .host( connectionString.getHost() )
            .jdbcUrl( connectionString.getJdbcUrl() )
            .password( TextCipher.encrypt( connectionString.getPassword() ) )
            .port( connectionString.getPort() )
            .schema( connectionString.getSchema() )
            .type( connectionString.getType() )
            .user( connectionString.getUser() )
            .build();
    }
    
    @Override
    public boolean equals( Object obj )
    {
        if ( obj == null )
        {
            return false;
        }
        if ( obj == this )
        {
            return true;
        }
        if ( !( obj instanceof ConnectionString ) )
        {
            return false;
        }

        ConnectionString other = (ConnectionString) obj;
        return new EqualsBuilder()
            .append( this.catalog, other.catalog )
            .append( this.schema, other.schema )
            .append( this.host, other.host )
            .append( this.port, other.port )
            .append( this.type, other.type )
            .append( this.user, other.user )
            .append( this.password, other.password )
            .isEquals();
    }

    @Override
    public int hashCode()
    {
        return new HashCodeBuilder()
            .append( this.catalog )
            .append( this.schema )
            .append( this.host )
            .append( this.port )
            .append( this.type )
            .append( this.user )
            .append( this.password )
            .toHashCode();
    }

    @Override
    public String toString()
    {
        StringBuilder s = new StringBuilder()
            .append( getClass().getSimpleName() )
            .append( "( " )
            .append( catalog ).append( "@" ).append( host );

        if ( host != null )
        {
            s.append( ":" ).append( host );
        }
        return s.append( ")" ).toString();
    }

}
