package com.suncode.dbexplorer.alias.util;

import java.lang.reflect.Field;
import java.net.URI;

import javax.sql.DataSource;

import org.apache.commons.lang.StringUtils;

import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.pwfl.database.DataSourceFactory;
import com.zaxxer.hikari.HikariDataSource;

import lombok.SneakyThrows;

public class SystemConnectionStringResolver
{
    public static ConnectionString getSystemConnectionString()
    {
        DataSource systemDataSource = DataSourceFactory.getDataSource();
        if ( systemDataSource instanceof HikariDataSource )
        {
            return resolveHikariDataSource( (HikariDataSource) systemDataSource );
        }
        else
        {
            return resolveDataSourceReflect( systemDataSource );
        }
    }

    private static ConnectionString resolveHikariDataSource( HikariDataSource dataSource )
    {
        String connectionUrl = dataSource.getJdbcUrl();

        return ConnectionString.builder()
            .jdbcUrl( connectionUrl )
            .user( dataSource.getUsername() )
            .password( dataSource.getPassword() )
            .type( DatabaseType.parseScheme( parseJdbcUrl( connectionUrl ).getScheme() ) )
            .build();
    }

    private static ConnectionString resolveDataSourceReflect( DataSource dataSource )
    {
        String connectionUrl = readPrivateProperty( dataSource, "url", String.class );

        return ConnectionString.builder()
            .jdbcUrl( connectionUrl )
            .user( readPrivateProperty( dataSource, "username", String.class ) )
            .password( readPrivateProperty( dataSource, "password", String.class ) )
            .type( DatabaseType.parseScheme( parseJdbcUrl( connectionUrl ).getScheme() ) )
            .build();
    }

    @SneakyThrows
    @SuppressWarnings( "unchecked" )
    private static <T> T readPrivateProperty( Object obj, String propertyName, Class<T> clazz )
    {
        Field field = obj.getClass().getDeclaredField( propertyName );
        field.setAccessible( true );
        return (T) field.get( obj );
    }

    public static URI parseJdbcUrl( String url )
    {
        if ( url.contains( "\\" ))
        {
            url = manageBackslash( url );
        }
        return URI.create( url.replaceFirst( "jdbc:", StringUtils.EMPTY ) );
    }

    private static String manageBackslash( String url )
    {
        int backslashIndex = url.indexOf( "\\" );
        int colonIndex = url.indexOf( ":", backslashIndex );
        String instanceName = url.substring( backslashIndex + 1, colonIndex );
        return url.substring( 0, backslashIndex ) + url.substring( colonIndex ) + ";instanceName=" + instanceName;
    }
}
