package com.suncode.dbexplorer.alias.permission.internal;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import com.suncode.dbexplorer.alias.TablesSet;
import com.suncode.dbexplorer.alias.dto.SimpleTableDto;
import com.suncode.dbexplorer.alias.internal.TablesSetRepository;
import com.suncode.dbexplorer.alias.permission.AccessLevel;
import com.suncode.dbexplorer.alias.permission.AccessResource;
import com.suncode.dbexplorer.alias.permission.AccessResource.ResourceType;
import com.suncode.dbexplorer.alias.permission.PermissionsService;
import com.suncode.dbexplorer.alias.permission.SecuredTablesSet;
import com.suncode.dbexplorer.alias.permission.TablesSetPermission;
import com.suncode.dbexplorer.context.UserContext;
import com.suncode.pwfl.administration.user.User;
import com.suncode.pwfl.administration.user.UserService;
import lombok.RequiredArgsConstructor;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
@Transactional
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class PermissionsServiceImpl
    implements PermissionsService
{
    private final UserService userService;

    private final TablesSetRepository setRepository;

    private final TablesSetPermissionRepository setPermissionRepository;

    private final LoadingCache<String, List<SecuredTablesSet>> permissionsCache = CacheBuilder.newBuilder()
        .maximumSize( 1000 )
        .expireAfterWrite( 10, TimeUnit.SECONDS ) // parametr dodany w związku ze zgłoszeniem DBEX-189
        .build( new CacheLoader<>()
        {
            @Override
            public List<SecuredTablesSet> load( String key )
            {
                return getSecuredTablesSets( key );
            }
        } );

    @Override
    public AccessResource getResource( Long id, ResourceType type )
    {
        switch ( type )
        {
            case USER:
                return new AccessResource( userService.getUser( id ) );
            case GROUP:
                return new AccessResource( userService.getGroup( id ) );
        }
        throw new IllegalStateException( "Unknown resource type: " + type );
    }

    @Override
    public List<TablesSetPermission> getPermissions( Long setId )
    {
        DetachedCriteria criteria = DetachedCriteria.forClass( TablesSetPermission.class )
            .add( Restrictions.eq( "tablesSet.id", setId ) );

        List<TablesSetPermission> permissions = setPermissionRepository.findByCriteria( criteria );
        deleteRecordsWithNonExistingUser( permissions );
        return permissions;
    }

    private void deleteRecordsWithNonExistingUser( List<TablesSetPermission> permissions )
    {
        for ( int i = permissions.size() - 1; i >= 0; i-- )
        {
            TablesSetPermission permission = permissions.get( i );
            if ( permission.getResource().getUser() == null && permission.getResource().getGroup() == null )
            {
                deletePermission( permission.getId() );
                permissions.remove( permission );
            }
        }
    }

    @Override
    public TablesSetPermission newPermission( Long setId, AccessResource resource, AccessLevel level )
    {
        TablesSet set = setRepository.get( setId );
        if ( set == null )
        {
            throw new IllegalArgumentException( "Tables set with id [" + setId + "] does not exists!" );
        }

        TablesSetPermission permission = setPermissionRepository.getResourcePermission( set, resource );
        if ( permission != null )
        {
            permission.setLevel( level );
        }
        else
        {
            permission = new TablesSetPermission( set, level, resource );
            setPermissionRepository.save( permission );
        }
        permissionsCache.invalidateAll();
        return permission;
    }

    @Override
    public TablesSetPermission changePermissionLevel( Long permissionId, AccessLevel newLevel )
    {
        Assert.notNull( newLevel, "New level cannot be empty" );

        TablesSetPermission permission = setPermissionRepository.get( permissionId );
        if ( permission == null )
        {
            throw new IllegalArgumentException( "Tables set permission with id [" + permissionId + "] not found!" );
        }

        permission.setLevel( newLevel );
        permissionsCache.invalidateAll();
        return permission;
    }

    @Override
    public TablesSetPermission deletePermission( Long permissionId )
    {
        TablesSetPermission permission = setPermissionRepository.get( permissionId );
        if ( permission != null )
        {
            setPermissionRepository.delete( permission );
            permissionsCache.invalidateAll();
        }

        return permission;
    }

    @Override
    public List<SecuredTablesSet> getSecuredTablesSets( String username )
    {
        Map<String, SecuredTablesSet> sets = Maps.newHashMap();

        for ( TablesSetPermission permission : getPermissionsForUser( username ) )
        {
            TablesSet set = permission.getTablesSet();
            SecuredTablesSet curr = new SecuredTablesSet( set, permission.getLevel() );
            SecuredTablesSet old = sets.get( set.getName() );

            if ( old == null || curr.getAccessLevel().isHigher( old.getAccessLevel() ) )
            {
                sets.put( curr.getName(), curr );
            }
        }

        permissionsCache.put( username, new ArrayList<>( sets.values() ) );
        return new ArrayList<>( sets.values() );
    }

    @Override
    public boolean hasPermissionToTable( String username, String tableName, AccessLevel accessLevel )
    {
        Set<SimpleTableDto> accessFilteredTables;
        try
        {
            accessFilteredTables = permissionsCache.get( UserContext.userName() )
                .stream()
                .filter( securedTablesSet -> securedTablesSet.getAccessLevel().equals( accessLevel ) || securedTablesSet.getAccessLevel()
                    .isHigher( accessLevel ) )
                .map( SecuredTablesSet::getTables )
                .flatMap( Set::stream )
                .collect( Collectors.toSet() );
        }
        catch ( ExecutionException e )
        {
            throw new RuntimeException( e );
        }

        return accessFilteredTables.stream()
            .anyMatch( table -> Objects.equals( table.getName(), tableName ) );
    }

    @Override
    public boolean hasAnyPermissions()
    {
        try
        {
            return !permissionsCache.get( UserContext.userName() ).isEmpty();
        }
        catch ( ExecutionException e )
        {
            throw new RuntimeException( e );
        }
    }

    private List<TablesSetPermission> getPermissionsForUser( String username )
    {
        User user = userService.getUser( username, User.JOIN_GROUPS );
        return setPermissionRepository.getPermissionsForUser( user );
    }
}
