package com.suncode.dbexplorer.alias.permission;

import com.suncode.dbexplorer.alias.permission.AccessResource.ResourceType;
import com.suncode.dbexplorer.alias.permission.dto.TablesSetPermissionDto;
import com.suncode.dbexplorer.audit.AuditTypes;
import com.suncode.dbexplorer.context.UserContext;
import com.suncode.dbexplorer.util.authorization.AuthorizationHelper;
import com.suncode.dbexplorer.util.web.rest.RestController;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Controller
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class TablesSetPermissionsController
    extends RestController
{
    private final PermissionsService permissionsService;

    private final AuthorizationHelper authorizationHelper;

    @RequestMapping( value = "/permissions/tablessets/{setId}", method = RequestMethod.GET )
    public @ResponseBody List<TablesSetPermissionDto> get( @PathVariable Long setId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        return TablesSetPermissionDto.from( permissionsService.getPermissions( setId ) );
    }

    @RequestMapping( value = "/permissions/tablessets/{setId}", method = RequestMethod.POST )
    public @ResponseBody TablesSetPermissionDto add( @PathVariable Long setId,
                                                     @RequestBody TablesSetPermissionDto newPermission )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();

        Long id = newPermission.getResourceId();
        ResourceType type = newPermission.getResourceType();
        AccessResource resource = permissionsService.getResource( id, type );

        TablesSetPermission permission =
            permissionsService.newPermission( setId, resource, newPermission.getAccessLevel() );

        audit( started, AuditTypes.AUDIT_PERMISSIONS_ADD, permission );

        return new TablesSetPermissionDto( permission );
    }

    @RequestMapping( value = "/permissions/tablessets/{setId}/{permissionId}", method = RequestMethod.PUT )
    public @ResponseBody TablesSetPermissionDto update( @PathVariable Long permissionId,
                                                        @RequestBody TablesSetPermissionDto newPermission )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();

        TablesSetPermission permission =
            permissionsService.changePermissionLevel( permissionId, newPermission.getAccessLevel() );

        audit( started, AuditTypes.AUDIT_PERMISSIONS_CHANGE, permission );

        return new TablesSetPermissionDto( permission );
    }

    @RequestMapping( value = "/permissions/tablessets/{setId}/{permissionId}", method = RequestMethod.DELETE )
    public @ResponseBody ResponseEntity<Void> delete( @PathVariable Long permissionId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();

        TablesSetPermission permission = permissionsService.deletePermission( permissionId );

        audit( started, AuditTypes.AUDIT_PERMISSIONS_DELETE, permission );

        return new ResponseEntity<>( HttpStatus.NO_CONTENT );
    }

    private void audit( Date started, AuditTypes type, TablesSetPermission permission )
    {
        Map<String, Object> params = new LinkedHashMap<>();
        params.put( "dbex.audit.tablesetname", permission.getTablesSet().getName() );

        params.put( "dbex.audit.permissions.accesslevel", permission.getLevel() == AccessLevel.EDIT ?
            "dbex.audit.permissions.accesslevel.edit" :
            "dbex.audit.permissions.accesslevel.view" );

        if ( permission.getResource().getType() == ResourceType.GROUP )
        {
            params.put( "dbex.audit.permissions.resource.group", permission.getResource().getGroup().getName() );
        }
        else
        {
            params.put( "dbex.audit.permissions.resource.user", permission.getResource().getUser().getUserName());
        }

        ManualAuditBuilder.getInstance()
            .type( type.getValue() )
            .username( UserContext.userName() )
            .success( true )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }
}
