package com.suncode.dbexplorer.alias.data.util.importer;

import com.github.pjfanning.xlsx.StreamingReader;
import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.alias.data.util.importer.config.ImportStrategy;
import com.suncode.dbexplorer.alias.data.util.importer.config.ImportType;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.Record;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;

@Component
@Slf4j
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class ExcelStreamImporter
    implements ImportStrategy
{
    private final ImportHelper importHelper;

    @Override
    public ImportType importMode()
    {
        return ImportType.STREAM;
    }

    @Override
    public void importRecords( InputStream inputStream, Database database, Table table, boolean clear )
        throws IOException
    {
        File tempPoiFile = importHelper.getTempPoiFile();
        tempPoiFile.deleteOnExit();

        try ( FileOutputStream tempOutputStream = new FileOutputStream( tempPoiFile ) )
        {
            IOUtils.copy( inputStream, tempOutputStream );
        }
        catch ( Exception e )
        {
            tempPoiFile.delete();
            throw e;
        }

        try ( Workbook workbook = StreamingReader.builder()
            .rowCacheSize( 4000 )
            .open( tempPoiFile ) )
        {
            Sheet sheet = workbook.getSheetAt( 0 );

            Iterator<Row> rowIterator = sheet.iterator();
            Row headerRow = rowIterator.next();

            String[] columnNames = new String[headerRow.getLastCellNum()];
            for ( Cell cell : headerRow )
            {
                columnNames[cell.getColumnIndex()] = cell.getStringCellValue();
            }

            database.withinSession( session -> {
                if ( clear )
                {
                    session.delete().from( table.getTableSchema().getFullName() ).execute();
                }

                long rowsInBase = session
                    .select()
                    .from( table.getTableSchema().getFullName() )
                    .count();

                while ( rowIterator.hasNext() )
                {
                    Row row = rowIterator.next();
                    if ( importHelper.isRowEmpty( row ) )
                    {
                        continue;
                    }

                    Record record = importHelper.createRecord( database, table, columnNames, row, table.getTableSchema() );
                    importHelper.processRecord( session, record, clear, row.getRowNum(), rowsInBase );
                }
                return null;
            } );
        }
        finally
        {
            tempPoiFile.delete();
        }
    }
}
