package com.suncode.dbexplorer.alias;

import com.google.common.collect.ImmutableMap;
import com.suncode.dbexplorer.alias.data.DataController;
import com.suncode.dbexplorer.alias.dto.AliasDto;
import com.suncode.dbexplorer.alias.dto.ColumnDto;
import com.suncode.dbexplorer.alias.dto.SchemaDto;
import com.suncode.dbexplorer.alias.dto.SimpleTableDto;
import com.suncode.dbexplorer.alias.dto.TableDto;
import com.suncode.dbexplorer.alias.dto.TablesSetDto;
import com.suncode.dbexplorer.alias.exception.AliasExistsException;
import com.suncode.dbexplorer.alias.exception.AliasNotActiveException;
import com.suncode.dbexplorer.alias.exception.InvalidJdbcUrlException;
import com.suncode.dbexplorer.alias.settings.ColumnSettings;
import com.suncode.dbexplorer.alias.settings.TableSettings;
import com.suncode.dbexplorer.audit.AuditTypes;
import com.suncode.dbexplorer.context.UserContext;
import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseFactory;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.dbexplorer.database.exception.DatabaseNotAvailableException;
import com.suncode.dbexplorer.util.authorization.AuthorizationHelper;
import com.suncode.dbexplorer.util.web.rest.RestController;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import jakarta.servlet.http.HttpServletRequest;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.regex.Pattern;

/**
 * Rest API aliasów {@link Alias}
 *
 * @author Cezary Kozar 8 wrz 2015
 */
@Slf4j
@Controller
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class AliasController
    extends RestController
{
    private static final Pattern JDBC_URL_PATTERN = Pattern.compile("^jdbc:(postgresql|postgres|sqlserver|oracle|db2):.*", Pattern.CASE_INSENSITIVE);

    private final AliasService aliasService;

    private final DatabaseFactory databaseFactory;

    private final AuthorizationHelper authorizationHelper;

    @RequestMapping( value = "/aliases", method = RequestMethod.POST )
    public @ResponseBody ResponseEntity<?> add( @RequestBody AliasDto newAlias )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();
        try
        {
            boolean isJdbcUrlConfig = newAlias.getIsJdbcUrlConfig();
            if ( isJdbcUrlConfig )
            {
                validateJdbcUrl( newAlias.getJdbcUrl() );
            }
            Alias alias = new Alias( newAlias.getName(), ConnectionString.toDomain( newAlias.getConnectionString() ) );
            alias.setIsActive( newAlias.getIsActive() );
            alias.setIsJdbcUrlConfig( isJdbcUrlConfig );
            alias.setLogging( newAlias.getLogging() );
            aliasService.addAlias( alias );
            auditAliasOperation( started, AuditTypes.AUDIT_ALIAS_ADD, alias, null );
            AliasDto aliasDto = new AliasDto( alias );
            aliasDto.setDisplayedName( resolveDisplayedName( aliasDto ) );
            return new ResponseEntity<>( aliasDto, HttpStatus.OK );
        }
        catch ( AliasExistsException ex )
        {
            return new ResponseEntity<>( ImmutableMap.of( "aliasAlreadyExists", true ),
                                         HttpStatus.NOT_ACCEPTABLE );
        }
    }

    private void validateJdbcUrl( String jdbcUrl ) throws InvalidJdbcUrlException
    {
        if (StringUtils.isBlank(jdbcUrl) || !JDBC_URL_PATTERN.matcher(jdbcUrl).matches()) {
            throw new InvalidJdbcUrlException();
        }
    }

    @RequestMapping( value = "/system/aliases", method = RequestMethod.POST )
    public @ResponseBody ResponseEntity<?> addSystemAlias()
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();
        try
        {
            Alias systemAlias = aliasService.addSystemAlias();
            auditSystemAliasOperation( started, AuditTypes.AUDIT_ALIAS_ADD, systemAlias );
            AliasDto aliasDto = new AliasDto( systemAlias );
            aliasDto.setDisplayedName( resolveDisplayedName( aliasDto ) );
            return new ResponseEntity<>( aliasDto, HttpStatus.OK );
        }
        catch ( AliasExistsException ex )
        {
            return new ResponseEntity<>( ImmutableMap.of( "aliasAlreadyExists", true ),
                                         HttpStatus.NOT_ACCEPTABLE );
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}", method = RequestMethod.PUT )
    public @ResponseBody AliasDto update( @PathVariable Long aliasId, @RequestBody AliasDto updatedAlias )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();
        Alias oldAlias = getAlias( aliasId );

        Alias alias = getAlias( aliasId );
        boolean isJdbcUrlConfig = updatedAlias.getIsJdbcUrlConfig();
        if ( isJdbcUrlConfig )
        {
            validateJdbcUrl( updatedAlias.getJdbcUrl() );
        }
        alias.setName( updatedAlias.getName() );
        alias.setConnectionStringDomain( ConnectionString.toDomain( updatedAlias.getConnectionString() ) );
        alias.setIsActive( updatedAlias.getIsActive() );
        alias.setLogging( updatedAlias.getLogging() );
        alias.setIsJdbcUrlConfig( isJdbcUrlConfig );

        aliasService.updateAlias( alias );

        auditAliasOperation( started, AuditTypes.AUDIT_ALIAS_UPDATE, alias, oldAlias );

        AliasDto aliasDto = new AliasDto( alias );
        aliasDto.setDisplayedName( resolveDisplayedName( aliasDto ) );
        return aliasDto;
    }

    @RequestMapping( value = "/aliases/{aliasId}", method = RequestMethod.DELETE )
    public @ResponseBody ResponseEntity<Void> delete( @PathVariable Long aliasId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        Date started = new Date();

        Alias aliasToDelete = aliasService.getAlias( aliasId );
        aliasService.deleteAlias( aliasId );

        if ( aliasToDelete.getIsSystemAlias() )
        {
            auditSystemAliasOperation( started, AuditTypes.AUDIT_ALIAS_DELETE, aliasToDelete );
        }
        else
        {
            auditAliasOperation( started, AuditTypes.AUDIT_ALIAS_DELETE, aliasToDelete, null );
        }

        return new ResponseEntity<>( HttpStatus.NO_CONTENT );
    }

    @RequestMapping( value = "/aliases", method = RequestMethod.GET )
    public @ResponseBody List<AliasDto> getAll(HttpServletRequest request)
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        log.info( "Requesting {}" + request.getRequestURI() );
        List<Alias> aliases = aliasService.getAliases();

        List<AliasDto> aliasDtos = aliases.stream()
            .map( alias -> {
                try
                {
                    return AliasDto.from( alias );
                }
                catch ( Exception e )
                {
                    log.info( "Exception while fetching alias: {}, message: {}", alias.getName(), e.getMessage() );
                    return null;
                }
            } )
            .filter( Objects::nonNull )
            .collect( Collectors.toList() );

        aliasDtos.forEach( aliasDto -> aliasDto.setDisplayedName( resolveDisplayedName( aliasDto ) ) );

        return aliasDtos;
    }

    @RequestMapping( value = "/aliases/{aliasId}/refreshschema", method = RequestMethod.POST )
    public @ResponseBody void refreshSchema( @PathVariable Long aliasId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Alias alias = aliasService.getAlias( aliasId );
        if ( databaseFactory.isAvailable( alias.getWrappedConnectionString() ) )
        {
            Database database = databaseFactory.create( alias.getWrappedConnectionString(), true );
            database.updateSchema();
        }
        else
        {
            throw new DatabaseNotAvailableException();
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}/tablessets", method = RequestMethod.GET )
    public @ResponseBody List<TablesSetDto> getTablesSets( @PathVariable Long aliasId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Alias alias = getAlias( aliasId );
        for ( TablesSet tableSet : alias.getTablesSets() )
        {
            normalizeTablesInSet( tableSet );
        }
        return TablesSetDto.from( alias );
    }

    @RequestMapping( value = "/aliases/{aliasId}/tablessets", method = RequestMethod.POST )
    public @ResponseBody TablesSetDto addTablesSet( @PathVariable Long aliasId, @RequestBody TablesSetDto newSet )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Date started = new Date();

        TablesSet set = new TablesSet( newSet.getName(), newSet.getTables() );

        Alias alias = getAlias( aliasId );
        alias.addTablesSet( set );

        aliasService.updateAlias( alias );

        auditTableSetOperation( started, AuditTypes.AUDIT_TABLE_SET_ADD, set, null );

        return new TablesSetDto( alias, set );
    }

    @RequestMapping( value = "/aliases/{aliasId}/tablessets/{setId}", method = RequestMethod.PUT )
    public @ResponseBody TablesSetDto updateTablesSet( @PathVariable Long aliasId, @PathVariable Long setId,
                                                       @RequestBody TablesSetDto newSet )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Date started = new Date();

        Alias alias = getAlias( aliasId );

        // Usuniecię poprzednio zapisanych tabel w celu uniknięcia powstawania duplikatów (w
        // przypadku gdy brak wartości w kolumnie schema_name)
        TablesSet set = alias.getTablesSet( setId );

        Map<String, Object> params = new LinkedHashMap<String, Object>();
        params.put( "dbex.audit.tableset.oldtables", set.getTables()
            .stream()
            .map( SimpleTableDto::toString )
            .collect( Collectors.joining( ", " ) ) );

        set.setTablesNames( new ArrayList<>() );
        aliasService.updateAlias( alias );

        set.setName( newSet.getName() );
        set.setTablesNames( newSet.getTables() );
        aliasService.updateAlias( alias );

        auditTableSetOperation( started, AuditTypes.AUDIT_TABLE_SET_UPDATE, set, params );

        return new TablesSetDto( alias, set );
    }

    @RequestMapping( value = "/aliases/{aliasId}/tablessets/{setId}", method = RequestMethod.DELETE )
    public @ResponseBody void deleteTablesSet( @PathVariable Long aliasId, @PathVariable Long setId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Date started = new Date();

        Alias alias = getAlias( aliasId );

        TablesSet set = alias.getTablesSet( setId );
        if ( set != null )
        {
            alias.removeTablesSet( set );
        }
        aliasService.updateAlias( alias );

        auditTableSetOperation( started, AuditTypes.AUDIT_TABLE_SET_DELETE, set, null );
    }

    @RequestMapping( value = "/aliases/{aliasId}/tables", method = RequestMethod.GET )
    public @ResponseBody List<TableDto> getTables( @PathVariable Long aliasId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Alias alias = getAlias( aliasId );

        Set<Table> tables = alias.getTables( databaseFactory );
        return TableDto.from( alias, tables );
    }

    @RequestMapping( value = "/aliases/{aliasId}/schemas", method = RequestMethod.GET )
    public @ResponseBody List<SchemaDto> getSchemas( @PathVariable Long aliasId )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Alias alias = getAlias( aliasId );
        boolean available = isAliasAvailable( alias );

        if ( available )
        {
            return SchemaDto.from( alias, alias.getSchemas( databaseFactory ) );
        }
        else
        {
            return Arrays.asList( SchemaDto.unavailable() );
        }
    }

    private boolean isAliasAvailable( Alias alias )
    {
        return databaseFactory.isAvailable( alias.getWrappedConnectionString() );
    }

    @RequestMapping( value = "/aliases/{aliasId}/tables", method = RequestMethod.PUT )
    public @ResponseBody void updateTable( @PathVariable Long aliasId, @RequestBody TableDto newTable )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Date started = new Date();
        Alias alias = getAlias( aliasId );
        TableSettings oldSettings = getAlias( aliasId ).getSettings().getTableSettings( newTable.getSchema(), newTable.getName() );

        TableSettings settings = alias.getSettings().getTableSettings( newTable.getSchema(), newTable.getName() );

        if ( newTable.getDisplayName() != null )
        {
            settings.setDisplayName( newTable.getDisplayName().trim() );
        }

        if ( newTable.getLogging() != null )
        {
            settings.setLogging( newTable.getLogging() );
        }

        aliasService.updateAlias( alias );
        auditTableUpdate( started, alias.getId(), oldSettings, settings );
    }

    @RequestMapping( value = "/aliases/{aliasId}/columns", method = RequestMethod.PUT )
    public @ResponseBody void updateColumn( @PathVariable Long aliasId, @RequestBody ColumnDto newColumn )
    {
        authorizationHelper.assertFullAdministrationRights( () -> {
        } );

        validateAliasIsActive( aliasId );

        Date started = new Date();
        Alias alias = getAlias( aliasId );
        ColumnSettings oldSettings = getAlias( aliasId ).getSettings()
            .getTableSettings( newColumn.getSchemaName(), newColumn.getTableName() )
            .getColumnSettings( newColumn.getName() );

        TableSettings tableSettings =
            alias.getSettings().getTableSettings( newColumn.getSchemaName(), newColumn.getTableName() );
        ColumnSettings settings = tableSettings.getColumnSettings( newColumn.getName() );

        if ( newColumn.getDisplayName() != null )
        {
            settings.setDisplayName( newColumn.getDisplayName().trim() );
        }

        aliasService.updateAlias( alias );
        auditColumnUpdate( started, alias.getId(), tableSettings, oldSettings, settings, true );
    }

    @ResponseBody
    @ExceptionHandler( { AliasNotActiveException.class } )
    public ResponseEntity<?> handleAliasNotActive( AliasNotActiveException e )
    {
        return new ResponseEntity<>( e.getMessage(), HttpStatus.BAD_REQUEST );
    }

    private void validateAliasIsActive( Long aliasId )
    {
        Alias alias = aliasService.getAlias( aliasId );

        if ( !alias.getIsActive() )
        {
            Translator translator = Translators.get( DataController.class );
            throw new AliasNotActiveException( translator.getMessage( "dbex.alias.notActive.exception", alias.getName() ) );
        }
    }

    private Alias getAlias( Long aliasId )
    {
        Alias alias = aliasService.getAlias( aliasId );
        if ( alias == null )
        {
            throw new AliasNotFoundException( aliasId );
        }
        return alias;
    }

    private void normalizeTablesInSet( TablesSet set )
    {
        Alias alias = set.getAlias();
        Set<SimpleTableDto> tables = set.getTables();

        for ( SimpleTableDto table : tables )
        {
            if ( table.getSchema() == null )
            {
                table.setSchema( getDefaultSchema( alias ) );
            }
        }
    }

    private String getDefaultSchema( Alias alias )
    {
        Database database = databaseFactory.create( alias.getWrappedConnectionString() );
        return database.getDefaultSchemaName();
    }

    @ResponseBody
    @ExceptionHandler( value = DatabaseNotAvailableException.class )
    public ResponseEntity<?> handleDatabaseNotAvailable( Exception ex )
    {
        return handle( ex, ImmutableMap.of( "databaseNotAvailable", true ) );
    }

    private ResponseEntity<?> handle( Exception ex, Object responseObject )
    {
        log.error( ex.getMessage(), ex );
        return new ResponseEntity<>( responseObject, HttpStatus.NOT_ACCEPTABLE );
    }

    private void auditAliasOperation( Date started, AuditTypes type, Alias alias, Alias oldAlias )
    {
        Map<String, Object> params = new LinkedHashMap<>();
        String changePattern = "%s -> %s";

        String aliasName;
        String aliasRealName;
        String databaseAddress;
        String databasePort;
        String user;
        String password;
        String databaseType;
        String isActive;
        String logging;
        String isJdbcUrlConfigParameter;
        String jdbcUrl;

        boolean isJdbcUrlConfig = alias.getIsJdbcUrlConfig();

        Integer port = alias.getConnectionStringDomain().getPort();
        DatabaseType dbType = alias.getConnectionStringDomain().getType();
        if ( oldAlias == null )
        {
            aliasName = alias.getName();
            aliasRealName = alias.getConnectionStringDomain().getCatalog();
            databaseAddress = alias.getConnectionStringDomain().getHost();
            databasePort = port != null ? port.toString() : StringUtils.EMPTY;
            user = alias.getConnectionStringDomain().getUser();
            password = "********";
            databaseType = isJdbcUrlConfig ? StringUtils.EMPTY : dbType.name();
            isActive = alias.getIsActive().toString();
            logging = alias.getLogging().toString();
            isJdbcUrlConfigParameter = String.valueOf( isJdbcUrlConfig);
            jdbcUrl = isJdbcUrlConfig ? alias.getConnectionStringDomain().getJdbcUrl() : StringUtils.EMPTY;
        }
        else
        {
            aliasName = String.format( changePattern, oldAlias.getName(), alias.getName() );
            aliasRealName =
                String.format( changePattern, oldAlias.getConnectionStringDomain().getCatalog(), alias.getConnectionStringDomain().getCatalog() );
            databaseAddress =
                String.format( changePattern, oldAlias.getConnectionStringDomain().getHost(), alias.getConnectionStringDomain().getHost() );

            Integer oldPort = oldAlias.getConnectionStringDomain().getPort();
            databasePort = String.format( changePattern, oldPort != null ? oldPort.toString() : StringUtils.EMPTY,
                                          port != null ? port.toString() : StringUtils.EMPTY );

            user = String.format( changePattern, oldAlias.getConnectionStringDomain().getUser(), alias.getConnectionStringDomain().getUser() );
            password = String.format( changePattern, "********", "********" );

            DatabaseType oldDbType = oldAlias.getConnectionStringDomain().getType();
            databaseType = String.format( changePattern, oldAlias.getIsJdbcUrlConfig() ? StringUtils.EMPTY : oldDbType.name(),
                                          isJdbcUrlConfig ? StringUtils.EMPTY : dbType.name() );
            isActive = String.format( changePattern, oldAlias.getIsActive(), alias.getIsActive() );
            logging = String.format( changePattern, oldAlias.getLogging(), alias.getLogging() );
            isJdbcUrlConfigParameter = String.format( changePattern, oldAlias.getIsJdbcUrlConfig(),isJdbcUrlConfig );
            String oldJdbcUrl = oldAlias.getIsJdbcUrlConfig() ? oldAlias.getConnectionStringDomain().getJdbcUrl() : StringUtils.EMPTY;
            String newJdbcUrl = isJdbcUrlConfig ? alias.getConnectionStringDomain().getJdbcUrl() : StringUtils.EMPTY;
            jdbcUrl = String.format( changePattern, oldJdbcUrl, newJdbcUrl );


        }

        params.put( "dbex.audit.alias.id", alias.getId() );
        params.put( "dbex.audit.alias.name", aliasName );
        params.put( "dbex.audit.alias.realName", aliasRealName );
        params.put( "dbex.audit.alias.databaseAddress", databaseAddress );
        params.put( "dbex.audit.alias.databasePort", databasePort );
        params.put( "dbex.audit.alias.isSystem", alias.getIsSystemAlias() );
        params.put( "dbex.audit.alias.user", user );
        params.put( "dbex.audit.alias.password", password );
        params.put( "dbex.audit.alias.databaseType", databaseType );
        params.put( "dbex.audit.alias.active", isActive );
        params.put( "dbex.audit.alias.table.logging", logging );
        params.put( "dbex.audit.alias.isJdbcUrlConfig", isJdbcUrlConfigParameter );
        params.put( "dbex.audit.alias.jdbcUrl", jdbcUrl );

        ManualAuditBuilder.getInstance()
            .type( type.getValue() )
            .username( UserContext.userName() )
            .success( true )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }

    private void auditSystemAliasOperation( Date started, AuditTypes type, Alias alias )
    {
        Map<String, Object> params = new LinkedHashMap<>();

        params.put( "dbex.audit.alias.id", alias.getId() );
        params.put( "dbex.audit.alias.name", alias.getName() );
        params.put( "dbex.audit.alias.connectionString", alias.getWrappedConnectionString().getJdbcUrl() );
        params.put( "dbex.audit.alias.isSystem", alias.getIsSystemAlias() );
        params.put( "dbex.audit.alias.user", alias.getWrappedConnectionString().getUser() );
        params.put( "dbex.audit.alias.password", "********" );
        params.put( "dbex.audit.alias.databaseType", alias.getWrappedConnectionString().getType().name() );
        params.put( "dbex.audit.alias.active", alias.getIsActive() );
        params.put( "dbex.audit.alias.table.logging", alias.getLogging() );

        ManualAuditBuilder.getInstance()
            .type( type.getValue() )
            .username( UserContext.userName() )
            .success( true )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }

    private void auditTableSetOperation( Date started, AuditTypes type, TablesSet set, Map<String, Object> moreParams )
    {
        Map<String, Object> params = new LinkedHashMap<>();

        params.put( "dbex.audit.tablesetname", set.getName() );
        params.put( "dbex.audit.tableset.tables", set.getTables().stream().map( SimpleTableDto::toString ).collect( Collectors.joining( ", " ) ) );

        if ( moreParams != null )
        {
            params.putAll( moreParams );
        }

        ManualAuditBuilder.getInstance()
            .type( type.getValue() )
            .username( UserContext.userName() )
            .success( true )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }

    private void auditTableUpdate( Date started, Long id, TableSettings oldSettings, TableSettings settings )
    {
        Map<String, Object> params = new LinkedHashMap<>();
        String changePattern = "%s -> %s";

        params.put( "dbex.audit.alias.id", id );
        params.put( "dbex.audit.alias.table.tableName", settings.getTableName() );
        if ( StringUtils.isNotBlank( oldSettings.getDisplayName() ) || StringUtils.isNotBlank( settings.getDisplayName() ) )
        {
            params.put( "dbex.audit.alias.table.displayedName",
                        String.format( changePattern, StringUtils.defaultString( oldSettings.getDisplayName() ), settings.getDisplayName() ) );
        }
        params.put( "dbex.audit.alias.table.logging", String.format( changePattern, oldSettings.isLogging(), settings.isLogging() ) );

        ManualAuditBuilder.getInstance()
            .type( AuditTypes.AUDIT_TABLE_CHANGE.getValue() )
            .username( UserContext.userName() )
            .success( true )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }

    private void auditColumnUpdate( Date started, Long id, TableSettings tableSettings, ColumnSettings oldColumnSettigns,
                                    ColumnSettings columnSettings,
                                    boolean success )
    {
        Map<String, Object> params = new LinkedHashMap<>();
        String changePattern = "%s -> %s";

        params.put( "dbex.audit.alias.id", id );
        params.put( "dbex.audit.alias.column.tableName", tableSettings.getTableName() );
        params.put( "dbex.audit.alias.column.columnName", columnSettings.getColumnName() );
        params.put( "dbex.audit.alias.column.displayedName",
                    String.format( changePattern, oldColumnSettigns.getDisplayName(), columnSettings.getDisplayName() ) );

        ManualAuditBuilder.getInstance()
            .type( AuditTypes.AUDIT_COLUMN_CHANGE.getValue() )
            .username( UserContext.userName() )
            .success( success )
            .params( params )
            .started( started )
            .stopped( new Date() )
            .build()
            .log();
    }

    private String resolveDisplayedName( AliasDto alias )
    {
        Translator translator = Translators.get( AliasController.class );

        return alias.getIsActive() ?
            alias.getName() :
            alias.getName() + " (" + translator.getMessage( "dbex.alias.notActive" ) + ")";
    }

    @ResponseStatus( HttpStatus.BAD_REQUEST )
    @ExceptionHandler( InvalidJdbcUrlException.class )
    @ResponseBody
    public ResponseEntity<?> handleNotFullRightsException()
    {
        return new ResponseEntity<>( ImmutableMap.of( "invalidJdbcUrl", true ), HttpStatus.BAD_REQUEST );
    }
}
