package com.suncode.dbexplorer.alias;

import com.google.common.collect.Sets;
import com.suncode.dbexplorer.alias.settings.Settings;
import com.suncode.dbexplorer.alias.settings.TableSettings;
import com.suncode.dbexplorer.alias.util.SystemConnectionStringResolver;
import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.ConnectionStringDomain;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseFactory;
import com.suncode.dbexplorer.database.schema.DatabaseSchema;
import com.suncode.dbexplorer.database.schema.TableSchema;
import com.suncode.dbexplorer.util.persistence.BaseEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Embedded;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.OneToMany;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.Set;

/**
 * Alias bazy danych.
 * -- Alias
 * -- nazwa
 * -- dane do połaczenia
 * -- zestawy tabel
 * -- ustawienia globalne
 * -- ustawienia tabel
 * -- ustawienia kolumn
 * -- uprawnienia do tabel
 * -- uprawnienia do aliasów
 * -- Pobranie zestawów:
 * Alias alias = aliasService.getAlias(id);
 * alias.getTablesSets();
 * -- Pobranie ustawień tabel:
 * List<TableSettings> settings = alias.getSettings().getTableSettings();
 * 
 * @author Cezary Kozar 27 lis 2015
 */
@Entity
@NoArgsConstructor( access = AccessLevel.PRIVATE )
public class Alias
    extends BaseEntity
{
    public static final String SYSTEM_ALIAS_NAME = "PlusWorkflow";

    @Getter
    @Column( nullable = false, unique = true )
    private String name;

    @Getter
    @Embedded
    private ConnectionStringDomain connectionStringDomain;

    @Embedded
    private Settings settings = new Settings();

    @Getter
    @Setter
    @Column( nullable = false )
    private Boolean isSystemAlias;

    @Getter
    @Setter
    @Column( nullable = false )
    private Boolean isActive;

    @Getter
    @Setter
    @Column( nullable = false )
    private Boolean logging;

    @Getter
    @Setter
    @Column( nullable = false, name = "is_jdbc_url_config")
    private Boolean isJdbcUrlConfig;

    @OneToMany( mappedBy = "alias", cascade = CascadeType.ALL, orphanRemoval = true, fetch = FetchType.EAGER )
    private Set<TablesSet> tablesSets = Sets.newHashSet();

    public Alias( String name, ConnectionStringDomain connectionStringDomain )
    {
        Assert.hasText( name, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.notNull( connectionStringDomain, "[Assertion failed] - this argument is required; it must not be null" );

        this.name = name;
        this.connectionStringDomain = connectionStringDomain;
        isSystemAlias = false;
    }

    public static Alias createSystemAlias()
    {
        Alias systemAlias = new Alias();
        systemAlias.setName( Alias.SYSTEM_ALIAS_NAME );
        systemAlias.setIsSystemAlias( true );
        systemAlias.setIsJdbcUrlConfig( false );
        systemAlias.setIsActive( true );
        systemAlias.setLogging( true );
        return systemAlias;
    }

    public void setName( String name )
    {
        Assert.hasText( name, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        this.name = name;
    }

    public void setConnectionStringDomain( ConnectionStringDomain connectionStringDomain )
    {
        Assert.notNull( connectionStringDomain, "[Assertion failed] - this argument is required; it must not be null" );
        this.connectionStringDomain = connectionStringDomain;
    }

    public Settings getSettings()
    {
        if ( settings == null )
        {
            settings = new Settings();
        }
        return settings;
    }

    public void addTablesSet( TablesSet set )
    {
        Assert.notNull( set, "[Assertion failed] - this argument is required; it must not be null" );
        set.setAlias( this );
        tablesSets.add( set );
    }

    public void removeTablesSet( TablesSet set )
    {
        Assert.notNull( set, "[Assertion failed] - this argument is required; it must not be null" );
        tablesSets.remove( set );
    }

    public Set<TablesSet> getTablesSets()
    {
        return Collections.unmodifiableSet( tablesSets );
    }

    public TablesSet getTablesSet( Long id )
    {
        for ( TablesSet set : tablesSets )
        {
            if ( set.getId().equals( id ) )
            {
                return set;
            }
        }
        return null;
    }

    public Set<Table> getTables( DatabaseFactory databaseFactory )
    {
        Database database = databaseFactory.create( getWrappedConnectionString() );

        Set<Table> tables = Sets.newLinkedHashSet();

        for ( DatabaseSchema databaseSchema : database.getSchemas() )
        {
            for ( TableSchema tableSchema : databaseSchema.getTables().values() )
            {
                String schemaName = tableSchema.getSchema();
                String tableName = tableSchema.getName();
                TableSettings tableSettings = settings.getTableSettings( schemaName, tableName );

                Table table = new Table( tableSchema, tableSettings );
                tables.add( table );
            }
        }

        return tables;
    }

    public Set<Schema> getSchemas( DatabaseFactory databaseFactory )
    {
        Database database = databaseFactory.create( getWrappedConnectionString() );
        Set<Schema> schemas = Sets.newLinkedHashSet();

        for ( DatabaseSchema databaseSchema : database.getSchemas() )
        {
            Set<Table> tables = Sets.newLinkedHashSet();
            for ( TableSchema tableSchema : databaseSchema.getTables().values() )
            {
                String schemaName = tableSchema.getSchema();
                String tableName = tableSchema.getName();
                TableSettings tableSettings = settings.getTableSettings( schemaName, tableName );

                Table table = new Table( tableSchema, tableSettings );
                tables.add( table );
            }
            schemas.add( new Schema( databaseSchema.getName(), tables ) );
        }

        return schemas;
    }

    public Table getTable( String schema, String tableName, DatabaseFactory databaseFactory )
    {
        for ( Table table : getTables( databaseFactory ) )
        {
            if ( table.getSchema().equals( schema ) && table.getName().equals( tableName ) )
            {
                return table;
            }
        }
        return null;
    }

    public ConnectionString getWrappedConnectionString()
    {
        if ( isSystemAlias )
        {
            return SystemConnectionStringResolver.getSystemConnectionString();
        }
        else if ( isJdbcUrlConfig )
        {
            return ConnectionString.fromJdbcUrl( connectionStringDomain );
        }
        else
        {
            return ConnectionString.fromDomain( connectionStringDomain );
        }
    }
}