package com.suncode.dbexplorer.database.query;

import java.util.Date;

import org.joda.time.LocalDateTime;

import com.suncode.dbexplorer.database.RecordId;
import com.suncode.dbexplorer.database.query.AndOrCondition.Type;
import com.suncode.dbexplorer.database.query.OperatorCondition.Operator;

/**
 * Fabryka podstawowych {@linkplain Condition} do wykorzystania w zapytaniach np.
 * {@linkplain SelectQuery}.
 * 
 * @author Cezary Kozar 28 wrz 2015
 */
public abstract class Conditions
{
    /**
     * Warunek negacji
     */
    public static Condition not( Condition condition )
    {
        return new NotCondition( condition );
    }

    /**
     * Warunek na nieuzupełnioną kolumnę
     */
    public static Condition isNull( String column )
    {
        return new NullCondition( column );
    }

    /**
     * Warunek na podaną wartośc lub nieuzupełnioną kolumnę
     */
    public static Condition eqOrIsNull( String column, Object value )
    {
        return or( eq( column, value ), isNull( column ) );
    }

    /**
     * Warunek równości wartości kolumny z podaną stałą wartością.
     */
    public static Condition eq( String column, Object value )
    {
        if ( value == null )
        {
            return isNull( column );
        }
        return new OperatorCondition( column, Operator.EQ, value );
    }

    /**
     * Warunek na idetyfikator rekordu (klucz główny tabeli).
     */
    public static Condition idEq( RecordId id )
    {
        return new RecordIdEqCondition( id );
    }

    /**
     * Warunek łączący wszystkie podane warunki logicznym AND
     */
    public static Condition and( Condition... conditions )
    {
        return new AndOrCondition( Type.AND, conditions );
    }

    /**
     * Warunek łączący wszystkie podane warunki logicznym OR
     */
    public static Condition or( Condition... conditions )
    {
        return new AndOrCondition( Type.OR, conditions );
    }

    /**
     * Warunek "większa lub równa" od podanej wartości
     */
    public static Condition ge( String column, Object value )
    {
        return new OperatorCondition( column, Operator.GE, value );
    }
    
    /**
     * Warunek "większa" od podanej wartości
     */
    public static Condition gt( String column, Object value )
    {
        return new OperatorCondition( column, Operator.GT, value );
    }

    /**
     * Warunek "mniejsza lub równa" od podanej wartości
     */
    public static Condition le( String column, Object value )
    {
        return new OperatorCondition( column, Operator.LE, value );
    }
    
    /**
     * Warunek "mniejsza" od podanej wartości
     */
    public static Condition lt( String column, Object value )
    {
        return new OperatorCondition( column, Operator.LT, value );
    }

    /**
     * Warunek LIKE
     */
    public static Condition like( String column, String value )
    {
        return new LikeCondition( column, value, true );
    }

    /**
     * Warunek LIKE (pomija wielkośc liter)
     */
    public static Condition ilike( String column, String value )
    {
        return new LikeCondition( column, value, false );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinMinutes( String column, int minutesAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusMinutes( minutesAmount );
        return withinDate( column, now.toDate(), withinPast( minutesAmount ) );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinHours( String column, int hoursAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusHours( hoursAmount );
        return withinDate( column, now.toDate(), withinPast( hoursAmount ) );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinDays( String column, int daysAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusDays( daysAmount );
        return withinDate( column, now.toDate(), withinPast( daysAmount ) );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinWeeks( String column, int weeksAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusWeeks( weeksAmount );
        return withinDate( column, now.toDate(), withinPast( weeksAmount ) );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinMonths( String column, int monthsAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusMonths( monthsAmount );
        return withinDate( column, now.toDate(), withinPast( monthsAmount ) );
    }

    /**
     * Warunek dla typów datowych. Spełniony jeżeli wartośc kolumny mieści się w podanym przedziale.
     */
    public static Condition withinYears( String column, int yearsAmount )
    {
        LocalDateTime now = LocalDateTime.now().minusYears( yearsAmount );
        return withinDate( column, now.toDate(), withinPast( yearsAmount ) );
    }

    private static Condition withinDate( String column, Date date, boolean past )
    {
        return past ? fromDateToNow( column, date ) : fromNowToDate( column, date );
    }

    private static Condition fromDateToNow( String column, Date date )
    {
        return and( ge( column, date ), le( column, new Date() ) );
    }

    private static Condition fromNowToDate( String column, Date date )
    {
        return and( le( column, date ), ge( column, new Date() ) );
    }

    private static boolean withinPast( int value )
    {
        return value >= 0;
    }
}
