package com.suncode.dbexplorer.database.internal.support;

import java.sql.Driver;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.util.Assert;

import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.dbexplorer.database.internal.ConnectionUrlResolver;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.schema.SchemaLoader;
import com.suncode.dbexplorer.database.internal.type.DataTypeRegistry;
import com.suncode.dbexplorer.database.schema.DatabaseSchema;
import org.springframework.util.ClassUtils;

public abstract class AbstractDatabaseImplementor
    implements DatabaseImplementor
{
    private final DatabaseType implementedType;

    private final SchemaLoader schemaLoader;

    private final DataTypeRegistry typeRegistry;

    private final ConnectionUrlResolver urlResolver;

    public AbstractDatabaseImplementor( DatabaseType databaseType, SchemaLoader schemaLoader,
                                        ConnectionUrlResolver urlResolver, DataTypeRegistry typeRegistry )
    {
        Assert.notNull( databaseType, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( schemaLoader, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( urlResolver, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( typeRegistry, "[Assertion failed] - this argument is required; it must not be null" );

        this.implementedType = databaseType;
        this.schemaLoader = schemaLoader;
        this.urlResolver = urlResolver;
        this.typeRegistry = typeRegistry;
    }

    @Override
    @SuppressWarnings( "unchecked" )
    public Class<? extends Driver> getDriverClass()
    {
        try
        {
            Class<?> driverClass = Class.forName( getDriverClassName(), true, ClassUtils.getDefaultClassLoader() );
            if ( !Driver.class.isAssignableFrom( driverClass ) )
            {
                throw new ClassNotFoundException( getDriverClassName() );
            }
            return (Class<? extends Driver>) driverClass;
        }
        catch ( ClassNotFoundException e )
        {
            // TODO: inny
            throw new RuntimeException( e );
        }
    }

    protected abstract String getDriverClassName();

    @Override
    public boolean handles( DatabaseType type )
    {
        return implementedType == type;
    }

    @Override
    public DataTypeRegistry getTypeRegistry()
    {
        return typeRegistry;
    }

    @Override
    public String buildConnectionUrl( ConnectionString connectionString )
    {
        if ( StringUtils.isNotEmpty( connectionString.getJdbcUrl() ) )
        {
            return connectionString.getJdbcUrl();
        }
        else
        {
            return urlResolver.resolve( connectionString );
        }
    }

    @Override
    public String getCatalog( DatabaseSession session )
    {
        try
        {
            return session.getConnection().getCatalog();
        }
        catch ( SQLException e )
        {
            // TODO wyjatek inny
            throw new RuntimeException( e );
        }
    }

    @Override
    public List<DatabaseSchema> readSchemas( DatabaseSession session, List<String> schemasNames )
    {
        try
        {
            return schemaLoader.loadSchemas( session, schemasNames );
        }
        catch ( Exception e )
        {
            throw new RuntimeException( "Error while loading schemas", e );
        }
    }

    @Override
    public String escapeColumnName( String columnName )
    {
        return escapeString( columnName );
    }

    @Override
    public String escapeTableName( String tableName )
    {
        String[] splited = tableName.split( "\\." );

        if ( splited.length == 1 )
        {
            return escapeString( splited[0] );
        }
        else
        {
            return escapeString( splited[0] ) + "." + escapeString( splited[1] );
        }
    }

    protected String escapeString( String value )
    {
        return "\"" + value + "\"";
    }
}
