package com.suncode.dbexplorer.database.internal.query;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.type.DataTypeHandler;
import com.suncode.dbexplorer.database.query.Condition;
import com.suncode.dbexplorer.database.query.QueryContext;
import com.suncode.dbexplorer.database.query.QueryParameter;
import com.suncode.dbexplorer.database.query.UpdateQuery;
import com.suncode.dbexplorer.database.schema.ColumnSchema;
import com.suncode.dbexplorer.database.type.BasicDataType;
import com.suncode.dbexplorer.database.type.DataType;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import com.suncode.pwfl.util.exception.ServiceException;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.jdbc.SQL;
import org.hibernate.SQLQuery;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

@Slf4j
public class UpdateQueryImpl
    extends AbstractQuery
    implements UpdateQuery
{
    private final List<Condition> whereConditions = new ArrayList<>();

    private final List<String> updateColumns = Lists.newArrayList();

    private final List<Object> updateValues = Lists.newArrayList();

    public UpdateQueryImpl( DatabaseSession session, DatabaseImplementor implementor )
    {
        super( session, implementor );
    }

    @Override
    public UpdateQuery table( String table )
    {
        Assert.hasText( table, "Table must have text" );

        TablePathInfo pathInfo = getPathInfo( table );
        return table( pathInfo.getSchema(), pathInfo.getName() );
    }

    @Override
    public UpdateQuery table( String schema, String table )
    {
        Assert.hasText( schema, "Schema must have text" );
        Assert.hasText( table, "Table must have text" );

        this.rootTable = session.getDatabase().getSchema( schema ).getTable( table );
        return this;
    }

    @Override
    public UpdateQuery set( String column, Object value )
    {
        Assert.hasText( column, "Column must have text" );
        this.updateColumns.add( column );
        this.updateValues.add( value );
        return this;
    }

    @Override
    public UpdateQuery where( Condition condition )
    {
        Assert.notNull( condition, "Condition cannot be null" );
        whereConditions.add( condition );
        return this;
    }

    @Override
    public int execute()
    {
        QueryContext queryContext = new QueryContextImpl( rootTable, implementor );
        SQL rawSql = buildSql( queryContext );

        StringBuilder sql = new StringBuilder();
        StringTokenizer tokenizer = new StringTokenizer( rawSql.toString().replace( "\n", " " ), " \"=><(),\n", true );

        List<QueryParameter> parameters = Lists.newArrayList();
        for ( int i = 0; i < updateColumns.size(); i++ )
        {
            String column = updateColumns.get( i );
            Object value = updateValues.get( i );

            ColumnSchema columnSchema = rootTable.getColumn( column );
            if ( columnSchema.isAutoIncrement() )
            {
                continue;
            }

            // FIXME: typy potrzebują trochę miłości
            if ( BasicDataType.BINARY.is( columnSchema.getType() )
                || BasicDataType.UNKNOWN.is( columnSchema.getType() ) )
            {
                continue;
            }

            if ( BasicDataType.STRING.is( columnSchema.getType() ) &&
                value instanceof String s )
            {
                int maxLength = columnSchema.getType().getNativeType().getLength();
                int valueLength = s.length();

                if ( maxLength > 0 && valueLength > maxLength )
                {
                    Translator translator = Translators.get( InsertQueryImpl.class );
                    String message = translator.getMessage( "dbex.data.column.exceeded.exception", columnSchema.getName(), s );

                    log.info( message );
                    throw new ServiceException( message );
                }
            }

            QueryParameter parameter = new QueryParameter( value, queryContext.getTypeOf( column ) );
            parameter.setAssociatedColumn( column );
            parameters.add( parameter );
        }

        for ( Condition condition : whereConditions )
        {
            Collections.addAll( parameters, condition.getParameters( queryContext ) );
        }

        int paramCount = 0;
        List<BindParam> binded = Lists.newArrayList();
        while ( tokenizer.hasMoreTokens() )
        {
            String token = tokenizer.nextToken();
            if ( token.equals( "?" ) )
            {
                QueryParameter param = parameters.get( paramCount );
                String paramName = "param" + paramCount;
                paramCount++;
                binded.add( new BindParam( paramName, param ) );

                sql.append( ":" ).append( paramName );
                continue;
            }
            sql.append( token );
        }

        SQLQuery sqlQuery = session.hibernateSession().createSQLQuery( sql.toString() );

        for ( BindParam param : binded )
        {
            DataType type = param.parameter.getType();
            DataTypeHandler typeHandler = implementor.getTypeRegistry().getTypeHandler( type );

            try
            {
                typeHandler.bindParameter( type, param.name, param.parameter.getValue(), sqlQuery );
            }
            catch ( RuntimeException ex )
            {
                log.error( "Error while binding value of column {}", param.parameter.getAssociatedColumn() );
                throw ex;
            }
        }

        return sqlQuery.executeUpdate();
    }

    private SQL buildSql( QueryContext queryContext )
    {
        SQL sql = new SQL()
            .UPDATE( implementor.escapeTableName( rootTable.getFullName() ) );

        for ( String updateColumn : updateColumns )
        {
            String columnName = implementor.escapeColumnName( updateColumn );
            ColumnSchema column = rootTable.getColumn( updateColumn );
            if ( column.isAutoIncrement() )
            {
                continue;
            }
            if ( BasicDataType.BINARY.is( column.getType() ) || BasicDataType.UNKNOWN.is( column.getType() ) )
            {
                continue;
            }

            sql.SET( columnName + "=?" );
        }
        for ( Condition condition : whereConditions )
        {
            String conditionSql = condition.toSql( queryContext );
            sql.WHERE( conditionSql );
        }
        return sql;
    }

    @AllArgsConstructor
    private static class BindParam
    {
        private final String name;

        private final QueryParameter parameter;
    }
}
