package com.suncode.dbexplorer.database.internal.query;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.type.DataTypeHandler;
import com.suncode.dbexplorer.database.query.InsertQuery;
import com.suncode.dbexplorer.database.query.QueryContext;
import com.suncode.dbexplorer.database.query.QueryParameter;
import com.suncode.dbexplorer.database.schema.ColumnSchema;
import com.suncode.dbexplorer.database.type.BasicDataType;
import com.suncode.dbexplorer.database.type.DataType;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import com.suncode.pwfl.util.exception.ServiceException;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.jdbc.SQL;
import org.hibernate.SQLQuery;
import org.springframework.util.Assert;

import java.util.Collection;
import java.util.List;
import java.util.StringTokenizer;

@Slf4j
public class InsertQueryImpl
extends AbstractQuery
    implements InsertQuery
{
    private final List<String> insertColumns = Lists.newArrayList();

    private final List<Object> insertValues = Lists.newArrayList();

    public InsertQueryImpl( DatabaseSession session, DatabaseImplementor implementor )
    {
        super( session, implementor );
    }

    @Override
    public InsertQuery into( String table )
    {
        Assert.hasText( table, "Table must have text" );
        
        TablePathInfo pathInfo = getPathInfo( table );
        return into( pathInfo.getSchema(), pathInfo.getName() );
    }
    
    @Override
    public InsertQuery into( String schema, String table )
    {
        Assert.hasText( schema, "Schema must have text" );
        Assert.hasText( table, "Table must have text" );

        this.rootTable = session.getDatabase().getSchema( schema ).getTable( table );
        return this;
    }

    @Override
    public InsertQuery value( String column, Object value )
    {
        Assert.hasText( column, "Column must have text" );

        this.insertValues.add( value );
        this.insertColumns.add( column );
        return this;
    }

    @Override
    public InsertQuery values( Collection<String> columns, Collection<Object> values )
    {
        Assert.isTrue( columns.size() == values.size(), "Column size must be equal to values size" );

        this.insertValues.addAll( values );
        this.insertColumns.addAll( columns );
        return this;
    }

    @Override
    public int execute()
    {
        QueryContext queryContext = new QueryContextImpl( rootTable, implementor );
        SQL rawSql = buildSql( queryContext );

        StringBuilder sql = new StringBuilder();
        StringTokenizer tokenizer = new StringTokenizer( rawSql.toString().replace( "\n", " " ), " \"=><(),", true );

        List<QueryParameter> parameters = Lists.newArrayList();
        for ( int i = 0; i < insertColumns.size(); i++ )
        {
            String column = insertColumns.get( i );
            ColumnSchema columnSchema = rootTable.getColumn( column );
            if ( columnSchema.isAutoIncrement() )
            {
                continue;
            }

            // FIXME: typy potrzebują trochę miłości
            if ( BasicDataType.BINARY.is( columnSchema.getType() ) || BasicDataType.UNKNOWN.is( columnSchema.getType() ) )
            {
                continue;
            }

            Object value = insertValues.get( i );
            if ( BasicDataType.STRING.is( columnSchema.getType() ) &&
                value instanceof String s )
            {
                int maxLength = columnSchema.getType().getNativeType().getLength();
                int valueLength = s.length();

                if ( maxLength > 0 && valueLength > maxLength )
                {
                    Translator translator = Translators.get( InsertQueryImpl.class );
                    String message = translator.getMessage( "dbex.data.column.exceeded.exception", columnSchema.getName(), s );

                    log.info( message );
                    throw new ServiceException( message );
                }
            }

            QueryParameter parameter = new QueryParameter( value, queryContext.getTypeOf( column ) );
            parameter.setAssociatedColumn( column );
            parameters.add( parameter );
        }

        int paramCount = 0;
        List<BindParam> binded = Lists.newArrayList();
        while ( tokenizer.hasMoreTokens() )
        {
            String token = tokenizer.nextToken();
            if ( token.equals( "?" ) )
            {
                QueryParameter param = parameters.get( paramCount );
                String paramName = "param" + paramCount;
                paramCount++;
                binded.add( new BindParam( paramName, param ) );

                sql.append( ":" ).append( paramName );
                continue;
            }
            sql.append( token );
        }

        SQLQuery sqlQuery = session.hibernateSession().createSQLQuery( sql.toString() );
        for ( BindParam param : binded )
        {
            DataType type = param.parameter.getType();
            DataTypeHandler typeHandler = implementor.getTypeRegistry().getTypeHandler( type );

            try
            {
                typeHandler.bindParameter( type, param.name, param.parameter.getValue(), sqlQuery );
            }
            catch ( RuntimeException ex )
            {
                log.error( "Error while binding value of column {}", param.parameter.getAssociatedColumn() );
                throw ex;
            }
        }

        return sqlQuery.executeUpdate();
    }

    private SQL buildSql( QueryContext queryContext )
    {
        SQL sql = new SQL()
            .INSERT_INTO( implementor.escapeTableName( rootTable.getFullName() ) );

        for ( String columnName : insertColumns )
        {
            ColumnSchema column = rootTable.getColumn( columnName );
            if ( column.isAutoIncrement() )
            {
                continue;
            }
            if ( BasicDataType.BINARY.is( column.getType() ) || BasicDataType.UNKNOWN.is( column.getType() ) )
            {
                continue;
            }

            sql.VALUES( implementor.escapeColumnName( columnName ), "?" );
        }
        return sql;
    }

    @AllArgsConstructor
    private static class BindParam
    {
        private final String name;

        private final QueryParameter parameter;
    }
}
