package com.suncode.dbexplorer.database.internal.query;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.ibatis.jdbc.SQL;
import org.hibernate.SQLQuery;
import org.springframework.util.Assert;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.type.DataTypeHandler;
import com.suncode.dbexplorer.database.query.Condition;
import com.suncode.dbexplorer.database.query.DeleteQuery;
import com.suncode.dbexplorer.database.query.QueryContext;
import com.suncode.dbexplorer.database.query.QueryParameter;
import com.suncode.dbexplorer.database.type.DataType;

public class DeleteQueryImpl
    extends AbstractQuery
    implements DeleteQuery
{

    private List<Condition> whereConditions = new ArrayList<Condition>();

    public DeleteQueryImpl( DatabaseSession session, DatabaseImplementor implementor )
    {
        super( session, implementor );
    }

    @Override
    public DeleteQuery from( String table )
    {
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        TablePathInfo pathInfo = getPathInfo( table );
        return from( pathInfo.getSchema(), pathInfo.getName() );
    }
    
    @Override
    public DeleteQuery from( String schema, String table )
    {
        Assert.hasText( schema, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        this.rootTable = session.getDatabase().getSchema( schema ).getTable( table );
        return this;
    }

    @Override
    public DeleteQuery where( Condition condition )
    {
        Assert.notNull( condition, "[Assertion failed] - this argument is required; it must not be null" );
        whereConditions.add( condition );
        return this;
    }

    @Override
    public int execute()
    {

        QueryContext queryContext = new QueryContextImpl( rootTable, implementor );
        SQL rawSql = buildSql( queryContext );

        StringBuilder sql = new StringBuilder();
        StringTokenizer tokenizer = new StringTokenizer( rawSql.toString().replace( "\n", " " ), " \"=><(),", true );

        List<QueryParameter> parameters = Lists.newArrayList();
        for ( Condition condition : whereConditions )
        {
            Collections.addAll( parameters, condition.getParameters( queryContext ) );
        }

        int paramCount = 0;
        List<BindParam> binded = Lists.newArrayList();
        while ( tokenizer.hasMoreTokens() )
        {
            String token = tokenizer.nextToken();
            if ( token.equals( "?" ) )
            {
                QueryParameter param = parameters.get( paramCount );
                String paramName = "param" + paramCount;
                paramCount++;
                binded.add( new BindParam( paramName, param ) );

                sql.append( ":" + paramName );
                continue;
            }
            sql.append( token );
        }

        SQLQuery sqlQuery = session.hibernateSession().createSQLQuery( sql.toString() );

        for ( BindParam param : binded )
        {

            DataType type = param.parameter.getType();
            DataTypeHandler typeHandler = implementor.getTypeRegistry().getTypeHandler( type );

            typeHandler.bindParameter( type, param.name, param.parameter.getValue(), sqlQuery );
        }

        return sqlQuery.executeUpdate();
    }

    private SQL buildSql( QueryContext queryContext )
    {
        SQL sql = new SQL()
            .DELETE_FROM( implementor.escapeTableName( rootTable.getFullName() ) );

        for ( Condition condition : whereConditions )
        {
            String conditionSql = condition.toSql( queryContext );
            sql.WHERE( conditionSql );
        }
        return sql;
    }

    private static class BindParam
    {
        String name;

        QueryParameter parameter;

        public BindParam( String name, QueryParameter parameter )
        {
            this.name = name;
            this.parameter = parameter;
        }
    }
}
