package com.suncode.dbexplorer.database.internal.query;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.SQLQuery;
import org.springframework.util.Assert;

import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.query.support.BuildCreateQueryOracle;
import com.suncode.dbexplorer.database.internal.query.support.BuildCreateQueryPostgres;
import com.suncode.dbexplorer.database.internal.query.support.BuildCreateQuerySqlserver;
import com.suncode.dbexplorer.database.internal.query.support.BuildSqlQuery;
import com.suncode.dbexplorer.database.query.CreateQuery;
import com.suncode.dbexplorer.database.type.BasicDataType;
import com.suncode.dbexplorer.database.type.support.TypeConventer;

import lombok.AllArgsConstructor;
import lombok.Getter;

public class CreateQueryImpl
    extends AbstractQuery
    implements CreateQuery
{

    private List<Column> columns = new ArrayList<>();

    private String schemaName;

    private String tableName;

    private DatabaseType database;

    private Set<Integer> primaryKeyIndexes = new HashSet<>();

    private Map<String, String[]> foreignKeys = new HashMap<>();

    public CreateQueryImpl( DatabaseSession session, DatabaseImplementor implementor )
    {
        super( session, implementor );

    }

    @Override
    public CreateQuery table( String table )
    {
        return table( session.getDatabase().getDefaultSchemaName(), table );
    }

    @Override
    public CreateQuery table( String schema, String table )
    {
        Assert.hasText( schema, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        String driver = implementor.getDriverClass().getName();
        database = DatabaseType.parseScheme( driver );
        
        this.schemaName = schema;
        this.tableName = table;
        return this;
    }

    @Override
    public CreateQuery column( String name, BasicDataType dataType )
    {
        return column( name, dataType, true, false );
    }

    @Override
    public CreateQuery column( String name, BasicDataType dataType, boolean nullable )
    {
        return column( name, dataType, nullable, false );
    }

    @Override
    public CreateQuery column( String name, BasicDataType dataType, boolean nullable,
                               boolean autoincrement )
    {
        Assert.hasText( name, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        String type = TypeConventer.convert( dataType, database );

        this.columns.add( new Column( name, type, nullable, autoincrement ) );
        return this;
    }

    @Override
    public CreateQuery setAsPrimary()
    {
        primaryKeyIndexes.add( this.columns.size() - 1 );
        return this;
    }

    @Override
    public CreateQuery setAsForeign( String foreignTableName, String foreignColumnName )
    {
        String columnName = this.columns.get( this.columns.size() - 1 ).getName();
        this.foreignKeys.put( columnName, new String[] { columnName, foreignTableName, foreignColumnName } );
        return this;
    }

    @Override
    public int execute()
    {
        StringBuilder sql = new StringBuilder();
        sql.append( buildSql() );

        SQLQuery sqlQuery = session.hibernateSession().createSQLQuery( sql.toString() );

        return sqlQuery.executeUpdate();
    }

    private String buildSql()
    {
        StringBuilder sql = new StringBuilder( "CREATE TABLE " + this.schemaName + "." + this.tableName + " ( \n" );

        BuildSqlQuery buildSqlQuery;

        if ( database == DatabaseType.POSTGRES )
        {
            buildSqlQuery =
                new BuildCreateQueryPostgres( this.tableName, this.columns, this.primaryKeyIndexes, this.foreignKeys );
            sql.append( buildSqlQuery.buildSql() );
        }
        else if ( database == DatabaseType.ORACLE )
        {
            buildSqlQuery =
                new BuildCreateQueryOracle( this.tableName, this.columns, this.primaryKeyIndexes, this.foreignKeys );
            sql.append( buildSqlQuery.buildSql() );
        }
        else if ( database == DatabaseType.SQLSERVER )
        {
            buildSqlQuery =
                new BuildCreateQuerySqlserver( this.tableName, this.columns, this.primaryKeyIndexes, this.foreignKeys );
            sql.append( buildSqlQuery.buildSql() );
        }
        return sql.toString();
    }

    @Getter
    @AllArgsConstructor
    public class Column
    {
        private String name;

        private String type;

        private boolean nullable;

        private boolean autoincrement;
    }
}