package com.suncode.dbexplorer.database.internal;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.StatelessSession;
import org.hibernate.Transaction;
import org.springframework.util.Assert;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.alias.data.CreateTable;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.Record;
import com.suncode.dbexplorer.database.RecordId;
import com.suncode.dbexplorer.database.internal.query.CreateQueryImpl;
import com.suncode.dbexplorer.database.internal.query.DeleteQueryImpl;
import com.suncode.dbexplorer.database.internal.query.DropQueryImpl;
import com.suncode.dbexplorer.database.internal.query.InsertQueryImpl;
import com.suncode.dbexplorer.database.internal.query.SelectQueryImpl;
import com.suncode.dbexplorer.database.internal.query.UpdateQueryImpl;
import com.suncode.dbexplorer.database.query.Conditions;
import com.suncode.dbexplorer.database.query.CreateQuery;
import com.suncode.dbexplorer.database.query.DeleteQuery;
import com.suncode.dbexplorer.database.query.DropQuery;
import com.suncode.dbexplorer.database.query.InsertQuery;
import com.suncode.dbexplorer.database.query.SelectQuery;
import com.suncode.dbexplorer.database.query.UpdateQuery;
import com.suncode.dbexplorer.database.schema.TableSchema;

/**
 * @author Cezary Kozar 31 lip 2015
 */
public class DatabaseSessionImpl
    implements DatabaseSession
{
    private final DatabaseImpl database;

    private final Connection connection;

    private final StatelessSession hibernateSession;

    private final Transaction transaction;

    public DatabaseSessionImpl( Connection connection, DatabaseImpl database )
    {
        this.database = database;
        this.connection = connection;
        this.hibernateSession = database.getSessionFactory().openStatelessSession( connection );

        // start transaction
        this.transaction = hibernateSession.beginTransaction();
    }

    @Override
    public Database getDatabase()
    {
        ensureOpen();
        return database;
    }

    @Override
    public Connection getConnection()
    {
        ensureOpen();
        return connection;
    }

    public StatelessSession hibernateSession()
    {
        ensureOpen();
        return hibernateSession;
    }

    @Override
    public boolean isActive()
    {
        return transaction.isActive();
    }

    @Override
    public void commit()
    {
        ensureOpen();

        try
        {
            transaction.commit();
        }
        catch ( RuntimeException e )
        {
            // TODO: transactione xception + logs
            throw new RuntimeException( e );
        }
        catch ( Error e )
        {
            // TODO: logs
            throw e;
        }
        finally
        {
            close();
        }
    }

    @Override
    public void rollback()
    {
        ensureOpen();

        try
        {
            transaction.rollback();
        }
        catch ( RuntimeException e )
        {
            // TODO: logs
            throw e;
        }
        catch ( Error e )
        {
            // TODO: logs
            throw e;
        }
        finally
        {
            close();
        }
    }

    private void close()
    {
        try
        {
            hibernateSession.close();
            connection.close();
        }
        catch ( SQLException e )
        {
            // TODO logs
        }
    }

    private void ensureOpen()
    {
        Assert.state( isActive(), "Session is closed!" );
    }

    @Override
    public Record createRecord( String table )
    {
        String defaultSchema = database.getDefaultSchemaName();
        return createRecord( defaultSchema, table );
    }

    @Override
    public Record createRecord( String schema, String table )
    {
        return new Record( schema, table, database );
    }

    @Override
    public Record get( String table, RecordId id )
    {
        String defaultSchema = database.getDefaultSchemaName();
        return get( defaultSchema, table, id );
    }

    @Override
    public Record get( String schema, String table, RecordId id )
    {
        return select()
            .from( schema + "." + table )
            .where( Conditions.idEq( id ) )
            .uniqueRecord();
    }

    @Override
    public void insert( Record record )
    {
        List<String> columns = Lists.newArrayList();
        List<Object> values = Lists.newArrayList();
        for ( String column : record.getData().keySet() )
        {
            columns.add( column );
            values.add( record.get( column ) );
        }

        insert().into( record.getTable().getFullName() )
            .values( columns, values )
            .execute();
    }

    @Override
    public boolean update( Record record )
    {
        Assert.notNull( record, "[Assertion failed] - this argument is required; it must not be null" );

        UpdateQuery update = update().table( record.getTable().getFullName() );

        boolean modified = false;
        for ( String column : record.getData().keySet() )
        {
            Object value = record.get( column );
            update.set( column, value );
            modified = true;
        }

        if ( !modified )
        {
            return false;
        }
        return update.where( Conditions.idEq( record.getId() ) )
            .execute() > 0;
    }

    @Override
    public void delete( Record record )
    {
        Assert.notNull( record, "[Assertion failed] - this argument is required; it must not be null" );

        delete().from( record.getTable().getFullName() )
            .where( Conditions.idEq( record.getId() ) )
            .execute();
    }

    @Override
    public void createTable( CreateTable table )
    {
        Assert.notNull( table, "[Assertion failed] - this argument is required; it must not be null" );
        CreateQuery createTableQuery;
        if ( StringUtils.isBlank( table.getSchema() ) )
        {
            createTableQuery = create().table( table.getName() );
        }
        else
        {
            createTableQuery = create().table( table.getSchema(), table.getName() );
        }

        Set<CreateTable.ForeignKey> foreignKeys = table.getForeignKeys();

        for ( CreateTable.Column column : table.getColumns() )
        {
            String name = column.getName();
            createTableQuery.column( name, column.getDataType(), column.isNullable(), column.isAutoIncrement() );
            if ( column.isPrimaryKey() )
            {
                createTableQuery.setAsPrimary();
            }

            for ( CreateTable.ForeignKey fk : foreignKeys )
            {
                if ( name.equals( fk.getColumnName() ) )
                {
                    createTableQuery.setAsForeign( fk.getForeignTableName(), fk.getForeignColumnName() );
                }
            }
        }
        createTableQuery.execute();
    }

    @Override
    public void drop( String table )
    {
        String defaultSchema = database.getDefaultSchemaName();
        drop( defaultSchema, table );
    }

    @Override
    public void drop( TableSchema table )
    {
        Assert.notNull( table, "[Assertion failed] - this argument is required; it must not be null" );

        drop( table.getSchema(), table.getName() );
    }

    @Override
    public void drop( String schema, String table )
    {
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.hasText( schema, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        drop().table( schema, table ).execute();
    }

    @Override
    public SelectQuery select()
    {
        return new SelectQueryImpl( this, database.getImplementor() );
    }

    @Override
    public InsertQuery insert()
    {
        return new InsertQueryImpl( this, database.getImplementor() );
    }

    @Override
    public UpdateQuery update()
    {
        return new UpdateQueryImpl( this, database.getImplementor() );
    }

    @Override
    public DeleteQuery delete()
    {
        return new DeleteQueryImpl( this, database.getImplementor() );
    }

    @Override
    public CreateQuery create()
    {
        return new CreateQueryImpl( this, database.getImplementor() );
    }

    @Override
    public DropQuery drop()
    {
        return new DropQueryImpl( this, database.getImplementor() );
    }
}
