package com.suncode.dbexplorer.database.internal;

import com.google.common.collect.Maps;
import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseFactory;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.plugin.framework.service.Provides;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.jdbc.datasource.SimpleDriverDataSource;
import org.springframework.stereotype.Component;

import javax.annotation.PreDestroy;
import javax.sql.DataSource;
import java.util.Map;

import static com.suncode.dbexplorer.alias.util.SystemConnectionStringResolver.parseJdbcUrl;

@Component
@Provides( DatabaseFactory.class )
public class DatabaseFactoryImpl
                implements DatabaseFactory
{
    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private DatabaseAvailabilityResolver availabilityResolver;

    // TODO: przy wyłączeniu wtyczki trzeba zamknąc wszystkie bazy
    private Map<ConnectionString, Database> dbCache = Maps.newHashMap();

    @Override
    public Database create( ConnectionString connectionString )
    {
        return create( connectionString, false );
    }

    @Override
    public Database create( ConnectionString connectionString, boolean force )
    {
        Database database = dbCache.get( connectionString );
        if ( database != null )
        {
            if ( force )
            {
                closeConnection( database );
                dbCache.remove( connectionString );
            }
            else
            {
                return database;
            }
        }

        DatabaseImplementor implementor = resolveDatabaseType( connectionString );
        DataSource dataSource = prepareDataSource( connectionString, implementor );

        database = new DatabaseImpl( dataSource, implementor, connectionString.getSchema(),
                                     availabilityResolver.testConnection( connectionString, dataSource ).getSuccess() );
        dbCache.put( connectionString, database );
        return database;
    }

    @Override
    public DatabaseConnectionTestResult testConnection( ConnectionString connectionString )
    {
        DatabaseImplementor implementor = resolveDatabaseType( connectionString );
        DataSource dataSource = prepareDataSource( connectionString, implementor );
        return availabilityResolver.testConnection( connectionString, dataSource );
    }

    @Override
    public boolean isAvailable( ConnectionString connectionString )
    {
        return testConnection(connectionString).getSuccess();
    }

    private void closeConnection( Database db )
    {
        ( (DatabaseImpl) db ).close();
    }

    public DatabaseImplementor resolveDatabaseType( ConnectionString connectionString )
    {
        // TODO: cache
        Map<String, DatabaseImplementor> impls = applicationContext.getBeansOfType( DatabaseImplementor.class );
        for ( DatabaseImplementor impl : impls.values() )
        {
            DatabaseType type;
            if( StringUtils.isNotBlank(connectionString.getJdbcUrl()))
            {
                type = DatabaseType.parseScheme( parseJdbcUrl( connectionString.getJdbcUrl() ).getScheme() );
            }
            else {
                type = connectionString.getType();
            }
            if ( impl.handles( type ) )
            {
                return impl;
            }
        }
        throw new IllegalStateException( "Not yet supported" );
    }

    private DataSource prepareDataSource( ConnectionString connectionString, DatabaseImplementor implementor )
    {

        SimpleDriverDataSource dataSource = new SimpleDriverDataSource();
        String url = StringUtils.isNotBlank( connectionString.getJdbcUrl() )
            ? connectionString.getJdbcUrl() : implementor.buildConnectionUrl( connectionString );
        dataSource.setUrl( url );
        dataSource.setUsername( connectionString.getUser() );
        dataSource.setPassword( connectionString.getPassword() );
        dataSource.setDriverClass( implementor.getDriverClass() );

        return dataSource;
    }

    @PreDestroy
    public void cleanup()
    {
        for ( Database db : dbCache.values() )
        {
            closeConnection( db );
        }
    }
}
