package com.suncode.dbexplorer.database.internal;

import com.suncode.dbexplorer.database.ConnectionString;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.datasource.SimpleDriverDataSource;
import org.springframework.stereotype.Service;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.*;

@Slf4j
@Service
public class DatabaseAvailabilityResolver
{
    private static final Long MIN_TIMEOUT_SEC = 5L;

    private static final Long TIMEOUT_EXTRA_TIME_SEC = 15L;

    private final Map<Integer, Long> timeoutCache = new HashMap<>();

    public DatabaseConnectionTestResult testConnection( ConnectionString connectionString, DataSource dataSource )
    {
        Long timeout = getTimeout( connectionString );
        String url = getUrl( dataSource );

        try
        {
            log.debug( "Testing connection for connection string [{}] with timeout {} sec", url,
                       timeout );
            return runAsync( connectionString, dataSource, timeout );
        }
        catch ( TimeoutException ex )
        {
            log.error( "Connection timeout {} sec for connection string [{}]", timeout, url );
            return DatabaseConnectionTestResult.failure( ex );
        }
        catch ( Exception ex )
        {
            log.error( "Cannot connect for connection string: [{}]", url );
            log.error( ex.getMessage(), ex );

            return DatabaseConnectionTestResult.failure( ex );
        }
    }

    private DatabaseConnectionTestResult runAsync( ConnectionString connectionString, DataSource dataSource, Long timeout )
                    throws InterruptedException, java.util.concurrent.ExecutionException,
                    java.util.concurrent.TimeoutException
    {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        try
        {
            Future<DatabaseConnectionTestResult> future = executor.submit( () -> runTest( connectionString, dataSource ) );
            return future.get( timeout, TimeUnit.SECONDS );
        }
        finally
        {
            executor.shutdown();
        }
    }

    private Long getTimeout( ConnectionString connectionString )
    {
        synchronized ( timeoutCache )
        {
            return timeoutCache.getOrDefault( connectionString.hashCode(), MIN_TIMEOUT_SEC );
        }
    }

    private void updateTimeout( ConnectionString connectionString, DataSource dataSource, Long connectionTimeMs )
    {
        synchronized ( timeoutCache )
        {
            String url = getUrl( dataSource );
            Long connectionTimeSec = connectionTimeMs / 1000;

            if ( connectionTimeSec > MIN_TIMEOUT_SEC )
            {
                Long newTimeoutSec = connectionTimeSec + TIMEOUT_EXTRA_TIME_SEC;
                log.debug( "Updating timeout for connection string [{}] to {} sec", url,
                           newTimeoutSec );
                timeoutCache.put( connectionString.hashCode(), newTimeoutSec );
            }
            else
            {
                log.debug( "Resetting timeout for connection string [{}]", url );
                timeoutCache.remove( connectionString.hashCode() );
            }
        }
    }

    private void deleteTimeout( ConnectionString connectionString, DataSource dataSource )
    {
        synchronized ( timeoutCache )
        {
            log.debug( "Resetting timeout for connection string [{}]", getUrl( dataSource ) );
            timeoutCache.remove( connectionString.hashCode() );
        }
    }

    private DatabaseConnectionTestResult runTest( ConnectionString connectionString, DataSource dataSource )
    {
        String url = getUrl( dataSource );
        Connection connection = null;
        try
        {
            long startedTime = System.currentTimeMillis();
            connection = dataSource.getConnection();
            long finishedTime = System.currentTimeMillis();

            long finalTime = finishedTime - startedTime;
            log.debug( "Connected with {} after {} ms", url, finalTime );
            updateTimeout( connectionString, dataSource, finalTime );

            return DatabaseConnectionTestResult.success();
        }
        catch ( SQLException ex )
        {
            log.error( "Cannot connect with connection string [{}]", url );
            log.error( ex.getMessage(), ex );

            deleteTimeout( connectionString, dataSource );
            return DatabaseConnectionTestResult.failure( ex );
        }
        finally
        {
            if ( connection != null )
            {
                try
                {
                    connection.close();
                }
                catch ( SQLException e )
                {
                    // ignore
                }
            }
        }
    }

    private String getUrl( DataSource dataSource )
    {
        if ( dataSource instanceof SimpleDriverDataSource )
        {
            SimpleDriverDataSource simpleDriverDataSource = (SimpleDriverDataSource) dataSource;
            return simpleDriverDataSource.getUrl();
        }
        else
        {
            log.warn( "Cannot find database URL. DataSource class [{}]", dataSource.getClass().getName() );
            return "UNKNOWN";
        }
    }
}
