package com.suncode.dbexplorer.database;

import java.util.LinkedHashMap;
import java.util.Map;

import org.springframework.util.Assert;

import com.google.common.collect.Iterables;

/**
 * Identyfikator rekordu.
 * <p>
 * W zależności od tabeli rekord może byc identyfikowany jedną lub wieloma kolumnami. Odpowiedni
 * identyfikator możemy stworzyc poprzez {@linkplain RecordId#single(Object)} lub
 * {@linkplain RecordId#composite(Map)}.
 * 
 * @author Cezary Kozar 6 paź 2015
 */
public class RecordId
{
    private boolean composite;

    private Object singleValue;

    private Map<String, Object> compositeValues;

    private RecordId( Object value )
    {
        this.composite = false;
        this.singleValue = value;
    }

    private RecordId( Map<String, Object> values )
    {
        Assert.notEmpty( values, "[Assertion failed] - this map must not be empty; it must contain at least one entry" );
        Assert.isTrue( values.size() > 1, "[Assertion failed] - this expression must be true" );

        this.composite = true;
        this.compositeValues = values;
    }

    public static RecordId single( Object value )
    {
        Assert.notNull( value, "[Assertion failed] - this argument is required; it must not be null" );
        return new RecordId( value );
    }

    public static RecordId composite( Map<String, Object> values )
    {
        if ( values.size() == 1 )
        {
            return new RecordId( Iterables.getFirst( values.values(), null ) );
        }
        return new RecordId( values );
    }

    public static Composite buildComposite()
    {
        return new Composite();
    }

    public Object getSingleValue()
    {
        Assert.state( !composite, "This IS composite id: check this with isComposite() method first!" );
        return singleValue;
    }

    public Map<String, Object> getCompositeValues()
    {
        Assert.state( composite, "This is NOT composite id: check this with isComposite() method first!" );
        return compositeValues;
    }

    public boolean isComposite()
    {
        return composite;
    }

    public static class Composite
    {

        private Map<String, Object> values = new LinkedHashMap<String, Object>();

        public Composite()
        {
        }

        public Composite add( String column, Object value )
        {
            values.put( column, value );
            return this;
        }

        public RecordId get()
        {
            if ( values.size() > 1 )
            {
                return composite( values );
            }
            return single( Iterables.getFirst( values.values(), null ) );
        }
    }
}
