package com.suncode.dbexplorer.configurationtransfer;

import com.suncode.dbexplorer.alias.Alias;
import com.suncode.dbexplorer.alias.AliasService;
import com.suncode.dbexplorer.alias.TablesSet;
import com.suncode.dbexplorer.configurationtransfer.dto.ConfigurationDbExplorerRootDto;
import com.suncode.dbexplorer.configurationtransfer.dto.alias.ConfigurationAliasConverter;
import com.suncode.dbexplorer.configurationtransfer.dto.alias.ConfigurationAliasDto;
import com.suncode.dbexplorer.configurationtransfer.dto.alias.tableset.ConfigurationTablesSetConverter;
import com.suncode.dbexplorer.configurationtransfer.dto.alias.tableset.ConfigurationTablesSetDto;
import com.suncode.plugin.framework.Plugin;
import com.suncode.plugin.framework.PluginsException;
import com.suncode.pwfl.administration.user.User;
import com.suncode.pwfl.administration.user.UserFinder;
import com.suncode.pwfl.administration.user.UserGroup;
import com.suncode.pwfl.administration.user.UserGroupFinder;
import com.suncode.pwfl.configuration.audit.ConfigurationTransferAudit;
import com.suncode.pwfl.configuration.dto.plugins.PluginConfigurationDtoRoot;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

@Slf4j
@Component
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class ConfigurationImporter
{
    private final Plugin plugin;

    private final AliasService aliasService;

    private final UserFinder userFinder;

    private final UserGroupFinder userGroupFinder;

    private final ConfigurationAliasConverter configurationAliasConverter;

    private final ConfigurationTablesSetConverter configurationTablesSetConverter;

    @Transactional
    public void importConfig( PluginConfigurationDtoRoot configuration, ConfigurationTransferAudit importAudit )
    {
        validatePlugin( configuration );

        ConfigurationDbExplorerRootDto config = (ConfigurationDbExplorerRootDto) configuration;

        if ( !config.getMetadata().isSelected() )
        {
            return;
        }

        List<ConfigurationAliasDto> selectedAliases = config.getAliases().getOnlySelectedElements();
        if ( !selectedAliases.isEmpty() )
        {
            importAliases( config.getAliases().getOnlySelectedElements(), importAudit );
        }
    }

    private void validatePlugin( PluginConfigurationDtoRoot configuration )
    {
        if ( !plugin.getKey().equals( configuration.getPluginId() ) )
        {
            throw new PluginsException( "The configuration object is not the " + plugin.getName() + " plugin configuration" );
        }
    }

    private void importAliases( List<ConfigurationAliasDto> aliases, ConfigurationTransferAudit importAudit )
    {
        List<Alias> aliasesInBase = aliasService.getAliases();
        List<User> allUsersInBase = userFinder.getAll();
        List<UserGroup> allGroupsInBase = userGroupFinder.getAll();

        for ( ConfigurationAliasDto aliasDto : aliases )
        {
            if ( aliasDto.isSystemAlias() )
            {
                Optional<Alias> systemAlias = findSystemAlias( aliasesInBase );
                systemAlias.ifPresent( alias -> addTableSets( alias, aliasDto, allUsersInBase, allGroupsInBase ) );
            }
            else
            {
                processAlias( aliasDto, allUsersInBase, allGroupsInBase, aliasesInBase, importAudit );
            }
        }
    }

    private void addTableSets( Alias existentAlias, ConfigurationAliasDto aliasDto,
                               List<User> users, List<UserGroup> groups )
    {
        List<ConfigurationTablesSetDto> selectedTableSets = aliasDto.getTableSets().getList()
            .stream()
            .filter( tableSet -> tableSet.getMetadata().isSelected() )
            .filter( tableSet -> existentAlias.getTablesSets().stream()
                .map( TablesSet::getName )
                .noneMatch( setName -> StringUtils.equals( setName, tableSet.getName() ) ) )
            .toList();

        for ( ConfigurationTablesSetDto tablesSetDto : selectedTableSets )
        {
            existentAlias.addTablesSet( configurationTablesSetConverter.convertToEntity( tablesSetDto, users, groups ) );
        }
    }

    private void processAlias( ConfigurationAliasDto aliasDto, List<User> users, List<UserGroup> groups, List<Alias> aliasesInBase,
                               ConfigurationTransferAudit importAudit )
    {
        Optional<Alias> existingAlias = checkAliasExistsInBase( aliasDto.getName(), aliasesInBase );
        if ( existingAlias.isPresent() )
        {
            log.info( "Alias with name {} already exists. Importing table sets...", existingAlias.get().getName() );
            addTableSets( existingAlias.get(), aliasDto, users, groups );
        }
        else
        {
            Alias alias = configurationAliasConverter.convertToEntity( aliasDto, users, groups );
            aliasService.addAlias( alias );
        }
        importAudit.addElement( "DB EXPLORER ALIAS", aliasDto.getName() );
    }

    private Optional<Alias> findSystemAlias( List<Alias> aliasesInBase )
    {
        return aliasesInBase.stream().filter( Alias::getIsSystemAlias ).findFirst();
    }

    private Optional<Alias> checkAliasExistsInBase( String aliasName, List<Alias> aliasesInBase )
    {
        return aliasesInBase.stream().filter( alias -> alias.getName().equals( aliasName ) ).findFirst();
    }
}
