package com.suncode.dbexplorer.alias.internal;

import com.suncode.dbexplorer.alias.Alias;
import com.suncode.dbexplorer.alias.AliasService;
import com.suncode.dbexplorer.alias.exception.AliasExistsException;
import com.suncode.plugin.framework.service.Provides;
import lombok.RequiredArgsConstructor;
import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.stream.Collectors;

@Service
@Transactional
@Provides( AliasService.class )
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class AliasServiceImpl
                implements AliasService
{
    private final AliasRepository aliasRepository;

    @Override
    public void addAlias( Alias alias )
    {
        Assert.notNull( alias, "[Assertion failed] - this argument is required; it must not be null" );
        if ( aliasRepository.findOneByProperty( "name", alias.getName() ) != null )
        {
            throw new AliasExistsException();
        }

        alias.setName( Jsoup.clean( alias.getName().trim(), Safelist.basic() ) );
        aliasRepository.save( alias );
    }

    @Override
    public Alias addSystemAlias()
    {
        if ( aliasRepository.findOneByProperty( "isSystemAlias", true ) != null
                        || aliasRepository.findOneByProperty( "name", Alias.SYSTEM_ALIAS_NAME ) != null )
        {
            throw new AliasExistsException();
        }

        Alias systemAlias = Alias.createSystemAlias();
        aliasRepository.save( systemAlias );
        return systemAlias;
    }

    @Override
    public void deleteAlias( Long id )
    {
        Alias alias = getAlias( id );
        if ( alias != null )
        {
            aliasRepository.delete( alias );
        }
    }

    @Override
    public void updateAlias( Alias alias )
    {
        Assert.notNull( alias, "[Assertion failed] - this argument is required; it must not be null" );
        alias.setName( Jsoup.clean( alias.getName().trim(), Safelist.basic() ) );
        aliasRepository.update( alias );
    }

    @Override
    public Alias getAlias( Long id )
    {
        return aliasRepository.get( id );
    }

    @Override
    public Alias getAlias( String name )
    {
        return aliasRepository.getByField( "name", name );
    }

    @Override
    public List<Alias> getAliases()
    {
        return aliasRepository.getAll();
    }

    @Override
    public List<Alias> getActiveAliases()
    {
        return getAliases()
            .stream()
            .filter( Alias::getIsActive )
            .collect( Collectors.toList() );
    }
}
