package com.suncode.dbexplorer.alias.data.util.importer;

import com.github.pjfanning.xlsx.StreamingReader;
import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.Record;
import com.suncode.dbexplorer.database.schema.ColumnSchema;
import com.suncode.dbexplorer.database.schema.TableSchema;
import com.suncode.dbexplorer.database.type.BasicDataType;
import com.suncode.pwfl.config.Environment;
import com.suncode.pwfl.config.home.HomeDirectory;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.openxml4j.opc.PackageAccess;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFFormulaEvaluator;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

@Component
@Slf4j
public class ImportHelper
{
    private static final Path TEMP_POI_PATH = Paths
        .get( Environment.get( HomeDirectory.HOME_DIRECTORY ), "temp", "poi" );

    public void processRecord(DatabaseSession session, Record record, boolean clear, int rowNum, long rowsInBase) {
        try {
            if (shouldUpdateRecord(clear, record, rowsInBase)) {
                if (tryUpdateRecord(session, record, rowNum)) {
                    return;
                }
            }
            session.insert(record);
        } catch (Exception e) {
            log.error("Invalid data in row number {}", rowNum);
            throw e;
        }
    }

    private boolean shouldUpdateRecord(boolean clear, Record record, long rowsInBase) {
        return !clear && record.hasId() && rowsInBase > 0;
    }

    private boolean tryUpdateRecord(DatabaseSession session, Record record, int rowNum) {
        ColumnSchema[] primaryKeyColumns = record.getTable().getPrimaryKeyColumns();
        if (!recordHasPkValue(primaryKeyColumns, record)) {
            if (recordHasAutoIncrementColumn(primaryKeyColumns)) {
                return false;
            } else {
                Translator translator = Translators.get(ImportHelper.class);
                throw new IllegalStateException(translator.getMessage("dbex.data.import.null.exception"));
            }
        }
        return session.update(record);
    }

    private boolean recordHasAutoIncrementColumn( ColumnSchema[] pkColumns )
    {
        return Arrays.stream( pkColumns )
            .anyMatch( col -> col.isAutoIncrement() );
    }

    public Record createRecord( Database database, Table table, XSSFFormulaEvaluator evaluator, String[] columnNames, Row row,
                                TableSchema tableSchema )
    {
        Record record = new Record( tableSchema.getSchema(), tableSchema.getName(), database );
        for ( int i = 0; i < columnNames.length; i++ )
        {
            String columnName = columnNames[i];
            if ( columnName == null )
            {
                continue;
            }
            if ( !table.getTableSchema().hasColumn( columnName ) )
            {
                continue;
            }

            Cell cell = row.getCell( i, Row.MissingCellPolicy.CREATE_NULL_AS_BLANK );
            ColumnSchema columnSchema = table.getTableSchema().getColumn( columnName );
            BasicDataType basicType = BasicDataType.of( columnSchema.getType() );

            if ( evaluator != null )
            {
                evaluator.evaluateInCell( cell );
            }

            Object value = readCellValue( cell, basicType );
            record.set( columnName, value );
        }
        return record;
    }

    public Record createRecord( Database database, Table table, String[] columnNames, Row row,
                                TableSchema tableSchema )
    {
        return createRecord( database, table, null, columnNames, row, tableSchema );
    }

    private Object readCellValue( Cell cell, BasicDataType basicType )
    {
        Object value = null;

        CellType cellType = cell.getCellType();
        switch ( cellType )
        {
            case STRING:
                value = cell.getStringCellValue();
                break;
            case NUMERIC:

                if ( basicType == BasicDataType.DATE || basicType == BasicDataType.DATETIME
                    || basicType == BasicDataType.TIME )
                {
                    value = cell.getDateCellValue();
                }
                else
                {
                    value = cell.getNumericCellValue();
                }
                break;
            case BOOLEAN:
                value = cell.getBooleanCellValue();
                break;
        }
        return value;
    }

    public OPCPackage getOpcPackage( File tempPoiFile )
        throws IOException
    {
        OPCPackage opcPackage;
        try
        {
            opcPackage = OPCPackage.open( tempPoiFile, PackageAccess.READ );
        }
        catch ( InvalidFormatException e1 )
        {
            tempPoiFile.delete();
            throw new IOException( e1 );
        }
        catch ( Exception e )
        {
            tempPoiFile.delete();
            throw e;
        }
        return opcPackage;
    }

    public File getTempPoiFile()
        throws IOException
    {
        Path tempPoiDirectoryPath = Files.createDirectories( TEMP_POI_PATH );
        Path tempPoiFilePath = Files.createTempFile( tempPoiDirectoryPath, null, null );

        return tempPoiFilePath.toFile();
    }

    public List<String> readColumns( InputStream inputStream )
        throws IOException
    {
        File tempPoiFile = getTempPoiFile();
        tempPoiFile.deleteOnExit();

        try ( FileOutputStream tempOutputStream = new FileOutputStream( tempPoiFile ) )
        {
            IOUtils.copy( inputStream, tempOutputStream );
        }
        catch ( Exception e )
        {
            tempPoiFile.delete();
            throw e;
        }

        try ( Workbook workbook = StreamingReader.builder()
            .rowCacheSize( 1 )
            .open( tempPoiFile ) )
        {
            List<String> columnNames = new ArrayList<>();
            Sheet sheet = workbook.getSheetAt( 0 );
            Iterator<Row> rowIterator = sheet.iterator();
            Row headerRow = rowIterator.next();

            for ( Cell cell : headerRow )
            {
                columnNames.add( cell.getStringCellValue() );
            }

            return columnNames;
        }
        finally
        {
            tempPoiFile.delete();
        }
    }

    public SheetData readSheetData( InputStream inputStream )
        throws IOException
    {
        File tempPoiFile = getTempPoiFile();
        tempPoiFile.deleteOnExit();

        try ( FileOutputStream tempOutputStream = new FileOutputStream( tempPoiFile ) )
        {
            IOUtils.copy( inputStream, tempOutputStream );
        }
        catch ( Exception e )
        {
            tempPoiFile.delete();
            throw e;
        }

        try ( Workbook workbook = StreamingReader.builder()
            .rowCacheSize( 4000 )
            .open( tempPoiFile ) )
        {
            List<String> columnNames = new ArrayList<>();
            Sheet sheet = workbook.getSheetAt( 0 );
            Iterator<Row> rowIterator = sheet.iterator();
            Row headerRow = rowIterator.next();

            long rowCount = 1;
            for ( Cell cell : headerRow )
            {
                columnNames.add( cell.getStringCellValue() );
            }

            while ( rowIterator.hasNext() )
            {
                rowCount++;
                rowIterator.next();
            }

            return new SheetData( columnNames, rowCount );
        }
        finally
        {
            tempPoiFile.delete();
        }
    }

    public boolean recordHasPkValue( ColumnSchema[] pkColumns, Record record )
    {
        for ( ColumnSchema schema : pkColumns )
        {
            if ( record.get( schema.getName() ) == null )
                return false;
        }
        return true;
    }

    public boolean isRowEmpty( Row row )
    {
        for ( Cell cell : row )
        {
            if ( !cell.toString().isEmpty() )
            {
                return false;
            }
        }
        return true;
    }

    @RequiredArgsConstructor
    @Getter
    public static class SheetData
    {
        private final List<String> columnNames;

        private final long rowCount;
    }
}
