package com.suncode.dbexplorer.alias.data.util.importer;

import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.alias.data.util.importer.config.ImportStrategy;
import com.suncode.dbexplorer.alias.data.util.importer.config.ImportType;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.Record;
import com.suncode.pwfl.util.exception.ServiceException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.xssf.usermodel.XSSFFormulaEvaluator;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

@Slf4j
@Component
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class ExcelXSSFImporter
    implements ImportStrategy
{
    //https://lists.apache.org/thread/5zd7xwocbcm9s5jlc88y5g8hnd36rhqr
    public static final int XSSF_CELL_SIZE_BYTES = 630;

    private final ImportHelper importHelper;

    @Override
    public ImportType importMode()
    {
        return ImportType.STANDARD;
    }

    @Override
    public void validate( InputStream fileStream )
        throws IOException
    {
        ImportHelper.SheetData sheetData = importHelper.readSheetData( fileStream );

        long expectedMemoryToUse = (sheetData.getColumnNames().size() * sheetData.getRowCount() * XSSF_CELL_SIZE_BYTES) / (1024 * 1024);
        long freeMemory = Runtime.getRuntime().freeMemory() / (1024 * 1024);

        if ( freeMemory < expectedMemoryToUse )
        {
            throw new ServiceException( "dbex.grid.import.validation.size" );
        }
    }

    @Override
    public void importRecords( InputStream inputStream, Database database, Table table, boolean clear )
        throws IOException
    {
        File tempPoiFile = importHelper.getTempPoiFile();
        tempPoiFile.deleteOnExit();

        try ( FileOutputStream tempOutputStream = new FileOutputStream( tempPoiFile ) )
        {
            IOUtils.copy( inputStream, tempOutputStream );
        }
        catch ( Exception e )
        {
            tempPoiFile.delete();
            throw e;
        }

        try ( OPCPackage opcPackage = importHelper.getOpcPackage( tempPoiFile ) )
        {
            XSSFWorkbook workbook = new XSSFWorkbook( opcPackage );
            XSSFFormulaEvaluator evaluator = workbook.getCreationHelper().createFormulaEvaluator();

            // 1. Odczyt pierwszego arkusza (jedyny)
            XSSFSheet sheet = workbook.getSheetAt( 0 );

            // 2. Odczyt nagłówka i pobranie nazw kolumn
            XSSFRow headerRow = sheet.getRow( 0 );

            String[] columnNames = new String[headerRow.getLastCellNum()];
            for ( Cell cell : headerRow )
            {
                columnNames[cell.getColumnIndex()] = cell.getStringCellValue();
            }

            database.withinSession( session -> {
                if ( clear )
                {
                    session.delete().from( table.getTableSchema().getFullName() ).execute();
                }

                long rowsInBase = session
                    .select()
                    .from( table.getTableSchema().getFullName() )
                    .count();


                for ( Row row : sheet )
                {
                    if ( row.getRowNum() == 0 || importHelper.isRowEmpty( row ) )
                    {
                        continue;
                    }

                    Record record = importHelper.createRecord( database, table, evaluator, columnNames, row, table.getTableSchema() );
                    importHelper.processRecord( session, record, clear, row.getRowNum(), rowsInBase );
                }
                return null;
            } );
        }
        finally
        {
            tempPoiFile.delete();
        }
    }
}
