package com.suncode.dbexplorer.alias.data.util.exporter;

import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.database.Record;
import com.suncode.dbexplorer.database.schema.ColumnSchema;
import com.suncode.dbexplorer.database.type.BasicDataType;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.util.DateFormatConverter;
import org.apache.poi.xssf.streaming.SXSSFCell;
import org.apache.poi.xssf.streaming.SXSSFRow;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.NumberUtils;

import java.util.Date;
import java.util.List;

@Component
public class ExcelExporter
{

    public SXSSFWorkbook exportRecords(Table table, List<Record> records )
    {
        SXSSFWorkbook workbook = new SXSSFWorkbook();
        SXSSFSheet sheet = workbook.createSheet( table.getName() );
        sheet.trackAllColumnsForAutoSizing();

        // HEADER
        int columnsCount = addHeaderRow( sheet, table );

        String dateFormat = DateFormatConverter.convert( LocaleContextHolder.getLocale(), "yyyy-MM-dd" );
        String timeFormat = DateFormatConverter.convert( LocaleContextHolder.getLocale(), "hh:mm:ss" );
        String dateTimeFormat = DateFormatConverter.convert( LocaleContextHolder.getLocale(), "yyyy-MM-dd hh:mm:ss" );

        CellStyle dateStyle =  workbook.createCellStyle();
        dateStyle.setDataFormat( workbook.createDataFormat().getFormat( dateFormat ) );

        CellStyle timeStyle = workbook.createCellStyle();
        timeStyle.setDataFormat( workbook.createDataFormat().getFormat( timeFormat ) );

        CellStyle dateTimeStyle = workbook.createCellStyle();
        dateTimeStyle.setDataFormat( workbook.createDataFormat().getFormat( dateTimeFormat ) );

        // DATA
        int rowNr = 1;
        for ( Record record : records )
        {
            SXSSFRow row = sheet.createRow( rowNr++ );
            int colNr = 0;
            for ( ColumnSchema column : table.getTableSchema().getColumns() )
            {
                SXSSFCell cell = row.createCell( colNr++ );

                Object value = record.get( column.getName() );
                if ( value == null )
                {
                    continue;
                }

                BasicDataType basicDataType = BasicDataType.of( column.getType() );
                if ( basicDataType != null )
                {
                    switch ( basicDataType )
                    {
                        case STRING:
                            cell.setCellValue( value.toString() );
                            break;
                        case INTEGER:
                            cell.setCellValue( NumberUtils.convertNumberToTargetClass( (Number) value, Long.class ) );
                            break;
                        case FLOAT:
                            cell.setCellValue( NumberUtils.convertNumberToTargetClass( (Number) value, Double.class ) );
                            break;
                        case BOOLEAN:
                            cell.setCellValue( (Boolean) value );
                            break;
                        case DATE:
                            cell.setCellValue( (Date) value );
                            cell.setCellStyle( dateStyle );
                            break;
                        case TIME:
                            cell.setCellValue( (Date) value );
                            cell.setCellStyle( timeStyle );
                            break;
                        case DATETIME:
                            cell.setCellValue( (Date) value );
                            cell.setCellStyle( dateTimeStyle );
                            break;
                        case BINARY:
                        case UNKNOWN:
                            break;
                    }
                }
                else
                {
                    if ( value instanceof Boolean )
                    {
                        cell.setCellValue( (Boolean) value );
                    }
                    else if ( value instanceof Number )
                    {
                        cell.setCellValue( NumberUtils.convertNumberToTargetClass( (Number) value, Double.class ) );
                    }
                    else if ( value instanceof Date )
                    {
                        cell.setCellValue( (Date) value );
                        cell.setCellStyle( dateTimeStyle );
                    }
                    else
                    {
                        cell.setCellValue( value.toString() );
                    }
                }
            }
        }

        for ( int i = 0; i < columnsCount; i++ )
        {
            sheet.autoSizeColumn( i );
        }
        return workbook;
    }

    private int addHeaderRow( SXSSFSheet sheet, Table table )
    {
        int colNr = 0;
        SXSSFRow headerRow = sheet.createRow( 0 );
        for ( ColumnSchema column : table.getTableSchema().getColumns() )
        {
            SXSSFCell cell = headerRow.createCell( colNr++ );
            cell.setCellValue( column.getName() );
        }

        return colNr;
    }
}
