package com.suncode.dbexplorer.alias.data.dto;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.alias.Alias;
import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.alias.dto.SimpleTableDto;
import com.suncode.dbexplorer.alias.dto.TableDto;
import com.suncode.dbexplorer.alias.permission.AccessLevel;
import com.suncode.dbexplorer.alias.permission.SecuredTablesSet;
import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseFactory;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Getter
@Setter
@NoArgsConstructor
@Slf4j
public class SecuredTablesSetDto
{
    private String name;

    private AccessLevel accessLevel;

    private Boolean available;

    private List<TableDto> tables = Lists.newArrayList();

    public SecuredTablesSetDto( DatabaseFactory databaseFactory, SecuredTablesSet securedSet )
    {
        this.name = securedSet.getName();
        this.accessLevel = securedSet.getAccessLevel();
        this.available = true;

        Alias alias = securedSet.getSet().getAlias();

        Set<SimpleTableDto> setTables = securedSet.getTables();
        Set<Table> tables = alias.getTables( databaseFactory );
        String defaultSchema = getDefaultSchema( databaseFactory, alias );
        for ( SimpleTableDto tableDto : setTables )
        {
            if ( StringUtils.isEmpty( tableDto.getSchema() ) )
            {
                tableDto.setSchema( defaultSchema );
            }
            for ( Table table : tables )
            {
                if ( table.getSchema().equals( tableDto.getSchema() ) && table.getName().equals( tableDto.getName() ) )
                {
                    this.tables.add( new TableDto( alias, table ) );
                }
            }
        }
    }

    private String getDefaultSchema( DatabaseFactory databaseFactory, Alias alias )
    {
        Database database = databaseFactory.create( alias.getWrappedConnectionString() );
        return database.getDefaultSchemaName();
    }

    @SneakyThrows
    public static List<SecuredTablesSetDto> from( DatabaseFactory databaseFactory,
                                                  Collection<SecuredTablesSet> securedSets )
    {
        ExecutorService executor = Executors.newFixedThreadPool( securedSets.size() );
        Map<Integer, Future<Boolean>> isAvailableMap = securedSets.stream()
            .map( set ->
                  {
                      try
                      {
                          return SecuredTablesSetDto.getConnectionString( set );
                      }
                      catch ( Exception e )
                      {
                          log.info( "Exception while fetching alias: {}, message: {}", set.getName(), e.getMessage() );
                          return null;
                      }
                  } )
            .filter( Objects::nonNull )
                        .distinct()
                        .collect( Collectors.toMap( ConnectionString::hashCode,
                                                    connectionString -> executor.submit(
                                                                    () -> databaseFactory.isAvailable( connectionString )
                                                    ) ) );

        executor.shutdown();
        executor.awaitTermination( 1, TimeUnit.MINUTES );

        return securedSets.stream()
                        .map( set -> {
                            try
                            {
                                ConnectionString connectionString;
                                try
                                {
                                    connectionString = getConnectionString( set );
                                }
                                catch ( Exception e )
                                {
                                    log.info( "Exception while fetching alias: {}, message: {}", set.getName(), e.getMessage() );
                                    return null;
                                }
                                Future<Boolean> futureAvailable = isAvailableMap.get( connectionString.hashCode() );
                                if ( futureAvailable.get() )
                                {
                                    return new SecuredTablesSetDto( databaseFactory, set );
                                }
                                else
                                {
                                    return SecuredTablesSetDto.unavailable( set );
                                }
                            }
                            catch ( Throwable ex )
                            {
                                throw new RuntimeException( ex );
                            }
                        } )
            .filter( Objects::nonNull )
                        .collect( Collectors.toList() );
    }

    private static ConnectionString getConnectionString( SecuredTablesSet set )
    {
        return set.getSet().getAlias().getWrappedConnectionString();
    }

    public static SecuredTablesSetDto unavailable( SecuredTablesSet securedSet )
    {
        SecuredTablesSetDto dto = new SecuredTablesSetDto();
        dto.setName( securedSet.getName() );
        dto.setAccessLevel( securedSet.getAccessLevel() );
        dto.setAvailable( false );
        dto.setTables( new LinkedList<>() );
        return dto;
    }
}
