package com.suncode.dbexplorer.alias.data;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.dbexplorer.alias.Alias;
import com.suncode.dbexplorer.alias.AliasService;
import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.alias.data.dto.SecuredTablesSetDto;
import com.suncode.dbexplorer.alias.data.util.importer.ImportHelper;
import com.suncode.dbexplorer.alias.data.util.importer.config.ImportType;
import com.suncode.dbexplorer.alias.exception.AliasNotActiveException;
import com.suncode.dbexplorer.alias.permission.AccessLevel;
import com.suncode.dbexplorer.alias.permission.PermissionsService;
import com.suncode.dbexplorer.alias.permission.SecuredTablesSet;
import com.suncode.dbexplorer.audit.AuditTypes;
import com.suncode.dbexplorer.context.UserContext;
import com.suncode.dbexplorer.database.DatabaseFactory;
import com.suncode.dbexplorer.database.Record;
import com.suncode.dbexplorer.database.query.Page;
import com.suncode.dbexplorer.database.query.Pagination;
import com.suncode.dbexplorer.util.authorization.AuthorizationHelper;
import com.suncode.dbexplorer.util.web.Paging;
import com.suncode.dbexplorer.util.web.rest.ResourceNotFoundException;
import com.suncode.dbexplorer.util.web.rest.RestController;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import com.suncode.pwfl.util.TempFile;
import com.suncode.pwfl.util.exception.ServiceException;
import com.suncode.pwfl.web.support.io.DownloadResource;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import static com.suncode.dbexplorer.audit.AuditTypes.AUDIT_TABLE_EXPORT;
import static com.suncode.dbexplorer.audit.AuditTypes.AUDIT_TABLE_IMPORT;
import static com.suncode.dbexplorer.audit.AuditTypes.AUDIT_TABLE_ROW_DELETE;
import static com.suncode.dbexplorer.audit.AuditTypes.AUDIT_TABLE_ROW_INSERT;
import static com.suncode.dbexplorer.audit.AuditTypes.AUDIT_TABLE_ROW_UPDATE;

@Slf4j
@Controller
@RequiredArgsConstructor( onConstructor_ = { @Autowired } )
public class DataController
    extends RestController
{
    private final DatabaseFactory databaseFactory;

    private final PermissionsService permissionService;

    private final DataService dataService;

    private final AliasService aliasService;

    private final AuthorizationHelper authorizationHelper;

    private final static ConcurrentMap<UUID, NamedFile> oneTimeFiles = new ConcurrentHashMap<>();

    private final ImportHelper importHelper;

    @RequestMapping( value = "/data/tablessets", method = RequestMethod.GET )
    public @ResponseBody List<SecuredTablesSetDto> getTablesSets()
    {
        List<SecuredTablesSet> sets = permissionService.getSecuredTablesSets( UserContext.userName() )
            .stream()
            .filter( securedTablesSet -> securedTablesSet.getSet().getAlias().getIsActive() )
            .collect( Collectors.toList() );

        if ( sets.size() == 0 )
        {
            return new ArrayList<>();
        }

        return SecuredTablesSetDto.from( databaseFactory, sets );
    }

    @RequestMapping( value = "/aliases/{aliasId}/data", method = RequestMethod.GET )
    public @ResponseBody Page<Record> getData( @PathVariable Long aliasId,
                                               @RequestParam final String schema, @RequestParam final String table,
                                               @RequestParam final Long limit,
                                               final Paging paging, @RequestParam( required = false ) String filters )
        throws Exception
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.VIEW );

        validateAliasIsActive( aliasId );

        if ( filters == null )
        {
            filters = "[]";
        }

        ObjectMapper objectMapper = new ObjectMapper();
        final List<Filter> filters2 = objectMapper.readValue( filters, new TypeReference<List<Filter>>()
        {
        } );

        Pagination pagination = paging.pagination();
        if ( limit != null )
        {
            pagination.setPageSize( limit.intValue() );
        }

        return dataService.getPage( aliasId, schema, table, pagination, filters2 );
    }

    @RequestMapping( value = "/aliases/{aliasId}/data", method = RequestMethod.POST )
    public @ResponseBody void insertData( @PathVariable Long aliasId,
                                          @RequestParam String schema, @RequestParam String table,
                                          @RequestBody UpdateRecord updateRecord )
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.EDIT );

        boolean isSuccess = false;
        Date start = new Date();
        validateAliasIsActive( aliasId );

        Alias alias = aliasService.getAlias( aliasId );
        Table aliasTable = alias
            .getTable( schema, table, databaseFactory );

        updateRecord.setSchema( schema );
        updateRecord.setTable( table );

        try
        {
            dataService.addRecord( aliasId, updateRecord );
            isSuccess = true;
        }
        finally
        {
            if ( alias.getLogging() || aliasTable.getLogging() )
            {
                log.info( String.format( "Data insertion in alias: %s, schema: %s, table: %s, record: %s", alias.getName(), schema, table,
                                         updateRecord.getData().toString() ) );
                Map<String, Object> auditParams = prepareBaseParams( alias, aliasTable );
                auditParams.put( "dbex.audit.alias.table.record.value", updateRecord.getData().toString().replace( "=", "->" ) );
                logAudit( AUDIT_TABLE_ROW_INSERT, isSuccess, auditParams, start );
            }
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}/data", method = RequestMethod.PUT )
    public @ResponseBody void updateData( @PathVariable Long aliasId,
                                          @RequestParam String schema, @RequestParam String table,
                                          @RequestBody UpdateRecord updateRecord )
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.EDIT );

        Date start = new Date();
        boolean isSuccess = false;

        validateAliasIsActive( aliasId );

        Alias alias = aliasService.getAlias( aliasId );
        Table aliasTable = alias
            .getTable( schema, table, databaseFactory );

        updateRecord.setSchema( schema );
        updateRecord.setTable( table );
        Optional<Record> recordToUpdate = Optional.ofNullable( dataService.getRecord( alias, updateRecord ) );

        try
        {
            dataService.updateRecord( aliasId, updateRecord );
            isSuccess = true;
        }
        finally
        {
            if ( alias.getLogging() || aliasTable.getLogging() )
            {
                String key = updateRecord.getData().keySet().iterator().next();
                String oldValue = String.valueOf( recordToUpdate
                                                      .map( record -> record.getData().getOrDefault( key, "N/A" ) )
                                                      .orElse( "N/A" ) );
                String newValue = String.valueOf( updateRecord.getData().getOrDefault( key, "N/A" ) );

                log.info( String.format(
                    "Data update in alias: %s, schema: %s, table: %s, column: %s, value: %s -> %s",
                    alias.getName(), aliasTable.getSchema(), aliasTable.getName(), key, oldValue, newValue ) );

                Map<String, Object> params = prepareBaseParams( alias, aliasTable );
                params.put( "dbex.audit.alias.column.columnName", key );
                params.put( "dbex.audit.alias.table.record.value", String.format( "%s -> %s", oldValue, newValue ) );

                logAudit( AUDIT_TABLE_ROW_UPDATE, isSuccess, params, start );
            }
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}/data", method = RequestMethod.DELETE )
    public @ResponseBody void deleteData( @PathVariable Long aliasId,
                                          @RequestParam String schema, @RequestParam String table,
                                          @RequestBody UpdateRecord updateRecord )
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.EDIT );

        boolean isSuccess = false;
        Date start = new Date();
        validateAliasIsActive( aliasId );

        Alias alias = aliasService.getAlias( aliasId );
        Table aliasTable = alias
            .getTable( schema, table, databaseFactory );

        updateRecord.setSchema( schema );
        updateRecord.setTable( table );

        try
        {
            dataService.deleteRecord( aliasId, updateRecord );
            isSuccess = true;
        }
        finally
        {
            if ( alias.getLogging() || aliasTable.getLogging() )
            {
                log.info( String.format( "Data deletion in alias: %s, schema: %s, table: %s, record id: %s", alias.getName(), schema, table,
                                         updateRecord.getPrimaryKey().values().iterator().next() ) );

                Map<String, Object> params = prepareBaseParams( alias, aliasTable );
                params.put( "dbex.audit.alias.id", updateRecord.getPrimaryKey().values().iterator().next() );

                logAudit( AUDIT_TABLE_ROW_DELETE, isSuccess, params, start );
            }
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}/deletedata", method = RequestMethod.DELETE )
    public @ResponseBody void deleteData( @PathVariable Long aliasId,
                                          @RequestBody List<UpdateRecord> updateRecords )
    {
        if ( updateRecords.isEmpty() )
        {
            return;
        }

        boolean hasRecordsFromDifferentTables = updateRecords.stream()
            .collect( Collectors.groupingBy( UpdateRecord::getTable ) )
            .keySet()
            .size() != 1;

        boolean hasRecordsFromDifferentSchemas = updateRecords.stream()
            .collect( Collectors.groupingBy( UpdateRecord::getSchema ) )
            .keySet()
            .size() != 1;

        if ( hasRecordsFromDifferentTables || hasRecordsFromDifferentSchemas )
        {
            Translator translator = Translators.get( DataController.class );
            throw new ServiceException( translator.getMessage( "dbex.validation.bulk.delete.error" ) );
        }

        authorizationHelper.hasAccessToTable( UserContext.userName(), updateRecords.get( 0 ).getTable(),
                                              AccessLevel.EDIT );

        validateAliasIsActive( aliasId );

        dataService.deleteRecords( aliasId, updateRecords );
    }

    @RequestMapping( value = "/aliases/{aliasId}/data/import/validate", method = RequestMethod.POST )
    public @ResponseBody ResponseEntity<?> validateColumnsMatch(
        @PathVariable Long aliasId,
        @RequestParam MultipartFile file,
        @RequestParam String schema, @RequestParam String table )
        throws IOException
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.EDIT );
        validateAliasIsActive( aliasId );
        List<String> uniqueColumns;

        try ( InputStream fileStream = file.getInputStream() )
        {
            uniqueColumns = dataService.getUniqueColumnsForFileAndTable( aliasId, schema, table, importHelper.readColumns( fileStream ) );
        }

        if ( uniqueColumns.isEmpty() )
        {
            return new ResponseEntity<>( HttpStatus.OK );
        }


        return new ResponseEntity<>( uniqueColumns.stream()
                                         .map( StringUtils::stripAccents )
                                         .collect( Collectors.joining( ", " ) ), HttpStatus.EXPECTATION_FAILED );
    }

    @RequestMapping( value = "/aliases/{aliasId}/data/import", method = RequestMethod.POST )
    public @ResponseBody void importTable(
        @PathVariable Long aliasId,
        @RequestParam MultipartFile file,
        @RequestParam String schema,
        @RequestParam String table,
        @RequestParam( required = false, defaultValue = "false" ) boolean clear,
        @RequestParam( required = false, defaultValue = "STANDARD" ) ImportType importType )
        throws IOException
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.EDIT );
        validateAliasIsActive( aliasId );

        Date started = new Date();
        Alias alias = aliasService.getAlias( aliasId );
        Table aliasTable = alias
            .getTable( schema, table, databaseFactory );

        Map<String, Object> auditParams = prepareBaseParams( alias, aliasTable );

        try
        {
            dataService.importTable( aliasId, schema, table, Boolean.TRUE.equals( clear ), file::getInputStream, importType );
            logAudit( AUDIT_TABLE_IMPORT, true, auditParams, started );
        }
        catch ( Exception e )
        {
            logAudit( AUDIT_TABLE_IMPORT, false, auditParams, started );
            throw new ServiceException( e.getMessage(), e );
        }
    }

    @RequestMapping( value = "/aliases/{aliasId}/data/export", method = RequestMethod.GET )
    public @ResponseBody String exportTable(
        @PathVariable Long aliasId,
        @RequestParam String schema,
        @RequestParam String table,
        @RequestParam String filename,
        final Paging paging,
        @RequestParam( required = false ) String filters
    )
        throws IOException
    {
        authorizationHelper.hasAccessToTable( UserContext.userName(), table, AccessLevel.VIEW );

        validateAliasIsActive( aliasId );

        Date started = new Date();
        Alias alias = aliasService.getAlias( aliasId );
        Table aliasTable = alias
            .getTable( schema, table, databaseFactory );

        if ( filters == null )
        {
            filters = "[]";
        }
        ObjectMapper objectMapper = new ObjectMapper();
        final List<Filter> filters2 = objectMapper.readValue( filters, new TypeReference<>()
        {
        } );

        SXSSFWorkbook workbook;
        try
        {
            workbook = dataService.exportTable( aliasId, schema, table, paging.pagination(), filters2 );
        }
        catch ( Exception e )
        {
            log.info( e.toString() );
            Throwable cause = e.getCause();
            throw new ServiceException( cause != null ? cause.getLocalizedMessage() : e.getLocalizedMessage(), e );
        }

        TempFile tempFile = new TempFile();
        tempFile.getFile().deleteOnExit();

        try ( FileOutputStream fs = new FileOutputStream( tempFile.getFile() ) )
        {
            workbook.write( fs );
        }

        UUID id = UUID.randomUUID();
        oneTimeFiles.put( id, new NamedFile( filename, tempFile ) );

        Map<String, Object> auditParams = prepareBaseParams( alias, aliasTable );
        auditParams.put( "dbex.audit.alias.table.export.filters", filters );
        logAudit( AUDIT_TABLE_EXPORT, true, auditParams, started );
        return id.toString();
    }

    private void validateAliasIsActive( Long aliasId )
    {
        Alias alias = aliasService.getAlias( aliasId );

        if ( !alias.getIsActive() )
        {
            Translator translator = Translators.get( DataController.class );
            throw new AliasNotActiveException( translator.getMessage( "dbex.alias.notActive.exception", alias.getName() ) );
        }
    }

    @RequestMapping( value = "/download/{uuid}", method = RequestMethod.GET )
    public @ResponseBody DownloadResource download( @PathVariable String uuid )
        throws IOException
    {
        UUID id = UUID.fromString( uuid );
        NamedFile namedFile = oneTimeFiles.remove( id );
        if ( namedFile == null )
        {
            throw new ResourceNotFoundException();
        }

        byte[] file = FileUtils.readFileToByteArray( namedFile.file.getFile() );
        namedFile.file.delete();

        ByteArrayResource resource = new ByteArrayResource( file );
        return new DownloadResource( namedFile.name, resource.contentLength(), resource );
    }

    @ResponseBody
    @ExceptionHandler( { AliasNotActiveException.class } )
    public ResponseEntity<?> handleBadRequest( Exception e )
    {
        return new ResponseEntity<>( e.getMessage(), HttpStatus.BAD_REQUEST );
    }

    @ResponseBody
    @ExceptionHandler( { ServiceException.class } )
    public ResponseEntity<?> handleServiceException( ServiceException e )
    {
        return new ResponseEntity<>( e.getMessage(), HttpStatus.EXPECTATION_FAILED );
    }

    private Map<String, Object> prepareBaseParams( Alias alias, Table aliasTable )
    {
        Map<String, Object> params = new LinkedHashMap<>();

        params.put( "dbex.audit.alias.name", alias.getName() );

        String displayName = aliasTable.getDisplayName();
        String tableName = displayName != null
            ? String.format( "%s (%s)", aliasTable.getName(), displayName )
            : aliasTable.getName();
        params.put( "dbex.audit.alias.table.tableName", tableName );

        return params;
    }

    private void logAudit( AuditTypes type, boolean success, Map<String, Object> auditParams, Date start )
    {
        ManualAuditBuilder.getInstance()
            .type( type.getValue() )
            .username( UserContext.userName() )
            .success( success )
            .params( auditParams )
            .started( start )
            .stopped( new Date() )
            .build()
            .log();
    }

    @AllArgsConstructor
    private static class NamedFile
    {
        private final String name;

        private final TempFile file;
    }
}
