import Assert from '../assert/Assert';
import { INPUT_TYPES } from '../../util/types/inputTypes';
import SourceResolver from './SourceResolver';
import DefaultValueService from '../api/DefaultValueService';

class InputResolver {
  async getInputs(view, menu) {
    if (view.inputs) {
      let inputs = await Promise.all(
        view.inputs.map(async input => {
          Assert.hasText(input.type, 'Input has to have a type');
          Assert.hasText(input.id, 'Input id cannot be empty');
          Assert.hasText(input.name, 'Input name cannot be empty');
          Assert.hasText(input.alias, 'Input alias cannot be empty');

          Assert.isTrue(this.validateType(input.type), 'Unknown type ' + input.type);

          let resultInput = {
            id: input.id,
            name: input.name,
            alias: input.alias,
            type: input.type,
            defaultValue: input.dynamicDefaultValue ? await this.getDynamicDefaultValue(view, input) : input.defaultValue,
            stickyViewIds: this.getStickyViewIds(input, menu),
          };

          if (input.type === INPUT_TYPES.LIST_TYPE) {
            Assert.notNull(input.source, 'Source for input type "list" cannot be empty');
            resultInput.source = SourceResolver.getSource(input);
            resultInput.allowMultipleValues = input.allowMultipleValues ?? false;
          }

          if (input.type === INPUT_TYPES.DATE_TYPE) {
            if (input.dateFormat) {
              resultInput.dateFormat = input.dateFormat;
            }

            if (input.returnType) {
              resultInput.returnType = input.returnType;
            }

            if (input.showTime !== undefined && input.showTime !== null) {
              resultInput.showTime = input.showTime;
            }
          }

          return resultInput;
        }),
      );

      this.validateInputs(inputs, view);
      return inputs;
    } else {
      return [];
    }
  }

  async getDynamicDefaultValue(view, input) {
    try {
      return await DefaultValueService.getDefaultValue(view, input);
    } catch (e) {
      console.warn(`Error on retrieving default value for input "${input.id}" in view "${view.id}"`);
      return null;
    }
  }

  validateInputs(inputs, view) {
    this.validateUniqueIds(inputs, view);
  }

  validateType(type) {
    try {
      INPUT_TYPES[type];
      return true;
    } catch (e) {
      return false;
    }
  }

  validateUniqueIds(inputs, view) {
    Assert.areUnique(
      inputs.map(input => input.id),
      'Input ids are not unique for id ' + view.id,
    );
  }

  getStickyViewIds(input, menu) {
    if (!input.stickyViewIds) {
      return null;
    }

    if (!Array.isArray(input.stickyViewIds)) {
      return null;
    }

    const validViewIds = input.stickyViewIds.filter(viewId => {
      const view = menu.views.find(view => view.id === viewId);
      if (!view) {
        console.warn(`Could not find view ${viewId} for the sticky configuration of input ${input.id}`);
        return false;
      }

      const mappedInput = view.inputs.find(otherInput => otherInput.id === input.id);
      if (!mappedInput) {
        console.warn(`Could not find input with id ${input.id} in view ${view.id} for the sticky configuration`);
        return false;
      }

      if (mappedInput.type !== input.type) {
        console.warn(`Input ${input.id} in view ${view.id} has a different type for the sticky configuration`);
        return false;
      }

      return true;
    });

    return validViewIds;
  }
}

export default new InputResolver();
