import ConfigurationService from '../api/ConfigurationService';
import { setQueryParams } from '../../menu/router/routerUtils';
import { absolutePath, createPath } from '../../util/urlUtils';
import { SHOW_VIEW } from '../resolver/action/ShowViewActionResolver';
import { SHOW_ACTIVITY_FORM } from '../resolver/action/ShowActivityFormActionResolver';
import { SHOW_ACTIVITY_DETAILS } from '../resolver/action/ShowActivityDetailsActionResolver';
import { SHOW_PROCESS_DETAILS } from '../resolver/action/ShowProcessDetailsActionResolver';
import { OPEN_URL } from '../resolver/action/OpenUrlActionResolver';
import { extend } from '../../util/objectUtils';

class ActionService {
  runAction(action, row, view) {
    switch (action.type) {
      case SHOW_VIEW:
        this.runShowView(action, row, view);
        break;
      case SHOW_ACTIVITY_FORM:
        this.runShowActivityForm(action, row, view);
        break;
      case SHOW_ACTIVITY_DETAILS:
        this.runShowActivityDetails(action, row, view);
        break;
      case SHOW_PROCESS_DETAILS:
        this.runShowProcessDetails(action, row, view);
        break;
      case OPEN_URL:
        this.runOpenUrl(action, row, view);
        break;
      default:
        throw 'Unknown action type ' + action.type;
    }
  }

  runMiddleClickAction(action, row, view) {
    switch (action.type) {
      case SHOW_ACTIVITY_FORM:
        this.runShowActivityForm(action, row, view, true);
        break;
      case SHOW_ACTIVITY_DETAILS:
        this.runShowActivityDetails(action, row, view, true);
        break;
      case SHOW_PROCESS_DETAILS:
        this.runShowProcessDetails(action, row, view, true);
        break;
      case OPEN_URL:
        action.openInNewTab = true;
        this.runOpenUrl(action, row, view, true);
        break;
      default:
        throw 'Unsupported type for middle click action: ' + action.type;
    }
  }

  iconByAction(action) {
    switch (action.type) {
      case SHOW_VIEW:
        return 'dvnt-icon-windows';
      case SHOW_ACTIVITY_FORM:
      case SHOW_ACTIVITY_DETAILS:
      case SHOW_PROCESS_DETAILS:
        return 'dvnt-icon-right-straight';
      case OPEN_URL:
        return 'dvnt-icon-search';
      default:
        return 'dvnt-icon-highlight';
    }
  }

  runShowView(action, row, view) {
    let outputValuePairs = this.getOutputValuePairs(row, view);

    ConfigurationService.getCurrentMenu().then(menu => {
      let actionView = menu.views.find(view => view.id == action.viewId);
      let inputValues = actionView.inputs.map(input => {
        let value = '';
        let mapping = action.mappings.find(mapping => mapping.inputId == input.id);

        if (mapping) {
          value = mapping.value;
          if (!value) {
            value = outputValuePairs.find(pair => pair.outputId == mapping.outputId).value;
          }
        }

        return {
          id: input.id,
          value,
        };
      });

      setQueryParams(
        {
          viewId: action.viewId,
          inputValues: JSON.stringify(inputValues),
        },
        true,
      );
    });
  }

  runShowActivityForm(action, row, view, openInNewTab) {
    if (!openInNewTab) {
      openInNewTab = false;
    }

    this.runOpenUrl(
      {
        url: 'ProcesActivity.do',
        absolute: false,
        openInNewTab: openInNewTab,
        parameters: [
          {
            name: 'processKey',
            outputId: action.mapping.processId,
          },
          {
            name: 'activityId',
            outputId: action.mapping.activityId,
          },
        ],
      },
      row,
      view,
      {
        redirect: window.location.href,
      },
    );
  }

  runShowActivityDetails(action, row, view, openInNewTab) {
    if (!openInNewTab) {
      openInNewTab = false;
    }

    this.runOpenUrl(
      {
        url: 'ShowDetailHistory.do',
        absolute: false,
        openInNewTab: openInNewTab,
        parameters: [
          {
            name: 'ProcessId',
            outputId: action.mapping.processId,
          },
          {
            name: 'histActivityId',
            outputId: action.mapping.activityId,
          },
        ],
      },
      row,
      view,
    );
  }

  runShowProcessDetails(action, row, view, openInNewTab) {
    if (!openInNewTab) {
      openInNewTab = false;
    }

    this.runOpenUrl(
      {
        url: 'ShowProcessHistory.do',
        absolute: false,
        openInNewTab: openInNewTab,
        parameters: [
          {
            name: 'ProcessId',
            outputId: action.mapping.processId,
          },
        ],
      },
      row,
      view,
    );
  }

  runOpenUrl(action, row, view, additionalParameters) {
    let parameters = {};
    let outputValuePairs = this.getOutputValuePairs(row, view);
    action.parameters.forEach(parameter => {
      let value = parameter.value;
      if (parameter.outputId) {
        value = outputValuePairs.find(item => item.outputId == parameter.outputId).value;
      }
      parameters[parameter.name] = value;
    });

    let path = createPath(action.url, extend(parameters, additionalParameters));
    if (!action.absolute) {
      path = absolutePath(path);
    }

    const windowEl = window.gadgetId ? parent.window : window;
    if (action.openInNewTab) {
      windowEl.open(path, '_blank');
    } else {
      windowEl.location.href = path;
    }
  }

  getOutputValuePairs(row, view) {
    return view.outputs.map(output => ({
      outputId: output.id,
      value: this.findOutputInRow(view.id, row, output.alias),
    }));
  }

  findOutputInRow(viewId, row, outputAlias) {
    let outputInRow = row.find(result => result.alias == outputAlias);
    if (!outputInRow) {
      throw new Error(
        "Zdefiniowany output alias '" + outputAlias + "' dla widoku '" + viewId + "' nie ma zmapowanej wartości w źródle danych",
      );
    }
    return outputInRow.value;
  }
}

export default new ActionService();
