import InputResolver from './InputResolver';
import OutputResolver from './OutputResolver';
import Assert from '../assert/Assert';
import ActionResolver from './action/ActionResolver';
import SourceResolver from './SourceResolver';
import LinkResolver from './LinkResolver';
import SummaryResolver from './SummaryResolver';
import DocumentExtrasResolver from './DocumentExtrasResolver';
import ConditionResolver from './ConditionResolver';

class ViewResolver {
  async getViews(menu) {
    let views = await Promise.all(
      menu.views.map(async view => {
        Assert.hasText(view.id, 'View id cannot be empty');
        Assert.hasText(view.name, 'View name cannot be empty');

        const outputs = OutputResolver.getOutputs(view);
        Assert.notEmpty(outputs, 'Outputs cannot be empty');

        return {
          id: view.id,
          name: view.name,
          default: view.default,
          source: SourceResolver.getSource(view),
          inputs: await InputResolver.getInputs(view, menu),
          outputs: OutputResolver.getOutputs(view),
          action: ActionResolver.getAction(view),
          actionCondition: ConditionResolver.getCondition(view.actionCondition),
          middleClickAction: ActionResolver.getMiddleClickAction(view),
          middleClickActionCondition: ConditionResolver.getCondition(view.middleClickActionCondition),
          links: LinkResolver.getLinks(view),
          comments: !!view.comments,
          summary: SummaryResolver.getSummary(view),
          pageSize: view.pageSize,
          searchOnLoad: !!view.searchOnLoad,
          hideFiltersOnLoad: !!view.hideFiltersOnLoad,
          documentExtras: DocumentExtrasResolver.getDocumentExtras(view),
          disabledViewIds: this.getDisabledViewIds(view, menu),
          nameTemplate: this.getNameTemplate(view),
          disableSorting: view.disableSorting,
        };
      })
    );

    this.validateViews(views);
    return views;
  }

  validateViews(views) {
    this.validateUniqueIds(views);
    this.validateSingleDefault(views);
    this.validateActions(views);
  }

  validateUniqueIds(views) {
    Assert.areUnique(
      views.map(view => view.id),
      'View ids are not unique'
    );
  }

  validateSingleDefault(views) {
    let defaultViews = views.filter(view => !!view.default);

    if (defaultViews.length > 1) {
      throw 'There are more than 1 default view';
    }
  }

  validateActions(views) {
    ActionResolver.validate(views);
  }

  getDisabledViewIds(view, menu) {
    if (!view.disabledViewIds) {
      return [];
    }

    view.disabledViewIds.forEach(viewId => {
      if (viewId === view.id) {
        throw `View with id '${viewId}' cannot be disabled on itself`;
      }

      if (!menu.views.some(otherView => otherView.id === viewId)) {
        throw `Cannot find view with id '${viewId}' to disable on view '${view.id}'`;
      }
    });

    return view.disabledViewIds;
  }

  getNameTemplate(view) {
    if (!view.nameTemplate || !view.nameTemplate.trim().length) {
      return null;
    }

    const variablePattern = /\${([^}]+)}/g;
    let match;
    let variableIds = [];
    while ((match = variablePattern.exec(view.nameTemplate))) {
      variableIds.push(match[1]);
    }

    variableIds.forEach(variableId => {
      if (!view.inputs.some(input => input.id === variableId)) {
        console.warn(`There is no input "${variableId}" in view "${view.name}"`);
      }
    });

    return view.nameTemplate;
  }
}

export default new ViewResolver();
