import axios from 'axios';
import { absolutePath } from '../../util/urlUtils';

class DocumentService {
  async getDocumentsForPreview(processId, activityId, start, limit, sortBy, sortDirection) {
    return (
      await axios.get(absolutePath(`/api/documents/preview`), {
        params: {
          processId: processId,
          activityId: activityId,
          start: start,
          limit: limit,
          sortBy: sortBy,
          sortDirection,
        },
      })
    ).data;
  }

  uploadDocument(file) {
    return axios.post(absolutePath(`/api/documents/uploadFile`), file).then(response => response.data);
  }

  async uploadDocumentWithStatusUpdates(file) {
    const documentUploadUUID = await this.startDocumentUpload(file);

    await new Promise(r => setTimeout(r, 500));
    let documentUploadStatus = await this.getDocumentUploadStatus(documentUploadUUID);
    while (documentUploadStatus.processingState === DocumentUploadProcessingState.PROCESSING) {
      await new Promise(r => setTimeout(r, 2000));
      documentUploadStatus = await this.getDocumentUploadStatus(documentUploadUUID);
    }

    return documentUploadStatus;
  }

  async startDocumentUpload(file) {
    return (await axios.post(absolutePath(`/api/documents/startDocumentUpload`), file)).data;
  }

  async getDocumentUploadStatus(documentUploadUUID) {
    return (
      await axios.get(absolutePath(`/api/documents/documentUploadStatus`), {
        params: {
          documentUploadUUID,
        },
      })
    ).data;
  }

  checkFileSize(file) {
    return axios.post(absolutePath(`/api/form/documents/checkSize`), file).then(response => response.data);
  }

  async getIndicesSync(processId, documentClassId) {
    return (
      await axios.get(absolutePath(`/api/form/documents/indexesForProcessId`), {
        params: {
          processId,
          documentClassId,
        },
      })
    ).data;
  }

  getProcessDocumentCount(processId) {
    return axios
      .get(absolutePath(`/api/documents/process/count`), {
        params: {
          processId: processId,
        },
      })
      .then(response => response.data);
  }

  getActivityDocumentCount(processId, activityId) {
    return axios
      .get(absolutePath(`/api/documents/activity/count`), {
        params: {
          processId: processId,
          activityId: activityId,
        },
      })
      .then(response => response.data);
  }

  async getDocumentCountForProcesses(processIds) {
    return (await axios.post(absolutePath(`/api/documents/process/count`), processIds)).data;
  }

  async getDocumentCountForActivities(activities) {
    return (await axios.post(absolutePath(`/api/documents/activity/count`), activities)).data;
  }

  async exportToZip(activityId, processId, processKey) {
    const data = (
      await axios.post(absolutePath(`/api/form/documents/exportToZip`), null, {
        params: {
          activityId,
          processId,
          processKey,
        },
      })
    ).data;
    window.location.href = absolutePath(`/api/form/documents/download/${data}`);
  }

  async hasPreviewFeature() {
    return (await axios.get(absolutePath('/api/search/feature/preview'))).data;
  }

  async hasAttachFeature() {
    return (await axios.get(absolutePath('/api/search/feature/attach'))).data;
  }
}

export const DocumentUploadProcessingState = Object.freeze({
  PROCESSING: 'PROCESSING',
  FINISHED: 'FINISHED',
  ERROR: 'ERROR',
});

export default new DocumentService();
