package com.suncode.plugin.dataviewer.web.configuration;

import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.plusmpm.util.Authorization;
import com.suncode.plugin.dataviewer.configuration.*;
import com.suncode.plugin.dataviewer.service.configuration.ConfigurationService;
import com.suncode.plugin.dataviewer.service.configuration.ConfigurationTranslator;
import com.suncode.plugin.dataviewer.service.datasupplier.DataSupplier;
import com.suncode.plugin.dataviewer.service.datasupplier.DataSupplierFactory;
import com.suncode.plugin.dataviewer.service.persmission.UserContext;
import com.suncode.plugin.dataviewer.web.util.CacheUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.sql.SQLException;
import java.util.Map;

@Slf4j
@Controller
@RequestMapping( "api/configuration" )
public class ConfigurationController
{
    @Autowired
    private ConfigurationService configurationService;

    @Autowired
    private ConfigurationTranslator configurationTranslator;

    @Autowired
    private DataSupplierFactory dataSupplierFactory;

    private final ObjectMapper objectMapper;

    @Autowired
    public ConfigurationController()
    {
        this.objectMapper = JsonMapper.builder().enable( MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS ).build();
    }

    @ResponseBody
    @Transactional
    @RequestMapping( method = RequestMethod.GET )
    public Configuration getConfiguration( HttpServletResponse response )
    {
        CacheUtil.noCache( response );

        try
        {
            Configuration configuration = configurationService.getConfiguration();
            applyTranslations( configuration );
            applyInputTypes( configuration );
            return configuration;
        }
        catch ( IllegalArgumentException ex )
        {
            throw ex;
        }
        catch ( Exception ex )
        {
            log.error( ex.getMessage(), ex );
            return null;
        }
    }

    private void applyTranslations( Configuration configuration )
    {
        try
        {
            configurationTranslator.apply( configuration );
        }
        catch ( Exception ex )
        {
            log.error( ex.getMessage(), ex );
        }
    }

    private void applyInputTypes( Configuration configuration )
    {
        configuration.getMenus().stream()
                        .flatMap( menu -> menu.getViews().stream() )
                        .forEach( this::applyInputTypes );
    }

    private void applyInputTypes( View view )
    {
        SourceData source = view.getSource();
        DataSupplier dataSupplier = dataSupplierFactory.getDataSupplier( source.getType(), source.getId() );
        Map<String, String> aliasTypes = dataSupplier.getAliasTypes();

        view.getInputs().stream()
                        .filter( input -> input.getType() == null )
                        .forEach( input -> {
                            if ( aliasTypes.containsKey( input.getAlias() ) )
                            {
                                String aliasType = aliasTypes.get( input.getAlias() );

                                try
                                {
                                    input.setType( InputType.forValue( aliasType ) );
                                }
                                catch ( RuntimeException ex )
                                {
                                    log.warn( "Cannot parse alias type " + aliasType + " to InputType" );
                                }
                            }
                            else
                            {
                                input.setType( Input.DEFAULT_INPUT_TYPE );
                            }
                        } );
    }

    @ResponseBody
    @RequestMapping( value = "json", method = RequestMethod.GET )
    public Configuration getJson( HttpServletResponse response )
    {
        CacheUtil.noCache( response );

        try
        {
            return configurationService.getConfiguration();
        }
        catch ( Exception e )
        {
            return null;
        }
    }

    @ResponseStatus( HttpStatus.OK )
    @RequestMapping( method = RequestMethod.POST )
    public ResponseEntity<?> save( @RequestBody String rawConfiguration )
        throws IOException, SQLException
    {
        Configuration configuration = null;
        try
        {
            configuration = objectMapper.readValue( rawConfiguration, Configuration.class );
        }
        catch ( Exception e )
        {
            log.info( "Can't map JSON to Configuration class", e );
            return new ResponseEntity<>( HttpStatus.BAD_REQUEST );
        }
        
        String username = UserContext.userName();
        Authorization.checkRight( "system", username, false, false );

        configurationService.save( configuration );
        return new ResponseEntity<>( HttpStatus.OK );
    }
}
