package com.suncode.plugin.dataviewer.service.export;

import com.suncode.plugin.dataviewer.configuration.format.Format;
import com.suncode.plugin.dataviewer.web.dto.CommentDto;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.xssf.streaming.SXSSFCell;
import org.apache.poi.xssf.streaming.SXSSFRow;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;

public class CellFactory
{
    private final CellStyleFactory cellStyleFactory;

    private CellFactory( SXSSFWorkbook workbook )
    {
        this.cellStyleFactory = CellStyleFactory.create( workbook );
    }

    public static CellFactory create( SXSSFWorkbook workbook )
    {
        return new CellFactory( workbook );
    }

    public SXSSFCell createHeaderCell( SXSSFRow row, int columnIndex, String value )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellStyle( cellStyleFactory.createHeaderCellStyle() );
        itemCell.setCellValue( new XSSFRichTextString( value ) );
        return itemCell;
    }

    public void createCellByType( SXSSFRow row, int columnIndex, Object value, String stringValue, Format format )
    {
        if ( value instanceof Number )
        {
            try
            {
                createCell( row, columnIndex, new BigDecimal( stringValue ), format );
            }
            catch ( Exception e )
            {
                createCell( row, columnIndex, stringValue, format );
            }
        }
        else if ( value instanceof Date )
        {
            createCell( row, columnIndex, (Date) value );
        }
        else if ( value instanceof Calendar )
        {
            createCell( row, columnIndex, (Calendar) value );
        }
        else
        {
            createCell( row, columnIndex, stringValue, format );
        }
    }

    public void createCell( SXSSFRow row, Integer columnIndex, String value )
    {
        createCell( row, columnIndex, value, null );
    }

    public void createCell( SXSSFRow row, Integer columnIndex, String value, Format format )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellStyle( cellStyleFactory.createCellStyle( format ) );

        String replacedValue = value
            .replace( ",", "." )
            .replace( " ", "" );

        if ( NumberUtils.isCreatable( replacedValue ) && replacedValue.matches( "^-?(0|[1-9][0-9]*)(\\.[0-9]+)?$" ) && replacedValue.length() <= 16 )
        {
            Number number = toNumber( replacedValue );
            itemCell.setCellValue( number.doubleValue() );
            return;
        }

        itemCell.setCellValue( new XSSFRichTextString( value ) );
    }

    private Number toNumber( String sTest )
    {
        if ( StringUtils.isBlank( sTest ) )
        {
            return null;
        }

        try
        {
            return Integer.valueOf( sTest );
        }
        catch ( NumberFormatException ignored )
        {
        }

        try
        {
            return Long.valueOf( sTest );
        }
        catch ( NumberFormatException ignored )
        {
        }

        try
        {
            return Float.valueOf( sTest );
        }
        catch ( NumberFormatException ignored )
        {
        }

        try
        {
            return Double.valueOf( sTest );
        }
        catch ( NumberFormatException ignored )
        {
        }

        return null;
    }

    private void createCell( SXSSFRow row, Integer columnIndex, Number value, Format format )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellValue( Double.parseDouble( value.toString() ) );
        itemCell.setCellStyle( cellStyleFactory.createCellStyle( format ) );
    }

    private void createCell( SXSSFRow row, Integer columnIndex, Date value )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellValue( value );
        itemCell.setCellStyle( cellStyleFactory.createDateCellStyle() );
    }

    private void createCell( SXSSFRow row, Integer columnIndex, Calendar value )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellValue( value );
        itemCell.setCellStyle( cellStyleFactory.createDateCellStyle() );
    }

    public void createCommentContentCell( SXSSFRow row, Integer columnIndex, CommentDto comment )
    {
        SXSSFCell itemCell = row.createCell( columnIndex );
        itemCell.setCellValue( new XSSFRichTextString( comment.getContent() ) );
        itemCell.setCellStyle( cellStyleFactory.createCommentStyle( comment ) );
    }
}
