package com.suncode.plugin.dataviewer.service.configuration;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.suncode.plugin.dataviewer.configuration.Configuration;
import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.service.persmission.PermissionService;
import com.suncode.plugin.framework.PluginStore;
import com.suncode.plugin.framework.web.WebFragmentsManager;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

@Slf4j
@Service
public class ConfigurationService
{
    public static final String FILENAME = "config.json";

    private final ObjectMapper mapper = JsonMapper.builder().enable( MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS ).build();

    @Autowired
    private WebFragmentsManager webFragmentsManager;

    @Autowired
    private ConfigurationCache configurationCache;

    @Autowired
    private PluginStore store;

    @Autowired
    private PermissionService permissionService;

    @PostConstruct
    private void init()
    {
        mapper.setSerializationInclusion( JsonInclude.Include.NON_NULL );
    }

    @SneakyThrows
    public Configuration getConfiguration()
    {
        try
        {
            String configurationJson = configurationCache.get();
            if ( StringUtils.isBlank( configurationJson ) )
            {
                configurationJson = "{}";
            }
            mapper.configure( DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            return mapper.readValue( configurationJson, Configuration.class );
        }
        catch ( Exception ex )
        {
            throw new ConfigurationLoadingException( ex );
        }
    }

    public void save( Configuration configuration )
                    throws IOException
    {
        String json = mapper.writerWithDefaultPrettyPrinter().writeValueAsString( configuration );
        try (InputStream stream = new ByteArrayInputStream( json.getBytes( StandardCharsets.UTF_8 ) ))
        {
            store.store( FILENAME, stream, true );
            configurationCache.invalidate();
            refreshMenus();
        }
    }

    private void refreshMenus()
    {
        webFragmentsManager.refreshInterface();
    }

    public List<Menu> getMenusAvailableForUser()
    {
        return getConfiguration()
            .getMenus()
            .stream()
            .filter( menu -> permissionService.hasPermissionToMenu( menu.getId() ) )
            .toList();
    }
}
