package com.suncode.plugin.dataviewer.web.api;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.suncode.plugin.dataviewer.web.dto.CellValueDto;
import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.configuration.View;
import com.suncode.plugin.dataviewer.configuration.format.DoubleFormat;
import com.suncode.plugin.dataviewer.configuration.format.Format;
import com.suncode.plugin.dataviewer.configuration.format.IntegerFormat;
import com.suncode.plugin.dataviewer.configuration.format.TimestampFormat;
import com.suncode.plugin.dataviewer.service.datasupplier.DataSupplierService;
import com.suncode.plugin.dataviewer.service.export.ExportRequest;
import com.suncode.plugin.dataviewer.service.export.ExportService;
import com.suncode.plugin.dataviewer.service.persmission.PermissionService;
import com.suncode.plugin.dataviewer.web.api.util.ConfigurationHelper;
import com.suncode.plugin.dataviewer.web.dto.DataResultDto;
import com.suncode.plugin.dataviewer.web.dto.SorterDto;
import com.suncode.pwfl.search.CountedResult;
import com.suncode.pwfl.search.Pagination;

@Controller
@RequestMapping( "api/data" )
public class ExportController
{
    @Autowired
    private ExportService exportService;

    @Autowired
    private PermissionService permissionService;

    @Autowired
    private ConfigurationHelper configurationHelper;

    @Autowired
    private DataSupplierService dataSupplierService;

    private ObjectMapper mapper = new ObjectMapper();

    @RequestMapping( "{viewId}/excel" )
    public HttpEntity<byte[]> excel( @PathVariable String viewId, @RequestParam( required = false ) String parameters,
                                     SorterDto sorter )
                    throws IOException
    {
        Menu menu = configurationHelper.findMenuByViewId( viewId );
        permissionService.validatePermission( menu.getId() );
        View view = configurationHelper.findView( viewId, menu );

        Map<String, String> params = mapper.readValue( parameters, new TypeReference<Map<String, String>>()
        {
        } );

        DataResultDto dataResult = dataSupplierService
            .getData( menu.getId(), view, params, Pagination.create( sorter.getSorter(), 0, Integer.MAX_VALUE ),
                      Arrays.asList( IntegerFormat.class, DoubleFormat.class, TimestampFormat.class ) );

        List<Map<String, Object>> rowsToExport = new ArrayList<>();
        for ( Map<String, CellValueDto> valueMap : dataResult.getData() )
        {
            Map<String, Object> rowToExport = new HashMap<>();
            for ( Map.Entry<String, CellValueDto> entry : valueMap.entrySet() )
            {
                rowToExport.put( entry.getKey(), sanitizeValue( entry.getValue().getValue() ) );
            }
            rowsToExport.add( rowToExport );
        }

        CountedResult<Map<String, Object>> summaryResult =
            getSummaryData( sorter, view, params,
                            Arrays.asList( IntegerFormat.class, DoubleFormat.class, TimestampFormat.class ) );

        ExportRequest exportRequest = new ExportRequest( view, rowsToExport, dataResult.getComments(), summaryResult.getData() );
        byte[] bytes = exportService.export( exportRequest );

        HttpHeaders header = new HttpHeaders();
        header.setContentType( MediaType.APPLICATION_OCTET_STREAM );
        header.set( "Content-Disposition",
                    "attachment; filename=export_" + view.getId() + ".xlsx" );
        header.setContentLength( bytes.length );

        return new HttpEntity<>( bytes, header );
    }

    private Object sanitizeValue( Object value )
    {
        if ( value == null )
        {
            return null;
        }

        if ( !(value instanceof String) )
        {
            return value;
        }

        return Jsoup.clean( (String) value, Safelist.simpleText() );
    }

    private CountedResult<Map<String, Object>> getSummaryData( SorterDto sorter, View view, Map<String, String> params,
                                                               List<Class<? extends Format>> skipFormatting )
    {
        if ( view.getSummary() != null )
        {
            return dataSupplierService
                .getSummaryData( view, params,
                                 Pagination.create( sorter.getSorter(), 0, Integer.MAX_VALUE ), skipFormatting );
        }
        else
        {
            return new CountedResult<>( 0, new LinkedList<>() );
        }
    }
}
