package com.suncode.plugin.dataviewer.web.api;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import com.suncode.plugin.dataviewer.configuration.SourceData;
import com.suncode.plugin.dataviewer.web.dto.CellValueDto;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.plugin.dataviewer.configuration.Input;
import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.configuration.View;
import com.suncode.plugin.dataviewer.service.datasupplier.DataSupplierService;
import com.suncode.plugin.dataviewer.service.persmission.PermissionService;
import com.suncode.plugin.dataviewer.web.api.util.ConfigurationHelper;
import com.suncode.plugin.dataviewer.web.dto.DataResultDto;
import com.suncode.plugin.dataviewer.web.dto.SorterDto;
import com.suncode.plugin.dataviewer.web.util.CacheUtil;
import com.suncode.pwfl.search.CountedResult;
import com.suncode.pwfl.search.Pagination;
import com.suncode.pwfl.search.SortDirection;
import com.suncode.pwfl.search.Sorter;
import com.suncode.pwfl.web.support.ajax.EntityRestResult;

@Controller
@RequestMapping( "api/data" )
public class DataController
{
    @Autowired
    private DataSupplierService dataSupplierService;

    @Autowired
    private PermissionService permissionService;

    @Autowired
    private ConfigurationHelper configurationHelper;

    private ObjectMapper mapper = new ObjectMapper();

    @RequestMapping( value = "{viewId}", method = RequestMethod.GET )
    public @ResponseBody ResponseEntity<?> data( @PathVariable String viewId,
                                                 @RequestParam( required = false ) String parameters,
                                                 @RequestParam Integer start, @RequestParam Integer limit,
                                                 SorterDto sorter, HttpServletResponse response )
        throws IOException
    {
        CacheUtil.noCache( response );

        Menu menu = configurationHelper.findMenuByViewId( viewId );
        permissionService.validatePermission( menu.getId() );
        View view = configurationHelper.findView( viewId, menu );

        Map<String, String> params = mapper.readValue( parameters, new TypeReference<>() {} );

        DataResultDto data = dataSupplierService.getData( menu.getId(), view, params,
                                                          Pagination.create( sorter.getSorter(), start, limit ),
                                                          new ArrayList<>() );
        return successResultWithFormattedData( data );
    }

    @RequestMapping( value = "{viewId}/summary", method = RequestMethod.GET )
    public @ResponseBody ResponseEntity<?> summary( @PathVariable String viewId,
                                                    @RequestParam( required = false ) String parameters,
                                                    @RequestParam Integer start, @RequestParam Integer limit,
                                                    SorterDto sorter, HttpServletResponse response )
        throws IOException
    {
        CacheUtil.noCache( response );

        View view = findView( viewId );

        Map<String, String> params = mapper.readValue( parameters, new TypeReference<>() {} );
        CountedResult<Map<String, Object>> summary = dataSupplierService
            .getSummaryData( view, params, Pagination.create( sorter.getSorter(), start, limit ), new ArrayList<>() );
        return successResult( summary );
    }

    @RequestMapping( value = "{viewId}/{inputId}/list", method = RequestMethod.GET )
    public @ResponseBody ResponseEntity<?> list( @PathVariable String viewId,
                                                 @PathVariable String inputId,
                                                 @RequestParam( "query" ) String query, HttpServletResponse response )
    {
        CacheUtil.noCache( response );

        Input input = getInput( viewId, inputId );

        Pagination pagination = createSorter();
        Map<String, String> parameters = new HashMap<>();
        parameters.put( "query", query );

        CountedResult<Map<String, Object>> listData = dataSupplierService
            .getFromSource( input.getSource(), parameters, pagination );
        return successResult( listData );
    }

    @RequestMapping( value = "{viewId}/{inputId}/defaultValue", method = RequestMethod.GET )
    public @ResponseBody ResponseEntity<?> defaultValue( @PathVariable String viewId,
                                                         @PathVariable String inputId,
                                                         HttpServletResponse response )
    {
        CacheUtil.noCache( response );

        Input input = getInput( viewId, inputId );
        Pagination pagination = createSorter();

        SourceData defaultValueSourceData = input.getDynamicDefaultValue();
        if ( defaultValueSourceData == null )
        {
            throw new IllegalArgumentException(
                String.format( "Input id=%s in view viewId=%s has no source configured for its default value", viewId, inputId ) );
        }

        CountedResult<Map<String, Object>> data = dataSupplierService.getFromSource( defaultValueSourceData, new HashMap<>(), pagination );

        if ( data.getData().size() == 0 )
        {
            throw new IllegalArgumentException(
                String.format( "Datasource return 0 values as default value for viewId=%s and inputId=%s", viewId,
                               inputId ) );
        }

        Map<String, Object> dataSourceResult = data.getData().get( 0 );
        if ( dataSourceResult.size() != 1 )
        {
            throw new IllegalArgumentException(
                String.format( "Datasource return %s values as default value for viewId=%s and inputId=%s", dataSourceResult.size(), viewId,
                               inputId ) );
        }
        Object defaultValue = data.getData().get( 0 ).values().iterator().next();
        return new ResponseEntity<>( defaultValue, HttpStatus.OK );
    }

    private Input getInput( String viewId, String inputId )
    {
        View view = findView( viewId );
        return findInput( inputId, view );
    }

    private View findView( String viewId )
    {
        Menu menu = configurationHelper.findMenuByViewId( viewId );
        permissionService.validatePermission( menu.getId() );
        return configurationHelper.findView( viewId, menu );
    }

    private Pagination createSorter()
    {
        Sorter sorter = new Sorter();
        sorter.setProperty( "value" );
        sorter.setDirection( SortDirection.ASC );
        return Pagination.create( sorter, 0, Integer.MAX_VALUE );
    }

    private Input findInput( String inputId, View view )
    {
        return view.getInputs().stream()
            .filter( input -> input.getId().equals( inputId ) )
            .findFirst()
            .orElseThrow( RuntimeException::new );
    }

    private ResponseEntity<?> successResultWithFormattedData( CountedResult<Map<String, CellValueDto>> data )
    {
        EntityRestResult<CountedResult<Map<String, CellValueDto>>> result = new EntityRestResult<>();
        result.setSuccess( true );
        result.setMessage( null );
        result.setEntity( data );
        return new ResponseEntity<>( result, HttpStatus.OK );
    }

    private ResponseEntity<?> successResult( CountedResult<Map<String, Object>> data )
    {
        EntityRestResult<CountedResult<Map<String, Object>>> result = new EntityRestResult<>();
        result.setSuccess( true );
        result.setMessage( null );
        result.setEntity( data );
        return new ResponseEntity<>( result, HttpStatus.OK );
    }
}
