package com.suncode.plugin.dataviewer.web.api;

import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.configuration.View;
import com.suncode.plugin.dataviewer.configuration.ViewNotExistsException;
import com.suncode.plugin.dataviewer.service.comment.Comment;
import com.suncode.plugin.dataviewer.service.comment.CommentKey;
import com.suncode.plugin.dataviewer.service.comment.CommentService;
import com.suncode.plugin.dataviewer.service.configuration.ConfigurationService;
import com.suncode.plugin.dataviewer.service.persmission.PermissionException;
import com.suncode.plugin.dataviewer.web.dto.CommentDto;
import com.suncode.plugin.dataviewer.web.dto.UserDto;
import com.suncode.pwfl.administration.user.UserContext;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

@Controller
@RequestMapping( "api/comments" )
public class CommentController
{
    @Autowired
    private CommentService commentService;
    
    @Autowired
    private ConfigurationService configurationService;
    
    @ResponseStatus( HttpStatus.OK )
    @RequestMapping( value = "{viewId}/{rowId}", method = RequestMethod.POST )
    public void create( @RequestBody CommentDto commentDto, @PathVariable String viewId,
                        @PathVariable String rowId )
    {
        Menu menu = findMenuByViewId( viewId );
        View view = findView( viewId, menu );
        CommentKey key = new CommentKey( menu.getId(), viewId, view.getSource().getType(), view.getSource().getId(), rowId );

        commentDto.setAuthor( UserDto.get( UserContext.current().getUser().getUserName() ) );
        Comment comment = commentDto.buildDomain();
        comment.setKey( key );
        comment.setAuthor( loggedUser() ); 
        commentService.addComment( comment );
    }
    
    private Menu findMenuByViewId( String viewId )
    {
        return configurationService.getConfiguration().getMenus().stream()
                        .filter( menu -> menu.getViews().stream()
                                        .anyMatch( view -> view.getId().equals( viewId ) ) )
                        .findFirst()
                        .orElseThrow( () -> new ViewNotExistsException( viewId ) );
    }

    private View findView( String viewId, Menu menu )
    {
        return menu.getViews().stream()
                        .filter( viewItem -> viewItem.getId().equals( viewId ) )
                        .findFirst()
                        .orElseThrow( () -> new ViewNotExistsException( viewId ) );
    }

    @Transactional
    @ResponseStatus( HttpStatus.OK )
    @RequestMapping( value = "{commentId}", method = RequestMethod.DELETE )
    public void delete( @PathVariable Long commentId )
    {
        Comment comment = commentService.getComment( commentId );
        if( isLoggedUser( comment.getAuthor() ) )
        {
            commentService.deleteComment( comment );
        }
        else
        {
            throw new PermissionException();
        }
    }

    @ResponseStatus( HttpStatus.OK )
    @RequestMapping( method = RequestMethod.PUT )
    public void change( @RequestBody CommentDto commentDto )
    {
        if( isLoggedUser( commentDto.getAuthor().getUserName() ) )
        {
            commentService.changeComment( commentDto.buildDomain() );
        }
        else
        {
            throw new PermissionException();
        }
    }
    
    private boolean isLoggedUser( String userName )
    {
        if( StringUtils.isBlank( userName ) )
        {
            return false;
        }
        
        return userName.equals( loggedUser() );
    }
    
    private String loggedUser()
    {
        if( UserContext.isActive() )
        {
            return UserContext.current().getUser().getUserName();
        }
        
        return null;
    }
}
