package com.suncode.plugin.dataviewer.service.persmission;

import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.configuration.Permissions;
import com.suncode.plugin.dataviewer.service.configuration.ConfigurationService;
import com.suncode.plugin.framework.support.Condition;
import com.suncode.plugin.framework.support.ConditionSupport;
import com.suncode.pwfl.administration.user.User;
import com.suncode.pwfl.administration.user.UserService;
import com.suncode.pwfl.transaction.TransactionManagerFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import java.util.List;
import java.util.Optional;

@Slf4j
@Service
public class PermissionService
{
    @Autowired
    private UserService userService;

    @Autowired
    private ConfigurationService configurationService;

    @Autowired
    private PermissionCache permissionCache;

    public void validatePermission( String menuId )
    {
        if ( !hasPermissionToMenu( menuId ) )
        {
            throw new PermissionException();
        }
    }

    public boolean hasPermissionToMenu( String menuId )
    {
        return createCondition( menuId ).isFulfilled();
    }

    public Condition createCondition( String menuId )
    {
        return new ConditionSupport()
        {
            @Override
            public boolean isFulfilled()
            {
                String userName = UserContext.userName();
                try
                {
                    MenuUserPair key = MenuUserPair.builder()
                                    .menuId( menuId )
                                    .username( userName )
                                    .build();
                    Optional<Boolean> optionalBoolean = permissionCache.get( key );
                    if ( optionalBoolean.isPresent() )
                    {
                        return optionalBoolean.get();
                    }
                    else
                    {
                        return isFullfilled( key );
                    }
                }
                catch ( Exception ex )
                {
                    log.error( "Error while checking permissions for menu " + menuId + "  for user " + userName,
                               ex );
                    return false;
                }
            }
        };
    }

    private Boolean isFullfilled( MenuUserPair pair )
    {
        Menu menu = getMenu( pair );
        Permissions permissions = menu.getPermissions();
        if ( permissions == null )
        {
            return false;
        }
        
        List<String> usernames = permissions.getUsers();
        List<String> groupNames = permissions.getGroups();

        Optional<Boolean> optionalValue = checkCachable( pair, usernames, groupNames );
        if ( optionalValue.isPresent() )
        {
            Boolean value = optionalValue.get();
            permissionCache.put( pair, value );
            return value;
        }

        Boolean weakValue = checkWeakCachable( pair, groupNames );
        permissionCache.putTemporarily( pair, weakValue );
        return weakValue;
    }

    private Optional<Boolean> checkCachable( MenuUserPair pair, List<String> usernames, List<String> groupNames )
    {
        if ( usernames.isEmpty() && groupNames.isEmpty() )
        {
            return Optional.of( false );
        }

        boolean usernameMatches = usernames.stream()
                        .anyMatch( username -> username.equals( pair.getUsername() ) );
        if ( usernameMatches )
        {
            return Optional.of( true );
        }

        return Optional.empty();
    }

    private Boolean checkWeakCachable( MenuUserPair pair, List<String> groupNames )
    {
        PlatformTransactionManager manager =
                        TransactionManagerFactory.getHibernateTransactionManager();
        return new TransactionTemplate( manager ).execute( status -> {
            User user = userService.getUser( pair.getUsername(), User.JOIN_GROUPS );
            return user.getGroups().stream()
                            .anyMatch( group -> groupNames.stream()
                                            .anyMatch( permissionGroupName -> permissionGroupName
                                                            .equals( group.getName() ) ) );
        } );
    }

    private Menu getMenu( MenuUserPair pair )
    {
        return configurationService.getConfiguration().getMenus().stream()
                        .filter( menuItem -> menuItem.getId().equals( pair.getMenuId() ) )
                        .findFirst()
                        .orElseThrow( IllegalArgumentException::new );
    }
}
