package com.suncode.plugin.dataviewer.service.export;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.suncode.plugin.dataviewer.configuration.Output;
import com.suncode.plugin.dataviewer.configuration.Summary;
import com.suncode.plugin.dataviewer.configuration.View;
import com.suncode.plugin.dataviewer.configuration.mapping.SummaryOutputMapping;
import com.suncode.plugin.dataviewer.support.TranslatorHolder;
import com.suncode.plugin.dataviewer.web.dto.CommentDto;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.streaming.SXSSFRow;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

@Service
public class ExportService
{
    private final ObjectMapper mapper = new ObjectMapper();

    private CellFactory cellFactory;

    public byte[] export( ExportRequest exportRequest )
        throws IOException
    {
        try ( SXSSFWorkbook workbook = new SXSSFWorkbook() )
        {
            this.cellFactory = CellFactory.create( workbook );
            buildExcel( exportRequest, workbook );

            try ( ByteArrayOutputStream stream = new ByteArrayOutputStream() )
            {
                workbook.write( stream );
                return stream.toByteArray();
            }
        }
    }

    private void buildExcel( ExportRequest exportRequest, SXSSFWorkbook workbook )
    {
        View view = exportRequest.getView();
        SXSSFSheet sheet = workbook.createSheet( view.getName() );
        sheet.trackAllColumnsForAutoSizing();

        createHeaderCells( view, sheet );

        int lastIndex = fillDataRows( exportRequest, sheet );
        fillSummaryRows( exportRequest, sheet, lastIndex );
        autoSizeColumns( view, sheet );
    }

    private void createHeaderCells( View view, SXSSFSheet sheet )
    {
        SXSSFRow row = sheet.createRow( 0 );
        cellFactory.createHeaderCell( row, 0, TranslatorHolder.get().getMessage( "data.viewer.item" ) );

        List<Output> outputs = getVisibleOutputs( view );
        for ( int i = 0; i < outputs.size(); i++ )
        {
            int columnIndex = i + 1;
            cellFactory.createHeaderCell( row, columnIndex, outputs.get( i ).getName() );
        }

        if ( view.isComments() )
        {
            cellFactory.createHeaderCell( row, 1 + outputs.size(),
                                          TranslatorHolder.get().getMessage( "data.viewer.comments" )
            );
            sheet.addMergedRegion( new CellRangeAddress( 0, 0, 1 + outputs.size(), 2 + outputs.size() ) );

        }
    }

    private int fillDataRows( ExportRequest exportRequest, SXSSFSheet sheet )
    {
        View view = exportRequest.getView();
        List<Map<String, Object>> dataRows = exportRequest.getData();
        Map<String, List<CommentDto>> commentsMap = exportRequest.getComments();

        int rowIndex = 1;
        List<Output> outputs = getVisibleOutputs( view );
        Output primaryKeyOutput = getPrimaryKeyOutput( view );
        int commentDateColumnIndex = 1 + outputs.size();
        int commentContentColumnIndex = 2 + outputs.size();
        for ( int dataIndex = 0; dataIndex < dataRows.size(); dataIndex++ )
        {
            SXSSFRow row = sheet.createRow( rowIndex );

            cellFactory.createCell( row, 0, String.valueOf( dataIndex + 1 ) );

            Map<String, Object> dataRow = dataRows.get( dataIndex );
            createDataCells( outputs, row, dataRow );

            Object rowIdObject = dataRow.get( primaryKeyOutput.getAlias() );
            if (rowIdObject != null)
            {
                List<CommentDto> comments = commentsMap.get( rowIdObject.toString() );
                if ( comments != null )
                {
                    for ( CommentDto comment : comments )
                    {
                        SXSSFRow commentRow = sheet.getRow( rowIndex );
                        if ( commentRow == null )
                        {
                            commentRow = sheet.createRow( rowIndex );
                        }

                        cellFactory.createCell( commentRow, commentDateColumnIndex, formatCommentDate( new Date( comment.getDate() ) ) );
                        cellFactory.createCommentContentCell( commentRow, commentContentColumnIndex, comment );
                        rowIndex++;
                    }

                    if ( comments.size() > 1 )
                    {
                        mergeDataCells( sheet, rowIndex, outputs, comments );
                    }
                }
                else
                {
                    rowIndex++;
                }
            }
            else
            {
                rowIndex++;
            }
        }

        return rowIndex;
    }

    private void fillSummaryRows( ExportRequest exportRequest, SXSSFSheet sheet, int lastIndex )
    {
        View view = exportRequest.getView();
        Summary summary = view.getSummary();
        List<Output> outputs = getVisibleOutputs( view );
        int rowIndex = lastIndex;

        for ( Map<String, Object> summaryRow : exportRequest.getSummary() )
        {
            SXSSFRow row = sheet.createRow( rowIndex );

            cellFactory.createCell( row, 0, StringUtils.EMPTY );

            Map<String, Object> dataRow = new HashMap<>();

            for ( Output output : outputs )
            {
                Optional<String> optionalAlias = summary.getOutputMappings().stream()
                    .filter( mapping -> mapping.getOutputId().equals( output.getId() ) )
                    .map( SummaryOutputMapping::getAlias )
                    .findFirst();
                if ( optionalAlias.isPresent() )
                {
                    String alias = optionalAlias.get();
                    dataRow.put( output.getAlias(), summaryRow.getOrDefault( alias, StringUtils.EMPTY ) );
                }
                else
                {
                    dataRow.put( output.getAlias(), StringUtils.EMPTY );
                }
            }

            createDataCells( outputs, row, dataRow );
            rowIndex++;
        }
    }

    private void createDataCells( List<Output> outputs, SXSSFRow row, Map<String, Object> dataRow )
    {
        for ( int outputIndex = 0; outputIndex < outputs.size(); outputIndex++ )
        {
            Output output = outputs.get( outputIndex );
            String alias = output.getAlias();
            Object value = dataRow.get( alias );
            String stringValue = formatDataValue( value );
            int columnIndex = outputIndex + 1;

            cellFactory.createCellByType( row, columnIndex, value, stringValue, output.getFormat() );
        }
    }

    private void mergeDataCells( SXSSFSheet sheet, int rowIndex, List<Output> outputs, List<CommentDto> comments )
    {
        int bound = outputs.size() + 1;
        for ( int columnIndexToMerge = 0; columnIndexToMerge < bound; columnIndexToMerge++ )
        {
            sheet.addMergedRegion( new CellRangeAddress( rowIndex - comments.size(), rowIndex - 1, columnIndexToMerge,
                                                         columnIndexToMerge ) );
        }
    }

    private Output getPrimaryKeyOutput( View view )
    {
        return view.getOutputs().stream()
                        .filter( Output::isPrimaryKey )
                        .findFirst()
                        .orElseThrow( RuntimeException::new );
    }

    private List<Output> getVisibleOutputs( View view )
    {
        return view.getOutputs().stream()
                        .filter( output -> !output.isHidden() )
                        .collect( Collectors.toList() );
    }

    private String formatCommentDate( Date date )
    {
        return new SimpleDateFormat( "yyyy-MM-dd" ).format( date );
    }

    @SneakyThrows
    private String formatDataValue( Object value )
    {
        if ( value == null )
        {
            return StringUtils.EMPTY;
        }
        else
        {
            if ( value instanceof Date )
            {
                return mapper.writeValueAsString( value ).replace( "\"", StringUtils.EMPTY );
            }
            else
            {
                return value.toString();
            }
        }
    }

    private void autoSizeColumns( View view, SXSSFSheet sheet )
    {
        int columnsCount = 1 + getVisibleOutputs( view ).size();
        if ( view.isComments() )
        {
            columnsCount += 2;
        }

        IntStream.range( 0, columnsCount ).forEach( sheet::autoSizeColumn );
    }
}
