package com.suncode.plugin.dataviewer.service.export;

import com.suncode.plugin.dataviewer.configuration.format.DoubleFormat;
import com.suncode.plugin.dataviewer.configuration.format.Format;
import com.suncode.plugin.dataviewer.web.dto.CommentDto;
import com.suncode.pwfl.util.constants.DateConstants;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFFont;

import javax.annotation.Nullable;

public class CellStyleFactory
{
    private final SXSSFWorkbook workbook;

    private CellStyleFactory( SXSSFWorkbook workbook )
    {
        this.workbook = workbook;
    }

    public static CellStyleFactory create( SXSSFWorkbook workbook )
    {
        return new CellStyleFactory( workbook );
    }

    public CellStyle createHeaderCellStyle()
    {
        CellStyle style = workbook.createCellStyle();

        style.setFillPattern( FillPatternType.FINE_DOTS );
        style.setFillBackgroundColor( HSSFColor.HSSFColorPredefined.BLUE_GREY.getIndex() );
        style.setAlignment( HorizontalAlignment.CENTER );
        style.setVerticalAlignment( VerticalAlignment.CENTER );

        Font font = createFontStyle();
        style.setFont( font );

        return style;
    }

    private Font createFontStyle()
    {
        Font font = workbook.createFont();
        font.setBold( true );
        font.setColor( HSSFColor.HSSFColorPredefined.WHITE.getIndex() );
        return font;
    }

    public CellStyle createCellStyle()
    {
        return createCellStyle( null );
    }

    public CellStyle createCellStyle( Format format )
    {
        CellStyle style = workbook.createCellStyle();
        style.setAlignment( HorizontalAlignment.CENTER );
        style.setVerticalAlignment( VerticalAlignment.CENTER );
        setDataFormat( style, format );
        return style;
    }

    private void setDataFormat( CellStyle style, @Nullable Format format )
    {
        if ( format == null )
        {
            return;
        }

        String precisionFormat = getPrecisionFormat( format );
        if ( precisionFormat == null )
        {
            return;
        }

        DataFormat dataFormat = workbook.createDataFormat();
        style.setDataFormat( dataFormat.getFormat( precisionFormat ) );
    }

    private String getPrecisionFormat( Format format )
    {
        if ( !(format instanceof DoubleFormat) )
        {
            return null;
        }

        DoubleFormat doubleFormat = (DoubleFormat) format;
        Integer decimalPrecision = doubleFormat.getDecimalPrecision();

        if ( decimalPrecision == null || decimalPrecision < 1 )
        {
            return null;
        }

        return "0." + "0".repeat( decimalPrecision );
    }

    public CellStyle createDateCellStyle()
    {
        CellStyle style = workbook.createCellStyle();
        style.setAlignment( HorizontalAlignment.CENTER );
        style.setVerticalAlignment( VerticalAlignment.CENTER );
        CreationHelper createHelper = workbook.getCreationHelper();
        style.setDataFormat( createHelper.createDataFormat().getFormat( DateConstants.DATE_TIME_FORMAT ) );
        return style;
    }

    public CellStyle createCommentStyle( CommentDto comment )
    {
        XSSFCellStyle style = (XSSFCellStyle) workbook.createCellStyle();
        
        style.setAlignment( HorizontalAlignment.CENTER );
        style.setVerticalAlignment( VerticalAlignment.CENTER );

        style.setFont( createFontForComment( comment ) );

        return style;
    }

    private XSSFFont createFontForComment( CommentDto comment )
    {
        XSSFFont font = (XSSFFont) workbook.createFont();

        if ( Boolean.TRUE.equals( comment.getBold() ) )
        {
            font.setBold( true );
        }

        if ( Boolean.TRUE.equals( comment.getItalic() ) )
        {
            font.setItalic( true );
        }

        if ( Boolean.TRUE.equals( comment.getUnderline() ) )
        {
            font.setUnderline( Font.U_SINGLE );
        }

        if ( StringUtils.isNotBlank( comment.getColor() ) )
        {
            font.setColor( new XSSFColor( comment.getColor().getBytes() ) );
        }

        return font;
    }
}
