package com.suncode.plugin.dataviewer.scheduledTask;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.plusmpm.util.CoreTools;
import com.suncode.plugin.dataviewer.configuration.*;
import com.suncode.plugin.dataviewer.configuration.format.DoubleFormat;
import com.suncode.plugin.dataviewer.configuration.format.IntegerFormat;
import com.suncode.plugin.dataviewer.configuration.format.TimestampFormat;
import com.suncode.plugin.dataviewer.service.datasupplier.DataSupplierService;
import com.suncode.plugin.dataviewer.service.export.ExportRequest;
import com.suncode.plugin.dataviewer.service.export.ExportService;
import com.suncode.plugin.dataviewer.util.MailUtils;
import com.suncode.plugin.dataviewer.web.api.util.ConfigurationHelper;
import com.suncode.plugin.dataviewer.web.dto.CellValueDto;
import com.suncode.plugin.dataviewer.web.dto.DataResultDto;
import com.suncode.plugin.dataviewer.web.dto.InputFilterDto;
import com.suncode.plugin.dataviewer.web.dto.SorterDto;
import com.suncode.pwfl.administration.scheduledtask.ScheduledTaskDefinitionBuilder;
import com.suncode.pwfl.administration.scheduledtask.annotation.ScheduledTask;
import com.suncode.pwfl.component.annotation.Define;
import com.suncode.pwfl.component.annotation.Param;
import com.suncode.pwfl.core.type.Types;
import com.suncode.pwfl.search.CountedResult;
import com.suncode.pwfl.search.Pagination;
import com.suncode.pwfl.search.SortDirection;
import org.apache.commons.lang3.time.DateUtils;
import org.joda.time.LocalDateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.jsoup.Jsoup;
import org.jsoup.safety.Safelist;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;

import jakarta.mail.*;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.*;

@ScheduledTask
public class SendDataViewerReportTask
{
    @Autowired
    private ExportService exportService;
    
    @Autowired
    private ConfigurationHelper configurationHelper;

    @Autowired
    private DataSupplierService dataSupplierService;

    private static final ObjectMapper mapper = new ObjectMapper();
    
    @Define
    public void definition( ScheduledTaskDefinitionBuilder builder ) {
        builder
                        .id( "send-data-viewer-report" )
                        .name( "data.viewer.scheduled.task.send.report.name" )
                        .description( "data.viewer.scheduled.task.send.report.desc" )
                        .cancelable()
                        .parameter()
                            .id( "mailTitle" )
                            .name( "data.viewer.scheduled.task.mail.title" )
                            .description( "data.viewer.scheduled.task.mail.title.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "mailPath" )
                            .name( "data.viewer.scheduled.task.mail.path" )
                            .description( "data.viewer.scheduled.task.mail.path.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "mailRecipients" )
                            .name( "data.viewer.scheduled.task.mail.recipients" )
                            .description( "data.viewer.scheduled.task.mail.recipients.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "JSONParams" )
                            .name( "data.viewer.scheduled.task.json.params" )
                            .description( "data.viewer.scheduled.task.json.params.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "viewId" )
                            .name( "data.viewer.scheduled.task.view.id" )
                            .description( "data.viewer.scheduled.task.view.id.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "sortProperty" )
                            .name( "data.viewer.scheduled.task.filter.sort.property" )
                            .description( "data.viewer.scheduled.task.filter.sort.property.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "sortDirection" )
                            .name( "data.viewer.scheduled.task.filter.sort.direction" )
                            .description( "data.viewer.scheduled.task.filter.sort.direction.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "reportFilename" )
                            .name( "data.viewer.scheduled.task.report.filename" )
                            .description( "data.viewer.scheduled.task.report.filename.description" )
                            .type( Types.STRING )
                            .create()
                        .parameter()
                            .id( "reportFileExtension" )
                            .name( "data.viewer.scheduled.task.report.extension" )
                            .description( "data.viewer.scheduled.task.report.extension.description" )
                            .type( Types.STRING )
                            .create();
    }
    
    public void execute(
                    @Param( value = "mailTitle" ) String mailTitle,
                    @Param( value = "mailPath" ) String mailPath,
                    @Param( value = "mailRecipients" ) String mailRecipients,
                    @Param( value = "JSONParams" ) String JSONParams,
                    @Param( value = "viewId" ) String viewId,
                    @Param( value = "sortProperty" ) String sortProperty,
                    @Param( value = "sortDirection" ) String sortDirection,
                    @Param( value = "reportFilename" ) String reportFilename,
                    @Param( value = "reportFileExtension" ) String reportFileExtension )
                    throws IOException, MessagingException, ParseException
    {
        Assert.isTrue( isSupportedExtension( reportFileExtension ), "Unsupported attachment extension" );
        byte[] reportFile = generateReportFile( viewId, JSONParams, sortProperty, sortDirection );
        String template = new String( Files.readAllBytes( Paths.get( mailPath ) ), StandardCharsets.UTF_8 );
        MailUtils.send( mailRecipients, mailTitle, reportFilename + "." + reportFileExtension, template, reportFile );
    }
        
    private byte[] generateReportFile(String viewId, String parameters, String sortProperty, String sortDirection)
                    throws IOException, ParseException
    {
        Menu menu = configurationHelper.findMenuByViewId( viewId );
        View view = configurationHelper.findView( viewId, menu );
    
        List<InputFilterDto> inputFilters = mapper.readValue( parameters, new TypeReference<List<InputFilterDto>>()
        {
        } );
        Map<String, String> params = new HashMap<>();
        for ( InputFilterDto filter : inputFilters )
        {
            params.put( filter.getInputId(), computeFilterValues( filter, view.getInputs() ) );
        }

        Output sortOutput = view.getOutputs()
                        .stream()
                        .filter( output -> output.getId().equals( sortProperty ) )
                        .findFirst()
            .orElseThrow( () -> new IllegalArgumentException( "Incorrect sorting property" ) );

        Assert.isTrue( isProperSortDirection( sortDirection ), "The parameter specifying the sort direction is incorrect" );
        SorterDto sorter = new SorterDto();
        sorter.setDirection( sortDirection );
        sorter.setProperty( sortOutput.getAlias() );

        DataResultDto dataResult = dataSupplierService
            .getData( menu.getId(), view, params, Pagination.create( sorter.getSorter(), 0, Integer.MAX_VALUE ),
                      Arrays.asList( IntegerFormat.class, DoubleFormat.class, TimestampFormat.class ) );

        List<Map<String, Object>> rowsToExport = new ArrayList<>();
        for ( Map<String, CellValueDto> valueMap : dataResult.getData() )
        {
            Map<String, Object> rowToExport = new HashMap<>();
            for ( Map.Entry<String, CellValueDto> entry : valueMap.entrySet() )
            {
                rowToExport.put( entry.getKey(), sanitizeValue( entry.getValue().getValue() ) );
            }
            rowsToExport.add( rowToExport );
        }

        CountedResult<Map<String, Object>> summaryResult = getSummaryData( sorter, view, params );

        ExportRequest exportRequest = new ExportRequest( view, rowsToExport, dataResult.getComments(),
                                                         summaryResult.getData() );

        return exportService.export( exportRequest );
    }

    private String computeFilterValues( InputFilterDto filter, List<Input> inputs )
                    throws ParseException
    {
        Optional<Input> input = inputs
                        .stream()
                        .filter( viewInput -> filter.getInputId().equals( viewInput.getId() ) )
                        .findAny();
        if ( input.isPresent() && input.get().getType() == InputType.DATE )
        {
            String[] DATE_MARKER_PATTERNS = { "yyyy-MM-dd", "yyyy-MM-dd HH:mm" };
            Date date = DateUtils.parseDate( CoreTools.replaceDateMarker( filter.getInputValue() ),
                                             DATE_MARKER_PATTERNS );

            DateTimeFormatter formatter = DateTimeFormat.forPattern( "yyyy-MM-dd HH:mm:ss" );
            return new Long( Timestamp.valueOf( new LocalDateTime( date ).toString( formatter ) ).getTime() )
                            .toString();
        }
        return filter.getInputValue();
    }

    private boolean isProperSortDirection( String sortDirection )
    {
        try
        {
            SortDirection.valueOf( sortDirection );
            return true;
        }
        catch ( IllegalArgumentException e )
        {
            return false;
        }
    }

    private Object sanitizeValue( Object value )
    {
        if ( value == null )
        {
            return null;
        }

        if ( !(value instanceof String) )
        {
            return value;
        }

        return Jsoup.clean( (String) value, Safelist.simpleText() );
    }

    private CountedResult<Map<String, Object>> getSummaryData( SorterDto sorter, View view, Map<String, String> params )
    {
        if ( view.getSummary() != null )
        {
            return dataSupplierService
                .getSummaryData( view, params,
                                 Pagination.create( sorter.getSorter(), 0, Integer.MAX_VALUE ),
                                 Arrays.asList( IntegerFormat.class, DoubleFormat.class, TimestampFormat.class ) );
        }
        else
        {
            return new CountedResult<>( 0, new LinkedList<>() );
        }
    }
    
    private boolean isSupportedExtension(String reportType)
    {
        return reportType.equals( "xlsx" );
    }
}
