package com.suncode.plugin.dataviewer.menu;

import com.google.common.base.Joiner;
import com.suncode.plugin.dataviewer.configuration.Configuration;
import com.suncode.plugin.dataviewer.configuration.Menu;
import com.suncode.plugin.dataviewer.service.configuration.ConfigurationService;
import com.suncode.plugin.dataviewer.service.persmission.PermissionService;
import com.suncode.plugin.framework.FrameworkConstants;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.net.URLEncoder;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class MenuResolver
{
    private static final String LINK_SUFFIX = "viewer";

    @Autowired
    private ConfigurationService configurationService;

    @Autowired
    private PermissionService permissionService;

    public List<DataViewerMenu> resolveMenus( String pluginKey )
    {
        try
        {
            Configuration configuration = configurationService.getConfiguration();
            return configuration.getMenus().stream()
                            .map( menu -> createMenu( menu, pluginKey ) )
                            .collect( Collectors.toList() );
        }
        catch ( Exception ex )
        {
            log.error( "Cannot build any menu tab", ex );
            return new LinkedList<>();
        }
    }

    private DataViewerMenu createMenu( Menu menu, String pluginKey )
    {
        return DataViewerMenu.builder()
                        .id( menu.getId() )
                        .link( () -> buildLink( menu.getId(), pluginKey ) )
                        .label( new I18nCustomLocalizedMessage( menu.getName() ) )
                        .conditions( Arrays.asList( permissionService.createCondition( menu.getId() ) ) )
                        .build();
    }

    @SneakyThrows
    private String buildLink( String id, String pluginKey )
    {
        String encodedKey = URLEncoder.encode( pluginKey, "UTF-8" );
        String encodedId = URLEncoder.encode( id, "UTF-8" );
        String link = Joiner.on( "/" )
                        .join( FrameworkConstants.PLUGIN_SERVLET_PATH, encodedKey, encodedId, LINK_SUFFIX );
        return link + "?decorator=noScripts";
    }
}
