/*
 * Decompiled with CFR 0.152.
 */
package com.suncode.pdfutils.resize;

import com.suncode.pdfutils.exception.CompressionFileSizeNotReachedException;
import com.suncode.pdfutils.exception.NotSupportedImageException;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.cos.COSName;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.PDResources;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.PDXObject;
import org.apache.pdfbox.pdmodel.graphics.image.JPEGFactory;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfResizer {
    private static final Logger log = LoggerFactory.getLogger(PdfResizer.class);
    private double scalar;
    private float compression = 0.75f;
    private String sourcePath;
    private String destinationPath;
    private boolean deleteSource = false;
    private ResizeMethod resizeMethod = ResizeMethod.ULTRA;

    public PdfResizer(String sourcePath, String destinationPath, double scalar) {
        this.sourcePath = sourcePath;
        this.destinationPath = destinationPath;
        this.scalar = scalar;
    }

    public PdfResizer(String sourcePath, String destinationPath, double scalar, float compression) {
        this.sourcePath = sourcePath;
        this.destinationPath = destinationPath;
        this.scalar = scalar;
        this.compression = compression;
    }

    public double getScalar() {
        return this.scalar;
    }

    public void setScalar(double scalar) {
        this.scalar = scalar;
    }

    public float getCompression() {
        return this.compression;
    }

    public void setCompression(float compression) {
        this.compression = compression;
    }

    public String getSourcePath() {
        return this.sourcePath;
    }

    public void setSourcePath(String sourcePath) {
        this.sourcePath = sourcePath;
    }

    public String getDestinationPath() {
        return this.destinationPath;
    }

    public void setDestinationPath(String destinationPath) {
        this.destinationPath = destinationPath;
    }

    public boolean isDeleteSource() {
        return this.deleteSource;
    }

    public void setDeleteSource(boolean deleteSource) {
        this.deleteSource = deleteSource;
    }

    public ResizeMethod getResizeMethod() {
        return this.resizeMethod;
    }

    public void setResizeMethod(ResizeMethod resizeMethod) {
        this.resizeMethod = resizeMethod;
    }

    public void resize() throws IOException, NotSupportedImageException {
        File sourceFile = new File(this.sourcePath);
        PDDocument doc = Loader.loadPDF((File)sourceFile);
        try (PDDocument output = new PDDocument();){
            for (PDPage page : doc.getDocumentCatalog().getPages()) {
                PDResources resources = page.getResources();
                HashMap<String, PDImageXObject> images = new HashMap<String, PDImageXObject>();
                for (COSName cOSName : resources.getXObjectNames()) {
                    PDXObject xObject = resources.getXObject(cOSName);
                    if (!(xObject instanceof PDImageXObject)) continue;
                    images.put(cOSName.getName(), (PDImageXObject)xObject);
                }
                if (images.isEmpty()) {
                    output.addPage(page);
                    continue;
                }
                for (Map.Entry entry : images.entrySet()) {
                    BufferedImage inputImage = this.getInputImage((PDImageXObject)entry.getValue());
                    if (inputImage == null) {
                        doc.save(this.destinationPath);
                        throw new NotSupportedImageException("Pdf zawiera obrazy w nieobs\u0142ugiwanym formacie. Skalowanie zostaje pomini\u0119te.");
                    }
                    int scaledWidth = (int)((double)inputImage.getWidth() * this.scalar);
                    int scaledHeight = (int)((double)inputImage.getHeight() * this.scalar);
                    BufferedImage scaledImage = Scalr.resize((BufferedImage)inputImage, (Scalr.Method)this.resizeMethod.getMethod(), (int)scaledWidth, (int)scaledHeight, (BufferedImageOp[])new BufferedImageOp[0]);
                    PDImageXObject img = JPEGFactory.createFromImage((PDDocument)output, (BufferedImage)scaledImage, (float)this.compression);
                    PDPage newPage = new PDPage(new PDRectangle((float)scaledWidth, (float)scaledHeight));
                    PDPageContentStream contentStream = new PDPageContentStream(output, newPage);
                    contentStream.drawImage(img, 0.0f, 0.0f);
                    contentStream.close();
                    output.addPage(newPage);
                }
            }
            output.save(this.destinationPath);
        }
        catch (Exception e) {
            doc.save(this.destinationPath);
            throw new NotSupportedImageException("Nie uda\u0142o si\u0119 przeskalowa\u0107 dokumentu.");
        }
        finally {
            doc.close();
            if (this.deleteSource) {
                FileUtils.deleteQuietly((File)sourceFile);
            }
        }
    }

    private void compressImagesKeepingDocumentPagesStructure(PDDocument inputDoc, OutputStream outputStream) throws IOException {
        try (PDDocument output = new PDDocument();){
            for (PDPage page : inputDoc.getDocumentCatalog().getPages()) {
                output.importPage(page);
                PDPage importedPage = output.getPage(output.getNumberOfPages() - 1);
                this.processPageResources(page, importedPage, output);
            }
            output.save(outputStream);
        }
    }

    private void processPageResources(PDPage page, PDPage importedPage, PDDocument output) throws IOException {
        PDResources resources = page.getResources();
        if (resources == null || !this.hasImages(resources)) {
            return;
        }
        PDResources newResources = this.createNewResources(resources, output);
        importedPage.setResources(newResources);
    }

    private boolean hasImages(PDResources resources) throws IOException {
        for (COSName xObjectName : resources.getXObjectNames()) {
            PDXObject xObject = resources.getXObject(xObjectName);
            if (!(xObject instanceof PDImageXObject)) continue;
            return true;
        }
        return false;
    }

    private PDResources createNewResources(PDResources resources, PDDocument output) throws IOException {
        PDResources newResources = new PDResources();
        this.copyFonts(resources, newResources);
        this.processXObjects(resources, newResources, output);
        this.copyColorSpaces(resources, newResources);
        return newResources;
    }

    private void copyFonts(PDResources resources, PDResources newResources) throws IOException {
        for (COSName fontName : resources.getFontNames()) {
            newResources.put(fontName, resources.getFont(fontName));
        }
    }

    private void processXObjects(PDResources resources, PDResources newResources, PDDocument output) throws IOException {
        for (COSName xObjectName : resources.getXObjectNames()) {
            PDXObject xObject = resources.getXObject(xObjectName);
            if (!(xObject instanceof PDImageXObject)) {
                newResources.put(xObjectName, xObject);
                continue;
            }
            PDImageXObject processedImage = this.processImageXObject((PDImageXObject)xObject, output);
            newResources.put(xObjectName, (PDXObject)processedImage);
        }
    }

    private PDImageXObject processImageXObject(PDImageXObject imageXObject, PDDocument output) throws IOException {
        BufferedImage inputImage = this.getInputImage(imageXObject);
        if (inputImage == null) {
            return imageXObject;
        }
        int scaledWidth = (int)((double)inputImage.getWidth() * this.scalar);
        int scaledHeight = (int)((double)inputImage.getHeight() * this.scalar);
        BufferedImage scaledImage = Scalr.resize((BufferedImage)inputImage, (Scalr.Method)this.resizeMethod.getMethod(), (int)scaledWidth, (int)scaledHeight, (BufferedImageOp[])new BufferedImageOp[0]);
        return JPEGFactory.createFromImage((PDDocument)output, (BufferedImage)scaledImage, (float)this.compression);
    }

    private void copyColorSpaces(PDResources resources, PDResources newResources) throws IOException {
        if (resources.getColorSpaceNames() != null) {
            for (COSName colorSpaceName : resources.getColorSpaceNames()) {
                newResources.put(colorSpaceName, resources.getColorSpace(colorSpaceName));
            }
        }
    }

    private BufferedImage getInputImage(PDImageXObject image) throws IOException {
        return image.getImage();
    }

    public void determineCompressionParametersForFile(float thresholdMB, InputStream inputPdf) throws IOException, NotSupportedImageException, CompressionFileSizeNotReachedException {
        int bytesRead;
        ByteArrayOutputStream fileBuffer = new ByteArrayOutputStream();
        byte[] buffer = new byte[8192];
        while ((bytesRead = inputPdf.read(buffer)) != -1) {
            fileBuffer.write(buffer, 0, bytesRead);
        }
        byte[] originalData = fileBuffer.toByteArray();
        double fileSizeMB = (double)originalData.length / 1048576.0;
        double targetReductionRatio = (double)thresholdMB / fileSizeMB;
        boolean targetAchieved = false;
        int maxIterations = 5;
        int iteration = 0;
        double resultSizeMB = fileSizeMB;
        this.setInitialCompressionParameters(targetReductionRatio);
        while (!targetAchieved && iteration < maxIterations) {
            InputStream testResult = this.resize(new ByteArrayInputStream(originalData));
            byte[] resultData = IOUtils.toByteArray((InputStream)testResult);
            resultSizeMB = (double)resultData.length / 1048576.0;
            log.debug("Iteration {}: scalar={}, compression={}, file size={} MB", new Object[]{++iteration, String.format("%.2f", this.scalar), String.format("%.2f", Float.valueOf(this.compression)), String.format("%.3f", resultSizeMB)});
            if (resultSizeMB <= (double)thresholdMB) {
                targetAchieved = true;
            }
            if (iteration >= maxIterations) continue;
            this.adjustParametersMoreAggressive(resultSizeMB, thresholdMB, iteration + 1);
        }
        if (!targetAchieved) {
            throw new CompressionFileSizeNotReachedException(thresholdMB, resultSizeMB);
        }
    }

    private void setInitialCompressionParameters(double targetReductionRatio) {
        double correctionFactor = 1.8;
        double targetCombinedEffect = targetReductionRatio * correctionFactor;
        if ((targetCombinedEffect = Math.min(targetCombinedEffect, 0.98)) > 0.95) {
            this.scalar = 1.0;
            this.compression = 0.95f;
        } else if (targetCombinedEffect > 0.85) {
            this.scalar = 0.99;
            this.compression = (float)Math.max(0.85, targetCombinedEffect / (this.scalar * this.scalar));
        } else if (targetCombinedEffect > 0.75) {
            this.scalar = 0.97;
            this.compression = (float)Math.max(0.75, targetCombinedEffect / (this.scalar * this.scalar));
        } else if (targetCombinedEffect > 0.65) {
            this.scalar = 0.94;
            this.compression = (float)Math.max(0.65, targetCombinedEffect / (this.scalar * this.scalar));
        } else if (targetCombinedEffect > 0.5) {
            this.scalar = 0.9;
            this.compression = (float)Math.max(0.55, targetCombinedEffect / (this.scalar * this.scalar));
        } else {
            this.scalar = 0.85;
            this.compression = (float)Math.max(0.45, targetCombinedEffect / (this.scalar * this.scalar));
        }
    }

    private void adjustParametersMoreAggressive(double currentSizeMB, double targetSizeMB, int iteration) {
        double currentRatio = targetSizeMB / currentSizeMB;
        double overshoot = currentSizeMB / targetSizeMB;
        if (overshoot <= 1.1) {
            double adjustmentFactor = Math.pow(currentRatio, 0.3);
            this.scalar = Math.max(0.85, this.scalar * adjustmentFactor);
            this.compression = Math.max(0.6f, this.compression * (float)adjustmentFactor);
        } else if (overshoot <= 1.3) {
            double adjustmentFactor = Math.pow(currentRatio, 0.5);
            this.scalar = Math.max(0.7, this.scalar * adjustmentFactor);
            this.compression = Math.max(0.4f, this.compression * (float)adjustmentFactor);
        } else {
            double adjustmentFactor = Math.sqrt(currentRatio);
            this.scalar = Math.max(0.5, this.scalar * adjustmentFactor);
            this.compression = Math.max(0.3f, this.compression * (float)adjustmentFactor);
        }
        if (iteration == 4) {
            this.scalar = Math.max(0.4, this.scalar * 0.75);
            this.compression = Math.max(0.2f, this.compression * 0.75f);
        } else if (iteration == 5) {
            this.scalar = Math.max(0.3, this.scalar * 0.65);
            this.compression = Math.max(0.1f, this.compression * 0.65f);
        }
    }

    public InputStream resize(InputStream inputStream) throws IOException, NotSupportedImageException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try (PDDocument doc = Loader.loadPDF((byte[])IOUtils.toByteArray((InputStream)inputStream));){
            this.compressImagesKeepingDocumentPagesStructure(doc, outputStream);
        }
        return new ByteArrayInputStream(outputStream.toByteArray());
    }

    public PdfResizer() {
    }

    public static enum ResizeMethod {
        AUTO(Scalr.Method.AUTOMATIC),
        SPEED(Scalr.Method.SPEED),
        BALANCED(Scalr.Method.BALANCED),
        QUALITY(Scalr.Method.QUALITY),
        ULTRA(Scalr.Method.ULTRA_QUALITY);

        private Scalr.Method method;

        private ResizeMethod(Scalr.Method method) {
            this.method = method;
        }

        public Scalr.Method getMethod() {
            return this.method;
        }
    }
}

