( function( Ext ) {
Ext.define('CUF', {
  singleton: true,
  init: function () {
    if (!Ext.isIE && CUF.cfg.Properties.debug == true) {
      this.log = console.log.bind(console);
      this.log('CUF log enabled');
    }
    this.loadCss();
  },
  loadCss: function () {
    for (var i = 0; i < CUF.cfg.Properties.css.files.length; i++) {
      var file = CUF.cfg.Properties.css.files[i];
      jQuery('<link>')
        .appendTo('head')
        .attr({ type: 'text/css', rel: 'stylesheet' })
        .attr('href', CUF.cfg.Properties.css.prefix + file);
    }
  },
  log: function () {},
  t: function (key) {
    return PW.t(key);
  },
  localeLoaded: false,
  initLocale: function () {},
  exception: {
    NullPointer: 'NullPointer',
    FilterTypeNotSupported: function (type) {
      return 'FilterTypeNotSupported: ' + type;
    },
    FilterFinderNotSupported: function (type) {
      return 'FilterFinderNotSupported: ' + type;
    },
  },
});

Ext.define('CUF.cfg.Properties', {
  singleton: true,
  dateFormat: 'Y-m-d H:i:s',
  debug: false,
  css: {
    prefix: 'style/cuf/',
    files: ['filter-grid.css', 'buttons.css'],
  },
});

CUF.init();

/**
 * Serwis umożliwiający generowanie dokumentów z szablonów w formacie 'docx' z aktualnych wartości zmiennych procesu.
 */
Ext.define('CUF.template.WordTemplateService', {
  singleton: true,
  alternateClassName: ['CUF.WordTemplateService'],

  /**
   * Generuje i pobiera dokument wygnerowany z podanego szablonu oraz
   */
  download: function (config) {
    var ais = ServiceFactory.getActivityInfoService();

    config = config || {};
    Ext.applyIf(config, {
      format: 'PDF',
      documentClass: undefined,
      template: undefined,
      attachToArchive: false,
      attachToProcess: false,
      newVersion: false,
      documentName: '',
    });

    Ext.apply(config, {
      processId: ais.getProcessId(),
      activityId: ais.getActivityId(),
    });

    var params = jQuery.param(config);
    var url = Suncode.getContextPath() + '/api/cuf/template/word/download?' + params;

    ServiceFactory.getFormService().save({
      success: function () {
        // TODO: wyskakujące okienka
        window.open(url, '_self');
        CUF.document.core.DocumentView.reloadDocumentsGrid();
      },
    });
  },

  /**
   * Generuje dokument na podstawie aktualnych danych procesu.
   */
  generate: function (config) {
    var ais = ServiceFactory.getActivityInfoService();

    config = config || {};
    Ext.applyIf(config, {
      format: 'PDF',
      show: true,
      documentClass: undefined,
      template: undefined,
      attachToProcess: false,
      newVersion: false,
      previewOnly: false,
      successfulFileGenerationMessage: false,
    });

    if ((config.show || config.previewOnly) && !MobileService.isMobile()) {
      var documentPreview = window.open('', '_blank', this._buildWindowParams());
    }

    Ext.apply(config, {
      processId: ais.getProcessId(),
      activityId: ais.getActivityId(),
    });
    ServiceFactory.getFormService().save({
      scope: this,
      success: function () {
        this._doGenerate(config, documentPreview);
      },
    });
  },

  _doGenerate: function (config, documentPreview) {
    var me = this;
    var fs = ServiceFactory.getFormService();
    var ms = ServiceFactory.getMessageService();

    if (!config.show && !config.previewOnly) {
      fs.mask(config.message || 'Generowanie dokumentu z szablonu...');
    }
    if (config.previewOnly) {
      me._previewOnly(config, documentPreview);
    } else {
      me._generateWithSaveToArchive(config, documentPreview);
    }
  },

  _previewOnly: function (config, documentPreview) {
    var me = this;
    var ms = ServiceFactory.getMessageService();
    jQuery.ajax({
      url: Suncode.getContextPath() + '/api/cuf/template/word/generate-document-uuid/',
      method: 'POST',
      data: config,
      success: function (documentUUID) {
        me._previewDocument(documentUUID, documentPreview);
      },
      error: function () {
        ms.showFailure('Błąd generowania dokumentu z szablonu.');
      },
    });
  },

  _generateWithSaveToArchive: function (config, documentPreview) {
    var me = this;
    var fs = ServiceFactory.getFormService();
    var ms = ServiceFactory.getMessageService();
    jQuery.ajax({
      url: Suncode.getContextPath() + '/api/cuf/template/word/',
      method: 'GET',
      data: config,
      success: function (response) {
        if (config.show) {
          me._showDocument(response.encodedId, documentPreview);
        } else {
          fs.unmask();
        }
        if (config.successfulFileGenerationMessage) {
          ms.showSuccess(config.messageContent);
        }
        fs.refreshDocuments();
      },
      error: function () {
        fs.unmask();
        ms.showFailure('Błąd generowania dokumentu z szablonu.');
      },
    });
  },

  _previewDocument: function (uuid, documentPreview) {
    if (MobileService.isMobile()) {
      PW.Logger.debug('Ignoring WordTemplateService#_previewDocument for mobile');
      return;
    }

    var dataURI = Suncode.getContextPath() + '/api/cuf/template/word/preview/' + uuid;
    documentPreview.open(dataURI, '_self', 'viewer').focus();
  },

  _showDocument: function (encodedFileId, documentPreview) {
    if (MobileService.isMobile()) {
      PW.Logger.debug('Ignoring WordTemplateService#_showDocument for mobile');
      return;
    }

    var ais = ServiceFactory.getActivityInfoService();
    var processId = ais.getProcessId();
    var activityId = ais.getActivityId();
    documentPreview
      .open('ShowFile.do?fileName=' + encodedFileId + '&processId=' + processId + '&activityId=' + activityId, '_self', 'viewer')
      .focus();
  },

  _buildWindowParams: function () {
    var body = Ext.getBody();
    var windowParams = 'menubar=no,toolbar=no,location=no,';
    windowParams += 'directories=no,status=no,scrollbars=yes,';
    windowParams += 'resizable=yes,fullscreen=no,channelmode=no,';
    windowParams += 'width=' + body.getWidth() / 2 + ',height=' + body.getHeight() + ',';
    windowParams += 'left=' + body.getWidth() / 2 + ',top=0';

    return windowParams;
  },
});

Ext.define('CUF.util.Arrays', {
  singleton: true,
  each: function (iterable, funcName) {
    var args = Ext.Array.slice(arguments, 2);
    Ext.Array.each(iterable, function (el) {
      el[funcName].apply(el, args);
    });
  },
});

Ext.define('CUF.util.Encoder', {
  singleton: true,
  base64: {
    base64s: 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/',
    encode: function (decStr) {
      if (typeof btoa === 'function') {
        return btoa(decStr);
      }
      var base64s = this.base64s;
      var bits;
      var dual;
      var i = 0;
      decStr = String(decStr);
      var encOut = '';
      while (decStr.length >= i + 3) {
        bits = ((decStr.charCodeAt(i++) & 0xff) << 16) | ((decStr.charCodeAt(i++) & 0xff) << 8) | (decStr.charCodeAt(i++) & 0xff);
        encOut +=
          base64s.charAt((bits & 0x00fc0000) >> 18) +
          base64s.charAt((bits & 0x0003f000) >> 12) +
          base64s.charAt((bits & 0x00000fc0) >> 6) +
          base64s.charAt(bits & 0x0000003f);
      }
      if (decStr.length - i > 0 && decStr.length - i < 3) {
        dual = Boolean(decStr.length - i - 1);
        bits = ((decStr.charCodeAt(i++) & 0xff) << 16) | (dual ? (decStr.charCodeAt(i) & 0xff) << 8 : 0);
        encOut +=
          base64s.charAt((bits & 0x00fc0000) >> 18) +
          base64s.charAt((bits & 0x0003f000) >> 12) +
          (dual ? base64s.charAt((bits & 0x00000fc0) >> 6) : '=') +
          '=';
      }
      return encOut;
    },
    decode: function (encStr) {
      if (typeof atob === 'function') {
        return atob(encStr);
      }
      var base64s = this.base64s;
      var bits;
      var decOut = '';
      var i = 0;
      for (; i < encStr.length; i += 4) {
        bits =
          ((base64s.indexOf(encStr.charAt(i)) & 0xff) << 18) |
          ((base64s.indexOf(encStr.charAt(i + 1)) & 0xff) << 12) |
          ((base64s.indexOf(encStr.charAt(i + 2)) & 0xff) << 6) |
          (base64s.indexOf(encStr.charAt(i + 3)) & 0xff);
        decOut += String.fromCharCode((bits & 0xff0000) >> 16, (bits & 0xff00) >> 8, bits & 0xff);
      }
      if (encStr.charCodeAt(i - 2) == 61) {
        return decOut.substring(0, decOut.length - 2);
      } else if (encStr.charCodeAt(i - 1) == 61) {
        return decOut.substring(0, decOut.length - 1);
      } else {
        return decOut;
      }
    },
  },
});

Ext.define('CUF.util.GridUtils', {
  singleton: true,
  getColumnByDataIndex: function (grid, dataIndex) {
    var cols = grid.headerCt.getGridColumns();
    for (var i = 0; i < cols.length; i++) {
      if (dataIndex == cols[i].dataIndex) return cols[i];
    }
    return null;
  },
  hasHeader: function (grid) {
    if (grid.getHeader()) return true;
    return false;
  },
  hasBorder: function (grid) {
    if (grid.border) return true;
    return false;
  },
});

Ext.define('CUF.util.MessageService', {
  singleton: true,
  warn: function (msg) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: msg,
      minWidth: 300,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
  },
  info: function (msg) {
    Ext.Msg.show({
      title: CUF.t('information'),
      msg: msg,
      minWidth: 300,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.INFO,
    });
  },
});

Ext.define('CUF.util.Renderers', {
  singleton: true,
  dateRenderer: function (time) {
    if (Ext.isEmpty(time)) return '';
    return Ext.Date.format(new Date(time), CUF.cfg.Properties.dateFormat);
  },
  fileSizeRenderer: function (byteSize) {
    if (Ext.isEmpty(byteSize)) return '';
    return Math.floor(byteSize / 1000) + ' KB';
  },

  /**
   * Renderer wykorzystujący mechanizm tłumaczeń CUF (CUF.t()). W przypadku podania prefixu
   * jest on łączony z wartością za pomocą "-" np.: translationRenderer('somekey', 'prefix') - somekey-prefix
   *
   * @param String value tłumaczona wartość
   * @param String prefix (optional) prefix dodany do wartości, połączony z nim za pomocą "-"
   */
  translationRenderer: function (value, prefix) {
    if (!Ext.isEmpty(prefix)) {
      return CUF.t(prefix + '-' + value);
    }
    return CUF.t(value);
  },

  /**
   * Renderer wyświetlający domyślną wartość, jeżeli podana wartość jest pusta (null, undefined, "")
   *
   * @param String value wartość
   * @param String defaultValue domyślna wartość w przypadku pustej wartości
   */
  defaultValueRenderer: function (value, defaultValue) {
    if (Ext.isEmpty(value)) {
      return defaultValue;
    }
    return value;
  },
});

Ext.define('CUF.util.StoreUtils', {
  singleton: true,
  propertyList: function (records, property) {
    var list = [];
    for (var i = 0; i < records.length; i++) {
      list.push(records[i].data[property]);
    }
    return list;
  },
  getStoreData: function (store) {
    var data = [];
    for (var i = 0; i < store.getCount(); i++) {
      data.push(this._clone(store.getAt(i).data));
    }
    return data;
  },
  _clone: function (obj) {
    var cl = {};
    for (var i in obj) {
      if (obj.hasOwnProperty(i)) {
        cl[i] = obj[i];
      }
    }
    return cl;
  },
});

Ext.define('CUF.util.UtilService', {
  singleton: true,
  alternateClassName: ['CUF.UtilService'],
  execute: function (definition) {
    var cfg = Ext.apply({}, definition);
    return Ext.Ajax.request(cfg);
  },
  parseBoolean: function (v) {
    if (Ext.isEmpty(v)) return false;
    if (v == 'true') return true;
    if (v == true) return true;
    return false;
  },
  encode: function (data) {
    if (data instanceof Array) {
      var ret = [];
      for (var i = 0; i < data.length; i++) {
        ret[i] = this.encode(data[i]);
      }
      return ret;
    } else if (data instanceof Date) {
      return data.format('Y-m-d h:i:s');
    } else if (data instanceof Object) {
      var ret = {};
      for (var i in data) {
        if (data.hasOwnProperty(i)) {
          ret[i] = this.encode(data[i]);
        }
      }
      return ret;
    } else {
      return encodeURIComponent(data);
    }
  },
  buildURLParams: function (params) {
    var sParams = '';
    for (var i in params) {
      if (params.hasOwnProperty(i)) {
        if (sParams) {
          sParams += '&';
        }
        sParams += i + '=' + params[i];
      }
    }
    return sParams;
  },
  convertData: function (data, mappings) {
    var records = [];
    for (var i = 0; i < data.length; i++) {
      var row = {};
      var dataRow = data[i];
      for (var property in dataRow) {
        if (dataRow.hasOwnProperty(property)) {
          row[mappings[property]] = dataRow[property];
        }
      }
      for (var mapping in mappings) {
        if (!dataRow.hasOwnProperty(mapping)) {
          row[mappings[mapping]] = '';
        }
      }
      records.push(row);
    }
    return records;
  },
  download: function (url, params) {
    var urlParams = this.buildURLParams(params);
    window.open(Suncode.getContextPath() + '/' + url + '?' + urlParams, 'Download');
  },
});

Ext.define('CUF.data.core.FinderTypeResolver', {
  singleton: true,
  finders: {},
  initialized: false,
  init: function () {
    var finder = CUF.data.core.finder;
    for (var i in finder) {
      if (finder.hasOwnProperty(i)) {
        this.finders[finder[i].getName()] = finder[i];
      }
    }
    this.initialized = true;
  },
  resolve: function (name) {
    if (this.initialized == false) this.init();
    return this.finders[name];
  },
});

Ext.define('CUF.data.core.OperatorResolver', {
  singleton: true,
  operators: {},
  initialized: false,
  defaultOperator: null,
  init: function () {
    var operators = CUF.data.core.operator;
    for (var i in operators) {
      if (operators.hasOwnProperty(i)) {
        this.operators[operators[i].getName()] = operators[i];
      }
    }
    this.defaultOperator = this.operators['EQ'];
    this.initialized = true;
  },
  resolve: function (name) {
    if (this.initialized == false) this.init();

    var op = this.operators[name];
    if (Ext.isEmpty(op)) {
      return this.defaultOperator;
    }
    return op;
  },
});

Ext.define('CUF.data.service.DataFilters', {
  singleton: true,
  alternateClassName: ['CUF.DataFilters'],
  betweenDate: function () {
    return this._betweenFilter(this.date);
  },
  date: function (cfg) {
    return this._build('datefield', 'date', Ext.apply({ format: 'Y-m-d', startDay: 1 }, cfg));
  },
  intNumber: function (cfg) {
    return this._build('numberfield', 'integer', Ext.apply({ allowDecimals: false }, cfg));
  },
  longNumber: function (cfg) {
    return this._build('numberfield', 'long', Ext.apply({ allowDecimals: false }, cfg));
  },
  doubleNumber: function (cfg) {
    return this._build('numberfield', 'double', cfg);
  },
  betweenInt: function () {
    return this._betweenFilter(this.intNumber);
  },
  betweenLong: function () {
    return this._betweenFilter(this.longNumber);
  },
  betweenDouble: function () {
    return this._betweenFilter(this.doubleNumber);
  },
  like: function () {
    return {
      xtype: 'textfield',
      operator: 'LIKE',
      valueConverter: function (v) {
        if (Ext.isEmpty(v)) return null;
        return '%' + v + '%';
      },
    };
  },
  _build: function (xtype, type, cfg) {
    if (!Ext.isEmpty(cfg) && cfg.isList) type += 'List';
    return Ext.apply(
      {
        xtype: xtype,
        valueConverter: function (v) {
          if (Ext.isEmpty(v)) return null;
          var obj = {};
          obj[type] = v;
          return obj;
        },
      },
      cfg
    );
  },
  _betweenFilter: function (creator) {
    var field1 = creator.call(this, { isList: true });
    Ext.applyIf(field1, {
      emptyText: CUF.t('from'),
      operator: 'BETWEEN',
      secondField: 'secondField',
    });
    var field2 = creator.call(this);
    Ext.applyIf(field2, {
      emptyText: CUF.t('to'),
      name: 'secondField',
      depend: true,
    });
    return [field1, field2];
  },
});

Ext.define('CUF.util.validation.ParamChecker', {
  singleton: true,
  notEmpty: function () {
    for (var i = 0; i < arguments.length; i++) {
      if (Ext.isEmpty(arguments[i])) {
        throw CUF.exception.NullPointer;
      }
    }
  },
});

// @req ../core/FinderTypeResolver.js
// @req ../../util/validation/ParamChecker.js
Ext.define('CUF.data.service.DataService', {
  singleton: true,
  alternateClassName: ['CUF.DataService'],
  finderResolver: CUF.data.core.FinderTypeResolver,
  checker: CUF.util.validation.ParamChecker,
  showDataWin: function (cfg) {
    this.checker.notEmpty(cfg.type);
    var finder = this.finderResolver.resolve(cfg.type);
    if (!Ext.isEmpty(cfg.gridCfg.pageSize) && cfg.gridCfg.pageSize >= 1) cfg.gridCfg.paging = true;
    else cfg.gridCfg.paging = false;
    var grid = finder.buildGrid(cfg);
    var win = Ext.widget(
      'cuf_win',
      Ext.apply(
        {
          layout: 'fit',
          maxSize: true,
          items: [grid],
        },
        cfg.winCfg
      )
    );
    return win;
  },
});

Ext.define('CUF.comment.core.CommentService', {
  singleton: true,
  getActivityComments: function (activityId) {
    CUF.util.validation.ParamChecker.notEmpty(activityId);
    var resp = CUF.util.UtilService.execute({
      url: 'api/comments/query',
      async: false,
      method: 'GET',
      params: {
        activityId: activityId,
      },
    }).responseText;
    return Ext.decode(resp);
  },
});

Ext.define('CUF.comment.service.CommentService', {
  singleton: true,
  alternateClassName: ['CUF.CommentService'],
  //---public
  //zwraca true jeżeli w otwartym zadaniu dodano komentarz, false w przeciwnym razie
  checkComment: function () {
    var comments = CUF.comment.core.CommentService.getActivityComments(activityId);
    if (Ext.isEmpty(comments)) return false;
    return true;
  },
  //Sprawdza czy w otwartym zadaniu dodano komentarz.
  //Jeżeli nie to otwiera okno z możliwością dodania komentarza
  checkCommentWindow: function (cfg) {
    if (Ext.isEmpty(cfg)) cfg = {};
    if (!this.checkComment()) {
      if (Ext.isEmpty(cfg.title)) {
        cfg.title = CUF.t('message');
      }
      if (Ext.isEmpty(cfg.msg)) {
        cfg.msg = CUF.t('comment-is-required');
      }
      this.addCommentWindow(cfg);
    } else {
      if (Ext.isFunction(cfg.handler)) {
        cfg.handler(true, cfg);
      }
    }
  },
  addCommentWindow: function (cfg) {
    var win = this._buildCommentWin(cfg);
  },
  _buildCommentWin: function (cfg) {
    Ext.widget('cuf_commentwin', cfg);
  },
});

Ext.define('CUF.comment.view.CommentForm', {
  extend: 'Ext.form.Panel',
  alias: 'widget.cuf_commentform',
  initComponent: function () {
    var cfg = this.initialConfig;
    this.callParent(arguments);
    if (!Ext.isEmpty(cfg.msg)) {
      this.down('textarea').setFieldLabel(cfg.msg);
      this.down('textarea').focus(false, 100);
    }
  },
  frame: false,
  border: 0,
  items: [
    {
      margin: 10,
      allowBlank: false,
      xtype: 'textarea',
      labelAlign: 'top',
      anchor: '100% 100%',
      name: 'comment',
    },
  ],
});

Ext.define('CUF.extension.Base.Window', {
  extend: 'Ext.Window',
  alias: 'widget.cuf_win',
  maxSize: false,
  vMargin: 50,
  hMargin: 50,
  autoShow: true,
  modal: true,
  style: {
    'z-index': 20000,
  },
  initComponent: function () {
    this.configureSize();
    this.callParent();
  },
  //ustawia rozmiar okna na maksymalny w zależności od rozmiaru body
  //uwzględnia maxSizeMargin(odstęp od krawędzi)
  configureSize: function () {
    if (this.maxSize == false) return;
    var width = this.calculateWidth();
    var height = this.calculateHeight();
    this.setSize(width, height);
  },
  calculateWidth: function () {
    var w = Ext.getBody().getViewSize().width;
    return w - this.hMargin * 2;
  },
  calculateHeight: function () {
    var w = Ext.getBody().getViewSize().height;
    return w - this.vMargin * 2;
  },
});

// @req ../../extension/base/BaseWin.js
Ext.define('CUF.comment.view.CommentWin', {
  extend: 'CUF.extension.Base.Window',
  alias: 'widget.cuf_commentwin',
  layout: 'fit',
  width: 550,
  modal: true,
  height: 350,
  padding: 10,
  added: false,
  title: CUF.t('new-comment'),
  initComponent: function () {
    var cfg = this.initialConfig;
    if (!Ext.isEmpty(cfg.title)) {
      this.title = cfg.title;
    }
    this.callParent(arguments);
    this.add(
      Ext.widget('cuf_commentform', {
        msg: cfg.msg,
      })
    );
    if (!Ext.isEmpty(cfg.buttonName)) {
      this.down('cuf_addbtn').setText(cfg.buttonName);
    }
  },
  listeners: {
    close: function (win) {
      var cfg = this.initialConfig;
      if (win.added == false && Ext.isFunction(cfg.handler)) {
        cfg.handler(false, cfg);
      }
    },
  },
  buttons: [
    {
      xtype: 'cuf_addbtn',
      handler: function (btn) {
        var form = btn.getWinForm();
        var comment = form.getForm().getValues().comment;
        if (form.isValid()) {
          var cfg = btn.getWin().initialConfig;
          ServiceFactory.getCommentsService().add(comment);
          btn.getWin().added = true;
          btn.getWin().close();
          if (Ext.isFunction(cfg.handler)) {
            cfg.handler(true, cfg);
          }
        } else {
          CUF.util.MessageService.warn(CUF.t('comment-is-required'));
        }
      },
    },
    {
      xtype: 'cuf_cancelbtn',
    },
  ],
});

Ext.define('CUF.extension.Base.Grid', {
  extend: 'Ext.grid.Panel',
  alias: 'widget.cuf_grid',
  numbered: false,
  filtered: true,
  showRowCount: false,
  checkboxSelection: false,
  selectionType: 'MULTI',
  viewConfig: {
    enableTextSelection: true,
  },
  constructor: function (cfg) {
    if (cfg.showRowCount) {
      this.bbar = [
        '->',
        {
          xtype: 'label',
          text: CUF.t('row-count') + ': ',
        },
        {
          xtype: 'label',
          name: 'totalCount',
          text: '0',
        },
        ' ',
      ];
      cfg.store.on('datachanged', this.setRowCount, this);
    }
    if (cfg.cfgOrg && cfg.cfgOrg.columns) {
      this.columns = cfg.cfgOrg.columns;
    }
    if (cfg.cfgOrg && cfg.cfgOrg.selectionType) {
      this.selectionType = cfg.cfgOrg.selectionType;
    }
    this.callParent([cfg]);
  },
  initComponent: function () {
    this.initNumbered();
    this.initPlugins();
    this.initCheckboxSelection();
    this.initRowCount();
    this.callParent();
  },
  initPlugins: function () {
    if (this.filtered) {
      var plugin = new CUF.extension.plugin.HeaderFilters();
      if (Ext.isEmpty(this.plugins)) {
        this.plugins = [plugin];
      } else {
        this.plugins.push(plugin);
      }
    }
  },
  reload: function () {
    this.getStore().load();
  },
  initNumbered: function () {
    if (this.numbered) {
      Ext.Array.insert(this.columns, 0, [{ xtype: 'rownumberer', text: CUF.t('lp'), resizable: true, minWidth: 50 }]);
    }
  },
  initCheckboxSelection: function () {
    var injectCheckbox = 0;
    if (this.numbered) injectCheckbox = 1;
    if (this.checkboxSelection) {
      this.selModel = {
        selType: 'checkboxmodel',
        mode: this.selectionType,
        injectCheckbox: injectCheckbox,
        pruneRemoved: false,
        checkOnly: true,
      };
    }
  },
  initRowCount: function () {
    if (this.showRowCount) {
      this.on('afterrender', this.setRowCount, this);
    }
  },
  setRowCount: function () {
    var store = this.getStore();
    var count = 0;
    if (store.buffered) count = store.getTotalCount();
    else count = store.getCount();
    this.down('toolbar label[name=totalCount]').setText(count);
  },
});

Ext.define('CUF.extension.base.Buttons.BaseBtn', {
  extend: 'Ext.button.Button',
  alias: 'widget.cuf_basebtn',
  getWinForm: function () {
    var win = this.up('window');
    var form = win.down('form');
    return form;
  },
  getWin: function () {
    var win = this.up('window');
    return win;
  },
});

Ext.define('CUF.extension.base.Buttons.CancelBtn', {
  extend: 'CUF.extension.base.Buttons.BaseBtn',
  alias: 'widget.cuf_cancelbtn',
  text: CUF.t('cancel'),
  handler: function (btn) {
    btn.getWin().close();
  },
});

Ext.define('CUF.extension.base.Buttons.AddBtn', {
  extend: 'CUF.extension.base.Buttons.BaseBtn',
  alias: 'widget.cuf_addbtn',
  text: CUF.t('add'),
  iconCls: 'dvnt-icon-plus-symbol',
});

Ext.define('CUF.extension.base.Buttons.SaveBtn', {
  extend: 'CUF.extension.base.Buttons.BaseBtn',
  alias: 'widget.cuf_savebtn',
  text: CUF.t('save'),
  iconCls: 'dvnt-icon-save',
});

Ext.define('CUF.extension.base.Buttons.ExcelBtn', {
  extend: 'CUF.extension.base.Buttons.BaseBtn',
  alias: 'widget.cuf_excelbtn',
  iconCls: 'dvnt-icon-excel',
});

Ext.define('CUF.extension.PagedGrid', {
  extend: 'CUF.extension.Base.Grid',
  alias: 'widget.cuf_paged_grid',
  constructor: function (cfg) {
    this.rememberedSelections = new Array();
    this.selectedItems = [];
    this.dockedItems = [
      {
        xtype: 'pagingtoolbar',
        store: cfg.store,
        dock: 'bottom',
        displayInfo: true,
      },
    ];
    this.callParent([cfg]);

    if (cfg.hasOwnProperty('rememberSelection') && cfg.rememberSelection === true) {
      this.handleRememberedSelection();
    }
  },
  _mayRecordBeUnselected: function (index) {
    var outOfCurrentPageRecord = -1;
    if (index == outOfCurrentPageRecord) return false;
    return true;
  },
  handleRememberedSelection: function () {
    var selectionModel = this.getSelectionModel();
    selectionModel.getSelection = function () {
      return this.selectedItems;
    }.bind(this);
    selectionModel.deselectAll = function () {
      this.selectedItems = [];
      this.store.load();
    }.bind(this);
    this.store.on(
      'load',
      function () {
        var intersection = this.store.data.items.filter(
          function (item) {
            return (
              this.selectedItems.find(function (selectedItem) {
                item.raw.internalId = undefined;
                selectedItem.raw.internalId = undefined;
                return JSON.stringify(item.raw) === JSON.stringify(selectedItem.raw);
              }) !== undefined
            );
          }.bind(this)
        );
        selectionModel.doMultiSelect(intersection, true);
      }.bind(this)
    );
  },
  listeners: {
    beforedeselect: function (selectionModel, record, index, eOpts) {
      this.selectedItems = this.selectedItems.filter(function (item) {
        item.raw.internalId = undefined;
        record.raw.internalId = undefined;
        return JSON.stringify(item.raw) !== JSON.stringify(record.raw);
      });

      return this._mayRecordBeUnselected(index);
    },
    beforeselect: function (selectionModel, record, index, eOpts) {
      if (
        this.selectedItems.find(function (item) {
          item.raw.internalId = undefined;
          record.raw.internalId = undefined;
          return JSON.stringify(item.raw) === JSON.stringify(record.raw);
        }) === undefined
      ) {
        if (selectionModel.getSelectionMode() === 'SINGLE') {
          this.selectedItems = [];
        }
        this.selectedItems.push(record);
      }
    },
  },
});

Ext.define('CUF.extension.data.HibernateStoreCreator', {
  singleton: true,
  alternateClassName: ['CUF.HibernateStoreCreator'],
  create: function (storeName, cfg) {
    return Ext.create(storeName, {
      cfgOrg: cfg,
      finderPath: '/api/cuf/finder/hibernate',
      queryObject: function () {
        return this.getProxy().jsonData;
      },
      jsonData: cfg.query,
    });
  },
});

Ext.define('CUF.extension.data.JsonBaseStore', {
  extend: 'Ext.data.Store',
  buffered: false,
  constructor: function (cfg) {
    (this.fields = cfg.fields),
      (this.remoteSort = true),
      (this.proxy = {
        type: 'memory',
        enablePaging: true,
        data: cfg.data,
        reader: {
          idProperty: cfg.gridCfg.idProperty,
        },
      });
    if (cfg.gridCfg.paging) this.pageSize = cfg.gridCfg.pageSize;
    this.callParent([cfg]);
  },
});

Ext.define('CUF.extension.data.JsonFilterBaseStore', {
  extend: 'Ext.data.Store',
  baseFilters: [],
  baseSorters: [],
  remoteFilter: true,
  remoteSort: true,
  buffered: false,
  constructor: function (cfg) {
    var query = cfg.cfgOrg.query;
    if (!Ext.isEmpty(query)) {
      if (!Ext.isEmpty(query.filters)) {
        this.baseFilters = query.filters;
      }
      if (!Ext.isEmpty(query.sorters)) {
        this.baseSorters = query.sorters;
      }
    }
    this.initialQuery = query;
    this.getQueryObject = cfg.queryObject;
    this.proxy = {
      type: 'jsonproxy',
      jsonData: cfg.jsonData,
      timeout: 900000,
      api: {
        read: cfg.cfgOrg.url ? cfg.cfgOrg.url : Suncode.getContextPath() + cfg.finderPath,
      },
      reader: {
        root: 'data',
        totalProperty: 'total',
        idProperty: cfg.cfgOrg.gridCfg.idProperty,
      },
    };
    if (cfg.cfgOrg.gridCfg.paging) this.pageSize = cfg.cfgOrg.gridCfg.pageSize;
    this.callParent([cfg.cfgOrg]);
  },

  listeners: {
    beforeload: function (store, operation, opts) {
      var filters = store.filters;
      var sorters = store.sorters;
      var query = store.getQueryObject();
      if (filters.getCount() == 0) {
        query.filters = [];
      } else {
        var f = [];
        for (var i = 0; i < filters.getCount(); i++) {
          var filter = filters.getAt(i);
          f.push(store.parseFilters(filter));
        }
        query.filters = f;
      }

      if (sorters.getCount() == 0) {
        query.sorters = [];
      } else {
        var s = [];
        for (var i = 0; i < sorters.getCount(); i++) {
          var sorter = sorters.getAt(i);
          s.push(store.parseSorters(sorter));
        }
        query.sorters = s;
      }
      query.start = operation.start;
      query.limit = operation.limit;
      query.filters = query.filters.concat(store.baseFilters);
      query.sorters = query.sorters.concat(store.baseSorters);
    },
    beforeprefetch: function (store, operation, opts) {
      var query = store.getQueryObject();
      query.start = operation.start;
      query.limit = operation.limit;
    },
  },
  parseFilters: function (filter) {
    return {
      property: filter.property,
      value: filter.value,
      operator: filter.operator,
    };
  },
  parseSorters: function (sorter) {
    return {
      property: sorter.property,
      direction: sorter.direction,
    };
  },
});

Ext.define('CUF.extension.data.JsonFilterStore', {
  extend: 'CUF.extension.data.JsonFilterBaseStore',
  buffered: true,
  leadingBufferZone: 100,
  trailingBufferZone: 100,
  pageSize: 100,
});

Ext.define('CUF.extension.data.JsonProxy', {
  extend: 'Ext.data.proxy.Rest',
  alias: 'proxy.jsonproxy',
  buildRequest: function (operation) {
    var request = this.callParent(arguments);
    request.jsonData = this.jsonData;
    request.params = {};
    return request;
  },
  actionMethods: {
    read: 'POST',
  },
});

Ext.define('CUF.extension.data.SimpleStoreCreator', {
  singleton: true,
  alternateClassName: ['CUF.SimpleStoreCreator'],
  create: function (storeName, cfg) {
    return Ext.create(storeName, {
      cfgOrg: cfg,
      finderPath: '/api/cuf/finder/simple',
      queryObject: function () {
        return this.getProxy().jsonData.query;
      },
      jsonData: {
        className: cfg.query.className,
        parameters: cfg.query.parameters,
        query: {},
      },
    });
  },
});

Ext.selection.Model.override({
  storeHasSelected: function (record) {
    var store = this.store,
      records,
      len,
      id,
      i;

    if (record.hasId() && store.getById(record)) {
      return true;
    } else {
      records = store.data.items;
      len = records ? records.length : 0;
      id = record.internalId;

      for (i = 0; i < len; ++i) {
        if (id === records[i].internalId) {
          return true;
        }
      }
    }
    return false;
  },
});

Ext.define('CUF.extension.plugin.HeaderFilters', {
  ptype: 'gridheaderfilters',
  grid: null,
  fields: null,
  filterDefs: null,
  containers: null,
  fieldLines: 0,
  storeLoaded: false,
  lastApplyFilters: null,

  reloadOnChange: true,

  rendered: false,

  constructor: function (cfg) {
    if (cfg) {
      Ext.apply(this, cfg);
    }
  },

  init: function (grid) {
    this.grid = grid;

    this.grid.on({
      scope: this,
      columnresize: this.resizeFilterContainer,
      beforedestroy: this.onDestroy,
      afterlayout: this.adjustFilterWidth,
    });

    this.grid.headerCt.on({
      scope: this,
      afterrender: this.renderFilters,
    });

    this.grid.getStore().on({
      scope: this,
      load: this.onStoreLoad,
    });

    // Enable new grid methods
    Ext.apply(this.grid, {
      headerFilterPlugin: this,
      setHeaderFilter: function (sName, sValue) {
        if (!this.headerFilterPlugin) return;
        var fd = {};
        fd[sName] = sValue;
        this.headerFilterPlugin.setFilters(fd);
      },
      /**
       * Returns a collection of filters corresponding to enabled
       * header filters. If a filter field is disabled, the filter
       * is not included. <b>This method is enabled on Grid</b>.
       *
       * @method
       * @return {Array[Ext.util.Filter]} An array of
       *         Ext.util.Filter
       */
      getHeaderFilters: function () {
        if (!this.headerFilterPlugin) return null;
        return this.headerFilterPlugin.getFilters();
      },
      /**
       * Set header filter values <b>Method enabled on Grid</b>
       *
       * @method
       * @param {Object
       *            or Array[Object]} filters An object with
       *            key/value pairs or an array of Ext.util.Filter
       *            objects (or corresponding configuration). Only
       *            filters that matches with header filters names
       *            will be set
       */
      setHeaderFilters: function (obj) {
        if (!this.headerFilterPlugin) return;
        this.headerFilterPlugin.setFilters(obj);
      },
      getHeaderFilterField: function (fn) {
        if (!this.headerFilterPlugin) return;
        if (this.headerFilterPlugin.fields[fn]) return this.headerFilterPlugin.fields[fn];
        else return null;
      },
      resetHeaderFilters: function () {
        if (!this.headerFilterPlugin) return;
        this.headerFilterPlugin.resetFilters();
      },
      clearHeaderFilters: function () {
        if (!this.headerFilterPlugin) return;
        this.headerFilterPlugin.clearFilters();
      },
      applyHeaderFilters: function () {
        if (!this.headerFilterPlugin) return;
        this.headerFilterPlugin.applyFilters();
      },
    });
  },
  valueConverter: function (filters) {
    return filters;
  },
  setFieldValue: function (field, value) {
    var column = field.column;
    if (!Ext.isEmpty(value)) {
      field.setValue(value);
      if (!Ext.isEmpty(value) && column.hideable && !column.isVisible() && !field.isDisabled()) {
        column.setVisible(true);
      }
    } else {
      field.setValue('');
    }
  },
  buildFilterDefs: function () {
    var columns = this.grid.headerCt.getGridColumns();
    for (var i = 0; i < columns.length; i++) {
      var col = columns[i];
      var filters = [];
      if (!Ext.isEmpty(col.filter)) {
        if (Ext.isArray(col.filter)) {
          filters = filters.concat(col.filter);
        } else {
          filters.push(col.filter);
        }
      }

      col.filters = filters;
    }
    this.initFieldLines(columns);
    this.fillEmptyFields(columns);
  },
  fillEmptyFields: function (columns) {
    for (var c = 0; c < columns.length; c++) {
      var col = columns[c];
      var filters = col.filters;
      for (var i = filters.length; i < this.fieldLines; i++) {
        filters.push({
          xtype: 'displayfield',
          depend: true,
        });
      }
    }
  },
  renderFilters: function () {
    this.destroyFilters();

    this.fields = {};
    this.containers = {};
    this.buildFilterDefs();
    var store = this.grid.getStore();
    var columns = this.grid.headerCt.getGridColumns();
    for (var c = 0; c < columns.length; c++) {
      var column = columns[c];
      column.componentLayout.notPublishInnerHeight = true;
      if (!column.el) {
        continue;
      }
      var filterContainerConfig = {
        itemId: column.id + '-filtersContainer',
        layout: 'anchor',
        bodyStyle: {
          'background-color': 'transparent',
        },
        border: false,
        padding: '0 2 0 2',
        width: column.getWidth(),
        listeners: {
          scope: this,
          element: 'el',
          mousedown: function (e) {
            e.stopPropagation();
          },
          click: function (e) {
            e.stopPropagation();
          },
          keydown: function (e) {
            e.stopPropagation();
          },
          keypress: function (e) {
            e.stopPropagation();
          },
          keyup: function (e) {
            e.stopPropagation();
          },
        },
        items: [],
      };

      for (var ci = 0; ci < column.filters.length; ci++) {
        var fc = column.filters[ci];
        Ext.applyIf(fc, {
          filterName: column.dataIndex,
          depend: false,
          hideLabel: true,
        });
        Ext.apply(fc, {
          itemId: fc.filterName + '-' + ci,
          anchor: '-1',
          enableKeyEvents: true,
          checkChangeBuffer: 500,
          listeners: {
            scope: this,
            keypress: function (field, e) {
              if (e.getKey() == Ext.EventObject.ENTER) {
                this.onFilterContainerEnter();
              }
            },
            change: function () {
              this.applyFilters();
            },
          },
        });
        Ext.apply(fc, {
          operator: CUF.data.core.OperatorResolver.resolve(fc.operator),
        });
        var filterField = Ext.ComponentManager.create(fc);
        filterField.column = column;
        if (Ext.isEmpty(this.fields[filterField.filterName])) {
          this.fields[filterField.filterName] = [filterField];
        } else {
          this.fields[filterField.filterName].push(filterField);
        }

        filterContainerConfig.items.push(filterField);
      }

      var filterContainer = Ext.create('Ext.container.Container', filterContainerConfig);
      filterContainer.render(column.el);
      this.containers[column.id] = filterContainer;
      column.setPadding = Ext.Function.createInterceptor(column.setPadding, function (h) {
        return false;
      });
    }

    this.applyFilters();
    this.rendered = true;
  },
  initFieldLines: function (columns) {
    var fieldCount = 0;
    for (var i = 0; i < columns.length; i++) {
      var col = columns[i];
      if (col.filters.length > fieldCount) fieldCount = col.filters.length;
    }
    this.fieldLines = fieldCount;
  },
  onStoreLoad: function () {
    this.storeLoaded = true;
  },

  onFilterContainerEnter: function () {
    this.applyFilters();
  },

  resizeFilterContainer: function (headerCt, column, w, opts) {
    if (!this.containers) return;
    var cnt = this.containers[column.id];
    if (cnt) {
      cnt.setWidth(w);
      cnt.doLayout();
    }
  },

  destroyFilters: function () {
    this.rendered = false;
    if (this.fields) {
      for (var f in this.fields) Ext.destroy(this.fields[f]);
      delete this.fields;
    }

    if (this.containers) {
      for (var c in this.containers) Ext.destroy(this.containers[c]);
      delete this.containers;
    }
  },

  onDestroy: function () {
    this.destroyFilters();
    Ext.destroy(this.tooltip, this.tooltipTpl);
  },

  adjustFilterWidth: function () {
    if (!this.containers) return;
    var columns = this.grid.headerCt.getGridColumns();
    for (var c = 0; c < columns.length; c++) {
      var column = columns[c];
      if (column.filter && column.flex) {
        this.containers[column.id].setWidth(column.getWidth() - 1);
      }
    }
  },

  resetFilters: function () {
    if (!this.fields) return;
    for (var fn in this.fields) {
      var f = this.fields[fn];
      if (!f.isDisabled() && !f.readOnly && Ext.isFunction(f.reset)) f.reset();
    }
    this.applyFilters();
  },

  clearFilters: function () {
    if (!this.fields) return;
    for (var fn in this.fields) {
      var f = this.fields[fn];
      if (!f.isDisabled() && !f.readOnly) f.setValue('');
    }
    this.applyFilters();
  },

  setFilters: function (filters) {
    if (!filters) return;

    if (Ext.isArray(filters)) {
      var conv = {};
      Ext.each(filters, function (filter) {
        if (filter.property) {
          conv[filter.property] = filter.value;
        }
      });
      filters = conv;
    } else if (!Ext.isObject(filters)) {
      return;
    }

    this.initFilterFields(filters);
    this.applyFilters();
  },

  getFilters: function () {
    return this.lastApplyFilters;
  },
  parseFilters: function () {
    var filters = [];
    if (!this.fields) return filters;
    for (var fn in this.fields) {
      var fieldList = this.fields[fn];

      Ext.each(
        fieldList,
        function (field) {
          if (!field.isDisabled() && field.isValid() && field.depend == false) {
            var value = field.operator.parse(field.getSubmitValue(), field);
            if (Ext.isEmpty(value)) return;
            var filter = {
              value: value,
              operator: field.operator.getName(),
              property: field.filterName,
            };
            if (Ext.isDefined(field.valueConverter)) {
              filter.rawValue = filter.value;
              filter.value = field.valueConverter(filter.value);
            }
            filters.push(filter);
          }
        },
        this
      );
    }
    return filters;
  },
  initFilterFields: function (filters) {
    if (!this.fields) return;

    for (var fn in filters) {
      var value = filters[fn];
      var field = this.fields[fn];
      if (field) {
        this.setFieldValue(filterField, initValue);
      }
    }
  },

  applyFilters: function () {
    var filters = this.parseFilters();

    this.grid.getStore().clearFilter(true);
    var store = this.grid.getStore();
    var storeFilters = this.grid.getStore().filters;
    if (!Ext.isEmpty(filters)) {
      if (store.remoteFilter) {
        store.filter(filters);
      } else {
        var localeFilters = this._prepareLocaleFilters(filters);
        store.filter(localeFilters);
      }
    } else {
      if (store.remoteFilter) {
        store.load();
      } else {
        store.clearFilter();
        this.grid.getView().refresh();
      }
    }
    store.fireEvent('datachanged', store);
  },

  _prepareLocaleFilters: function (filters) {
    var me = this;
    var localeFilters = [];
    Ext.each(filters, function (filter) {
      var obj = {};
      obj.filterFn = me._getFilterFunction(filter);
      localeFilters.push(obj);
    });
    return localeFilters;
  },

  _getFilterFunction: function (filter) {
    var value = this._getFilterValue(filter);
    if (filter.operator == 'EQ') {
      return function (item) {
        return item.get(filter.property) == value;
      };
    } else if (filter.operator == 'BETWEEN') {
      if (value[1] == null) {
        return function (item) {
          return item.get(filter.property) >= value[0];
        };
      } else {
        return function (item) {
          return item.get(filter.property) >= value[0] && item.get(filter.property) <= value[1];
        };
      }
    } else if (filter.operator == 'LIKE') {
      return function (item) {
        return item.get(filter.property).indexOf(value) > -1;
      };
    }
  },

  _getFilterValue: function (filter) {
    if (filter.value.dateList) {
      var result = [];
      var dates = filter.value.dateList;
      Ext.each(dates, function (date) {
        if (date == null) {
          result.push(null);
        }
        result.push(Ext.Date.parse(date, 'Y-m-d'));
      });
      return result;
    } else {
      return filter.rawValue;
    }
  },
});

Ext.define('CUF.table.core.TableService', {
  singleton: true,

  vs: ServiceFactory.getVariableSetService(),

  exportSchemeToExcel: function (grid, config) {
    config.suffix = ' - schemat';
    var cfg = this._buildExcelDef(grid, config);
    cfg.headers = CUF.UtilService.encode(cfg.headers);
    cfg.headerMappings = CUF.UtilService.encode(cfg.headerMappings);
    CUF.UtilService.download('CUF.ExportDTSchemeToExcel.customServlet', cfg);
  },

  exportDataToExcel: function (grid, config) {
    try {
      config.suffix = ' - eksport';
      var cfg = this._buildExcelDef(grid, config);

      var data = CUF.util.StoreUtils.getStoreData(grid.getStore());
      // filtering of data
      if (cfg.filters && cfg.filters.length > 0) {
        data = this.filterData(data, cfg.filters, cfg.columnsToExport);
      }
      data = Ext.encode(data);

      var msg = this._exportDataToExcel(
        Ext.apply(
          {
            data: data,
            processId: processId,
          },
          cfg
        )
      );

      var resp = msg.data;
      CUF.UtilService.download('CUF.ExportDTToExcel.customServlet', {
        title: encodeURIComponent(resp[3]),
        extension: encodeURIComponent(resp[1]),
        size: encodeURIComponent(resp[2]),
        path: encodeURIComponent(resp[0]),
        mode: 'download',
        checksum: encodeURIComponent(resp[4]),
      });
    } catch (e) {
      CUF.log(e);
    }
  },

  importDataFromExcel: function (grid, config) {
    Ext.widget('cuf_importexcelwin', {
      grid: grid,
      cfg: config,
    });
  },

  writeData: function (grid, data, suspendAutoMappings) {
    ServiceFactory.getVariableSetService().addRow(grid, false, data, suspendAutoMappings);
  },

  getTableDefinition: function (grid, includeHiddenColumns, includedColumns) {
    var st = grid.getStore();
    var gridInfo = this.vs.getInfo(grid);
    var cols = gridInfo.columns;
    var title = gridInfo.title;

    var headers = [];
    var headersIds = [];
    var headerMappings = {};
    var reverseMappings = {};
    var allColumnsMappings = {};
    for (var i = 0; i < cols.length; i++) {
      var col = cols[i],
        isHidden = col.hidden,
        hText = col.name.unescapeHTML(),
        shouldBeExported =
          (includedColumns && (jQuery.inArray(col.id, includedColumns) != -1 || jQuery.inArray(hText, includedColumns) != -1)) ||
          !includedColumns ||
          includedColumns.length == 0;

      if (shouldBeExported && (isHidden == false || includeHiddenColumns == true)) {
        headersIds.push(col.id);
        headers.push(hText);
        headerMappings[col.id] = hText;
        reverseMappings[hText] = col.id;
      }
      allColumnsMappings[col.id] = hText;
    }

    return {
      headersIds: headersIds,
      headers: headers,
      headerMappings: headerMappings,
      reverseMappings: reverseMappings,
      allColumnsMappings: allColumnsMappings,
      title: title,
    };
  },

  _buildExcelDef: function (grid, config) {
    if (Ext.isEmpty(config)) config = {};
    var ext = config.extension,
      includeHiddenColumns = config.includeHiddenColumns,
      extension = 'xlsx',
      columnsToExport = config.columnsToExport;

    if (!Ext.isEmpty(ext) && ext == 'xls') extension = 'xls';

    var td = this.getTableDefinition(grid, includeHiddenColumns, columnsToExport);
    var headers = td.headers;
    var title = td.title + config.suffix;
    if (!Ext.isEmpty(config.title)) title = config.title;
    var headerMappings = td.headerMappings;

    headers = Ext.encode(headers);
    headerMappings = Ext.encode(headerMappings);

    return {
      headers: headers,
      title: title,
      extension: extension,
      headerMappings: headerMappings,
      filters: config.filters || [],
    };
  },

  _exportDataToExcel: function (data) {
    var msg = CUF.UtilService.execute({
      url: Suncode.getContextPath() + '/CUF.ExportDTToExcel.customServlet',
      method: 'POST',
      async: false,
      params: data,
    }).responseText;

    msg = Ext.decode(msg);
    return msg;
  },

  filterData: function (dataFromExcel, filters) {
    var result = [];
    for (var i = 0; i < dataFromExcel.length; i++) {
      if (this.checkRow(dataFromExcel[i], filters)) {
        result.push(dataFromExcel[i]);
      }
    }
    return result;
  },

  checkRow: function (row, filters) {
    for (var i = 0; i < filters.length; i++) {
      var filter = filters[i];
      var fieldValue = typeof row[filter.colName] === 'string' ? row[filter.colName].replace(/,/g, '.') : row[filter.colName];
      var filterValue = filter.value.replace(/,/g, '.');
      if (!isNaN(fieldValue) && !isNaN(filterValue)) {
        fieldValue = Number(fieldValue);
        filterValue = Number(filterValue);
      } else {
        fieldValue = row[filter.colName];
        filterValue = filter.value;
      }

      if (fieldValue instanceof Date) {
        fieldValue = fieldValue.dateFormat('Y-m-d');
      }
      var filterPassed = this.filtersCheck[filter.type](fieldValue, filterValue);
      if (!filterPassed) {
        return filterPassed;
      }
    }
    return true;
  },

  filtersCheck: {
    '==': function (cellValue, expectedValue) {
      return cellValue == expectedValue;
    },
    '!=': function (cellValue, compareValue) {
      return cellValue != compareValue;
    },
    '<': function (cellValue, compareValue) {
      return cellValue < compareValue;
    },
    '>': function (cellValue, compareValue) {
      return cellValue > compareValue;
    },
    '<=': function (cellValue, compareValue) {
      return cellValue <= compareValue;
    },
    '>=': function (cellValue, compareValue) {
      return cellValue >= compareValue;
    },
    regexp: function (cellValue, pattern) {
      return pattern.test(cellValue);
    },
    fn: function (cellValue, fn) {
      return fn(cellValue);
    },
  },
});

// @req ../core/TableService.js
/**
 * Definicje funkcji przycisków tabelki
 * Funkcje tej klasy mogą być podawane w CUFdefs.js
 */
Ext.define('CUF.table.service.TableButton', {
  singleton: true,
  tableService: CUF.table.core.TableService,
  //importuje dane z excela
  importFromExcel: function (btn, cfg) {
    var grid = btn.up('variableset');
    var conf = CUF.table.service.TableButton._buildImportConfig(grid, cfg);
    CUF.table.core.TableService.importDataFromExcel(grid, conf);
  },
  //eksportuje dane do excela
  exportToExcel: function (btn, cfg) {
    var grid = btn.up('variableset');
    var conf = CUF.table.service.TableButton._buildExportConfig(grid, cfg);
    CUF.table.core.TableService.exportDataToExcel(grid, conf);
  },
  //eksportuje schemat(nagłówki) do excela
  exportSchemaToExcel: function (btn, cfg) {
    var grid = btn.up('variableset');
    var conf = CUF.table.service.TableButton._buildExportConfig(grid, cfg);
    CUF.table.core.TableService.exportSchemeToExcel(grid, conf);
  },
  //-----------private
  _buildImportConfig: function (grid, cfg) {
    var conf = {};
    if (!Ext.isEmpty(cfg.functionName)) conf.callback = window[cfg.functionName];
    if (!Ext.isEmpty(cfg.showHidden)) conf.includeHiddenColumns = CUF.UtilService.parseBoolean(cfg.showHidden);
    if (!Ext.isEmpty(cfg.loadButtonLabel)) conf.loadButtonLabel = cfg.loadButtonLabel;
    if (!Ext.isEmpty(cfg.fileButtonLabel)) conf.fileButtonLabel = cfg.fileButtonLabel;
    if (!Ext.isEmpty(cfg.windowTitle)) conf.title = cfg.windowTitle;
    if (!Ext.isEmpty(cfg.saveData)) conf.fileButtosaveDatanLabel = CUF.UtilService.parseBoolean(cfg.saveData);
    if (!Ext.isEmpty(cfg.filters)) conf.filters = cfg.filters;
    if (!Ext.isEmpty(cfg.columnsToImport)) conf.columnsToImport = cfg.columnsToImport;
    return conf;
  },
  _buildExportConfig: function (grid, cfg) {
    var conf = {};
    if (!Ext.isEmpty(cfg.showHidden)) conf.includeHiddenColumns = CUF.UtilService.parseBoolean(cfg.showHidden);
    if (!Ext.isEmpty(cfg.extension)) conf.extension = cfg.extension;
    if (!Ext.isEmpty(cfg.saveData)) conf.saveData = CUF.UtilService.parseBoolean(cfg.saveData);
    conf.title = cfg.title;
    return conf;
  },
});

Ext.define('CUF.table.view.ImportExcelForm', {
  extend: 'Ext.form.Panel',
  alias: 'widget.cuf_importexcelform',
  frame: false,
  border: false,
  layout: {
    type: 'vbox',
    align: 'stretch',
  },
  tableService: null,
  initComponent: function () {
    var cfg = Ext.clone(this.initialConfig.cfg);
    this.tableService = CUF.table.core.TableService;
    var hideOverride = false;
    if (cfg.overrideCheckBox != undefined) {
      hideOverride = !cfg.overrideCheckBox;
    }
    this.items = [
      {
        xtype: 'filefield',
        allowBlank: false,
        emptyText: CUF.t('choose-excel-file'),
        fieldLabel: CUF.t('excel-file'),
        name: 'file',
        buttonText: '',
        labelWidth: 150,
        alowBlank: false,
        padding: '0px 1px 0px 0px',
      },
      {
        xtype: 'checkbox',
        labelWidth: 150,
        name: 'override',
        fieldLabel: CUF.t('clean-current-data'),
        checked: false,
        hidden: hideOverride,
      },
    ];
    this.callParent(arguments);
  },
  importExcel: function (callback, columnsToImport, scope) {
    var me = this;
    FormService.save({
      success: me.doImport,
      successParams: [callback, columnsToImport, scope],
      failure: me.doImport,
      failureParams: [callback, columnsToImport, scope],
      scope: me,
    });
  },
  doImport: function (callback, columnsToImport, scope) {
    var headers = CUF.UtilService.encode(columnsToImport);
    headers = Ext.encode(headers);
    var params = CUF.UtilService.buildURLParams({
      processId: encodeURIComponent(processId),
      activityId: encodeURIComponent(activityId),
      headers: encodeURIComponent(headers),
      columns: encodeURIComponent(JSON.stringify(scope.tableDefinition.allColumnsMappings)),
      columnsToImport: encodeURIComponent(JSON.stringify(scope.tableDefinition.headersIds)),
      dataChoosersWithSuspendedAutomappings: encodeURIComponent(JSON.stringify(scope.dataChoosersWithSuspendedAutomappings)),
      override: this.getForm().getFieldValues().override,
    });
    this.submit({
      url: 'CUF.ImportDTFromExcel.customServlet?' + params,
      waitMsg: CUF.t('processing-file'),
      waitTitle: CUF.t('please-wait'),
      success: function (form, action) {
        var data = action.result.data;
        if (Ext.isEmpty(data)) data = [];
        callback.call(scope, data, form.getFieldValues().override);
        var msg = action.result.message;
        if (msg) {
          CUF.util.MessageService.warn(msg);
        }
      },
      failure: function (form, action) {
        var msg = action.result.errorMsg;
        if (/^Niezn/.test(msg)) {
          msg = 'Wystapił błąd podczas przetwarzania pliku. \nSprawdź format pliku oraz nazwy kolumn w pliku.';
        }
        CUF.util.MessageService.warn(msg);
      },
    });
  },
});

// @req ../../extension/base/BaseWin.js
Ext.define('CUF.table.view.ImportExcelWin', {
  extend: 'CUF.extension.Base.Window',
  alias: 'widget.cuf_importexcelwin',
  layout: 'fit',
  width: 580,
  modal: true,
  tableService: null,
  tableDefinition: null,
  defaultCfg: {
    saveData: true,
    title: CUF.t('import-excel-file'),
    loadButtonLabel: CUF.t('import'),
    beforeImport: null,
    callback: null,
    filters: [],
  },
  initComponent: function () {
    var cfg = Ext.clone(this.initialConfig.cfg);
    Ext.applyIf(cfg, this.defaultCfg);
    this.tableService = CUF.table.core.TableService;
    this.tableDefinition = this.tableService.getTableDefinition(this.grid, cfg.includeHiddenColumns, cfg.columnsToImport);
    this.title = cfg.title;
    this.saveData = cfg.saveData;
    this.callParent(arguments);
    this.beforeImport = cfg.beforeImport;
    this.importCallback = cfg.callback;
    this.filters = cfg.filters || [];
    this.columnsToImport = cfg.columnsToImport || [];
    this.dataChoosersWithSuspendedAutomappings = cfg.dataChoosersWithSuspendedAutomappings || [];
    this.add(
      Ext.widget('cuf_importexcelform', {
        cfg: this.initialConfig.cfg,
        grid: this.grid,
      })
    );
    this.down('cuf_excelbtn').setText(cfg.loadButtonLabel);
  },
  buttons: [
    {
      xtype: 'cuf_excelbtn',
      handler: function (btn) {
        btn.getWin().readExcel(btn);
      },
      cls: 'x4-accept-button',
    },
    {
      xtype: 'cuf_cancelbtn',
    },
  ],
  readExcel: function (btn) {
    var form = btn.getWinForm();
    form.importExcel(btn.getWin().afterReadExcel, this.columnsToImport, btn.getWin());
  },
  afterReadExcel: function (data, override) {
    if (this.filters.length > 0) {
      data = this.filterData(data);
    }

    this.callBeforeImport(data, this.grid);

    if (override == true) {
      ServiceFactory.getVariableSetService().clear(this.grid);
    }

    this.tableService.writeData(this.grid, data, true);
    if (this.saveData == true) {
      ServiceFactory.getVariableSetService().save(this.grid);
    }
    this.callCallback(data, this.grid);
    this.close();
  },
  callBeforeImport: function (data, grid) {
    if (this.beforeImport) {
      var action = this.resolveFunction(this.beforeImport, [data, grid]);
      action.fn.apply(action.scope, action.arguments);
    }
  },
  callCallback: function (data, grid) {
    if (this.importCallback) {
      var action = this.resolveFunction(this.importCallback, [data, grid]);
      action.fn.apply(action.scope, action.arguments);
    }
  },
  resolveFunction: function (fn, args) {
    var arguments = args || [];
    var action = {
      fn: Ext.isObject(fn) ? fn.fn : fn,
      scope: fn.scope || window,
      arguments: fn.arguments ? arguments.concat(fn.arguments) : arguments,
    };
    return action;
  },
  filterData: function (dataFromExcel) {
    var result = [];
    for (var i = 0; i < dataFromExcel.length; i++) {
      if (this.checkRow(dataFromExcel[i])) {
        result.push(dataFromExcel[i]);
      }
    }
    return result;
  },
  checkRow: function (row) {
    for (var i = 0; i < this.filters.length; i++) {
      var filter = this.filters[i];
      var fieldValue = typeof row[filter.colName] === 'string' ? row[filter.colName].replace(/,/g, '.') : row[filter.colName];
      var filterValue = filter.value.replace(/,/g, '.');
      if (!isNaN(fieldValue) && !isNaN(filterValue)) {
        fieldValue = Number(fieldValue);
        filterValue = Number(filterValue);
      } else {
        fieldValue = row[filter.colName];
        filterValue = filter.value;
      }

      if (fieldValue instanceof Date) {
        fieldValue = fieldValue.dateFormat('Y-m-d');
      }
      var filterPassed = this.filtersCheck[filter.type](fieldValue, filterValue);
      if (!filterPassed) {
        return filterPassed;
      }
    }
    return true;
  },
  filtersCheck: {
    '==': function (cellValue, expectedValue) {
      return cellValue == expectedValue;
    },
    '!=': function (cellValue, compareValue) {
      return cellValue != compareValue;
    },
    '<': function (cellValue, compareValue) {
      return cellValue < compareValue;
    },
    '>': function (cellValue, compareValue) {
      return cellValue > compareValue;
    },
    '<=': function (cellValue, compareValue) {
      return cellValue <= compareValue;
    },
    '>=': function (cellValue, compareValue) {
      return cellValue >= compareValue;
    },
    regexp: function (cellValue, pattern) {
      return pattern.test(cellValue);
    },
    fn: function (cellValue, fn) {
      return fn(cellValue);
    },
  },
  onlySelectedColumns: function (dataFromExcel) {
    var result = [];
    for (var i = 0; i < dataFromExcel.length; i++) {
      var row = this.getRowWithSelectedColumns(dataFromExcel[i]);
      result.push(row);
    }
    return result;
  },
  getRowWithSelectedColumns: function (fullRow) {
    var row = {};
    for (var i = 0; i < this.columnsToImport.length; i++) {
      row[this.columnsToImport[i]] = fullRow[this.columnsToImport[i]];
    }
    return row;
  },
});

Ext.define('CUF.document.controller.DocumentGridController', {
  extend: 'Ext.app.Controller',
  init: function (grid) {
    grid.on('cellclick', this.cellClick, this);
    grid.on('beforeitemcontextmenu', this.beforeItemContextmenu, this);
  },
  cellClick: function (gridView, td, cellIndex, record, tr, rowIndex, e, eOpts) {
    var cols = gridView.headerCt.getVisibleGridColumns();
    var col = cols[cellIndex];

    if (col.isXType('documentinfocolumn')) {
      this.showDocumentInfo(gridView, record);
    } else if (col.isXType('documentactionscolumn')) {
      this.showDocumentActionsMenu(gridView, record, e);
    }
  },
  showDocumentInfo: function (gridView, record) {
    var grid = gridView.up('cuf_documentgrid');
    var window = {
      xtype: 'window',
      id: 'showDocumentInfoWindow',
      title: CUF.t('document-info'),
      modal: true,
      listeners: {
        resize: function () {
          Ext.getCmp('showDocumentInfoWindow').center();
        },
      },
    };
    window = Ext.apply(window, {
      loader: {
        url: 'ShowDocumentInformation.do',
        autoLoad: true,
        scripts: true,
        params: {
          fileId: CUF.util.Encoder.base64.encode(record.get('id')),
          docClassId: record.get('documentClass').id,
        },
      },
    });
    Ext.create('Ext.window.Window', window).show();
  },
  beforeItemContextmenu: function (gridView, record, item, index, e) {
    e.stopEvent();
    var grid = gridView.up('cuf_documentgrid');
    var menu = grid.buildContextMenu(record.data);
    menu.showAt(e.getX() - 50, e.getY());
  },
  showDocumentActionsMenu: function (gridView, record, e) {
    this.beforeItemContextmenu(gridView, record, null, null, e);
  },
});

Ext.define('CUF.document.core.DocumentService', {
  singleton: true,
  hasRightsToDetachDocument: function (fileId, processId, activityId) {
    CUF.util.validation.ParamChecker.notEmpty(fileId, processId, activityId);
    var resp = CUF.util.UtilService.execute({
      url: 'api/documents/' + fileId + '/canDetach',
      async: false,
      method: 'GET',
      params: {
        processId: processId,
        activityId: activityId,
      },
    });
    return resp.responseText == 'true';
  },
  detachDocument: function (fileId, processId, activityId, cfg) {
    CUF.util.validation.ParamChecker.notEmpty(fileId, processId, activityId);
    CUF.util.UtilService.execute(
      Ext.apply(
        {
          url: 'api/documents/' + fileId + '/' + processId,
          method: 'DELETE',
        },
        cfg
      )
    );
  },
  detachDocuments: function (fileIds, processId, activityId, cfg) {
    CUF.util.validation.ParamChecker.notEmpty(fileIds, processId, activityId);
    CUF.util.UtilService.execute(
      Ext.apply(
        {
          url: 'api/documents/detach/' + processId + '/' + activityId,
          params: {
            fileIds: fileIds,
          },
          method: 'POST',
        },
        cfg
      )
    );
  },
  deleteDocuments: function (fileIds, cfg) {
    CUF.util.validation.ParamChecker.notEmpty(fileIds);
    CUF.util.UtilService.execute(
      Ext.apply(
        {
          url: 'api/cuf/document/delete',
          params: {
            fileIds: fileIds,
          },
          method: 'POST',
        },
        cfg
      )
    );
  },
  verifyDocuments: function (verifyDef, cfg) {
    var ais = ServiceFactory.getActivityInfoService();
    verifyDef.processId = ais.getProcessId();
    verifyDef.activityId = ais.getActivityId();
    var result;
    CUF.util.validation.ParamChecker.notEmpty(verifyDef);
    CUF.util.validation.ParamChecker.notEmpty(verifyDef.conditions);
    var def = this._convertVerifyDef(verifyDef);
    CUF.util.UtilService.execute(
      Ext.apply(
        {
          url: 'api/cuf/document/verifier',
          jsonData: def,
          method: 'POST',
          async: false,
          success: function (response) {
            result = Ext.JSON.decode(response.responseText);
          },
        },
        cfg
      )
    );

    return result;
  },
  changeDescription: function (fileId, description, cfg) {
    CUF.util.validation.ParamChecker.notEmpty(fileId);
    CUF.util.UtilService.execute(
      Ext.apply(
        {
          url: 'api/cuf/document/description/change',
          params: {
            fileId: fileId,
            description: description,
          },
          method: 'POST',
        },
        cfg
      )
    );
  },
  _convertVerifyDef: function (def) {
    this._setVerifyDefaults(def);

    def.mode = {
      type: def.verifyMode,
    };
    for (var i = 0; i < def.conditions.length; i++) {
      var c = def.conditions[i];
      c.conditionType = {
        type: c.conditionType,
      };
    }
    return def;
  },
  _setVerifyDefaults: function (def) {
    Ext.applyIf(def, {
      verifyMode: 'activity',
    });
    var conds = [];
    for (var i = 0; i < def.conditions.length; i++) {
      var cond = def.conditions[i];
      Ext.applyIf(cond, {
        documentCount: 1,
        conditionType: '=',
      });
    }
  },
});

Ext.define('CUF.document.core.DocumentView', {
  singleton: true,
  setDocumentsCount: function (count) {
    var title = Ext.getCmp('documents_tab_panel').title;
    var idx = title.indexOf('(');
    title = title.substr(0, idx + 1) + count + ')';
    Ext.getCmp('documents_tab_panel').setTitle(title);
  },
  reloadDocumentsGrid: function () {
    FormService.refreshDocuments();
  },
  refreshDocuments: function () {
    FormService.refreshDocuments();
  },
});

Ext.define('CUF.document.model.Document', {
  extend: 'Ext.data.Model',
  fields: [
    {
      name: 'fileName',
      type: 'string',
    },
    {
      name: 'description',
      type: 'string',
    },
    {
      name: 'fileDate',
      type: 'int',
    },
    {
      name: 'size',
      type: 'int',
    },
    {
      name: 'uploader',
      type: 'string',
    },
    {
      name: 'id',
      type: 'integer',
    },
    {
      name: 'encodedId',
      type: 'integer',
      convert: function (v, model) {
        return CUF.util.Encoder.base64.encode(model.get('id'));
      },
    },
    {
      name: 'documentClass',
    },
    {
      name: 'attacher',
      type: 'string',
    },
  ],
});

Ext.define('CUF.document.service.DocumentService', {
  singleton: true,
  alternateClassName: ['CUF.DocumentService'],
  //---public
  showAddDocumentsFromDiscForm: function (cfg) {
    if (MobileService.isMobile()) {
      var savingAsNewVersionChecked = cfg.savingAsNewVersion ? cfg.savingAsNewVersion.checked : false;
      var savingAsNewVersionReadOnly = cfg.savingAsNewVersion ? cfg.savingAsNewVersion.readOnly : false;

      MobileService.emit("openAddDocumentCufPage", {
        autoSelectFirst: this.defaultBoolean(cfg.autoSelectFirst, true),
        defaultClassId: cfg.defaultClassId,
        defaultClassName: cfg.defaultClassName,
        documentClassIdList: this.defaultNonEmptyArray(cfg.documentClassIdList),
        documentClassNameList: this.defaultNonEmptyArray(cfg.documentClassNameList),
        savingAsNewVersionChecked: this.defaultBoolean(savingAsNewVersionChecked, false ),
        savingAsNewVersionReadOnly: this.defaultBoolean(savingAsNewVersionReadOnly, false ),
        showIndexes: this.defaultBoolean(cfg.showIndexes, false),
      });
    }
    else
    {
      var url = this._buildAddDocumentFromDiscUrl();
      var plugins = [new CUF.document.view.form.DocumentFormPlugin(cfg)];
      this._showAddDocumentsForm(url, plugins);
    }
  },
  defaultNonEmptyArray: function(param) {
    if (!Ext.isEmpty(param) && Ext.isArray(param)) {
      return param;
    }
    return null;
  },
  defaultBoolean: function(param, defaultValue) {
    return param !== null && param !== undefined
      ? !!param
      : defaultValue;
  },
  showAddDocumentsFromArchive: function (cfg) {
    //default
    var settings = new Object();
    settings.indexes = new Object();
    settings.documentClassId = 0;
    Ext.apply(settings, cfg);
    if (!isHistory) {
      this._showSearchInArchiveWindow(settings.documentClassId, settings.indexes);
    }
  },
  showAddDocumentsFromArchiveExtendedWindow: function (cfg) {
    const settings = {
      indexes: {},
      documentClassId: 0,
      sortBy: "",
      sortDirection: "",
      displayedIndexNames: {}
    };
    Ext.apply(settings, cfg);
    if (!isHistory) {
      this._showSearchInArchiveExtendedWindow(settings);
    }
  },
  showProcessDocumentManagement: function (cfg) {
    var gridCfg = Ext.apply({}, cfg);
    var grid = Ext.widget('cuf_processdocumentgrid', gridCfg);
    var win = Ext.widget('cuf_documentmanagementwin');
    win.add(grid);

    let store = grid.getStore();
    let docClassNameFilter = gridCfg.documentClassNameList;
    if (docClassNameFilter?.length) {
      store.filter(item => docClassNameFilter.includes(item.get("documentClass").name));
    }
    store.load();
  },
  showFile: function (fileId) {
    if (MobileService.isMobile()) {
      PW.Logger.debug('Ignoring DocumentService#showFile for mobile');
      return;
    }

    window
      .open(
        'ShowFile.do?fileName=' + CUF.util.Encoder.base64.encode(fileId) + '&processId=' + processId + '&activityId=' + activityId,
        'viewer',
        this._buildWindowParams()
      )
      .focus();
  },
  deleteDocuments: function (fileIds, cfg) {
    CUF.document.core.DocumentService.deleteDocuments(fileIds, cfg);
  },
  hasRightsToDetachDocument: function (fileId, processId, activityId) {
    return CUF.document.core.DocumentService.hasRightsToDetachDocument(fileId, processId, activityId);
  },
  detachDocuments: function (fileIds, processId, activityId, cfg) {
    CUF.document.core.DocumentService.detachDocuments(fileIds, processId, activityId, cfg);
  },
  detachDocument: function (fileId, processId, activityId, cfg) {
    CUF.document.core.DocumentService.detachDocument(fileId, processId, activityId, cfg);
  },
  changeDescription: function (fileId, description, cfg) {
    CUF.document.core.DocumentService.changeDescription(fileId, description, cfg);
  },
  verifyDocuments: function (verifyDef, cfg) {
    var result = CUF.document.core.DocumentService.verifyDocuments(verifyDef, cfg);

    if (!result.valid && (Ext.isEmpty(verifyDef.showMessage) || verifyDef.showMessage == true))
      CUF.util.MessageService.warn(result.message);
    return result.valid;
  },

  addDocumentFromArchive: function (cfg) {
    //default
    var settings = new Object();
    settings.fileId;
    Ext.apply(settings, cfg);
    if (!isHistory) {
      ServiceFactory.getFormService().save({
        scope: this,
        success: this._addDocumentFromArchive,
        successParams: [settings.fileId],
        failure: function (response) {
          ServiceFactory.getMessageService().showFailure(ServiceFactory.getFormService().self.DEFAULT_ERROR_MESSAGE);
        },
      });
    }
  },
  showSendDocumentsForm: function (fileId, cfg) {
    var url = Suncode.getContextPath() + '/' + ServiceFactory.getFormService().self.SEND_DOCUMENTS_AFTER_CHANGE_FROM_DISC;
    this._showSendDocumentsForm(fileId, url, cfg);
  },
  //---private
  _buildAddDocumentFromDiscUrl: function () {
    return Suncode.getContextPath() + '/' + ServiceFactory.getFormService().self.ADD_DOCUMENTS_FROM_DISC_URL;
  },
  _buildWindowParams: function () {
    var body = Ext.getBody();
    var windowParams = 'menubar=no,toolbar=no,location=no,';
    windowParams += 'directories=no,status=no,scrollbars=yes,';
    windowParams += 'resizable=yes,fullscreen=no,channelmode=no,';
    windowParams += 'width=' + body.getWidth() / 2 + ',height=' + body.getHeight() + ',';
    windowParams += 'left=' + body.getWidth() / 2 + ',top=0';
    return windowParams;
  },
  _showAddDocumentsForm: function (url, plugins) {
    if (Ext.isEmpty(plugins)) {
      plugins = [];
    }
    Ext.Ajax.request({
      url: url,
      method: 'GET',
      params: {
        processId: processId,
        activityId: activityId,
      },
      scope: this,
      success: function (response, options) {
        var result = Ext.JSON.decode(response.responseText);
        var window = Ext.JSON.decode(result.json);
        CUF.util.Arrays.each(plugins, 'initWinCfg', window);
        CUF.util.Arrays.each(plugins, 'initFormCfg', window.items[0]);
        var win = Ext.create('Ext.window.Window', window);
        var form = win.down('form');
        form.bodyPadding = 10;
        CUF.util.Arrays.each(plugins, 'initForm', form);
        win.show();
      },
      failure: function (response, options) {
        ServiceFactory.getMessageService().showFailure(ServiceFactory.getFormService().self.DEFAULT_ERROR_MESSAGE);
      },
    });
  },
  _showSearchInArchiveWindow: function (documentClassId, indexes) {
    var params = {
      activityId: activityId,
      documentClassId: documentClassId,
      processId: processId,
    };

    for (var index in indexes) {
      if (indexes.hasOwnProperty(index)) {
        var name = 'indexes[' + index + ']';
        var value = indexes[index];
        params[name] = value;
      }
    }

    Ext.Ajax.request({
      url: 'api/form/documents/showSearchInArchiveWindow',
      method: 'POST',
      params: params,
      scope: this,
      success: function (response, options) {
        var form = Ext.JSON.decode(response.responseText);
        var window = Ext.JSON.decode(form.json);
        Ext.create('Ext.window.Window', window).show();
      },
      failure: function (response, options) {
        ServiceFactory.getMessageService().showFailure(ServiceFactory.getFormService().self.DEFAULT_ERROR_MESSAGE);
      },
    });
  },
  _showSearchInArchiveExtendedWindow: function (settings) {
    const params = {
      activityId: activityId,
      documentClassId: settings.documentClassId,
      processId: processId,
      sortBy: settings.sortBy,
      sortDirection: settings.sortDirection,
      displayedIndexNames: settings.displayedIndexNames
    }
    for (const index in settings.indexes) {
      if (settings.indexes.hasOwnProperty(index)) {
        const name = `indexes[${index}]`;
        params[name] = settings.indexes[index];
      }
    }
    Ext.Ajax.request({
      url: 'api/form/documents/showSearchInArchiveExtendedWindow',
      method: 'POST',
      params: params,
      scope: this,
      success: function (response, options) {
        var form = Ext.JSON.decode(response.responseText);
        var window = Ext.JSON.decode(form.json);
        Ext.create('Ext.window.Window', window).show();
      },
      failure: function (response, options) {
        ServiceFactory.getMessageService().showFailure(ServiceFactory.getFormService().self.DEFAULT_ERROR_MESSAGE);
      },
    });
  },
  _addDocumentFromArchive: function (fileId) {
    var connection = Ext.create('Ext.data.Connection');
    connection.request({
      standardSubmit: true,
      url: 'AddDocumentToProcess.do',
      params: {
        processId: processId,
        activityId: activityId,
        fileId: fileId,
      },
      scope: this,
      callback: function (options, success) {
        if (success) {
          FormService.refreshDocuments();
        } else {
          ServiceFactory.getMessageService().showSuccess(ServiceFactory.getFormService().self.DEFAULT_ERROR_MESSAGE);
        }
      },
    });
  },
  _reloadDocumentsTable: function () {
    var documentsTable = Ext.getCmp('documents_tab_panel').down('documentstable');
    var store = documentsTable.getStore();
    store.reload();
  },
  _refreshDocumentsViewTabPanel: function () {
    var tool = Ext.ComponentQuery.query('refreshdocumentsviewtool')[0];
    tool.fireEvent('click', tool);
  },
  _showSendDocumentsForm: function (fileId, url, cfg) {
    Ext.Ajax.request({
      url: url,
      method: 'GET',
      params: {
        fileId: fileId,
        activityId: activityId,
        processId: processId,
      },
      scope: this,
      success: function (response) {
        var result = Ext.JSON.decode(response.responseText);
        var window = Ext.JSON.decode(result.json);

        window = Ext.apply(
          {
            fileId: fileId,
            activityId: activityId,
            processId: processId,
            config: cfg,
          },
          window
        );

        var win = Ext.create('Suncode.form.view.window.CheckInDocumentWindow', window);
        win.show();
      },
      error: function (error) {
        console.log(error);
      },
    });
  },
});

Ext.define('CUF.document.store.DocumentClassStore', {
  extend: 'Ext.data.Store',
  requires: ['Suncode.form.model.DocumentClass'],
  model: 'Suncode.form.model.DocumentClass',
  proxy: {
    type: 'jsonproxy',
    api: {
      read: Suncode.getContextPath() + '/api/cuf/documentclass',
    },
    reader: {
      root: 'data',
      totalProperty: 'total',
    },
    noCache: false,
  },
});

// @req ParamChecker.js
Ext.define('CUF.util.validation.ViewParamChecker', {
  singleton: true,
  checker: CUF.util.validation.ParamChecker,
  ex: CUF.exception,
  msg: CUF.util.MessageService,
  selectionIsEmpty: function (recs) {
    try {
      this.checker.notEmpty(recs);
      return false;
    } catch (e) {
      this.msg.info(CUF.t('no-selection'));
      return true;
    }
  },
});

Ext.define('CUF.data.core.finder.HibernateFinder', {
  singleton: true,
  getName: function () {
    return 'hibernate';
  },
  buildGrid: function (cfg) {
    if (cfg.gridCfg.paging) {
      return Ext.widget(
        'cuf_paged_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.HibernateStoreCreator.create('CUF.extension.data.JsonFilterBaseStore', cfg),
          },
          cfg.gridCfg
        )
      );
    } else {
      return Ext.widget(
        'cuf_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.HibernateStoreCreator.create('CUF.extension.data.JsonFilterStore', cfg),
          },
          cfg.gridCfg
        )
      );
    }
  },
});

Ext.define('CUF.data.core.finder.LocalFinder', {
  singleton: true,
  getName: function () {
    return 'local';
  },
  buildGrid: function (cfg) {
    if (cfg.gridCfg.paging) {
      return Ext.widget(
        'cuf_paged_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: Ext.create('CUF.extension.data.JsonBaseStore', cfg),
          },
          cfg.gridCfg
        )
      );
    } else {
      return Ext.widget(
        'cuf_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: Ext.create('Ext.data.JsonStore', cfg),
          },
          cfg.gridCfg
        )
      );
    }
  },
});

Ext.define('CUF.data.core.finder.SimpleFinder', {
  singleton: true,
  getName: function () {
    return 'simple';
  },
  buildGrid: function (cfg) {
    if (cfg.gridCfg.paging) {
      return Ext.widget(
        'cuf_paged_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.SimpleStoreCreator.create('CUF.extension.data.JsonFilterBaseStore', cfg),
          },
          cfg.gridCfg
        )
      );
    } else {
      return Ext.widget(
        'cuf_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.SimpleStoreCreator.create('CUF.extension.data.JsonFilterStore', cfg),
          },
          cfg.gridCfg
        )
      );
    }
  },
});

Ext.define('CUF.data.core.finder.WebServiceFinder', {
  singleton: true,
  getName: function () {
    return 'webservice';
  },
  buildGrid: function (cfg) {
    if (cfg.gridCfg.paging) {
      return Ext.widget(
        'cuf_paged_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.HibernateStoreCreator.create('CUF.extension.data.JsonFilterBaseStore', cfg),
          },
          cfg.gridCfg
        )
      );
    } else {
      return Ext.widget(
        'cuf_grid',
        Ext.applyIf(
          {
            cfgOrg: cfg,
            store: CUF.HibernateStoreCreator.create('CUF.extension.data.JsonFilterStore', cfg),
          },
          cfg.gridCfg
        )
      );
    }
  },
});

Ext.define('CUF.data.core.operator.Between', {
  singleton: true,
  getName: function () {
    return 'BETWEEN';
  },
  parse: function (value, field) {
    var secField = field.next('[name=' + field.secondField + ']');
    var value = [this.readEmpty(value), this.readEmpty(secField.getSubmitValue())];
    if (Ext.isEmpty(value[0]) && Ext.isEmpty(value[1])) return null;
    return value;
  },
  readEmpty: function (str) {
    if (Ext.isEmpty(str)) {
      return null;
    }
    return str;
  },
});

Ext.define('CUF.data.core.operator.Eq', {
  singleton: true,
  getName: function () {
    return 'EQ';
  },
  parse: function (value) {
    return value;
  },
});

Ext.define('CUF.data.core.operator.Like', {
  singleton: true,
  getName: function () {
    return 'LIKE';
  },
  parse: function (value) {
    return value;
  },
});

// @req ../../../extension/base/BaseWin.js
/**
 * Kreator szablonu formularza.
 */
Ext.define('CUF.form.template.view.FormTemplateCreator', {
  extend: 'CUF.extension.Base.Window',
  layout: 'fit',
  title: CUF.t('form-template-creator-title'),
  alternateClassName: ['CUF.FormTemplateCreator'],
  autoShow: false,
  border: false,
  closeAction: 'hide',
  width: 750,
  maxHeight: 700,
  padding: 10,
  statics: {
    lastTemplateName: '',
  },
  constructor: function () {
    var ddGroupInclude = 'form-template-include',
      ddGroupExclude = 'form-template-exclude';

    Ext.apply(this, {
      items: [
        {
          xtype: 'form',
          border: false,
          defaultType: 'textfield',
          layout: {
            type: 'vbox',
            align: 'stretch',
          },
          defaults: {
            labelWidth: 150,
          },
          items: [
            {
              fieldLabel: CUF.t('form-template-name'),
              name: 'templateName',
              allowBlank: false,
              validator: function (value) {
                return value != 'default' && value != 'delete';
              },
            },
            {
              xtype: 'checkbox',
              fieldLabel: CUF.t('form-template-creator-setdefault'),
              name: 'setDefault',
              checked: true,
            },
            {
              xtype: 'checkbox',
              fieldLabel: CUF.t('form-template-creator-overwrite'),
              name: 'setOverwriteTemplate',
              checked: false,
            },
            {
              xtype: 'container',
              itemId: 'templateFields',
              layout: {
                type: 'hbox',
                align: 'stretch',
              },
              flex: 1,
              defaults: {
                xtype: 'grid',
                selType: 'checkboxmodel',
                columns: {
                  items: [
                    {
                      text: CUF.t('type'),
                      dataIndex: 'type',
                      renderer: this.templateEntryTypeRenderer,
                      flex: 1,
                    },
                    {
                      text: CUF.t('name'),
                      dataIndex: 'name',
                      renderer: this.templateNameRenderer,
                      flex: 1,
                    },
                  ],
                },
                features: [
                  {
                    ftype: 'grouping',
                    groupHeaderTpl: [
                      '<div>{name:this.formatName}</div>',
                      {
                        formatName: function (name) {
                          return CUF.util.Renderers.translationRenderer(name, 'form-template-creator');
                        },
                      },
                    ],
                    hideGroupedHeader: true,
                  },
                ],
              },
              items: [
                {
                  title: CUF.t('form-template-creator-includedfields'),
                  itemId: 'includeGrid',
                  flex: 1,
                  store: this.createIncludedVariablesStore(),
                  viewConfig: {
                    plugins: {
                      ptype: 'gridviewdragdrop',
                      dragGroup: ddGroupInclude,
                      dropGroup: ddGroupExclude,
                    },
                  },
                },
                this.createIncludeExcludeButtons(),
                {
                  title: CUF.t('form-template-creator-excludedfields'),
                  itemId: 'excludeGrid',
                  flex: 1,
                  store: this.createExcludedVariablesStore(),
                  viewConfig: {
                    plugins: {
                      ptype: 'gridviewdragdrop',
                      dragGroup: ddGroupExclude,
                      dropGroup: ddGroupInclude,
                    },
                  },
                },
              ],
            },
          ],
        },
      ],
      buttons: [
        {
          text: CUF.t('save'),
          iconCls: 'dvnt-icon-save',
          handler: this.saveTemplate,
          scope: this,
        },
        {
          text: CUF.t('cancel'),
          handler: this.close,
          scope: this,
        },
      ],
      listeners: {
        afterRender: function (thisForm, options) {
          this.keyNav = Ext.create('Ext.util.KeyNav', this.el, {
            enter: this.saveTemplate,
            scope: this,
          });
        },
      },
    });

    this.callParent(arguments);
  },

  createIncludeExcludeButtons: function () {
    return {
      xtype: 'container',
      width: 45,
      layout: {
        type: 'vbox',
        align: 'center',
        pack: 'center',
      },
      defaults: {
        width: 35,
        margin: '5 0',
        scope: this,
      },
      items: [
        {
          xtype: 'button',
          iconCls: 'dvnt-icon-chevron-right',
          handler: this.excludeSelectedVariable,
        },
        {
          xtype: 'button',
          iconCls: 'dvnt-icon-chevron-left',
          handler: this.includeSelectedVariable,
        },
      ],
    };
  },

  initComponent: function () {
    this.on('show', this.onShowEvent, this);
    this.callParent(arguments);
  },

  onShowEvent: function () {
    this.getForm().findField('templateName').focus(false, 200);

    if (!this.loadMask) {
      this.loadMask = new Ext.LoadMask({
        msg: CUF.t('form-template-creator-saving'),
        target: this,
      });
    }
  },

  createIncludedVariablesStore: function () {
    this.includedStore = this.includedStore || this.createVariablesStore();
    return this.includedStore;
  },

  createExcludedVariablesStore: function () {
    this.excludedStore = this.excludedStore || this.createVariablesStore();
    return this.excludedStore;
  },

  createVariablesStore: function () {
    return Ext.create('Ext.data.Store', {
      fields: ['type', 'id', 'name', 'value'],
      groupField: 'type',
      autoDestroy: true,
      proxy: {
        type: 'memory',
      },
    });
  },

  includeSelectedVariable: function () {
    this.moveVariables('#excludeGrid', '#includeGrid');
  },
  excludeSelectedVariable: function () {
    this.moveVariables('#includeGrid', '#excludeGrid');
  },
  moveVariables: function (sourceQry, destQry) {
    var source = this.down(sourceQry),
      dest = this.down(destQry);

    var records = source.getSelectionModel().getSelection();
    if (!Ext.isEmpty(records)) {
      source.getStore().remove(records);
      dest.getStore().add(records);
    }
  },

  setup: function (config) {
    // apply defaults
    this.config = Ext.applyIf(config || {}, {
      setDefault: false,
      allowFieldsSelection: true,
      category: undefined,
      fields: null,
      filter: undefined,
      scope: undefined,
    });

    // setup
    this.reset();
    this.includedStore.add(this.collectTemplateFields());
  },

  saveTemplate: function () {
    var form = this.getForm();

    if (form.isValid()) {
      this.loadMask.show();
      CUF.FormTemplateService.saveTemplate(
        {
          name: form.findField('templateName').getValue(),
          defaultTemplate: form.findField('setDefault').getValue(),
          category: this.config.category,
          fields: this.getTemplateFields(),
          grids: this.getTemplateGrids(),
        },
        function (success, message) {
          this.loadMask.hide();

          if (!success) {
            //Sprawdzenie czy chechbox "Nadpisz szablon" jest ustawionyna true
            if (form.findField('setOverwriteTemplate').getValue()) {
              //tworzenie zmiennej template
              var ais = ServiceFactory.getActivityInfoService();
              var templateToDelete = {
                activityType: ais.getActivityDefId(),
                category: this.config.category,
                fields: this.getTemplateFields(),
                grids: this.getTemplateFields(),
                name: form.findField('templateName').getValue(),
                owner: Suncode.getCurrentUser(),
                processType: ais.getProcessDefId(),
              };
              //usuniecie aktualnego templeta z Bazy
              CUF.FormTemplateService.deleteTemplate(
                templateToDelete,
                function (success, message) {
                  if (success) {
                    //PO popprawnym usunieciu zapisanie nowego szablonu o tej samej nazwie (nadpisanie)
                    CUF.FormTemplateService.saveTemplate(
                      {
                        name: form.findField('templateName').getValue(),
                        defaultTemplate: form.findField('setDefault').getValue(),
                        category: this.config.category,
                        fields: this.getTemplateFields(),
                        grids: this.getTemplateGrids(),
                      },
                      function (success, message) {
                        if (success) {
                          //Komunikat potwierdzający nadpisanie szablonu
                          var templateName = form.findField('templateName').getValue();
                          var info = CUF.t('msg-overwrite-template-confirm');
                          info = info.replace('@templateName@', templateName);
                          CUF.util.MessageService.warn(info);

                          return;
                        } else {
                          //Komunikat o błedzie zapisu
                          message = message || CUF.t('msg-unknownerror');
                          CUF.util.MessageService.warn(message);
                          return;
                        }
                      },
                      this
                    );
                  } else {
                    //Komunikat o błedzie usuniecia
                    message = message || CUF.t('msg-unknownerror');
                    CUF.util.MessageService.warn(message);
                    return;
                  }
                },
                this
              );
            } else {
              //Komunikat o błędzie zapisu
              message = message || CUF.t('msg-unknownerror');
              CUF.util.MessageService.warn(message);
              return;
            }
            //----------------------------------
          }
          this.close();
        },
        this
      );
    }
  },

  getTemplateFields: function () {
    var fields = [];

    this.includedStore.each(function (field) {
      if (field.get('type') == 'variable') {
        fields.push({
          id: field.get('id'),
          value: field.get('value'),
        });
      }
    });

    return fields;
  },

  getTemplateGrids: function () {
    var grids = [];

    this.includedStore.each(function (field) {
      if (field.get('type') == 'variableset') {
        grids.push({
          id: field.get('id'),
          title: field.get('name'),
          data: field.get('value'),
        });
      }
    });

    return grids;
  },

  reset: function () {
    var hideDefault = false;

    if (this.config && this.config.hideCheckboxDefault) {
      hideDefault = this.config.hideCheckboxDefault;
    }
    //ustawienia Checkbox "setOverwriteTemplate"
    var hideOverwriteTemplate = true;
    var setOverwriteTemplate = false;
    //!=undefined bo wartosc moze byc false, co chcemy ustawic
    if (this.config && this.config.hideCheckboxOverwriteTemplate != undefined) {
      hideOverwriteTemplate = this.config.hideCheckboxOverwriteTemplate;
    }
    if (this.config && this.config.setOverwriteTemplate) {
      setOverwriteTemplate = this.config.setOverwriteTemplate;
    }

    this.getForm().reset();
    //checkbox Zapisz domyślnie
    this.getForm().findField('setDefault').setValue(this.config.setDefault);
    this.getForm().findField('setDefault').setVisible(!hideDefault);
    //checkbox Nadpisz szablon
    this.getForm().findField('setOverwriteTemplate').setValue(setOverwriteTemplate);
    this.getForm().findField('setOverwriteTemplate').setVisible(!hideOverwriteTemplate);
    this.getTemplateFieldsContainer().setVisible(this.config.allowFieldsSelection);
    this.center();
    this.clearStores();
    //odczytyje ostatnio użytą nazwe szablonu (zapisana lub odczytana) i sutawia domyślną wartośc pola
    this.getForm().findField('templateName').setValue(CUF.FormTemplateCreator.lastTemplateName);
  },

  clearStores: function () {
    this.excludedStore.removeAll();
    this.includedStore.removeAll();
  },

  collectTemplateFields: function () {
    var fields = this.config.fields;
    if (Ext.isArray(fields) && Ext.isEmpty(fields)) {
      return [];
    }

    var variables = this.getVariables(fields),
      variableSets = this.getVariableSets(fields);

    return this.filterFields(variables.concat(variableSets));
  },

  getVariables: function (fields) {
    var fs = ServiceFactory.getFormService(),
      vs = ServiceFactory.getVariableService(),
      values = fs.getValues(),
      merge = Ext.isArray(fields),
      result = [];

    var vars = fs.getVariablesInfo();
    for (var varId in vars) {
      if (vars.hasOwnProperty(varId)) {
        var info = vars[varId];
        if (!info.inVariableSet) {
          var good = !merge;
          if (merge) {
            for (var i = 0; i < fields.length; i++) {
              var field = fields[i];
              if (field.type == 'variable' && field.id == varId) {
                good = true;
                continue;
              }
            }
          }

          if (good) {
            result.push({
              type: 'variable',
              id: varId,
              name: vs.getName(varId),
              value: values[varId],
            });
          }
        }
      }
    }
    return result;
  },

  getVariableSets: function (fields) {
    var vss = ServiceFactory.getVariableSetService(),
      merge = Ext.isArray(fields),
      variableSets = [];

    if (this.config != undefined && this.config.columnsId != undefined) {
      var columnsId = this.config.columnsId;
      var isArrayColumnId = Ext.isArray(columnsId);
    }

    Ext.each(vss.getAll(), function (variableSet, index) {
      var info = vss.getInfo(variableSet);
      var good = !merge;
      if (merge) {
        for (var i = 0; i < fields.length; i++) {
          var field = fields[i];
          if (field.type == 'variableset' && (field.id == info.originalName || field.id == info.id)) {
            good = true;
            continue;
          }
        }
      }

      var data = [];
      if (vss.getRowsCount(variableSet) > 0) {
        data = Ext.decode(vss.getVariableSetData(variableSet));
        if (columnsId != undefined) {
          var selectedData = [];
          Ext.each(data, function (obj, index) {
            var tempObject = new Object();
            for (var property in obj) {
              if (obj.hasOwnProperty(property)) {
                if (isArrayColumnId) {
                  if (Ext.Array.contains(columnsId, property)) {
                    tempObject[property] = obj[property];
                  }
                } else {
                  if (property == columnsId) {
                    tempObject[property] = obj[property];
                  }
                }
              }
            }
            selectedData.push(tempObject);
          });
          data = selectedData;
        }
      }

      if (good) {
        variableSets.push({
          type: 'variableset',
          id: info.id,
          name: info.originalName,
          value: data,
        });
      }
    });

    return variableSets;
  },

  filterFields: function (fields) {
    var filter = this.config.filter || Ext.emptyFn,
      filterFn = Ext.isFunction(filter) ? filter : this.createFieldTypeFilter(filter),
      scope = this || this.config.scope,
      result = [];

    Ext.each(
      fields,
      function (field) {
        if (filterFn.call(scope, field) !== false) {
          result.push(field);
        }
      },
      this
    );

    return result;
  },

  createFieldTypeFilter: function (type) {
    return function (field) {
      if (field.type != type) {
        return false;
      }
      return true;
    };
  },

  templateEntryTypeRenderer: function (value) {
    return CUF.util.Renderers.translationRenderer(value, 'form-template-creator');
  },

  templateNameRenderer: function (value, metaData, record) {
    if (!Ext.isEmpty(value)) {
      return value;
    }

    var type = record.get('type');
    if (type == 'variable') {
      return value;
    }

    return '<i>' + CUF.t('form-template-creator-notitle') + '</i>';
  },

  getForm: function () {
    this._form = this._form || this.down('form').getForm();
    return this._form;
  },

  getTemplateFieldsContainer: function () {
    this._templateFieldsContainer = this._templateFieldsContainer || this.down('container#templateFields');
    return this._templateFieldsContainer;
  },
});

// @req ../../../extension/base/BaseWin.js
/**
 * Kreator szablonu formularza.
 */
Ext.define('CUF.form.template.view.FormTemplateLoader', {
  extend: 'CUF.extension.Base.Window',
  layout: 'fit',
  title: CUF.t('form-template-loader-title'),
  border: false,
  width: 800,
  padding: 10,
  minHeight: 300,
  maxHeight: 500,

  constructor: function (config) {
    this.setup(config);
    var hideDefaultTemplate = false;

    if (config != undefined && config.hideDefaultTemplate != undefined) {
      hideDefaultTemplate = config.hideDefaultTemplate;
    }
    var hideOwner = false;

    if (config != undefined && config.hideOwner != undefined) {
      hideOwner = config.hideOwner;
    }

    var store = Ext.create('Ext.data.Store', {
      fields: ['name', 'category', 'owner', 'defaultTemplate', 'processType', 'activityType', 'fields', 'grids', 'version'],
      proxy: {
        type: 'ajax',
        url: 'api/cuf/form/templates/query',
        extraParams: this.buildParams(),
        timeout: 900000,
      },
      sorters: [
        {
          property: 'name',
          direction: 'ASC',
        },
      ],
      autoLoad: true,
    });

    Ext.apply(this, {
      items: [
        {
          xtype: 'grid',
          store: store,
          columns: {
            items: [
              {
                text: CUF.t('form-template-name'),
                dataIndex: 'name',
                flex: 2,
              },
              {
                text: CUF.t('owner'),
                dataIndex: 'owner',
                flex: 1,
                hidden: hideOwner,
              },
              {
                text: CUF.t('default'),
                dataIndex: 'defaultTemplate',
                renderer: function (value, metaData, record, rowIndex, colIndex, store) {
                  var cls = this._isDefault(record) ? 'dvnt-icon-check-symbol' : 'dvnt-icon-x-symbol';
                  metaData.style = 'text-align: center';
                  return '<span style="display: inline-block; width:20px" class="' + cls + '">&nbsp;</span>';
                },
                scope: this,
                width: 100,
                hidden: hideDefaultTemplate,
              },
            ],
          },
        },
      ],
      buttons: [
        {
          text: CUF.t('load'),
          iconCls: 'dvnt-icon-download',
          handler: this.loadTemplate,
          scope: this,
        },
        {
          text: CUF.t('form-template-loader-setasdefault'),
          iconCls: 'dvnt-icon-check-symbol',
          handler: this.setDefaultTemplate,
          scope: this,
          hidden: hideDefaultTemplate,
        },
        // TODO: podgląd
        {
          text: CUF.t('delete'),
          iconCls: 'dvnt-icon-x-symbol',
          handler: this.deleteTemplate,
          scope: this,
        },
        {
          text: CUF.t('cancel'),
          handler: this.close,
          scope: this,
        },
      ],
    });

    this.callParent(arguments);
  },

  setup: function (config) {
    this.config = Ext.applyIf(config || {}, {
      category: undefined,
      activityScope: true,
      userScope: true,
    });
  },

  buildParams: function () {
    var ais = ServiceFactory.getActivityInfoService(),
      params = {
        processId: encodeURIComponent(processId),
        activityId: encodeURIComponent(activityId),
        process: ais.getProcessDefId(),
        category: this.config.category,
      };

    if (this.config.userScope == true) {
      params.user = Suncode.getCurrentUser();
    }

    if (this.config.activityScope == true) {
      params.activity = ais.getActivityDefId();
    }

    return params;
  },

  loadTemplate: function () {
    var template = this._selectedTemplate();
    if (!template) {
      return;
    }
    var templateDto = template.getData();
    FormService.mask();
    CUF.util.UtilService.execute({
      url: 'api/cuf/form/templates/one',
      method: 'GET',
      params: {
        user: templateDto.owner,
        processId: encodeURIComponent(processId),
        activityId: encodeURIComponent(activityId),
        processType: templateDto.processType,
        activityType: templateDto.activityType,
        templateName: templateDto.name,
        category: templateDto.category,
      },
      callback: function (opts, success, response) {
        if (success) {
          var template = Ext.decode(response.responseText);
          CUF.FormTemplateService.loadTemplate(template);
        }
        FormService.unmask();
      },

      scope: this,
    });
    this.close();
  },

  setDefaultTemplate: function () {
    var template = this._selectedTemplate();
    if (!template) {
      return;
    }

    if (template.get('owner') != Suncode.getCurrentUser()) {
      CUF.util.MessageService.warn(CUF.t('form-template-not-owner'));
      return;
    }

    CUF.FormTemplateService.setDefaultTemplate(
      template.getData(),
      function (success, message) {
        if (success) {
          // wszystkie na false ten na true
          template.store.each(function (record) {
            if (this._isDefault(record)) {
              record.set('defaultTemplate', false);
            }
          }, this);

          template.set('defaultTemplate', true);
          template.store.commitChanges();
          return;
        }
      },
      this
    );
  },

  deleteTemplate: function () {
    var template = this._selectedTemplate();
    if (!template) {
      return;
    }

    if (template.get('owner') != Suncode.getCurrentUser()) {
      CUF.util.MessageService.warn(CUF.t('form-template-not-owner'));
      return;
    }

    CUF.FormTemplateService.deleteTemplate(
      template.getData(),
      function (success, message) {
        if (success) {
          template.store.remove(template);
          return;
        }
        CUF.util.MessageService.warn(message || CUF.t('msg-unknownerror'));
      },
      this
    );
  },

  _selectedTemplate: function () {
    var grid = this.down('grid'),
      sm = grid.getSelectionModel();

    if (sm.hasSelection()) {
      return sm.getSelection()[0];
    }
    return null;
  },

  _isDefault: function (template) {
    return template.get('defaultTemplate') === true && template.get('owner') == Suncode.getCurrentUser();
  },
});

// @req ../view/FormTemplateCreator.js
// @req ../view/FormTemplateLoader.js
/**
 * Szablony formularza.
 *
 * Serwis umożliwia tworzenie i wczytywanie szablonów formularza.
 */
Ext.define('CUF.form.template.service.FormTemplateService', {
  singleton: true,
  alternateClassName: ['CUF.FormTemplateService'],
  mixins: {
    observable: 'Ext.util.Observable',
  },

  constructor: function () {
    this.mixins.observable.constructor.call(this);
    this.addEvents(
      'beforeload',
      'load',
      'beforesetdefault',
      'setdefault',
      'beforedelete',
      'delete',
      'loader.open',
      'loader.close',
      'creator.open',
      'creator.close'
    );
  },

  /**
   * Wyświetla kreator tworzenia nowego szablonu formularza.
   *
   * @param Object config - obiekt konfiguracyjny kreatora szablonu:
   * 						- setDefault Boolean: czy ustawić ten szablon jako szablon domyślny (domyślnie true)
   * 						- category String: kategoria szablonu
   *						- allowFieldsSelection Boolean: czy pokazywać formualrz wyboru pól wchodzących w skład szablonu (domyślnie true)
   * 						- filter Function|String: funkcja filtrująca
   *						- scope Object: scope w którym wywołana będzie funkcja filtrujaca
   *						- fields Array: zbiór pól wchodzących w skład szablony (domyślnie wszystkie pola formularza)
   */
  openTemplateCreator: function (config) {
    if (!this.creator) {
      this.creator = Ext.create('CUF.form.template.view.FormTemplateCreator');
      this.creator.on(
        'close',
        function () {
          this.fireEvent('creator.close', config);
        },
        this
      );
    }

    if (this.creator.isHidden()) {
      this.creator.setup(config);
      this.creator.show();
      this.fireEvent('creator.open', config);
    }
  },

  /**
   * Wyświetla okno ładowania szablonów formularza.
   *
   * @param Object config - obiekt konfiguracyjny kreatora szablonu:
   * 					   - userScope Boolean: czy wyświetlić szablony tylko aktualnego użytkownika (domyślnie true)
   * 					   - activityScope Boolean: czy wyświetlić szablony tylko dla aktualnego zadania (domyślnie true)
   * 					   - category String: kategoria szablonu
   */
  openTemplateLoader: function (config) {
    var loader = Ext.create('CUF.form.template.view.FormTemplateLoader', config);
    loader.on(
      'close',
      function () {
        this.fireEvent('loader.close', config);
      },
      this
    );

    loader.show();
    this.fireEvent('loader.open', config);
  },

  /**
   * Zapisuje szablon formularza na podstawie podanej definicji. Definicja zawiera
   * pola formularza, których wartości mają zostać zapisane.
   *
   * Struktura szablonu jest następująca:
   * {
   *	   name: // String: unikalna nazwa szablonu
   *	   defaultTemplate: // Boolean: czy zapisać szblon jako domyślny
   *	   category: // String (optional): kategoria szablonu
   *     fields: [] // tablica pól formularza
   *     grids: [] // tablica tabelek formularza
   * }
   *
   * Struktura obiektu pól:
   * {
   *     id: // String: identyfikator pola (id zmiennej)
   *     value: // String: wartość tego pola
   * }
   *
   * Struktura obiektu tabelek:
   * {
   *     title: // String: identyfikator tabelki (tytuł)
   *     data: // Array: tablica wierszy tabelki, każdy wiersz to obiekt z nazwą zmiennej i jej wartością
   * }
   *
   * @param Object template (required) zapisywany szablon
   * @param Function callback funkcja wywołana po zapisaniu szablony, przyjmuje następujące parametry:
   * 					- Boolean success czy akcja zakończyła się sukcesem
   * 					- String message wiadomość serwera
   * @param Object scope scope w którym zostanie wywołany callback
   */
  saveTemplate: function (template, callback, scope) {
    var ais = ServiceFactory.getActivityInfoService();
    //zapisuje nazwe zapisanego szablonu do zmiennej static w danej klasie
    CUF.FormTemplateCreator.lastTemplateName = template.name;
    var formTemplate = Ext.apply(template, {
      processType: ais.getProcessDefId(),
      activityType: ais.getActivityDefId(),
    });

    // create template
    CUF.util.UtilService.execute({
      url: 'api/cuf/form/templates',
      method: 'POST',
      jsonData: formTemplate,
      timeout: 300000,
      callback: function (opts, success, response) {
        this._callCallback(callback, scope, success, response, this._readMessage(response));
      },
      scope: this,
    });
  },

  /**
   * Pobiera szablon podanego użytkownika o podanej nazwie. W przypadku braku użytkownika, pobierany jest szablon dla aktualnego użytkownika.
   *
   * @param String templateName (required) nazwa szablonu
   * @param String user nazwa użytkownika
   * @param String category kategoria szablonu
   * @param Function callback funkcja wywołana po pobraniu szablonu, przyjmuje następujące parametry:
   * 					- Boolean success czy akcja zakończyła się sukcesem
   * 					- Object template szablon formularza
   * @param Object scope scope w którym zostanie wywołany callback
   */
  getTemplate: function (templateName, user, category, callback, scope) {
    user = user || Suncode.getCurrentUser();

    CUF.util.UtilService.execute({
      url: this._getTemplateApiUrl(user, category) + '/' + templateName,
      method: 'GET',
      callback: function (opts, success, response) {
        if (success) {
          var template = this._readTemplate(response);
          this._callCallback(callback, scope, success, response, template);
          return;
        }

        this._callCallback(callback, scope, success, response, this._readMessage(response));
      },
      scope: this,
    });
  },

  /**
   * Pobiera domyślny szablon podanego użytkownika. W przypadku braku użytkownika, pobierany jest szablon dla aktualnego użytkownika.
   *
   * @param String user nazwa użytkownika
   * @param Function callback funkcja wywołana po pobraniu szablonu, przyjmuje następujące parametry:
   * 					- Boolean success czy akcja zakończyła się sukcesem
   * 					- Object template szablon formularza
   * @param Object scope scope w którym zostanie wywołany callback
   */
  getDefaultTemplate: function (user, category, callback, scope) {
    user = user || Suncode.getCurrentUser();

    CUF.util.UtilService.execute({
      url: this._getDefaultTemplateApiUrl(user, category),
      method: 'GET',
      callback: function (opts, success, response) {
        if (success) {
          var template = this._readTemplate(response);
          this._callCallback(callback, scope, success, response, template);
          return;
        }

        this._callCallback(callback, scope, success, response, this._readMessage(response));
      },
      scope: this,
    });
  },

  /**
   * Usuwa podany szablon aktualnego użytkownika.
   *
   * @param Object/String template (required) szablon
   * @param Function callback funkcja wywołana po zapisaniu szablony, przyjmuje następujące parametry:
   * 					- Boolean success czy akcja zakończyła się sukcesem
   * 					- String message wiadomość serwera
   * @param Object scope scope w którym zostanie wywołany callback
   */
  deleteTemplate: function (template, callback, scope) {
    if (false === this.fireEvent('beforedelete', template)) {
      return;
    }

    CUF.util.UtilService.execute({
      url: this._getTemplateApiUrl(template),
      method: 'DELETE',
      callback: function (opts, success, response) {
        if (success) {
          this.fireEvent('delete', template);
        }

        this._callCallback(callback, scope, success, response, this._readMessage(response));
      },
      scope: this,
    });
  },

  /**
   * Ustawia szablon jako domyślny aktualnego użytkownika
   *
   * @param String template (required) szablon
   * @param Function callback funkcja wywołana po zapisaniu szablony, przyjmuje następujące parametry:
   * 					- Boolean success czy akcja zakończyła się sukcesem
   * 					- String message wiadomość serwera
   * @param Object scope scope w którym zostanie wywołany callback
   */
  setDefaultTemplate: function (template, callback, scope) {
    if (false === this.fireEvent('beforesetdefault', template)) {
      return;
    }

    template.defaultTemplate = true;
    CUF.util.UtilService.execute({
      url: 'api/cuf/form/templates',
      method: 'PUT',
      jsonData: template,
      callback: function (opts, success, response) {
        if (success) {
          this.fireEvent('setdefault', template);
        }

        this._callCallback(callback, scope, success, response, this._readMessage(response));
      },
      scope: this,
    });
  },

  /**
   * Wczytuje domyślny szablon aktualnego użytkownika
   */
  loadDefaultTemplate: function (category) {
    CUF.FormTemplateService.getDefaultTemplate(
      Suncode.getCurrentUser(),
      category,
      function (success, template) {
        if (success) {
          CUF.FormTemplateService.loadTemplate(template);
        }
      },
      this
    );
  },

  /**
   * Wczytuje szablon formularza podanego użytkownika o podanej nazwie
   *
   * @param Object template (required) wczytany formularza zgodny (@see #saveTemplate)
   */
  loadTemplate: function (template) {
    if (false === this.fireEvent('beforeload', template)) {
      return;
    }

    var vs = ServiceFactory.getVariableService(),
      vss = ServiceFactory.getVariableSetService();
    //zapisuje nazwe wczytanego szablonu do zmiennej static w danej klasie
    CUF.FormTemplateCreator.lastTemplateName = template.name;
    // Ładowanie formularza
    if (!Ext.isEmpty(template.fields)) {
      Ext.each(
        template.fields,
        function (field) {
          vs.setValue(field.id, field.value);
        },
        this
      );
    }

    // Tabelki
    if (!Ext.isEmpty(template.grids)) {
      var variableSetsId = {};
      var variableSetsName = {};

      Ext.each(
        vss.getAll(),
        function (variableSet) {
          var info = vss.getInfo(variableSet);

          variableSetsId[info.id] = variableSet;
          if (!Ext.isEmpty(info.originalName)) {
            variableSetsName[info.originalName] = variableSet;
          }
        },
        this
      );

      Ext.each(
        template.grids,
        function (grid, index) {
          var variableSet = variableSetsId[grid.id];
          if (!Ext.isDefined(variableSet)) {
            variableSet = variableSetsName[grid.title];
          }

          if (!Ext.isDefined(variableSet)) {
            return;
          }

          // clear and fill
          vss.clear(variableSet);

          if (!Ext.isEmpty(grid.data)) {
            vss.addRow(variableSet, false, grid.data, true);
          }
        },
        this
      );
    }
    this.fireEvent('load', template);
  },

  _getTemplateApiUrl: function (user, category, prefix) {
    var url = 'api/cuf/form/templates',
      ais = ServiceFactory.getActivityInfoService(),
      enc = this._encode;

    if (!Ext.isEmpty(prefix)) {
      url += prefix;
    }

    // Template object
    if (Ext.isObject(user)) {
      var template = user;
      url += '/' + enc(template.processType) + '/' + enc(template.activityType);
      url += '/' + this._escapeUser(Ext.isEmpty(template.owner) ? Suncode.getCurrentUser() : template.owner);
      if (!Ext.isEmpty(template.category)) {
        url += '/' + enc(template.category);
      }

      url += '/' + this._escapeTemplateName(template.name);
    } else {
      url += '/' + enc(ais.getProcessDefId()) + '/' + enc(ais.getActivityDefId());
      url += '/' + this._escapeUser(Ext.isEmpty(user) ? Suncode.getCurrentUser() : user);

      if (!Ext.isEmpty(category)) {
        url += '/' + enc(category);
      }
    }

    return url;
  },

  _getDefaultTemplateApiUrl: function (user, category) {
    var url = 'api/cuf/form/templates/default',
      ais = ServiceFactory.getActivityInfoService(),
      enc = this._encode;

    url += '/' + enc(ais.getProcessDefId());
    url += '/' + this._escapeUser(Ext.isEmpty(user) ? Suncode.getCurrentUser() : user);

    if (!Ext.isEmpty(category)) {
      url += '/' + enc(category);
    }

    return url;
  },

  _escapeUser: function (user) {
    // użytkownicy domenowi mają / w nazwie
    // tomcat domyślnie nie zezwala na kodowanie tego znaku jako %2F
    // ze względów bezpieczeństwa

    // escape'ujemy ten znak sami innym znakiem
    return this._encode((user = user.replace(/\//g, ':')));
  },
  _escapeTemplateName: function (tamplateName) {
    // Nazwy mogą mieć znak /
    // tomcat domyślnie nie zezwala na kodowanie tego znaku jako %2F
    // ze względów bezpieczeństwa

    // Kodujemy całośc w base64
    return btoa(this._encode(tamplateName));
  },
  _encode: function (s) {
    return encodeURIComponent(s);
  },

  _readTemplate: function (response) {
    return Ext.decode(response.responseText);
  },

  _callCallback: function (callback, scope, success, response, params) {
    params = params || [];
    if (!Ext.isArray(params)) {
      params = [params];
    }

    if (Ext.isFunction(callback)) {
      callback.apply(scope || this, [success].concat(params));
    }
  },

  _readMessage: function (response) {
    try {
      return Ext.decode(response.responseText).message;
    } catch (Error) {
      // ignore
    }
  },
});

Ext.define('CUF.document.view.action.ActionFactory', {
  singleton: true,
  text: CUF.t('attach-document'),
  actions: {},
  getAction: function (actionDef, cfg) {
    var name = null;
    if (Ext.isString(actionDef)) {
      name = actionDef;
    } else if (Ext.isObject(actionDef)) {
      name = actionDef.name;
    } else {
      throw 'unsupported action type';
    }

    var me = CUF.document.view.action.ActionFactory;
    var fullClassName = null;
    if (Ext.isEmpty(me.actions[name])) {
      var actions = CUF.document.view.action;
      for (var actionClass in actions) {
        if (actions.hasOwnProperty(actionClass)) {
          if (Ext.isDefined(actions[actionClass].getName) && actions[actionClass].getName() == name) {
            fullClassName = 'CUF.document.view.action.' + actionClass;
            me.actions[name] = fullClassName;
            break;
          }
        }
      }
    } else {
      fullClassName = me.actions[name];
    }
    if (Ext.isEmpty(cfg)) {
      cfg = {};
    }
    if (Ext.isObject(actionDef)) {
      Ext.apply(cfg, actionDef.cfg);
    }
    return Ext.create(fullClassName, { cfg: cfg });
  },
});

Ext.define('CUF.document.view.action.AttachAction', {
  statics: {
    getName: function () {
      return 'attach';
    },
  },
  config: {
    cfg: null,
  },
  constructor: function (cfg) {
    this.initConfig(cfg);
  },
  position: ['tbar'],
  getAction: function () {
    var me = this;
    return {
      iconCls: 'dvnt-icon-file-plus',
      text: CUF.t('attach-document'),
      handler: function (btn) {
        me.execute(btn);
      },
    };
  },
  execute: function (btn) {
    var grid = btn.up('grid');
    this.addReloadListener(grid);
    CUF.DocumentService.showAddDocumentsFromDiscForm(this.cfg);
  },
  addReloadListener: function (grid) {
    var callback = this.cfg.callback;
    var afterAdd = function () {
      grid.reload();
    };
    if (Ext.isEmpty(callback)) {
      this.cfg.callback = afterAdd;
    } else if (Ext.isFunction(callback)) {
      this.cfg.callback = [this.cfg.callback, afterAdd];
    } else if (Ext.isArray(callback)) {
      this.cfg.callback.push(afterAdd);
    }
  },
});

// @req ../../../util/validation/ViewParamChecker.js
Ext.define('CUF.document.view.action.ChangeDescriptionAction', {
  statics: {
    getName: function () {
      return 'changeDescription';
    },
  },
  checker: CUF.util.validation.ViewParamChecker,
  config: {
    cfg: null,
  },
  constructor: function (cfg) {
    this.initConfig(cfg);
  },
  position: ['tbar', 'contextMenu'],
  getAction: function () {
    var me = this;
    return {
      iconCls: 'dvnt-icon-file-plus',
      text: CUF.t('change-description'),
      handler: function (btn) {
        me.execute(btn);
      },
    };
  },
  execute: function (btn) {
    var grid = btn.grid;
    var sels = grid.getSelectionModel().getSelection();
    if (this.checker.selectionIsEmpty(sels)) return;
    this.showWindow(sels[0].data, grid);
  },
  showWindow: function (file, grid) {
    Ext.widget('cuf_win', {
      title: CUF.t('change-description'),
      width: 400,
      autoShow: true,
      layout: 'fit',
      padding: 5,
      items: [
        {
          xtype: 'form',
          file: file,
          grid: grid,
          frame: false,
          border: false,
          items: [
            {
              xtype: 'textfield',
              anchor: '100%',
              name: 'description',
              fieldLabel: CUF.t('description'),
              value: file.description,
            },
          ],
        },
      ],
      buttons: [
        {
          xtype: 'cuf_savebtn',
          handler: this.changeDesc,
        },
        {
          xtype: 'cuf_cancelbtn',
        },
      ],
    });
  },
  changeDesc: function (btn) {
    var form = btn.getWinForm();
    var fileId = form.file.id;
    var desc = form.getForm().getFieldValues().description;
    CUF.DocumentService.changeDescription(fileId, desc, {
      success: function () {
        form.grid.reload();
        CUF.document.core.DocumentView.reloadDocumentsGrid();
        btn.getWin().close();
      },
    });
  },
});

// @req ../../../util/validation/ViewParamChecker.js
Ext.define('CUF.document.view.action.DeleteAction', {
  statics: {
    getName: function () {
      return 'delete';
    },
  },
  config: {
    cfg: null,
  },
  constructor: function (cfg) {
    this.initConfig(cfg);
  },
  checker: CUF.util.validation.ViewParamChecker,
  position: ['tbar', 'contextMenu'],
  getAction: function () {
    var me = this;
    return {
      iconCls: 'dvnt-icon-x-symbol',
      text: CUF.t('delete-document'),
      handler: function (btn) {
        me.execute(btn);
      },
    };
  },
  isAvailable: function (view, file) {
    if (!this.checkRights()) {
      return true;
    }
    return this.canDelete(file);
  },
  canDelete: function (file) {
    if (!Ext.isDefined(file)) return true;
    if (file.uploader == Suncode.getCurrentUser()) {
      return true;
    }
    return false;
  },
  execute: function (btn) {
    var grid = btn.grid;
    var sels = grid.getSelectionModel().getSelection();
    if (this.checker.selectionIsEmpty(sels)) return;
    var validRecs = this.getValidRecs(sels);
    if (Ext.isEmpty(validRecs)) {
      this.noDocsToDelete();
      return;
    }
    this.checkDetachRights(sels, validRecs);
    var fileIds = CUF.util.StoreUtils.propertyList(validRecs, 'id');
    CUF.DocumentService.deleteDocuments(fileIds, {
      success: function () {
        grid.getSelectionModel().deselectAll();
        grid.reload();
        CUF.document.core.DocumentView.reloadDocumentsGrid();
      },
    });
  },
  checkDetachRights: function (sels, validRecs) {
    if (sels.length > validRecs.length) {
      CUF.util.MessageService.info(CUF.t('some-documents-wont-be-deleted-because-rights'));
    }
  },
  noDocsToDelete: function () {
    CUF.util.MessageService.info(CUF.t('no-rights-to-delete-documents'));
  },
  getValidRecs: function (recs) {
    if (!this.checkRights()) {
      return recs;
    }
    var validRecs = [];
    for (var i = 0; i < recs.length; i++) {
      var file = recs[i].data;
      if (this.canDelete(file)) {
        validRecs.push(recs[i]);
      }
    }
    return validRecs;
  },
  checkRights: function () {
    if (!Ext.isEmpty(this.cfg)) {
      if (this.cfg.checkRights == false) return false;
    }
    return true;
  },
});

// @req ../../../util/validation/ViewParamChecker.js
Ext.define('CUF.document.view.action.DetachAction', {
  statics: {
    getName: function () {
      return 'detach';
    },
  },
  config: {
    cfg: null,
  },
  constructor: function (cfg) {
    this.initConfig(cfg);
  },
  checker: CUF.util.validation.ViewParamChecker,
  position: ['tbar', 'contextMenu'],
  getAction: function () {
    var me = this;
    return {
      iconCls: 'dvnt-icon-file-minus',
      text: CUF.t('detach-document'),
      handler: function (btn) {
        me.execute(btn);
      },
    };
  },
  isAvailable: function (view, file) {
    if (!this.checkRights()) {
      return true;
    }
    return this.canDetach(file);
  },
  canDetach: function (file) {
    if (!Ext.isDefined(file)) {
      return true;
    }

    if (CUF.DocumentService.hasRightsToDetachDocument(file.id, processId, activityId)) {
      return true;
    }

    return false;
  },
  execute: function (btn) {
    var grid = btn.grid;
    var sels = grid.getSelectionModel().getSelection();
    if (this.checker.selectionIsEmpty(sels)) return;
    var validRecs = this.getValidRecs(sels);
    if (Ext.isEmpty(validRecs)) {
      this.noDocsToDetach();
      return;
    }
    this.checkDetachRights(sels, validRecs);
    var fileIds = CUF.util.StoreUtils.propertyList(validRecs, 'id');
    CUF.DocumentService.detachDocuments(fileIds, processId, activityId, {
      success: function () {
        grid.getSelectionModel().deselectAll();
        grid.reload();
        CUF.document.core.DocumentView.reloadDocumentsGrid();
      },
    });
  },
  checkDetachRights: function (sels, validRecs) {
    if (sels.length > validRecs.length) {
      CUF.util.MessageService.info(CUF.t('some-documents-wont-be-detached-because-rights'));
    }
  },
  noDocsToDetach: function () {
    CUF.util.MessageService.info(CUF.t('no-rights-to-detach-documents'));
  },
  getValidRecs: function (recs) {
    if (!this.checkRights()) {
      return recs;
    }
    var validRecs = [];
    for (var i = 0; i < recs.length; i++) {
      var file = recs[i].data;
      if (this.canDetach(file)) {
        validRecs.push(recs[i]);
      }
    }
    return validRecs;
  },
  checkRights: function () {
    if (!Ext.isEmpty(this.cfg)) {
      if (this.cfg.checkRights == false) return false;
    }
    return true;
  },
});

Ext.define('CUF.document.view.action.ShowAction', {
  statics: {
    getName: function () {
      return 'show';
    },
  },
  position: ['tbar', 'contextMenu'],
  getAction: function () {
    return {
      iconCls: 'dvnt-icon-maximize',
      text: CUF.t('show-document'),
      handler: this.execute,
    };
  },
  execute: function (btn) {
    var grid = btn.grid;
    var sels = grid.getSelectionModel().getSelection();
    if (sels.length > 0) {
      var file = sels[0].data;
      CUF.document.service.DocumentService.showFile(file.id);
    } else {
      CUF.util.MessageService.info(CUF.t('no-selection'));
    }
  },
});

Ext.define('CUF.document.view.field.DocumentClassCombo', {
  requires: ['Suncode.form.view.documents.DocumentClassChooser'],
  extend: 'Suncode.form.view.documents.DocumentClassChooser',
  alias: 'widget.cuf_documentclasscombo',
  objectValue: null,
  listeners: {
    select: function (cmb, recs) {
      if (!Ext.isEmpty(recs)) cmb.objectValue = recs[0].data;
      else cmb.objectValue = null;
    },
  },
  initComponent: function () {
    this.callParent();
    this.store = Ext.create('CUF.document.store.DocumentClassStore');
  },
});

Ext.define('CUF.document.view.form.DocumentFormPlugin', {
  documentClassNameList: null, //lista nazw klas dokumentów dostępnych w liście rozwijanej
  documentClassIdList: null, //lista id klas dokumentów dostępnych w liście rozwijanej
  defaultClassId: null, //id klasy dokumentów, która ma zostać wybrana jako domyślna
  defaultClassName: null, //nazwa klasy dokumentów, która ma zostać wybrana jako domyślna
  checkRights: true, //true - pobierze tylko klasy dokumentów dla których użytkownika ma uprawnienie dodawania
  showIndexes: false, //wyświetla formularz z indeksami
  title: null, //domyślnie pobierane z systemu
  addButtonText: null, //domyślnie pobierane z systemu
  addNextDocumentText: null, //domyślnie pobierane z systemu
  callback: null, //funkcja, która zostanie
  attachToProcess: true,
  dynamicTableCfg: null,
  autoSelectFirst: true,
  documentClassFilters: [],
  files: [],
  fileNames: [],
  fileDescription: null,
  savingAsNewVersion: {
    checked: false,
    readOnly: false,
  },
  acceptedExtensions: [],

  constructor: function (cfg) {
    Ext.apply(this, cfg);
  },

  initWinCfg: function (winCfg) {
    if (!Ext.isEmpty(this.title)) {
      winCfg.title = this.title;
    }
    winCfg.items[0].items[2].xtype = 'cuf_documentclasscombo';
    if (this.showIndexes == false) winCfg.items[0].items[4].hidden = true;
    winCfg.layout = 'fit';
  },

  initWinSendCfg: function (winCfg) {
    if (!Ext.isEmpty(this.title)) {
      winCfg.title = this.title;
    }
    winCfg.layout = 'fit';
  },

  initFormCfg: function (formCfg) {
    var addBtn = formCfg.buttons[0];
    Ext.Array.splice(formCfg.buttons, 1, 1);
    addBtn.xtype = 'button';
    addBtn.action = 'add';
    if (this.files.length > 0) {
      for (var i = 0; i < this.files.length; i++) {
        formCfg.items[i].padding = '0px 1px 0px 0px';
        formCfg.items[i].fieldLabel = CUF.t('document-form-plugin-attached-file');
        formCfg.items[i].value = this.fileNames[i] ? this.fileNames[i] : this.files[i].name;
        formCfg.items[i].xtype = 'textfield';
        formCfg.items[i].readOnly = true;
        formCfg.items[i].name = 'fileName';
        if (i + 1 < this.files.length) {
          formCfg.items.splice(i + 1, 0, Ext.clone(formCfg.items[0]));
        }
      }
      formCfg.items.splice(this.files.length, 0, Ext.clone(formCfg.items[0]));
      formCfg.items[this.files.length].xtype = 'filefield';
      formCfg.items[this.files.length].name = 'files';
      formCfg.items[this.files.length].hidden = true;
    } else {
      let extensions = this.acceptedExtensions;
      formCfg.items[0].padding = '0px 1px 0px 0px';
      formCfg.items[0].name = 'file';
      const listeners = {
        afterrender: function (component) {
          component.fileInputEl.set({
            accept: extensions.join(', '),
          });
        },
      };
      formCfg.items[0].listeners = this.getMergedListeners(formCfg.items[0].listeners, listeners);
    }
    var descriptionField = formCfg.items.find(function (item) {
      return !!item.name && item.name == 'description';
    });
    if (descriptionField && this.fileDescription) {
      descriptionField.value = this.fileDescription;
    }
    var savingAsNewVersionField = formCfg.items.find(function (item) {
      return !!item.name && item.name == 'saveAsNewVersion';
    });

    if (savingAsNewVersionField && this.savingAsNewVersion.checked) {
      savingAsNewVersionField.checked = this.savingAsNewVersion.checked;
    }

    if (savingAsNewVersionField && this.savingAsNewVersion.readOnly) {
      savingAsNewVersionField.readOnly = this.savingAsNewVersion.readOnly;
    }

    if (!Ext.isEmpty(this.addButtonText)) {
      addBtn.text = this.addButtonText;
    }
    CUF.log(formCfg);
  },

  initWin: function (win) {},

  initForm: function (form) {
    var documentClassCombo = form.down('cuf_documentclasscombo');
    this.configureDocumentClassCombo(documentClassCombo);

    if (this.needToSetClass()) {
      var me = this;
      documentClassCombo.getStore().load({
        callback: function (recs) {
          var idx = -1;
          if (me.defaultClassId != null) {
            idx = documentClassCombo.getStore().find('id', me.defaultClassId);
          }
          if (me.defaultClassName != null) {
            idx = documentClassCombo.getStore().find('name', me.defaultClassName);
          }
          if (idx > -1) {
            documentClassCombo.select(recs[idx]);
            documentClassCombo.fireEvent('select', documentClassCombo, recs);
            return;
          }
          if (me.autoSelectFirst == true && !Ext.isEmpty(recs)) {
            documentClassCombo.select(recs[0]);
            documentClassCombo.fireEvent('select', documentClassCombo, recs);
          }
        },
      });
    }

    var addBtn = form.down('button[action=add]');
    addBtn.on('click', this.addDocuments, this);
  },

  needToSetClass: function () {
    if (this.autoSelectFirst == true || this.defaultClassName != null || this.defaultClassId != null) {
      return true;
    }
    return false;
  },

  addDocuments: function (button) {
      var formCmp = button.up('form');
      var form = formCmp.getForm();
      var checkSize = !!window.FileReader && !!window.FileList && !!window.Blob;
      var files = this.collectFiles(formCmp);

      if (!this.validateExtension(files)) {
          return;
      }

      var fileNames = this.fileNames;
      if (form.isValid() && files.length > 0) {
          formCmp.mask();
          ServiceFactory.getFormService().save({
              scope: this,
              success: checkSize ? this.checkFilesSize : this.fileUpload,
              successParams: [button, files, fileNames],
              failure: this.showSavingError.bind(formCmp),
          });
      }
  },

    validateExtension: function (files) {
      let acceptedExtensions = this.acceptedExtensions;

      if (!acceptedExtensions || acceptedExtensions.length === 0 || acceptedExtensions[0] === null) {
          return true;
      }

      acceptedExtensions = acceptedExtensions.toString().replace(/\s/g, "");
      const extensions = acceptedExtensions.split(',');
      for (let fileIndex = 0; fileIndex < files.length; fileIndex++) {
        const fileName = files[fileIndex].name.toString();
        const fileNameAndExtension = fileName.split(".");
        if (fileNameAndExtension.length < 2) {
          CUF.util.MessageService.warn(CUF.t('Plik_nie_posiada_rozszerzenia'));
          return false;
        } else {
          const extension = "." + fileNameAndExtension.pop();
                if (!extensions.includes(extension)) {
                    CUF.util.MessageService.warn(CUF.t('Niepoprawny_format') + ". " + CUF.t('Dopuszczalne_formaty') + ": " + acceptedExtensions);
                    return false;
                }
            }
        }
        return true;
    },

    collectFiles: function (form) {
        var files = this.files ? Array.prototype.slice.call(this.files) : new Array();
        var scopeId = form ? form.id : 'window';
        return files.concat(
            jQuery('#' + scopeId)
                .find('input[type="file"]')
                .map(function (index, fileInput) {
                    return jQuery.map(fileInput.files, function (file) {
                        return file;
                    });
                })
                .toArray()
        );
    },

  fileUpload: function (button, files, fileNames) {
    var url = Suncode.getContextPath() + '/api/cuf/document/upload';
    var formCmp = button.up('form');
    var form = formCmp.getForm();
    var me = this;
    var win = button.up('window');
    var callback = function (response, statusText) {
      var files = JSON.parse(response);
      if (!Ext.isEmpty(me.callback)) {
        var documentClassCombo = formCmp.down('cuf_documentclasscombo');
        var documentClass = documentClassCombo.objectValue;
        files.forEach(function (file) {
          file.documentClass = documentClass;
        });
        me.fireAfterAdd(files);
      }
      FormService.refreshDocuments();
      win.close();
    };

    jQuery.ajax({
      url: url,
      type: 'POST',
      data: this.collectFormData(form, button, files, fileNames),
      processData: false,
      contentType: false,
      success: callback,
      error: callback,
    });
  },

  collectFormData: function (form, button, files, fileNames) {
    var formData = new FormData();
    formData.append('processId', button.processId);
    formData.append('activityId', button.activityId);
    for (var i = 0; i < files.length; i++) {
      formData.append('files', files[i]);
    }
    if (fileNames) {
      for (var i = 0; i < fileNames.length; i++) {
        formData.append('fileNames', fileNames[i]);
      }
    }
    var formValues = form.getValues();
    for (var key in formValues) {
      if (formValues.hasOwnProperty(key)) {
        formData.append(key, formValues[key]);
      }
    }

    return formData;
  },

  checkFilesSize: function (button, files, fileNames) {
    var form = button.up('form'),
      documentClassId = form.down('field[name=documentClassId]').value;

    Ext.Ajax.request({
      url: 'api/form/documents/checkSize',
      jsonData: {
        documentClassId: documentClassId,
        files: this.filesToCheck(files),
      },
      scope: this,
      success: function (response) {
        var result = Ext.JSON.decode(response.responseText);
        if (result.success) {
          this.fileUpload(button, files, fileNames);
        } else {
          this.showCheckingSizeError(response, form);
        }
      },
      failure: function (response) {
        this.showCheckingSizeError(response, form);
      },
    });
  },

  filesToCheck: function (files) {
    var filesToCheck = [];
    jQuery.each(files, function (index, file) {
      filesToCheck.push({
        name: file.name,
        size: Math.floor(file.size / 1024),
      });
    });
    return filesToCheck;
  },

  showSavingError: function (form, action) {
    ServiceFactory.getMessageService().showFailure(CUF.t('msg-unknownerror'));
    this.unmask();
  },

  showCheckingSizeError: function (response, form) {
    if (response != null) {
      try {
        var feedback = Ext.JSON.decode(response.responseText);
        ServiceFactory.getMessageService().showFailure(feedback.message);
      } catch (e) {
        ServiceFactory.getMessageService().showFailure(CUF.t('msg-unknownerror'));
      }
    }
    form.unmask();
  },

  fireAfterAdd: function (files) {
    var callbackFilesArg;
    if (this.files.length > 0) {
      callbackFilesArg = files;
    } else {
      callbackFilesArg = files.length > 0 ? files[0] : undefined;
    }

    if (Ext.isArray(callback)) {
      for (var i = 0; i < this.callback.length; i++) {
        this.callback[i].call(this, callbackFilesArg);
      }
    } else {
      this.callback.call(this, callbackFilesArg);
    }
  },

  configureDocumentClassCombo: function (documentClassCombo) {
    var filters = this.documentClassFilters.concat([]);
    this.addFilters('name', 'stringList', this.documentClassNameList, filters);
    this.addFilters('id', 'longList', this.documentClassIdList, filters);
    documentClassCombo.getStore().getProxy().jsonData = {
      filters: filters,
    };
    if (this.checkRights == true) {
      documentClassCombo.getStore().getProxy().api.read = Suncode.getContextPath() + '/api/cuf/documentclass/protected';
    }
  },

  addFilters: function (property, type, list, filters) {
    if (!Ext.isEmpty(list)) {
      var filter = {
        property: property,
        operator: 'IN',
        value: {},
      };
      filter.value[type] = list;
      filters.push(filter);
    }
  },

  getMergedListeners: function (listener1, listener2) {
    if (listener1 && listener2) {
      var mergedListener = Ext.apply({}, listener1);

      for (var eventName in listener1) {
        var eventFunc1 = listener1[eventName];
        var eventFunc2 = listener2[eventName];
        if (Ext.isDefined(eventFunc2)) {
          mergedListener[eventName] = function () {
            eventFunc1.apply(this, arguments);
            eventFunc2.apply(this, arguments);
          };
        }
      }

      return Ext.applyIf(mergedListener, listener2);
    } else if (listener1) {
      return Ext.apply({}, listener1);
    } else if (listener2) {
      return Ext.apply({}, listener2);
    }

    return undefined;
  },
});

// @req ../../../extension/base/BaseGrid.js
Ext.define('CUF.document.view.grid.DocumentGrid', {
  extend: 'CUF.extension.Base.Grid',
  alias: 'widget.cuf_documentgrid',
  maxSize: true,
  numbered: true,
  columns: [],
  border: false,
  frame: false,
  checkboxSelection: true,
  documentActions: [],
  initComponent: function () {
    this.initActions();
    this.tbar = this.buildTbar();
    this.columns = this.buildColumns();
    this.store = this.buildStore();
    this.callParent();
    Ext.create('CUF.document.controller.DocumentGridController').init(this);
  },
  initActions: function () {
    var actions = [];
    if (!Ext.isEmpty(this.documentActionsCfg)) {
      for (var i = 0; i < this.documentActions.length; i++) {
        var action = this.documentActions[i];
        var name = null;
        if (Ext.isString(action)) {
          name = action;
        } else if (Ext.isObject(action)) {
          name = action.name;
        } else {
          throw 'unsupported action type';
        }
        var cfg = this.documentActionsCfg[name];
        if (!Ext.isEmpty(cfg)) {
          if (Ext.isString(action)) {
            actions.push({
              name: name,
              cfg: cfg,
            });
          } else if (Ext.isObject(action)) {
            actions.push(
              Ext.apply(action, {
                cfg: cfg,
              })
            );
          }
        } else {
          actions.push(action);
        }
      }
      this.documentActions = actions;
    }
  },
  getActions: function (view, additionals) {
    var actions = [];
    for (var i = 0; i < this.documentActions.length; i++) {
      var actionDef = this.documentActions[i];
      var action = CUF.document.view.action.ActionFactory.getAction(actionDef, this.initialConfig);
      if (this.canShowAction(action, view, additionals)) {
        var act = action.getAction();
        act.grid = this;
        actions.push(act);
      }
    }
    return actions;
  },
  canShowAction: function (action, view, additionals) {
    if (Ext.isEmpty(action) || 0 > action.position.indexOf(view)) {
      return false;
    }
    if (Ext.isDefined(action.isAvailable)) {
      if (action.isAvailable(view, additionals)) return true;
      else return false;
    }
    return true;
  },
  buildTbar: function () {
    var actions = this.getActions('tbar');
    this.insertSeparators(actions);
    return actions;
  },
  insertSeparators: function (actions) {
    var count = actions.length;
    for (var i = 0; i < count; i++) {
      Ext.Array.insert(actions, i + 1 + (count - (count - i)), ['-']);
    }
  },
  buildContextMenu: function (file) {
    var actions = this.getActions('contextMenu', file);
    return Ext.create('Ext.menu.Menu', { items: actions, border: false, shadow: false });
  },
  buildColumns: function () {
    return [
      {
        text: CUF.t('document-name'),
        dataIndex: 'fileName',
        align: 'center',
        flex: 1,
      },
      {
        text: CUF.t('document-description'),
        dataIndex: 'description',
        align: 'center',
        flex: 1,
      },
      {
        text: CUF.t('date'),
        dataIndex: 'fileDate',
        renderer: CUF.util.Renderers.dateRenderer,
        align: 'center',
        flex: 1,
      },
      {
        text: CUF.t('size'),
        dataIndex: 'size',
        renderer: CUF.util.Renderers.fileSizeRenderer,
        align: 'center',
        flex: 1,
      },
      {
        text: CUF.t('owner'),
        dataIndex: 'uploader',
        align: 'center',
        flex: 1,
      },
      {
        text: CUF.t('inf'),
        align: 'center',
        sortable: false,
        xtype: 'documentinfocolumn',
        width: 70,
      },
      {
        text: CUF.t('action'),
        align: 'center',
        sortable: false,
        xtype: 'documentactionscolumn',
        width: 70,
      },
    ];
  },
});

Ext.define('CUF.document.view.grid.ProcessDocumentGrid', {
  extend: 'CUF.document.view.grid.DocumentGrid',
  alias: 'widget.cuf_processdocumentgrid',
  documentActions: ['attach', 'detach', 'show'],
  buildStore: function () {
    return Ext.create('Ext.data.Store', {
      model: 'CUF.document.model.Document',
      proxy: {
        type: 'rest',
        extraParams: {
          processId: processId,
          activityId: activityId,
        },
        api: {
          read: Suncode.getContextPath() + '/api/documents/activity',
        },
        reader: {
          root: 'data',
          totalProperty: 'total',
        },
        noCache: false,
      },
    });
  },
});

// @req ../../../extension/base/BaseWin.js
Ext.define('CUF.document.view.window.DocumentManagementWin', {
  extend: 'CUF.extension.Base.Window',
  alias: 'widget.cuf_documentmanagementwin',
  maxSize: true,
  layout: 'fit',
  hMargin: 100,
  initComponent: function () {
    this.title = CUF.t('process-document-management');
    this.callParent();
  },
});

}( this.Ext4 ) );
//-------Parametry konfiguracyjne skryptów CUF'a------
var CUF_params = {
  logConfig: {
    enabled: false,
    level: 'trace',
  },
};

//---------Inicjalizacja CUF'a-----------------

//------Włączenie\Wyłączenie logów---
if (CUF_params.logConfig.enabled) {
  CUF_log = console.log;
  CUF_trace = console.log;
  CUF_debug = console.log;
  CUF_warn = console.log;
  CUF_error = console.log;
  CUF_depracated = function (args, newMethod) {
    console.log(
      '@Depracated ' + CUF.t('function') + ': ' + args.callee.name + CUF.t('function-is-deprecated') + ' ' + CUF.t('should-use') + newMethod
    );
  };
  CUF_debug('CUFScripts log turn on');
} else {
  CUF_log = function () {};
  CUF_trace = function () {};
  CUF_debug = function () {};
  CUF_warn = function () {};
  CUF_error = function () {};
  CUF_depracated = function () {};
}
//----------------------------

String.prototype.replaceAll = function (str1, str2, ignore) {
  return this.replace(
    new RegExp(str1.replace(/([\/\,\!\\\^\$\{\}\[\]\(\)\.\*\+\?\|\<\>\-\&])/g, '\\$&'), ignore ? 'gi' : 'g'),
    typeof str2 == 'string' ? str2.replace(/\$/g, '$$$$') : str2
  );
};

// /////////////////////////////////////
// /////////////////////////////////////
// ///////// dodanie funkcji wykonywanej po zaladowaniu okna
function CUF_addLoadEvent(func) {
  var oldonload = window.onload;

  if (typeof window.onload != 'function') {
    window.onload = function () {
      func();
    };
  } else {
    window.onload = function () {
      if (oldonload) {
        oldonload();
      }

      func();
    };
  }
}

function CUF_showLoadingMask(id, msg) {
  var maskDiv = Ext.DomHelper.append(
    Ext.getBody(),
    {
      tag: 'div',
      id: id,
    },
    true
  );
  maskDiv.setStyle({
    position: 'fixed',
    left: 0,
    top: 0,
    width: '100%',
    height: '100%',
    'z-index': 20000,
  });

  var mask = new Ext.LoadMask(maskDiv, {
    msg: msg,
  });
  mask.show();
}

function CUF_removeLoadingMask(id) {
  if (!Ext.isEmpty(Ext.get(id))) Ext.get(id).remove();
}

// ////////////// sprawdzenie komentarzy ////////////
function CUF_checkComment() {
  if (CUF.CommentService.checkComment()) {
    return true;
  } else {
    CUF.util.MessageService.warn(CUF.t('comment-is-required'));
    return false;
  }
}

function CUF_checkChangesInComments(activityId, processId, sTaskUserName) {
  var sServletName = 'CUF.CheckChangesInComments.customServlet';
  var arg = 'param=' + activityId + '&param1=' + processId + '&param2=' + sTaskUserName;
  return CUF_executeServlet(sServletName, arg);
}

// ////////////// wykonania servletu ////////////////
function CUF_executeServlet(servlet, arg) {
  var req = servlet;
  var text = '';

  try {
    request = new XMLHttpRequest();
    request.overrideMimeType('text/xml');
  } catch (e) {
    try {
      request = new ActiveXObject('Microsoft.XMLHTTP');
    } catch (e) {
      alert('XMLHttpRequest error: ' + e);
    }
  }

  bbb = CUF_addrequest(req, req, arg);
  var xmlDoc = bbb.responseXML;
  var response = xmlDoc.getElementsByTagName('value');
  text = response[0].firstChild.data;

  return text;
}

function CUF_executeServletGetResponse(servlet, arg) {
  var req = servlet;

  try {
    request = new XMLHttpRequest();
    request.overrideMimeType('text/xml');
  } catch (e) {
    try {
      request = new ActiveXObject('Microsoft.XMLHTTP');
    } catch (e) {
      alert('XMLHttpRequest error: ' + e);
    }
  }

  bbb = CUF_addrequest(req, req, arg);
  return bbb;
}

function CUF_addrequest(req, element, arg) {
  request.open('POST', element, false);
  request.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded; charset:UTF-8');
  request.send(arg);
  return request;
}

// /// okno wyszukiwania dla konkretnej wartości jednej ze zmiennych ////
function CUF_searchWindow(zmienna1, zmienna2, zmienna3, zmienna4, zmienna5, zmienna6, zmienna7) {
  var parameters = 'CUF.SearchWindow.customServlet?';
  var display = CUF.t('search-for');

  if (document.getElementById(zmienna1) != null && document.getElementById(zmienna1).value != '') {
    parameters = parameters + 'p1=' + document.getElementById(zmienna1).value;
    display = display + zmienna1 + '=' + document.getElementById(zmienna1).value + ', ';
  } else {
    parameters = parameters + 'p1=$';
  }
  if (zmienna1 != '') {
    parameters = parameters + '&n1=' + zmienna1 + '&';
  } else {
    parameters = parameters + '&n1=$' + '&';
  }

  if (document.getElementById(zmienna2) != null && document.getElementById(zmienna2).value != '') {
    parameters = parameters + 'p2=' + document.getElementById(zmienna2).value;
    display = display + zmienna2 + '=' + document.getElementById(zmienna2).value + ', ';
  } else {
    parameters = parameters + 'p2=$';
  }
  if (zmienna2 != '') {
    parameters = parameters + '&n2=' + zmienna2 + '&';
  } else {
    parameters = parameters + '&n2=$' + '&';
  }

  if (document.getElementById(zmienna3) != null && document.getElementById(zmienna3).value != '') {
    parameters = parameters + 'p3=' + document.getElementById(zmienna3).value;
    display = display + zmienna3 + '=' + document.getElementById(zmienna3).value + ', ';
  } else {
    parameters = parameters + 'p3=$';
  }
  if (zmienna3 != '') {
    parameters = parameters + '&n3=' + zmienna3 + '&';
  } else {
    parameters = parameters + '&n3=$' + '&';
  }

  if (document.getElementById(zmienna4) != null && document.getElementById(zmienna4).value != '') {
    parameters = parameters + 'p4=' + document.getElementById(zmienna4).value;
    display = display + zmienna4 + '=' + document.getElementById(zmienna4).value + ', ';
  } else {
    parameters = parameters + 'p4=$';
  }
  if (zmienna4 != '') {
    parameters = parameters + '&n4=' + zmienna4 + '&';
  } else {
    parameters = parameters + '&n4=$' + '&';
  }

  parameters = parameters + 'tableName=' + zmienna5;

  if (zmienna6 != '') {
    parameters = parameters + '&condition=' + zmienna6;
  } else {
    parameters = parameters + '&condition=$';
  }

  if (zmienna7 != '') {
    parameters = parameters + '&other=' + zmienna7;
  } else {
    parameters = parameters + '&other=$';
  }

  var tab = parameters.split(' ');
  parameters = '';
  var polskie = 'Ą;Ć;Ę;Ł;Ó;Ś;Ż;Ź;ą;ć;ę;ł;ó;ś;ż;ź';
  var polskieZnaki = polskie.split(';');
  var kodowanie = 'hA4;h8F;hA8;h9D;hE0;h97;hBD;h8D;hA5;h86;hA9;h88;hA2;h98;hBE;hAB';
  var koder = kodowanie.split(';');

  for (var i = 0; i < tab.length; i = i + 1) {
    if (i + 1 != tab.length) {
      parameters = parameters + tab[i] + '_';
    } else {
      parameters = parameters + tab[i];
    }
  }

  for (var j = 0; j < koder.length; j++) {
    for (var k = 0; k < parameters.length; k++) {
      if (parameters[k] == polskieZnaki[j]) {
        var zm = parameters.split(polskieZnaki[j]);
        parameters = '';
        for (var t = 0; t < zm.length; t++) {
          if (t + 1 != zm.length) {
            parameters = parameters + zm[t] + koder[j];
          } else {
            parameters = parameters + zm[t];
          }
        }
        k = k + 2;
      }
    }
  }

  display = display.substring(0, display.length - 2);
  show_data_window(parameters, display, CUF.t('launching'));
}
// // zapisanie wszystkich wartości zmiennych ////
function CUF_saveAllVariables() {
  ServiceFactory.getFormService().save();
}

function CUF_saveAllVariablesDisplayStatus() {
  CUF_saveAllVariables();
}

// //////// eksport schematu tabeli dynamicznej do xls //////////
// / omija ukryte kolumny
function CUF_exportDynamicTableScheme(oButton) {
  CUF_executeExportDynamicTableSchame(oButton, false);
}

// / uwzglednia ukryte kolumny
function CUF_exportDynamicTableSchemeWithHidden(oButton) {
  CUF_executeExportDynamicTableSchame(oButton, true);
}

// / pobieranie schematu tabeli dynamicznej
function CUF_executeExportDynamicTableSchame(oButton, showHidden) {
  CUF_exportDTSchemeToExcel({ showHidden: showHidden });
}

// //////// import linii do tabeli dynamnicznej z pliku xls ////////////
// // omija ukryte kolumny
function CUF_ShowXlsPanel(oButton) {
  CUF_showJExcelPanel(oButton, false);
}

// // uwzglednia ukryte kolumny
function CUF_ShowXlsPanelWithHidden(oButton) {
  CUF_showJExcelPanel(oButton, true);
}

// / wyswietlanie panelu do wyboru pliku i import linii tabeli dynamicznej -
// aktualna funkcja
function CUF_importDynamicTableFromXls(oButton, config) {
  CUF_importDTFromExcel(oButton, config);
}
// ustawia okno na środku okna przeglądarki
function CUF_adjustWinPosition(win) {
  try {
    var taskPanel = formPanel.items.first();

    if (taskPanel) {
      // var regulateWidth = Math.ceil( ( screen.width -
      // formPanel.getWidth() ) / 2 );
      var regulateWidth = Math.ceil((Ext.getBody().getViewSize().width - formPanel.getWidth()) / 2);
      var taskPanelWidth = taskPanel.getWidth();

      win.setPosition(Math.ceil(taskPanelWidth / 2 - win.getWidth() / 2 + regulateWidth), win.getPosition()[1]);
    }
  } catch (e) {}
}
// / wyswietlanie panelu do wyboru pliku i import linni tabeli dynamicznej -
// poprzednia funkcja
function CUF_showJExcelPanel(oButton, showHidden) {
  CUF_importDTFromExcel(oButton, config);
}

// import styli do formularza
function CUF_importCssForFrame(cssName) {
  if (!document.getElementById(cssName)) {
    var head = document.getElementsByTagName('head')[0];
    var link = document.createElement('link');
    link.id = cssName;
    link.rel = 'stylesheet';
    link.type = 'text/css';
    link.href = 'style/' + cssName + '.css';
    link.media = 'all';
    head.appendChild(link);
  }
}

function CUF_setDataInTable(colsId, data, removeOldRecords, tabStore, notSaveStoreAfterLoad, grid) {
  if (removeOldRecords == true) {
    ServiceFactory.getVariableSetService().clear(grid);
  }
  var records = [];
  for (var i = 0; i < data.length; i++) {
    var r = {};
    var d = data[i];
    for (var j in d) {
      if (d.hasOwnProperty(j)) {
        r[mappings[j]] = d[j];
      }
    }
    records.push(r);
  }
  ServiceFactory.getVariableSetService().addRow(grid, false, records);
}

// ////// eksportowanie zawartosci tabeli dynamicznej ///////////
// // omija ukryte kolumny
function CUF_exportDynamicTableToXls(oButton) {
  CUF_executeExportDynamicTableToXls(oButton, false, true, 'Eksport towarow');
}

// // uwzglednia ukryte kolumny
function CUF_exportDynamicTableToXlsWithHidden(oButton) {
  CUF_executeExportDynamicTableToXls(oButton, true, true, 'Eksport towarow');
}

// eksportuje linie z tabeli dynamicznej do pliku xls
function CUF_executeExportDynamicTableToXls(oButton, showHidden, askForFileName, fileName) {
  CUF_exportDTToExcel(oButton, { showHidden: showHidden });
}

// dodawanie dokumentow z archiwum do procesu
function CUF_attachDocumentFromArchive(config) {
  var processId = document.getElementsByName('processId')[0].value;
  var activityId = document.getElementsByName('activityId')[0].value;
  var windowTitle = config.addDocWindowTitle;
  var topPanelTitle = config.topPanelTitle;
  var bottomPanelTitle = config.bottomPanelTitle;
  var rightPanelTitle = config.rightPanelTitle;
  var attachButtonLabel = config.attachButtonLabel;
  var searchButtonLabel = config.searchButtonLabel;
  var funcNameAfterDocAction = config.funcNameAfterDocAction;
  var attachToDT = config.attachToDT;
  var skipGroupCheck = config.skipGroupCheck;
  var DTIndexes = config.DTIndexes;
  var gridNumber = config.gridNumber;
  var activityGrid = undefined;
  var maxNoOfIndexes = config.maxNoOfIndexes;
  var sortBy = config.sortBy;
  var sortDir = config.sortDir;
  var showRowContextMenu = config.showRowContextMenu;
  var contextMenuDef = config.contextMenuDef;
  var defaultSourceClassName = config.defaultSourceClassName;
  var defaultDestinationClassName = config.defaultDestinationClassName;
  if (attachToDT == undefined) {
    attachToDT = false;
  }

  if (attachToDT) {
    if (!gridNumber) {
      gridNumber = 1;
    }

    try {
      activityGrid = CUF_getActivityGrid(gridNumber);
    } catch (e) {}
  }

  if (skipGroupCheck == undefined) {
    skipGroupCheck = true;
  }

  if (maxNoOfIndexes == undefined) {
    maxNoOfIndexes = 0;
  }

  if (funcNameAfterDocAction == undefined) {
    funcNameAfterDocAction = '';
  }

  if (showRowContextMenu == undefined) {
    showRowContextMenu = true;
  }

  if (showRowContextMenu) {
    if (!contextMenuDef) {
      contextMenuDef = new Array();
    }

    var displayItem = new Object();
    displayItem.action = 'display';
    displayItem.text = CUF.t('show');
    contextMenuDef.push(displayItem);
    displayItem = new Object();
    displayItem.action = 'preview';
    displayItem.text = CUF.t('miniature');
    displayItem.paging = true;
    displayItem.height = 300;
    contextMenuDef.push(displayItem);
  }

  var docClassesServlet = 'CUF.GetAvailableDocClasses.customServlet';
  var docClassesArgs = 'userId=' + sTaskUserName + '&skipGroupCheck=' + skipGroupCheck;

  var docClassesResponse = CUF_executeServletGetResponse(docClassesServlet, docClassesArgs);
  var docClassesResponseTxt = docClassesResponse.responseText;
  var docClassesJsonData = Ext.util.JSON.decode(docClassesResponseTxt);
  var docClassesSuccess = docClassesJsonData.success;

  if (!docClassesSuccess) {
    Ext.Msg.show({
      title: CUF.t('error'),
      msg: docClassesJsonData.message,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.ERROR,
    });
    return;
  }

  var searchInDocClasses = docClassesJsonData.searchIn;
  var addToDocClasses = docClassesJsonData.addTo;

  if (searchInDocClasses.length == 0) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('docClass-search-no-available'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return;
  }

  if (addToDocClasses.length == 0) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('docClass-add-no-available'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return;
  }

  CUF_saveAllVariables();
  CUF_importCssForFrame('processdocuments');

  var lastWindowOnscroll = window.onscroll;
  var url = CUF_getUrl();
  var lastScrollPos = 0;
  var windowYOffset = 0;
  var doc_window = null;
  var mainPanel = null;
  var leftPanel = null;
  var topPanel = null;
  var bottomPanel = null;
  var rightPanel = null;
  var resultGrid = null;
  var resultLabel = null;
  var searchButton = null;
  var closeWindowButton = null;
  var docViewPanel = null;
  var docViewPanelCollapsed = true;
  var sourceDocClassItem = null;
  var destDocClassItem = null;
  var searchComboBoxConfig = new Object();
  var addComboBoxConfig = new Object();
  var gridConfig = new Object();
  gridConfig.processId = processId;
  gridConfig.activityId = activityId;
  gridConfig.funcNameAfterDocAction = funcNameAfterDocAction;
  gridConfig.sortBy = sortBy;
  gridConfig.sortDir = sortDir;
  gridConfig.reloadPage = config.reloadPage;
  gridConfig.attachToDT = attachToDT;
  gridConfig.DTIndexes = DTIndexes;
  gridConfig.activityGrid = activityGrid;

  if (formPanel.layout && formPanel.layout.east) {
    docViewPanel = formPanel.layout.east.panel;
  }

  if (docViewPanel != undefined && docViewPanel != null && !docViewPanel.hidden) {
    if (!docViewPanel.collapsed) {
      docViewPanelCollapsed = false;
      docViewPanel.collapse();
      CUF_refreshTaskPanel();
    }
  }

  var docClass = Ext.data.Record.create([
    {
      name: 'docClassName',
      type: 'string',
    },
  ]);

  var docClassesReader = new Ext.data.JsonReader({
    id: 'CUF_docclasses_reader',
    fields: docClass,
  });

  var sourceStore = new Ext.data.JsonStore({
    storeId: 'CUF_source_store',
    //reader: docClassesReader,
    data: searchInDocClasses,
    fields: ['docClassName'], //docClass
  });
  //sourceStore.loadData( searchInDocClasses );
  // sourceStore.commitChanges();

  var destStore = new Ext.data.Store({
    storeId: 'CUF_dest_store',
    reader: docClassesReader,
    fields: docClass,
  });
  destStore.loadData(addToDocClasses);
  destStore.commitChanges();

  Ext.apply(searchComboBoxConfig, {
    id: 'CUF_source_doc_class_name_item',
    store: sourceStore,
    listeners: {
      select: function (combo, record, index) {
        var docClassName = combo.getValue();

        if (docClassName == CUF.t('docClass-choose-class') || !combo.isValid()) {
          return;
        }

        if (this.actualDocClassName == docClassName) {
          return;
        }

        this.actualDocClassName = docClassName;
        var checkDocClassServlet = 'CUF.CheckDocClass.customServlet';
        var checkDocClassArgs = 'docClassName=' + docClassName + '&processId=' + processId + '&activityId=' + activityId;
        var checkDocClassResponse = CUF_executeServletGetResponse(checkDocClassServlet, checkDocClassArgs);
        var checkDocClassResponseTxt = checkDocClassResponse.responseText;
        var checkDocClassJsonData = Ext.util.JSON.decode(checkDocClassResponseTxt);
        var docClassId = checkDocClassJsonData.message;

        if (docClassId != '-1') {
          var docClassJsonRecs = checkDocClassJsonData.records;
          bottomPanel.addIndexes(docClassJsonRecs);
          rightPanel.createResultGrid(docClassJsonRecs);
        }
      },
    },
  });

  Ext.apply(addComboBoxConfig, {
    id: 'CUF_dest_doc_class_name_item',
    store: destStore,
  });

  sourceDocClassItem = new Ext.ux.plusmpm.DocClassesDataChooser(searchComboBoxConfig);
  destDocClassItem = new Ext.ux.plusmpm.DocClassesDataChooser(addComboBoxConfig);

  if (!attachButtonLabel || attachButtonLabel == '') {
    attachButtonLabel = CUF.t('attach');
  }

  if (!searchButtonLabel || searchButtonLabel == '') {
    searchButtonLabel = CUF.t('search');
  }

  if (!rightPanelTitle || rightPanelTitle == '') {
    rightPanelTitle = CUF.t('indexes');
  }

  if (!topPanelTitle || topPanelTitle == '') {
    topPanelTitle = CUF.t('document-searching');
  }

  if (!bottomPanelTitle || bottomPanelTitle == '') {
    bottomPanelTitle = CUF.t('search-results');
  }

  var viewSize = Ext.getBody().getViewSize();
  var mainPanelWidth = viewSize.width * 0.9;
  var mainPanelHeight = viewSize.height * 0.9;

  resultLabel = new Ext.form.Label({
    id: 'CUF_from_archive_result_label',
    fieldLabel: '',
    labelSeparator: '',
    width: 320,
  });

  searchButton = new Ext.Button({
    id: 'CUF_search_from_archive_button',
    text: searchButtonLabel,
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/folder_magnify.png',
    handler: function (btn, e) {
      var docClassName = sourceDocClassItem.getValue();

      if (docClassName == '' || docClassName == CUF.t('docClass-choose-class')) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('docClass-choose'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      var hmMapping = new Array();
      var counter = 1;

      var indexItem = Ext.getCmp('index' + counter);
      var noIdxMsg = CUF.t('docClass') + ' ' + docClassName + CUF.t('docClass-no-indexes');
      if (!indexItem) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: noIdxMsg,
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      while (indexItem != null && indexItem != undefined) {
        var indexName = indexItem.fieldLabel;
        var indexValue = indexItem.getValue();
        var obj = new Object();
        obj.indexName = indexName;

        if (indexItem.indexType == 'date') {
          obj.indexType = 'date';
          try {
            obj.startDate = indexItem.items.get(0).getValue().format('Y-m-d');
          } catch (err) {}
          try {
            obj.endDate = indexItem.items.get(1).getValue().format('Y-m-d');
          } catch (err) {}
          obj.indexValue = 'date';
        } else {
          obj.indexValue = indexValue;
        }
        hmMapping.push(obj);
        counter++;
        indexItem = Ext.getCmp('index' + counter);
      }

      var searchParams = new Array();
      searchParams['docClassName'] = docClassName;
      searchParams['conditions'] = Ext.util.JSON.encode(hmMapping);

      resultGrid.executeSearch(searchParams);
    },
  });

  closeWindowButton = new Ext.Button({
    id: 'CUF_close_from_archive_button',
    text: CUF.t('close'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/cross.png',
    hidden: true,
    handler: function (btn, e) {
      doc_window.close();
      window.onscroll = lastWindowOnscroll;
      window.scroll(0, lastScrollPos);

      if (docViewPanel != undefined && docViewPanel != null && !docViewPanel.hidden) {
        if (docViewPanelCollapsed && !docViewPanel.collapsed) {
          docViewPanel.collapse();
        } else if (!docViewPanelCollapsed && docViewPanel.collapsed) {
          docViewPanel.expand();
        }
      }

      window.setTimeout(function () {
        CUF_refreshTaskPanel();
      }, 500);
    },
  });

  var topButtonsPanel = new Ext.Container({
    id: 'CUF_top_buttons_panel',
    buttonAlign: 'center',
    layout: 'table',
    layoutConfig: {
      columns: 2,
    },
  });
  topButtonsPanel.add(searchButton);
  topButtonsPanel.add(closeWindowButton);

  topPanel = new Ext.FormPanel({
    id: 'CUF_from_archive_top_panel',
    width: 350,
    height: 150,
    collapsible: false,
    frame: true,
    hideBorders: true,
    title: "<center><img src='" + url + "/style/img/fam/page_white_magnify.png' align='top'>" + ' ' + topPanelTitle + '</center>',
    labelWidth: 300,
    labelAlign: 'top',
    region: 'north',
    items: [sourceDocClassItem, topButtonsPanel, resultLabel],
  });

  bottomPanel = new Ext.Panel({
    id: 'CUF_from_archive_bottom_panel',
    width: 350,
    collapsible: false,
    hideBorders: true,
    frame: true,
    title: "<center><img src='" + url + "/style/img/fam/page_white_wrench.png' align='top'>" + ' ' + rightPanelTitle + '</center>',
    region: 'center',
    layout: 'form',
    listeners: {
      resize: function (panel, adjWidth, adjHeight, rawWidth, rawHeight) {
        var counter = 1;
        var indexesItem = Ext.getCmp('CUF_indexes_panel');

        if (indexesItem) {
          indexesItem.labelWidth = adjWidth - 50;
          var indexItem = Ext.getCmp('index' + counter);

          while (indexItem) {
            indexItem.setWidth(adjWidth - 50);
            counter++;
            indexItem = Ext.getCmp('index' + counter);
          }
        }
      },
    },
    addIndexes: function (docClassJsonRecs) {
      this.remove('CUF_indexes_panel');
      var width = this.getWidth() + 15;

      var indexesItem = new Ext.FormPanel({
        id: 'CUF_indexes_panel',
        labelWidth: width - 50,
        height: this.getHeight() - 50,
        autoScroll: true,
        frame: false,
        labelAlign: 'top',
      });

      if (docClassJsonRecs && docClassJsonRecs.length > 0) {
        this.add(indexesItem);
        CUF_fillIndexesItem(indexesItem, docClassJsonRecs, width, 0, false, true);
      } else {
        var indexLabelItem = new Ext.form.Label({
          id: 'CUF_indexes_panel',
          fieldLabel: '',
          labelSeparator: '',
          width: width - 50,
        });
        indexLabelItem.addClass('x-CUFresultLabel-error');
        indexLabelItem.setText(CUF.t('no-indexes'), true);
        this.add(indexLabelItem);
      }

      this.doLayout();

      var counter = 1;
      var indexItem = Ext.getCmp('index' + counter);

      while (indexItem != null && indexItem != undefined) {
        indexItem.on('specialkey', function (field, e) {
          if (e.getKey() == e.ENTER) {
            searchButton.handler.call(searchButton, e);
          }
        });

        counter++;
        indexItem = Ext.getCmp('index' + counter);
      }
    },
  });

  leftPanel = new Ext.Panel({
    id: 'CUF_from_archive_left_panel',
    width: 350,
    height: mainPanelHeight,
    collapsible: false,
    hideBorders: true,
    layout: 'border',
    region: 'center',
    items: [topPanel, bottomPanel],
  });

  rightPanel = new Ext.Panel({
    id: 'CUF_from_archive_right_panel',
    collapsible: false,
    hideBorders: true,
    height: mainPanelHeight,
    region: 'east',
    layout: 'fit',
    title: "<center><img src='" + url + "/style/img/fam/table.png' align='top'>" + ' ' + bottomPanelTitle + '</center>',
    createResultGrid: function (docClassJsonRecs) {
      if (resultGrid != null) {
        this.remove(resultGrid);
      }

      if (!docClassJsonRecs || docClassJsonRecs.length == 0) {
        return;
      }

      var resultDef = new Array();
      var resultColumnModel = new Array();
      var colHidden = false;
      var resultCell = null;
      var resultColDef = null;

      for (var i = 0; i < docClassJsonRecs.length; i++) {
        var indexDef = docClassJsonRecs[i];
        var indexName = indexDef.indexName;
        var indexType = indexDef.indexType;
        var columnAlign = 'center';

        if (maxNoOfIndexes != 0 && maxNoOfIndexes == i) {
          colHidden = true;
        }

        resultCell = new Ext.data.Field({
          name: indexName,
          type: 'string',
        });
        resultDef.push(resultCell);

        if (indexType == 'int' || indexType == 'float') {
          columnAlign = 'right';
        }

        var colHeader = '';

        if (indexName == sortBy) {
          if (Ext.util.Format.lowercase(sortDir) == 'desc') {
            colHeader = indexName + "<img src='" + url + "/style/img/fam/bullet_arrow_down.png'>";
          } else {
            colHeader = indexName + "<img src='" + url + "/style/img/fam/bullet_arrow_up.png'>";
          }
        } else {
          colHeader = indexName;
        }

        resultColDef = new Ext.grid.Column({
          id: 'colindex' + (i + 1),
          header:
            '<center><img onclick="new Function(CUF_hideResultGridCol(\'' +
            'colindex' +
            (i + 1) +
            "'))\" src='" +
            url +
            "/style/img/fam/bullet_delete.png'>" +
            colHeader +
            '</center>',
          dataIndex: indexName,
          align: columnAlign,
          hidden: colHidden,
          sortable: false,
        });
        resultColumnModel.push(resultColDef);
      }

      resultCell = new Ext.data.Field({
        name: 'CUF_attach_doc_from_archive_btn',
        type: 'string',
      });
      resultDef.push(resultCell);
      resultColDef = new Ext.grid.Column({
        id: 'colindex' + (resultColumnModel.length + 1),
        header: '<center>' + CUF.t('attach-document') + '</center>',
        dataIndex: 'CUF_attach_doc_from_archive_btn',
        align: 'center',
        hidden: false,
        hideable: false,
        sortable: false,
        width: 40,
        menuDisabled: true,
        renderer: function (value, metaData, record, rowIndex, colIndex, store) {
          return "<img src='" + url + "/style/img/fam/add.png'>";
        },
      });
      resultColumnModel.push(resultColDef);

      resultCell = new Ext.data.Field({
        name: 'fileId',
        type: 'string',
      });
      resultDef.push(resultCell);
      resultColDef = new Ext.grid.Column({
        id: 'colindex' + (resultColumnModel.length + 1),
        header: 'fileId',
        dataIndex: 'fileId',
        align: 'center',
        hidden: true,
        hideable: false,
        sortable: false,
      });
      resultColumnModel.push(resultColDef);

      gridConfig.resultDef = resultDef;
      gridConfig.resultColumnModel = resultColumnModel;
      gridConfig.destDocClassItem = destDocClassItem;

      resultGrid = new Ext.ux.plusmpm.AddDocFromArchiveGrid(gridConfig);
      this.add(resultGrid);
      this.doLayout();

      var ctxMenu = new Ext.menu.Menu({
        id: 'CUF_context_menu_' + Ext.id(),
        renderTo: formPanel.id,
        docView: false,
      });

      CUF_createContextMenu(ctxMenu, contextMenuDef);

      resultGrid.on('rowcontextmenu', function (thisGrid, rowIndex, e) {
        e.stopEvent();
        var clickPos = e.getXY();
        var thisStore = thisGrid.getStore();
        e.rowIndex = rowIndex;
        e.tab = undefined;
        e.store = thisStore;
        e.fileId = thisStore.getAt(rowIndex).get('fileId');
        e.clickPos = clickPos;
        ctxMenu.showAt(clickPos);
      });
    },
  });

  if (!windowTitle || windowTitle == '') {
    windowTitle = CUF.t('document-add-archive');
  }

  mainPanel = new Ext.Panel({
    id: 'CUF_from_archive_main_panel',
    hideBorders: true,
    collapsible: true,
    width: mainPanelWidth,
    height: mainPanelHeight,
    titleCollapse: true,
    defaults: {
      split: true,
    },
    layout: 'border',
    title: "<center><img src='" + url + "/style/img/fam/page_white_add.png' align='top'>" + ' ' + windowTitle + '</center>',
    items: [leftPanel, rightPanel],
  });
  /*
    mainPanel.addTool( {
        id: 'close',
        handler: function( e, toolEl, panel, tc ) {
            closeWindowButton.handler.call( closeWindowButton, e );
        },
        qtip: CUF.t("close")
    } );*/

  doc_window = new Ext.Window({
    id: 'CUF_search_doc_from_archive_window',
    autoWidth: true,
    autoHeight: true,
    //frame: false,
    // closable: false,
    modal: true,
    // constrain: true,
    shadow: false,
    resizable: false,
    // border: false,
    // header: false,
    items: mainPanel,
    setCurrentScrollPos: function () {
      var pageYOffset = 0;

      if (Ext.isIE && Ext.isStrict) {
        pageYOffset = document.documentElement.scrollTop;
      } else {
        pageYOffset = window.pageYOffset;
      }

      lastScrollPos = pageYOffset;
      windowYOffset = doc_window.getPosition()[1] - pageYOffset;
    },
  });

  doc_window.show();
  doc_window.setCurrentScrollPos();

  if (searchInDocClasses.length == 1) {
    //jest tylko jedna klasa wiec ja ustawiam i wyswietlam pola indeksow
    sourceDocClassItem.setValue(searchInDocClasses[0].docClassName);
    sourceDocClassItem.fireEvent('select', sourceDocClassItem);
  } else if (!Ext.isEmpty(defaultSourceClassName)) {
    sourceDocClassItem.setValue(defaultSourceClassName);
    sourceDocClassItem.fireEvent('select', sourceDocClassItem);
  }

  if (destDocClassItem.getStore().getCount() == 1) {
    //jest tylko jedna klasa wiec ja ustawiam i wyswietlam pola indeksow
    destDocClassItem.setValue(destDocClassItem.getStore().getAt(0).data.docClassName);
  } else if (!Ext.isEmpty(defaultDestinationClassName)) {
    destDocClassItem.setValue(defaultDestinationClassName);
  }

  window.onscroll = function () {
    var pageYOffset = 0;

    if (Ext.isIE && Ext.isStrict) {
      pageYOffset = document.documentElement.scrollTop;
    } else {
      pageYOffset = window.pageYOffset;
    }

    doc_window.setPosition(doc_window.getPosition[0], pageYOffset + windowYOffset);
  };
}

Ext.ux.plusmpm.DocClassesDataChooser = function (config) {
  config = Ext.apply(
    {
      displayField: 'docClassName',
      valueField: 'docClassName',
      autoWidth: true,
      resizable: true,
      triggerAction: 'all',
      value: CUF.t('docClass-choose-class'),
      fieldLabel: CUF.t('docClass'),
      actualDocClassName: CUF.t('docClass-choose-class'),
      mode: 'local',
      forceSelection: true,
      emptyText: CUF.t('docClass-choose-class'),
      labelStyle: 'font-size: 12px;',
    },
    config
  );

  Ext.ux.plusmpm.DocClassesDataChooser.superclass.constructor.call(this, config);
};

Ext.extend(Ext.ux.plusmpm.DocClassesDataChooser, Ext.form.ComboBox, {
  initComponent: function () {
    Ext.ux.plusmpm.DocClassesDataChooser.superclass.initComponent.call(this);
  },
});

Ext.ux.plusmpm.AddDocFromArchiveGrid = function (config) {
  var resultReader = new Ext.data.JsonReader({
    id: 'CUF_from_archive_result_reader',
    fields: config.resultDef,
  });

  var proxy = new Ext.data.HttpProxy({
    method: 'GET',
    url: 'CUF.SearchDocsInArchive.customServlet',
  });

  var resultStore = new Ext.data.JsonStore({
    proxy: proxy,
    reader: resultReader,
    storeId: 'CUF_from_archive_result_store',
    autoDestroy: true,
    batch: true,
    root: 'documents',
    totalProperty: 'total',
    fields: config.resultDef,
    baseParams: {
      docClassName: '',
      conditions: '',
      sortBy: config.sortBy,
      sortDir: config.sortDir,
      start: 0,
      limit: 0,
    },
  });

  var view = new Ext.grid.GridView({
    forceFit: true,
    emptyText: "<img src='" + CUF_getUrl() + "/style/img/fam/error.png'><bold> " + CUF.t('no-documents') + '</bold>',
  });

  var selectionModel = new Ext.grid.RowSelectionModel({
    singleSelect: true,
    listeners: {
      beforerowselect: function (selModel, rowIndex, keepExisting, record) {
        return false;
      },
    },
  });

  var listeners = new Array();
  listeners.cellclick = function (grid, rowIndex, columnIndex, e) {
    var colModel = grid.getColumnModel();
    var colDataIdx = colModel.getDataIndex(columnIndex);

    if (colDataIdx != 'CUF_attach_doc_from_archive_btn') {
      return;
    }

    var store = grid.getStore();
    var record = store.getAt(rowIndex);
    var fileId = record.get('fileId');

    var addDocWindow = Ext.getCmp('CUF_add_doc_from_archive_window');
    var addButton = Ext.getCmp('CUF_add_from_archive_button');
    var destDocClassItem = config.destDocClassItem;

    if (!addDocWindow) {
      var form = new Ext.FormPanel({
        id: 'CUF_add_doc_from_archive_panel',
        frame: true,
        bodyStyle: 'padding: 10px 10px 0 10px;',
        labelWidth: 200,
        labelAlign: 'top',
        buttonAlign: 'center',
        items: [destDocClassItem],
      });

      addButton = new Ext.Button({
        id: 'CUF_add_from_archive_button',
        text: CUF.t('attach'),
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/add.png',
        hidden: false,
        handler: function (btn, e) {
          var docClassName = destDocClassItem.getValue();

          if (docClassName == CUF.t('docClass-choose-class') || !destDocClassItem.isValid()) {
            return;
          }

          var addFromArchiveServlet = 'CUF.AddDocFromArchive.customServlet';
          var addFromArchiveArgs =
            'processId=' +
            processId +
            '&activityId=' +
            activityId +
            '&docClassName=' +
            docClassName +
            '&fileId=' +
            this.fileId +
            '&uploader=' +
            sTaskUserName;
          var addFromArchiveResponse = CUF_executeServletGetResponse(addFromArchiveServlet, addFromArchiveArgs);
          var addFromArchiveResponseTxt = addFromArchiveResponse.responseText;
          var addFromArchiveJsonData = Ext.util.JSON.decode(addFromArchiveResponseTxt);
          var addFromArchiveSuccess = addFromArchiveJsonData.success;

          if (!addFromArchiveSuccess) {
            Ext.Msg.show({
              title: CUF.t('error'),
              msg: addFromArchiveJsonData.message,
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.ERROR,
            });
            return;
          }

          var newFileId = addFromArchiveJsonData.fileId;
          CUF_addDocumentsToView(newFileId);
          addDocWindow.hide();
          var attachToDT = config.attachToDT;
          var activityGrid = config.activityGrid;
          var DTIndexes = config.DTIndexes;

          if (attachToDT && activityGrid && DTIndexes) {
            var activityStore = activityGrid.getStore();

            try {
              activityStore.autoSave = false;
              var newRecord = new activityStore.recordType({});

              var getDocumentsInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
              var getDocumentsInfoArgs = 'fileIds=' + newFileId;
              var DocInfoResponse = CUF_executeServletGetResponse(getDocumentsInfoServlet, getDocumentsInfoArgs);
              var DocInfoResponseTxt = DocInfoResponse.responseText;
              var DocInfoJsonData = Ext.util.JSON.decode(DocInfoResponseTxt);
              var DocInfoSuccess = DocInfoJsonData.success;
              if (!DocInfoSuccess) {
                return;
              }
              var DocInfoRec = DocInfoJsonData.records[0];

              for (var idx in DTIndexes) {
                if (DTIndexes.hasOwnProperty(idx)) {
                  var tempIndexName = DTIndexes[idx];
                  var tempIndexValue = '';
                  var field = newRecord.fields.get(idx);
                  var fieldType = field.type.type;

                  if (tempIndexName == 'fileId') {
                    tempIndexValue = newFileId;
                  } else if (tempIndexName == 'processId') {
                    tempIndexName = processId;
                  } else if (tempIndexName == 'activityId') {
                    tempIndexName = activityId;
                  } else if (tempIndexName == 'orgFileName') {
                    tempIndexValue = DocInfoRec.orgName;
                  } else if (tempIndexName == 'description') {
                    tempIndexName = DocInfoRec.description;
                  } else if (tempIndexName == 'docClassName') {
                    tempIndexName = DocInfoRec.docClassName;
                  } else if (tempIndexName == 'owner') {
                    if (this.sRealUserName) {
                      tempIndexValue = sRealUserName;
                    } else if (this.sUserRealName) {
                      tempIndexValue = sUserRealName;
                    } else {
                      tempIndexValue = sTaskUserName;
                    }
                  } else if (tempIndexName == 'version') {
                    tempIndexValue = DocInfoRec.version;
                  } else if (tempIndexName == 'comment') {
                    tempIndexValue = DocInfoRec.comment;
                  } else {
                    tempIndexValue = record.get(tempIndexName);
                  }

                  if (!tempIndexValue) {
                    tempIndexValue = '';
                  }

                  if (fieldType == 'float' || fieldType == 'double') {
                    tempIndexValue = tempIndexValue.toString();
                    tempIndexValue = tempIndexValue.replace(',', '.');
                  }

                  newRecord.set(idx, tempIndexValue);
                }
              }

              activityStore.add(newRecord);
              activityStore.save();
            } catch (e) {
            } finally {
              activityStore.autoSave = true;
            }
          }

          try {
            window[config.funcNameAfterDocAction](processId, activityId, 'attach', docClassName, newFileId);
          } catch (e) {}

          if (config.reloadPage) {
            var win = Ext.getCmp('CUF_search_doc_from_archive_window');
            win.close();
            document.getElementById('saveBtn').click();
          } else {
            Ext.Msg.show({
              title: CUF.t('document-attached'),
              msg: CUF.t('document-attached-class') + docClassName + CUF.t('and-process'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.INFO,
            });
          }
        },
      });
      form.addButton(addButton);

      addDocWindow = new Ext.Window({
        id: 'CUF_add_doc_from_archive_window',
        autoWidth: true,
        autoHeight: true,
        modal: true,
        constrain: true,
        shadow: false,
        resizable: false,
        closeAction: 'hide',
        title: CUF.t('document-attaching'),
        iconCls: 'x-CUFwindow-adddoc',
        items: form,
      });
    }

    Ext.apply(addButton, {
      fileId: fileId,
    });

    addDocWindow.show();
  };

  Ext.apply(config, {
    listeners: listeners,
  });

  config = Ext.apply(
    {
      id: 'CUF_from_archive_result_grid',
      stateful: false,
      collapsible: false,
      store: resultStore,
      stripeRows: true,
      columnLines: true,
      autoScroll: true,
      colModel: new Ext.grid.ColumnModel({
        defaults: {
          sortable: true,
          menuDisabled: false,
        },
        columns: config.resultColumnModel,
      }),
      sm: selectionModel,
      loadMask: true,
      view: view,
      columnLines: true,
      bbar: new Ext.PagingToolbar({
        id: 'CUF_from_archive_result_grid_paging_toolbar',
        pageSize: 25,
        store: resultStore,
        displayInfo: true,
        displayMsg: '<bold>' + CUF.t('documents') + ' {0} - {1} ' + CUF.t('of') + ' {2}</bold>',
        emptyMsg: CUF.t('no-documents'),
      }),
    },
    config
  );

  Ext.ux.plusmpm.AddDocFromArchiveGrid.superclass.constructor.call(this, config);
};

Ext.extend(Ext.ux.plusmpm.AddDocFromArchiveGrid, Ext.grid.GridPanel, {
  initComponent: function () {
    Ext.ux.plusmpm.AddDocFromArchiveGrid.superclass.initComponent.call(this);
  },
  executeSearch: function (searchParams) {
    var store = this.getStore();
    store.setBaseParam('docClassName', searchParams['docClassName']);
    store.setBaseParam('conditions', searchParams['conditions']);
    store.setBaseParam('start', 0);
    store.setBaseParam('limit', 25);
    store.load({
      callback: this.afterLoad,
    });
  },
  afterLoad: function (records, options, success) {
    var resultLabel = Ext.getCmp('CUF_from_archive_result_label');

    if (!success) {
      var errorMsg = CUF.t('document-error-search-class') + docClassName + '.';

      Ext.Msg.show({
        title: CUF.t('error'),
        msg: errorMsg,
        buttons: Ext.Msg.OK,
        icon: Ext.Msg.ERROR,
      });

      resultLabel.removeClass('x-CUFresultLabel-correct ');
      resultLabel.addClass('x-CUFresultLabel-error');
      resultLabel.setText(errorMsg, true);
      return;
    }

    var foundDocs = this.getTotalCount();

    if (foundDocs == 0) {
      resultLabel.removeClass('x-CUFresultLabel-correct ');
      resultLabel.addClass('x-CUFresultLabel-error');
      searchMsg = CUF.t('document-search-no-found');
    } else {
      resultLabel.removeClass('x-CUFresultLabel-error');
      resultLabel.addClass('x-CUFresultLabel-correct ');
      searchMsg = CUF.t('document-search-found') + foundDocs;
    }

    resultLabel.setText(searchMsg, true);
  },
});

function CUF_hideResultGridCol(colId) {
  var resultGrid = Ext.getCmp('CUF_from_archive_result_grid');
  var colModel = resultGrid.getColumnModel();
  if (colModel.getColumnCount(true) > 1) {
    var columnIdx = colModel.getIndexById(colId);
    colModel.setHidden(columnIdx, true);
  }
}

function CUF_createIndexesItem(form, docClassData, docClassName, width, synchronize, doLayout) {
  var docClassJsonRecs = docClassData[docClassName];
  var labelStyle = 'margin:3px 0px 0px 0px !important;padding:0px 0px 0px 0px !important;';

  if (docClassJsonRecs && docClassJsonRecs.length > 0) {
    var indexLabelItem = new Ext.form.Label({
      id: 'CUF_index_label_item',
      fieldLabel: CUF.t('indexes'),
    });

    form.add(indexLabelItem);

    var indexesItem = new Ext.Panel({
      id: 'CUF_indexes_panel',
      // width : width - 20,
      anchor: '100%',
      autoScroll: true,
      frame: true,
      layout: 'form',
      labelWidth: 120,
      labelAlign: 'left',
      labelStyle: labelStyle,
    });

    CUF_fillIndexesItem(indexesItem, docClassJsonRecs, 120, 0, synchronize);

    form.add(indexesItem);
  } else {
    var indexLabelItem = new Ext.form.Label({
      id: 'CUF_index_label_item',
      fieldLabel: CUF.t('docClass-indexes-empty') + ' ' + docClassName,
      labelSeparator: '',
      anchor: '100%',
    });

    form.add(indexLabelItem);
  }

  if (Ext.isEmpty(doLayout) || doLayout == true) {
    form.doLayout();
  }
}

function CUF_fillIndexesItem(indexesItem, docClassJsonRecs, width, labelWidth, synchronize, dateRange) {
  var indexItem = null;
  var itemAnchor = '95%';
  var fieldStyle = 'margin-top:0px;margin-bottom:0px;margin-right:0px';
  var labelStyle = 'margin:3px 0px 0px 0px !important;padding:0px 0px 0px 0px !important;';
  for (var j = 0; j < docClassJsonRecs.length; j++) {
    var docClassJsonRec = docClassJsonRecs[j];

    if (docClassJsonRec.indexType == 'date' && dateRange == true) {
      var startValue = '';
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }
      var df1 = new Ext.form.DateField({
        format: 'Y-m-d',
        name: 'startDate',
        fieldLabel: CUF.t('from'),
        value: startValue,
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
      var df2 = new Ext.form.DateField({
        format: 'Y-m-d',
        value: startValue,
        name: 'endDate',
        fieldLabel: CUF.t('to'),
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
      var lbl1 = new Ext.form.Label({ text: CUF.t('from'), style: { marginTop: '5px' } });
      var lbl2 = new Ext.form.Label({ text: CUF.t('to'), style: { marginTop: '5px' } });
      indexItem = new Ext.form.CompositeField({
        id: 'index' + (j + 1),
        indexType: 'date',
        fieldLabel: docClassJsonRec.indexName,
        items: [lbl1, df1, lbl2, df2],
      });
    } else if (docClassJsonRec.indexType == 'date') {
      var startValue = '';
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }

      indexItem = new Ext.form.DateField({
        id: 'index' + (j + 1),
        fieldLabel: docClassJsonRec.indexName,
        format: 'Y-m-d',
        value: startValue,
        // width : width - labelWidth - 50,
        labelStyle: labelStyle,
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
    } else if (docClassJsonRec.indexType == 'integer') {
      var startValue = '';
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }

      indexItem = new Ext.form.NumberField({
        id: 'index' + (j + 1),
        fieldLabel: docClassJsonRec.indexName,
        allowDecimals: false,
        value: startValue,
        // width : width - labelWidth - 50,
        labelStyle: labelStyle,
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
    } else if (docClassJsonRec.indexType == 'float') {
      var startValue = '';
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }

      indexItem = new Ext.form.NumberField({
        id: 'index' + (j + 1),
        fieldLabel: docClassJsonRec.indexName,
        allowDecimals: true,
        decimalSeparator: ',',
        value: startValue,
        // width : width - labelWidth - 50,
        labelStyle: labelStyle,
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
    } else if (docClassJsonRec.indexType == 'list') {
      var startValue = '';
      var listValues = docClassJsonRec.indexListValues;
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }

      if (listValues && listValues != '') {
        var splitValues;
        if (listValues.indexOf('\n') != -1) {
          splitValues = listValues.split('\n');
        } else {
          splitValues = listValues.split(',');
        }

        var store = new Ext.data.SimpleStore();
        var rows = Ext.data.Record.create([
          {
            name: 'value',
            type: 'string',
          },
        ]);

        for (var i = 0; i < splitValues.size(); i++) {
          var record = new rows({
            value: splitValues[i],
          });

          store.add(record);
        }

        store.commitChanges();

        indexItem = new Ext.form.ComboBox({
          id: 'index' + (j + 1),
          displayField: 'value',
          valueField: 'value',
          store: store,
          triggerAction: 'all',
          emptyText: '',
          fieldLabel: docClassJsonRec.indexName,
          mode: 'local',
          forceSelection: true,
          // width : width - labelWidth - 50,
          labelStyle: labelStyle,
          style: fieldStyle,
          anchor: itemAnchor,
          value: startValue,
          enableKeyEvents: true,
        });
      } else {
        indexItem = new Ext.form.TextField({
          id: 'index' + (j + 1),
          inputType: 'text',
          fieldLabel: docClassJsonRec.indexName,
          // width : width - labelWidth - 50,
          labelStyle: labelStyle,
          style: fieldStyle,
          anchor: itemAnchor,
          value: startValue,
          enableKeyEvents: true,
        });
      }
    } else {
      var startValue = '';
      var fromResp = docClassJsonRec.indexValue;
      if (synchronize && fromResp) {
        startValue = fromResp;
      }

      indexItem = new Ext.form.TextField({
        id: 'index' + (j + 1),
        inputType: 'text',
        fieldLabel: docClassJsonRec.indexName,
        value: startValue,
        // width : width - labelWidth - 50,
        labelStyle: labelStyle,
        style: fieldStyle,
        anchor: itemAnchor,
        enableKeyEvents: true,
      });
    }

    indexesItem.add(indexItem);
  }
}

// ///////////////// zarzadanie dokumentami procesu /////////////
function CUF_processDocumentsManagement(config) {
  if (!config) {
    return;
  }

  var windowTitle = config.windowTitle;
  var addDoc = config.addDoc;
  var addDocButtonLabel = config.addDocButtonLabel;
  var attachToDT = config.attachToDT;
  var moveDoc = config.moveDoc;
  var moveDocButtonLabel = config.moveDocButtonLabel;
  var detach = config.detach;
  var detachButtonLabel = config.detachButtonLabel;
  var deleteDoc = config.deleteDoc;
  var deleteDocButtonLabel = config.deleteDocButtonLabel;
  var displayDoc = config.displayDoc;
  var displayDocButtonLabel = config.displayDocButtonLabel;
  var downloadDoc = config.downloadDoc;
  var downloadDocButtonLabel = config.downloadDocButtonLabel;
  var addToDocView = config.addToDocView;
  var addToDocViewButtonLabel = config.addToDocViewButtonLabel;
  var showIndexes = config.showIndexes;
  var showIndexesButtonLabel = config.showIndexesButtonLabel;
  var showIndexesEnableChanges = config.showIndexesEnableChanges;
  var changeDescr = config.changeDescr;
  var changeDescrButtonLabel = config.changeDescrButtonLabel;
  var addComment = config.addComment;
  var addCommentButtonLabel = config.addCommentButtonLabel;
  var extraFunc = config.extraFunc;
  var extraFuncButtonLabel = config.extraFuncButtonLabel;
  var extraFuncName = config.extraFuncName;
  var extraFuncClass = config.extraFuncClass;
  var funcNameAfterDocAction = config.funcNameAfterDocAction;
  var showDocToolTip = config.showDocToolTip;
  var checkDetachRights = config.checkDetachRights;
  var checkDeleteRights = config.checkDeleteRights;
  var width = screen.width - 100;

  if (!windowTitle || windowTitle == '') {
    windowTitle = CUF.t('document-process');
  }

  var data_window = new Ext.Window({
    id: 'CUF_process_doc_management_window',
    width: width + 20,
    autoHeight: true,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    title: windowTitle,
    iconCls: 'x-CUFwindow-management',
    listeners: {
      close: function (p) {
        Ext.getCmp('CUF_documents_item_ctx_menu').destroy();
      },
    },
  });

  var documentsPanel = null;
  var gridItem = null;
  var labelItem = null;
  var store = null;

  var bAnyDocAttached = false;
  var url = CUF_getUrl();

  var attachedDocIdsServlet = 'CUF.GetDocumentsAttachedToProcess.customServlet';
  var getDocumentsInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
  var dettachDocServlet = 'CUF.DetachDocumentFromProcess.customServlet';
  var attachedDocIdsArgs = 'sProcessId=' + processId + '&sActivityId=' + activityId;

  var DocIdsResponse = CUF_executeServletGetResponse(attachedDocIdsServlet, attachedDocIdsArgs);
  var DocIdsResponseTxt = DocIdsResponse.responseText;
  var DocIdsJsonData = Ext.util.JSON.decode(DocIdsResponseTxt);
  var DocIdsSuccess = DocIdsJsonData.success;

  if (!DocIdsSuccess) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: DocIdsJsonData.message,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return;
  }

  documentsPanel = new Ext.FormPanel({
    id: 'CUF_doc_management_panel',
    width: width + 10,
    frame: true,
    bodyStyle: 'padding: 10px 10px 0 10px;',
    labelWidth: width - 50,
  });

  labelItem = new Ext.form.Label({
    id: 'CUF_documents_item',
    fieldLabel: CUF.t('document-process-no-attached'),
    labelSeparator: '',
  });

  var docItemCtxMenu = new Ext.menu.Menu({
    id: 'CUF_documents_item_ctx_menu',
    renderTo: 'CUF_process_doc_management_window',
  });

  var docIdsForProcess = DocIdsJsonData.message;

  if (docIdsForProcess != null && docIdsForProcess != '') {
    bAnyDocAttached = true;

    var documentDef = Ext.data.Record.create([
      {
        name: 'lp',
        type: 'string',
      },
      {
        name: 'orgName',
        type: 'string',
      },
      {
        name: 'description',
        type: 'string',
      },
      {
        name: 'date',
        type: 'string',
      },
      {
        name: 'size',
        type: 'string',
      },
      {
        name: 'owner',
        type: 'string',
      },
      {
        name: 'moreInfo',
        type: 'string',
      },
      {
        name: 'fileId',
        type: 'string',
      },
    ]);

    var processInfoReader = new Ext.data.JsonReader({
      id: 'CUF_process_doc_reader',
      fields: documentDef,
    });

    store = new Ext.data.Store({
      storeId: 'CUF_process_documents_store',
      reader: processInfoReader,
      fields: documentDef,
    });

    var getDocumentsInfoArgs = 'fileIds=' + docIdsForProcess;
    var DocInfoResponse = CUF_executeServletGetResponse(getDocumentsInfoServlet, getDocumentsInfoArgs);
    var DocInfoResponseTxt = DocInfoResponse.responseText;
    var DocInfoJsonData = Ext.util.JSON.decode(DocInfoResponseTxt);
    var DocInfoSuccess = DocInfoJsonData.success;

    if (DocInfoSuccess) {
      Ext.each(DocInfoJsonData.records, function (record, index, allRecords) {
        Ext.apply(record, {
          moreInfo:
            '<a href="#info"><img onclick="new Function(CUF_moreInfoForDocument(\'' +
            record.fileId +
            '\'))" src="' +
            url +
            '/style/img/fam/information.png"></a>',
        });
      });

      store.loadData(DocInfoJsonData.records);
      store.commitChanges();
    }

    gridItem = new Ext.grid.GridPanel({
      id: 'CUF_documents_item',
      store: store,
      iconCls: 'x-CUFgrid-documents',
      enableColumnResize: false,
      enableColumnMove: false,
      enableColumnHide: false,
      stripeRows: true,
      columnLines: true,
      colModel: new Ext.grid.ColumnModel({
        defaults: {
          sortable: true,
          menuDisabled: false,
          align: 'center',
          css: 'vertical-align:middle;',
        },
        columns: [
          {
            id: 'lp',
            header: 'Lp.',
            width: 20,
            dataIndex: 'lp',
          },
          {
            id: 'docName',
            header: CUF.t('document-name'),
            dataIndex: 'orgName',
          },
          {
            id: 'descr',
            header: CUF.t('docClass-description'),
            dataIndex: 'description',
          },
          {
            id: 'date',
            header: CUF.t('date'),
            width: 65,
            dataIndex: 'date',
          },
          {
            id: 'size',
            header: CUF.t('size'),
            width: 35,
            dataIndex: 'size',
          },
          {
            id: 'owner',
            header: CUF.t('owner'),
            dataIndex: 'owner',
          },
          {
            id: 'more',
            header: CUF.t('info'),
            width: 20,
            dataIndex: 'moreInfo',
          },
          {
            id: 'fileId',
            header: CUF.t('file-id'),
            hidden: true,
            dataIndex: 'fileId',
          },
        ],
      }),
      view: new Ext.grid.GridView({
        forceFit: true,
      }),
      sm: new Ext.grid.RowSelectionModel({
        singleSelect: false,
        listeners: {
          beforerowselect: function (sm, rowIndex, keepExisting, record) {
            return keepExisting;
          },
        },
      }),
      width: width - 20,
      height: 300,
      frame: true,
      title: CUF.t('document-attached-process-amount') + store.getCount() + ')',
      listeners: {
        sortchange: function (grid, sortInfo) {
          var recordCounter = 1;

          store.each(function (record) {
            record.set('lp', recordCounter);
            recordCounter++;
          });

          store.commitChanges();
        },
        render: function (grid) {
          if (!showDocToolTip) {
            return;
          }

          var tempFileIdVar = config.fileIdVar;

          Ext.apply(config, {
            renderToolTipTo: 'CUF_process_doc_management_window',
            fileIdVar: 'fileId',
          });

          CUF_showDocToolTip(grid, config);
          config.fileIdVar = tempFileIdVar;
        },
        rowcontextmenu: function (grid, rowIndex, e) {
          e.stopEvent();
          docItemCtxMenu.showAt(e.getXY());
        },
        rowclick: function (grid, rowIndex, e) {
          var selModel = grid.getSelectionModel();
          if (selModel.isSelected(rowIndex)) {
            selModel.deselectRow(rowIndex);
          } else {
            selModel.selectRow(rowIndex, true);
          }
        },
      },
    });

    store.sort('fileId', 'ASC');
    CUF_handleRowDblClick(gridItem, config);
    documentsPanel.add(gridItem);
  } else {
    documentsPanel.add(labelItem);
  }

  if (addDoc) {
    if (!addDocButtonLabel || addDocButtonLabel == '') {
      addDocButtonLabel = CUF.t('attach-document');
    }

    var addDocButton = new Ext.Button({
      id: 'CUF_attach_doc_button',
      text: addDocButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/add.png',
      handler: function () {
        data_window.close();
        CUF_attachDocument(config);
      },
    });

    documentsPanel.addButton(addDocButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_add_doc',
        text: addDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/add.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          addDocButton.handler.call(addDocButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && moveDoc) {
    if (!moveDocButtonLabel || moveDocButtonLabel == '') {
      moveDocButtonLabel = CUF.t('move');
    }

    var moveDocButton = new Ext.Button({
      id: 'CUF_move_doc_button',
      text: moveDocButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/arrow_turn_right.png',
      hidden: true,
      handler: function () {
        if (gridItem != null) {
          var rows = gridItem.getSelectionModel().getSelections();

          if (rows.length == 0) {
            Ext.Msg.show({
              title: CUF.t('warning'),
              msg: CUF.t('document-choose-move'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.INFO,
            });
            return;
          }

          var documentsToMove = new Array();
          for (var i = 0; i < rows.length; i++) {
            var fileId = rows[i].get('fileId');
            documentsToMove.push(fileId);
          }

          config.documentsToMove = documentsToMove;
          CUF_moveDocumentBetweenDocClasses(config);
        }
      },
    });

    documentsPanel.addButton(moveDocButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_move_doc',
        text: moveDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/arrow_turn_right.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          moveDocButton.handler.call(moveDocButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && detach) {
    if (!detachButtonLabel || detachButtonLabel == '') {
      detachButtonLabel = CUF.t('detach');
    }

    var dettachButton = new Ext.Button({
      id: 'CUF_detach_doc_button',
      text: detachButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/delete.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-detach'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        Ext.Msg.show({
          title: CUF.t('document-process-detaching'),
          msg: CUF.t('document-selected') + rows.length + '. ' + CUF.t('detach-process'),
          buttons: Ext.Msg.YESNO,
          fn: function (buttonId) {
            if (buttonId != 'yes') {
              return;
            }

            for (var i = 0; i < rows.length; i++) {
              var fileId = rows[i].get('fileId');

              var dettachDocArgs =
                'sProcessId=' + processId + '&sActivityId=' + activityId + '&sFileId=' + fileId + '&checkRights=' + checkDetachRights;
              var detachResponse = CUF_executeServletGetResponse(dettachDocServlet, dettachDocArgs);
              var detachResponseTxt = detachResponse.responseText;
              var detachResponseJsonData = Ext.util.JSON.decode(detachResponseTxt);
              var detachResponseSuccess = detachResponseJsonData.success;

              if (!detachResponseSuccess) {
                if (detachResponseJsonData.message == 'noRights') Ext.Msg.alert(CUF.t('error'), CUF.t('document-detach-no-rights'));
                else Ext.Msg.alert(CUF.t('error'), CUF.t('document-error-process-detach'));
                break;
              }

              store.remove(rows[i]);
              CUF_deleteDocumentFromView(fileId);

              try {
                window[funcNameAfterDocAction](processId, activityId, 'detach', detachResponseJsonData.message, fileId);
              } catch (e) {}
            }

            if (store.getCount() == 0) {
              gridItem = null;
              var buttonIdsArray = new Array();
              buttonIdsArray.push('CUF_move_doc_button');
              buttonIdsArray.push('CUF_detach_doc_button');
              buttonIdsArray.push('CUF_delete_doc_button');
              buttonIdsArray.push('CUF_display_doc_button');
              buttonIdsArray.push('CUF_download_doc_button');
              buttonIdsArray.push('CUF_enlarge_doc_button');
              buttonIdsArray.push('CUF_show_doc_indexes_button');
              buttonIdsArray.push('CUF_change_doc_descr_button');
              buttonIdsArray.push('CUF_add_doc_comment_button');
              CUF_deactivateButtons(buttonIdsArray);
              documentsPanel.remove('CUF_documents_item');
              documentsPanel.insert(0, labelItem);
              documentsPanel.doLayout();
            } else {
              gridItem.setTitle(CUF.t('document-attached-process-amount') + store.getCount() + ')');

              for (var i = 0; i < store.getCount(); i++) {
                var row = store.getAt(i);

                row.set('lp', i + 1);
              }

              store.commitChanges();
            }

            if (attachToDT) {
              CUF_loadProcessDocsToDT(config);
            }
          },
          icon: Ext.MessageBox.QUESTION,
        });
      },
    });

    documentsPanel.addButton(dettachButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_detach_doc',
        text: detachButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/delete.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          dettachButton.handler.call(dettachButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && deleteDoc) {
    if (!deleteDocButtonLabel || deleteDocButtonLabel == '') {
      deleteDocButtonLabel = CUF.t('delete');
    }

    var deleteButton = new Ext.Button({
      id: 'CUF_delete_doc_button',
      text: deleteDocButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/cancel.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-detach-delete-archive'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        Ext.Msg.show({
          title: CUF.t('documents-deleting'),
          msg: CUF.t('document-selected') + rows.length + '. ' + CUF.t('detach-delete-process'),
          buttons: Ext.Msg.YESNO,
          fn: function (buttonId) {
            if (buttonId != 'yes') {
              return;
            }

            for (var i = 0; i < rows.length; i++) {
              var fileId = rows[i].get('fileId');
              var dettachDocArgs = 'sProcessId=' + processId + '&sActivityId=' + activityId + '&sFileId=' + fileId;
              if (checkDeleteRights) {
                var hasRightToDelete = CUF_exec(
                  'com.plusmpm.servlet.extension.CUF.CheckFileRights.customServlet',
                  'fileId=' + fileId + '&type=delete'
                );
                if (hasRightToDelete.status == 0) {
                  if (hasRightToDelete.data != 'full') {
                    Ext.Msg.alert(CUF.t('error'), CUF.t('document-delete-no-rights'));
                    continue;
                  }
                } else {
                  Ext.Msg.alert(CUF.t('error'), CUF.t('document-rights-error'));
                  continue;
                }
              }
              var detachResponse = CUF_executeServletGetResponse(dettachDocServlet, dettachDocArgs);
              var detachResponseTxt = detachResponse.responseText;
              var detachResponseJsonData = Ext.util.JSON.decode(detachResponseTxt);
              var detachResponseSuccess = detachResponseJsonData.success;

              if (!detachResponseSuccess) {
                Ext.Msg.alert(CUF.t('error'), CUF.t('document-error-process-detach'));
                break;
              }

              store.remove(rows[i]);
              CUF_deleteDocumentFromView(fileId);
              CUF_deleteDocumentFromArchive(fileId);

              try {
                window[funcNameAfterDocAction](processId, activityId, 'delete', detachResponseJsonData.message, fileId);
              } catch (e) {}
            }

            if (store.getCount() == 0) {
              gridItem = null;
              var buttonIdsArray = new Array();
              buttonIdsArray.push('CUF_move_doc_button');
              buttonIdsArray.push('CUF_detach_doc_button');
              buttonIdsArray.push('CUF_delete_doc_button');
              buttonIdsArray.push('CUF_display_doc_button');
              buttonIdsArray.push('CUF_download_doc_button');
              buttonIdsArray.push('CUF_enlarge_doc_button');
              buttonIdsArray.push('CUF_show_doc_indexes_button');
              buttonIdsArray.push('CUF_change_doc_descr_button');
              buttonIdsArray.push('CUF_add_doc_comment_button');
              CUF_deactivateButtons(buttonIdsArray);
              documentsPanel.remove('CUF_documents_item');
              documentsPanel.insert(0, labelItem);
              documentsPanel.doLayout();
            } else {
              gridItem.setTitle(CUF.t('document-attached-process-amount') + store.getCount() + ')');

              for (var i = 0; i < store.getCount(); i++) {
                var row = store.getAt(i);

                row.set('lp', i + 1);
              }

              store.commitChanges();
            }

            if (attachToDT) {
              CUF_loadProcessDocsToDT(config);
            }
          },
          icon: Ext.MessageBox.QUESTION,
        });
      },
    });

    documentsPanel.addButton(deleteButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_delete_doc',
        text: deleteDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/cancel.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          deleteButton.handler.call(deleteButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && displayDoc) {
    if (!displayDocButtonLabel || displayDocButtonLabel == '') {
      addDocButtonLabel = CUF.t('view');
    }

    var displayButton = new Ext.Button({
      id: 'CUF_display_doc_button',
      text: displayDocButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/zoom.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-view'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        if (rows.length > 1) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-view-single'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var fileId = rows[0].get('fileId');
        CUF_displayDocument(fileId);
      },
    });

    documentsPanel.addButton(displayButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_display_doc',
        text: displayDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/zoom.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          displayButton.handler.call(displayButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && downloadDoc) {
    if (!downloadDocButtonLabel || downloadDocButtonLabel == '') {
      downloadDocButtonLabel = CUF.t('download');
    }

    var downloadButton = new Ext.Button({
      id: 'CUF_download_doc_button',
      text: downloadDocButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/page_white_go.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-download'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }
        if (rows.length > 1) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-download-single'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var fileId = rows[0].get('fileId');
        CUF_downloadDocument(fileId);
      },
    });

    documentsPanel.addButton(downloadButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_download_doc',
        text: downloadDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/page_white_go.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          downloadButton.handler.call(downloadButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && addToDocView) {
    if (!addToDocViewButtonLabel || addToDocViewButtonLabel == '') {
      addToDocViewButtonLabel = CUF.t('enlarge');
    }

    var addToDocViewButton = new Ext.Button({
      id: 'CUF_enlarge_doc_button',
      text: addToDocViewButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/application_add.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-enlarge'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var fileIds = new Array();
        for (var i = 0; i < rows.length; i++) {
          fileIds.push(rows[i].get('fileId'));
        }
        CUF_addDocumentsToView(fileIds.join(','));
      },
    });

    documentsPanel.addButton(addToDocViewButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_enlarge_doc',
        text: addToDocViewButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/application_add.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          addToDocViewButton.handler.call(addToDocViewButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && showIndexes) {
    if (!showIndexesButtonLabel || showIndexesButtonLabel == '') {
      showIndexesButtonLabel = CUF.t('indexes');
    }

    var showIndexesButton = new Ext.Button({
      id: 'CUF_show_doc_indexes_button',
      text: showIndexesButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/page_white_wrench.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }
        if (rows.length > 1) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-single'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var fileId = rows[0].get('fileId');
        CUF_showDocIndexes(fileId, showIndexesEnableChanges);

        try {
          window[funcNameAfterDocAction](processId, activityId, 'updateIndexes', '', fileId);
        } catch (e) {}
      },
    });

    documentsPanel.addButton(showIndexesButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_show_indexes',
        text: showIndexesButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/page_white_wrench.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          showIndexesButton.handler.call(showIndexesButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && changeDescr) {
    if (!changeDescrButtonLabel || changeDescrButtonLabel == '') {
      changeDescrButtonLabel = CUF.t('description-change');
    }

    var changeDescrButton = new Ext.Button({
      id: 'CUF_change_doc_descr_button',
      text: changeDescrButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/arrow_refresh.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-change-description'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }
        if (rows.length > 1) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-change-description-single'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var record = rows[0];
        var fileId = record.get('fileId');
        CUF_changeDescriptionForDoc(fileId, record);
      },
    });

    documentsPanel.addButton(changeDescrButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_change_descr',
        text: changeDescrButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/arrow_refresh.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          changeDescrButton.handler.call(changeDescrButton, e);
        },
      })
    );
  }

  if (bAnyDocAttached && addComment) {
    if (addCommentButtonLabel) {
      addCommentButtonLabel = CUF.t('comment-add');
    }

    var addCommentButton = new Ext.Button({
      id: 'CUF_add_doc_comment_button',
      text: addCommentButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/comment.png',
      hidden: true,
      handler: function () {
        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-add-comment'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }
        if (rows.length > 1) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('document-choose-add-comment-single'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        var fileId = rows[0].get('fileId');
        CUF_saveCommentForDoc(fileId);
      },
    });

    documentsPanel.addButton(addCommentButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_add_comment',
        text: addCommentButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/comment.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          addCommentButton.handler.call(addCommentButton, e);
        },
      })
    );
  }

  if (extraFunc != null && extraFunc) {
    if (!extraFuncButtonLabel || extraFuncButtonLabel == '') {
      extraFuncButtonLabel = CUF.t('use');
    }

    var extraFuncButton = new Ext.Button({
      id: 'CUF_doc_extra_func_button',
      text: extraFuncButtonLabel,
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/link.png',
      hidden: true,
      handler: function () {
        if (!extraFuncName || extraFuncName == '') {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('function-error-no-execute'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.WARNING,
          });
          return;
        }

        var fileIds = '';

        if (gridItem == null) {
          return;
        }

        var rows = gridItem.getSelectionModel().getSelections();

        if (rows != null && rows.length > 0) {
          for (var h = 0; h < rows.length; h++) {
            var fileId = rows[h].get('fileId');

            if (fileIds == '') {
              fileIds = fileId;
            } else {
              fileIds += ',' + fileId;
            }
          }
        }

        if (extraFuncName == 'CUF_executeExtraMethodForDocs') {
          if (extraFuncClass != null && extraFuncClass != undefined && extraFuncClass != '') {
            CUF_executeExtraMethodForDocs(processId, activityId, fileIds, extraFuncClass);
          } else {
            Ext.Msg.show({
              title: CUF.t('warning'),
              msg: CUF.t('function-error-no-execute'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.WARNING,
            });
          }
        } else {
          try {
            window[extraFuncName](processId, activityId, fileIds);
          } catch (e) {
            Ext.Msg.show({
              title: CUF.t('warning'),
              msg: CUF.t('function-error-no-execute'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.WARNING,
            });
          }
        }
      },
    });

    documentsPanel.addButton(extraFuncButton);

    docItemCtxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_extra_func',
        text: extraFuncButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/link.png',
        handler: function (item, e) {
          docItemCtxMenu.hide();
          extraFuncButton.handler.call(extraFuncButton, e);
        },
      })
    );
  }

  var closeButton = new Ext.Button({
    id: 'CUF_close_doc_management_button',
    text: CUF.t('close'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/cross.png',
    handler: function () {
      data_window.close();
    },
  });

  documentsPanel.addButton(closeButton);

  docItemCtxMenu.addMenuItem(
    new Ext.menu.Item({
      id: 'CUF_ctx_menu_close',
      text: CUF.t('close'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/cross.png',
      handler: function (item, e) {
        docItemCtxMenu.hide();
        closeButton.handler.call(closeButton, e);
      },
    })
  );

  data_window.add(documentsPanel);
  data_window.show();
}

// dodatkowe informacje dla dokumentu
function CUF_moreInfoForDocument(fileId) {
  CUF_importCssForFrame('processdocuments');
  var getDocumentInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
  var getDocumentInfoArgs = 'fileIds=' + fileId;
  var DocInfoResponse = CUF_executeServletGetResponse(getDocumentInfoServlet, getDocumentInfoArgs);
  var DocInfoResponseTxt = DocInfoResponse.responseText;
  var DocInfoJsonData = Ext.util.JSON.decode(DocInfoResponseTxt);
  var DocInfoSuccess = DocInfoJsonData.success;

  if (!DocInfoSuccess) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: DocInfoJsonData.message,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.ERROR,
    });
    return;
  }

  var DocInfoRecord = DocInfoJsonData.records[0];
  var descr = DocInfoRecord.description;
  if (descr == null) {
    descr = '';
  }

  var archiveDate = DocInfoRecord.date;
  if (archiveDate == null) {
    archiveDate = '';
  }

  var size = DocInfoRecord.size;
  if (size == null) {
    size = '0 KB';
  } else {
    size = parseInt(size / 1024) + ' KB';
  }

  var owner = DocInfoRecord.owner;
  if (owner == null) {
    owner = '';
  }

  var docId = DocInfoRecord.docId;
  if (docId == null) {
    docId = '-1';
  }

  var docClassName = DocInfoRecord.docClassName;
  if (docClassName == null) {
    docClassName = '';
  }

  var expirationDate = DocInfoRecord.expirationDate;
  if (expirationDate == null) {
    expirationDate = '';
  }

  var version = DocInfoRecord.version;
  if (version == null) {
    version = '';
  }

  var state = DocInfoRecord.state;
  if (state == null) {
    state = '';
  }

  var comment = DocInfoRecord.comment;
  if (comment == null) {
    comment = '';
  }

  var width = 350;

  var moreDocInfoPanel = new Ext.Panel({
    id: 'CUF_more_document_info_panel',
    width: width + 10,
    frame: true,
    bodyStyle: 'padding: 10px 10px 0px 10px;',
    html:
      '<table width="' +
      parseInt(width - 20) +
      '"><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('document-id') +
      ':</b></p></td><td>' +
      docId +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('docClass') +
      ':</b></p></td><td>' +
      docClassName +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('archive-date') +
      ':</b></p></td><td>' +
      archiveDate +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('date-expired') +
      ':</b></p></td><td>' +
      expirationDate +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('owner') +
      ':</b></p></td><td>' +
      owner +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('description') +
      ':</b></p></td><td>' +
      descr +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('size') +
      ':</b></p></td><td>' +
      size +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('version') +
      ':</b></p></td><td>' +
      version +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('state') +
      ':</b></p></td><td>' +
      state +
      '</td></tr><tr><td width="120"><p style="color:#28529C"><b>' +
      CUF.t('comment') +
      ':</b></p></td><td>' +
      comment +
      '</td></tr></table>',
  });

  var data_window = new Ext.Window({
    id: 'CUF_more_document_info_window',
    width: width + 20,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    title: CUF.t('document-info'),
    iconCls: 'x-CUFwindow-moredocinfo',
    items: moreDocInfoPanel,
  });

  data_window.show();
}

function CUF_moveDocumentBetweenDocClasses(config) {
  CUF_importCssForFrame('processdocuments');
  var moveDocButtonLabel = config.moveDocButtonLabel;
  var moveDocDocClasses = config.moveDocDocClasses;
  var moveDocCheckProtection = config.moveDocCheckProtection;
  var moveDocCopyIndexes = config.moveDocCopyIndexes;
  var funcNameAfterDocAction = config.funcNameAfterDocAction;
  var documentsToMove = config.documentsToMove;

  var processId = document.getElementsByName('processId')[0].value;
  var activityId = document.getElementsByName('activityId')[0].value;

  if (!moveDocButtonLabel || moveDocButtonLabel == '') {
    moveDocButtonLabel = CUF.t('move');
  }

  Ext.Msg.show({
    title: CUF.t('document-moving'),
    msg: CUF.t('document-selected') + documentsToMove.length + '. ' + CUF.t('document-move-class-another'),
    buttons: Ext.Msg.YESNO,
    fn: function (buttonId) {
      if (buttonId != 'yes') {
        return;
      }

      if (moveDocDocClasses == null || moveDocDocClasses == undefined || moveDocDocClasses == '') {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('docClass-no-available'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      var docClassTab = moveDocDocClasses.split(';');
      var acceptedDocClasses = new Array();

      var checkDocClassServlet = 'CUF.CheckDocClass.customServlet';
      var checkDocClassProtectionServlet = 'CUF.CheckDocClassProtection.customServlet';
      var checkDocClassArgs = '';
      var checkDocClassProtectionArgs = '';
      var moveDocClassItem = null;

      for (var k = 0; k < docClassTab.length; k++) {
        checkDocClassArgs = 'docClassName=' + docClassTab[k];
        var docClassResponse = CUF_executeServletGetResponse(checkDocClassServlet, checkDocClassArgs);
        var docClassResponseTxt = docClassResponse.responseText;
        var docClassJsonData = Ext.util.JSON.decode(docClassResponseTxt);
        var docClassId = docClassJsonData.message;

        if (docClassId == '-1') {
          continue;
        }

        if (moveDocCheckProtection) {
          checkDocClassProtectionArgs = 'userId=' + sTaskUserName + '&docClassName=' + docClassTab[k];
          var protectionResponse = CUF_executeServletGetResponse(checkDocClassProtectionServlet, checkDocClassProtectionArgs);
          var protectionResponseTxt = protectionResponse.responseText;
          var protectionJsonData = Ext.util.JSON.decode(protectionResponseTxt);
          var protectionSuccess = protectionJsonData.success;

          if (protectionSuccess) {
            acceptedDocClasses.push(docClassTab[k]);
          }
        } else {
          acceptedDocClasses.push(docClassTab[k]);
        }
      }

      if (acceptedDocClasses.size() == 0) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('docClass-no-available'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      var moveDocWindow = null;

      var moveStore = new Ext.data.Store();
      var docClass = Ext.data.Record.create([
        {
          name: 'id',
          type: 'string',
        },
        {
          name: 'docClassName',
          type: 'string',
        },
      ]);

      for (var i = 0; i < acceptedDocClasses.size(); i++) {
        var record = new docClass({
          id: i,
          docClassName: acceptedDocClasses[i],
        });

        moveStore.add(record);
      }

      moveStore.commitChanges();

      moveDocClassItem = new Ext.form.ComboBox({
        id: 'CUF_doc_class_name_item',
        displayField: 'docClassName',
        valueField: 'docClassName',
        store: moveStore,
        triggerAction: 'all',
        value: acceptedDocClasses[0],
        fieldLabel: CUF.t('docClass-target'),
        mode: 'local',
        forceSelection: true,
        width: 350,
      });

      if (acceptedDocClasses.size() == 1) {
        moveDocClassItem.disable();
      }

      var moveDocAddActionItem = new Ext.form.Checkbox({
        id: 'CUF_move_doc_add_action_item',
        fieldLabel: CUF.t('docClass-actions-execute'),
        boxLabel: '',
        inputValue: 'true',
        checked: true,
      });

      var moveDocForm = new Ext.FormPanel({
        id: 'CUF_move_doc_panel',
        width: 400,
        frame: true,
        bodyStyle: 'padding: 10px 10px 0 10px;',
        labelWidth: 140,
        labelAlign: 'top',
        items: [moveDocClassItem, moveDocAddActionItem],
      });

      var moveDocActionButton = new Ext.Button({
        id: 'CUF_accept_move_doc_button',
        text: moveDocButtonLabel,
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/arrow_turn_right.png',
        handler: function () {
          var sChosenDocClassName = moveDocClassItem.getValue();
          var executeAddActions = moveDocAddActionItem.getValue();

          if (sChosenDocClassName == null || sChosenDocClassName == '') {
            Ext.Msg.show({
              title: CUF.t('warning'),
              msg: CUF.t('docClass-choose'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.WARNING,
            });
            return;
          }

          var moveDocServlet = 'CUF.MoveDocument.customServlet';
          var copyIndexes = 'false';
          if (
            moveDocCopyIndexes != null &&
            moveDocCopyIndexes != undefined &&
            (moveDocCopyIndexes == true || moveDocCopyIndexes == 'true')
          ) {
            copyIndexes = 'true';
          }

          var moved = 0;

          for (var i = 0; i < documentsToMove.length; i++) {
            var fileId = documentsToMove[i];
            var moveDocArgs =
              'fileId=' +
              fileId +
              '&destDocClassName=' +
              sChosenDocClassName +
              '&copyIndexes=' +
              copyIndexes +
              '&executeAddActions=' +
              executeAddActions;

            var moveDocResponse = CUF_executeServletGetResponse(moveDocServlet, moveDocArgs);
            var moveDocResponseTxt = moveDocResponse.responseText;
            var moveDocJsonData = Ext.util.JSON.decode(moveDocResponseTxt);
            var moveDocSuccess = moveDocJsonData.success;

            if (!moveDocSuccess) {
              continue;
            }

            moved++;

            try {
              window[funcNameAfterDocAction](processId, activityId, 'move', sChosenDocClassName, fileId);
            } catch (e) {}
          }

          moveDocWindow.close();

          Ext.Msg.show({
            title: CUF.t('stats'),
            msg: CUF.t('document-move-success-p1') + moved + CUF.t('document-move-success-p2') + sChosenDocClassName + '.',
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
        },
      });

      moveDocForm.addButton(moveDocActionButton);

      moveDocWindow = new Ext.Window({
        width: 410,
        modal: true,
        constrain: true,
        shadow: false,
        resizable: false,
        title: CUF.t('docClass-selection'),
        iconCls: 'x-CUFwindow-movedoc',
        items: moveDocForm,
      });

      moveDocWindow.show();
    },
  });
}

// odlaczanie dokumentu od procesu
function CUF_detachDocumentFromProcess(processId, activityId, fileId, checkRights) {
  try {
    var detachDocServlet = 'CUF.DetachDocumentFromProcess.customServlet';
    var detachDocArgs = 'sProcessId=' + processId + '&sActivityId=' + activityId + '&sFileId=' + fileId + '&checkRights=' + checkRights;
    var detachResponse = CUF_executeServletGetResponse(detachDocServlet, detachDocArgs);
    var detachResponseTxt = detachResponse.responseText;
    var detachResponseJsonData = Ext.util.JSON.decode(detachResponseTxt);
    CUF.DocumentView.reloadDocumentsGrid();
    return detachResponseJsonData.success;
  } catch (e) {
    return false;
  }
}

function CUF_reloadDocumentsTab() {
  CUF_depracated(arguments, 'CUF_reloadDocumentsGrid');
  CUF.DocumentView.reloadDocumentsGrid();
}

// usuwanie dokumentu z archiwum
function CUF_deleteDocumentFromArchive(fileId, checkRights) {
  var deleteDocServlet = 'CUF.DeleteDocument.customServlet';
  var deleteDocArgs = 'fileId=' + fileId + '&checkRights=' + checkRights;
  CUF_executeServletGetResponse(deleteDocServlet, deleteDocArgs);
}

// okno z indeksami dokumentu
function CUF_showDocIndexes(fileId, enableUpdate) {
  CUF_importCssForFrame('processdocuments');
  var indexesWindow = null;

  var showIndexesServlet = 'CUF.GetDocumentIndexesInfo.customServlet';
  var showIndexesArgs = 'fileId=' + fileId;

  var showIndexesResponse = CUF_executeServletGetResponse(showIndexesServlet, showIndexesArgs);
  var showIndexesResponseTxt = showIndexesResponse.responseText;
  var showIndexesJsonData = Ext.util.JSON.decode(showIndexesResponseTxt);
  var showIndexesSuccess = showIndexesJsonData.success;

  if (!showIndexesSuccess) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('indexes-error-info-loading'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.ERROR,
    });
    return;
  }

  var showIndexesRecords = showIndexesJsonData.records;

  var labelWidth = 450;
  var indexesExist = false;

  if (showIndexesRecords.length == 0) {
    indexesExist = false;
  } else {
    indexesExist = true;
  }

  var indexesForm = new Ext.FormPanel({
    id: 'CUF_indexes_panel',
    autoScroll: true,
    frame: true,
    layout: 'form',
    labelWidth: labelWidth,
    labelAlign: 'top',
  });

  if (!indexesExist) {
    var labelItem = new Ext.form.Label({
      id: 'CUF_index_label_item',
      fieldLabel: CUF.t('document-no-indexes'),
      labelSeparator: '',
    });

    indexesForm.add(labelItem);
  } else {
    var readOnlyIndex = true;

    if (enableUpdate) {
      readOnlyIndex = false;
    }

    for (var j = 0; j < showIndexesRecords.length; j++) {
      var indexName = showIndexesRecords[j].name;
      var indexType = showIndexesRecords[j].type;
      var indexValue = showIndexesRecords[j].value;

      if (indexType == 'date') {
        indexItem = new Ext.form.DateField({
          id: 'index' + (j + 1),
          fieldLabel: indexName,
          format: 'Y-m-d',
          value: indexValue,
          width: 450,
          readOnly: readOnlyIndex,
          labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
        });
      } else if (indexType == 'integer') {
        indexItem = new Ext.form.NumberField({
          id: 'index' + (j + 1),
          fieldLabel: indexName,
          allowDecimals: false,
          value: indexValue,
          width: 450,
          readOnly: readOnlyIndex,
          labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
        });
      } else if (indexType == 'float') {
        indexItem = new Ext.form.NumberField({
          id: 'index' + (j + 1),
          fieldLabel: indexName,
          allowDecimals: true,
          decimalSeparator: ',',
          value: indexValue,
          width: 450,
          readOnly: readOnlyIndex,
          labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
        });
      } else if (indexType == 'list') {
        var listValues = showIndexesRecords[j].listValues;

        if (listValues != null && listValues != undefined && listValues != '') {
          var splitValues = listValues.split(',');

          var store = new Ext.data.SimpleStore();
          var rows = Ext.data.Record.create([
            {
              name: 'value',
              type: 'string',
            },
          ]);

          for (var i = 0; i < splitValues.size(); i++) {
            var record = new rows({
              value: splitValues[i],
            });

            store.add(record);
            store.commitChanges();
          }

          indexItem = new Ext.form.ComboBox({
            id: 'index' + (j + 1),
            displayField: 'value',
            valueField: 'value',
            store: store,
            triggerAction: 'all',
            fieldLabel: indexName,
            value: indexValue,
            mode: 'local',
            forceSelection: true,
            width: 450,
            readOnly: readOnlyIndex,
            labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
          });
        } else {
          indexItem = new Ext.form.TextField({
            id: 'index' + (j + 1),
            inputType: 'text',
            fieldLabel: indexName,
            value: indexValue,
            width: 450,
            readOnly: readOnlyIndex,
            labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
          });
        }
      } else {
        indexItem = new Ext.form.TextField({
          id: 'index' + (j + 1),
          inputType: 'text',
          fieldLabel: indexName,
          value: indexValue,
          width: 450,
          readOnly: readOnlyIndex,
          labelStyle: 'margin-top:0.25em; margin-bottom:0.25em;',
        });
      }

      indexesForm.add(indexItem);
    }
  }

  var hideModifyButton = true;

  if (indexesExist == true && enableUpdate) {
    hideModifyButton = false;
  }

  var modifyButton = new Ext.Button({
    id: 'CUF_modify_indexes_button',
    text: CUF.t('update'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/page_refresh.png',
    hidden: hideModifyButton,
    handler: function () {
      Ext.Msg.show({
        title: CUF.t('indexes-updating'),
        msg: CUF.t('document-replace-indexes'),
        buttons: Ext.Msg.YESNO,
        fn: function (buttonId) {
          if (buttonId != 'yes') {
            return;
          }

          var hmMapping = '';
          var counter = 1;
          var indexItem = Ext.get('index' + counter);

          while (indexItem != null && indexItem != undefined) {
            var label = Ext.DomQuery.select(String.format('label[for="{0}"]', 'index' + counter));
            label = label[0].innerHTML;
            label = label.substring(0, label.length - 1);
            hmMapping = hmMapping + label + '=' + indexItem.getValue() + ';';
            counter++;
            indexItem = Ext.get('index' + counter);
          }

          if (hmMapping.length > 0) {
            hmMapping = hmMapping.substring(0, hmMapping.length - 1);
          }

          indexesWindow.close();

          var updateIndexesServlet = 'CUF.UpdateDocumentIndexes.customServlet';
          var updateIndexesArgs = 'fileId=' + fileId + '&hmMapping=' + hmMapping;

          var updateIndexesResponse = CUF_executeServletGetResponse(updateIndexesServlet, updateIndexesArgs);
          var updateIndexesResponseTxt = updateIndexesResponse.responseText;
          var updateIndexesJsonData = Ext.util.JSON.decode(updateIndexesResponseTxt);
          var updateIndexesSuccess = updateIndexesJsonData.success;

          if (!updateIndexesSuccess) {
            Ext.Msg.show({
              title: CUF.t('error'),
              msg: CUF.t('document-error-update-indexes'),
              buttons: Ext.Msg.OK,
              icon: Ext.Msg.ERROR,
            });
          }
        },
        icon: Ext.MessageBox.QUESTION,
      });
    },
  });

  indexesForm.addButton(modifyButton);

  var closeButton = new Ext.Button({
    id: 'CUF_close_indexes_panel_button',
    text: CUF.t('close'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/cross.png',
    handler: function () {
      indexesWindow.close();
    },
  });

  indexesForm.addButton(closeButton);

  indexesWindow = new Ext.Window({
    id: 'CUF_show_indexes_window',
    width: 500,
    autoHeight: true,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    title: CUF.t('document-indexes'),
    iconCls: 'x-CUFwindow-indexes',
    items: indexesForm,
    calculateHeights: function () {
      var viewportHeight = Ext.getBody().getViewSize().height;
      var windowHeight = this.getOuterSize().height + this.getFrameHeight();

      if (windowHeight > viewportHeight) {
        indexesForm.setHeight(viewportHeight - 50);
      }
    },
    disableIndexes: function (indexesNames) {
      this.investigateIndexes(indexesNames, true);
    },
    enableIndexes: function (indexesNames) {
      this.investigateIndexes(indexesNames, false);

      if (modifyButton.hidden) {
        modifyButton.show();
      }
    },
    investigateIndexes: function (indexesNames, readOnly) {
      if (!indexesNames || indexesNames.length == 0) {
        return;
      }

      var counter = 1;
      var label = Ext.DomQuery.select(String.format('label[for="{0}"]', 'index' + counter));

      while (label && label.length > 0) {
        label = label[0].innerHTML;
        label = label.substring(0, label.length - 1);

        if (indexesNames.indexOf(label) != -1) {
          Ext.getCmp('index' + counter).setReadOnly(readOnly);
        }

        counter++;
        label = Ext.DomQuery.select(String.format('label[for="{0}"]', 'index' + counter));
      }
    },
  });

  indexesWindow.show();

  try {
    indexesWindow.calculateHeights();
    var taskPanel = formPanel.items.first();
    var pageYOffset = 0;

    if (Ext.isIE && Ext.isStrict) {
      pageYOffset = document.documentElement.scrollTop;
    } else {
      pageYOffset = window.pageYOffset;
    }

    var windowYOffset = indexesWindow.getPosition()[1] - pageYOffset;

    if (taskPanel) {
      CUF_adjustWinPosition(data_window);
    } else {
      data_window.setPosition(indexesWindow.getPosition()[0], indexesWindow.getPosition()[1] - windowYOffset);
    }
  } catch (e) {}

  return indexesWindow;
}

// wykonanie dodatkowej funkcji dla dokumentu
function CUF_executeExtraMethodForDocs(processId, activityId, fileIds, extraFuncClass) {
  var extraFuncServlet = 'CUF.ExecuteExtraMethodForDocs.customServlet';
  var extraFuncArgs = 'processId=' + processId + '&activityId=' + activityId + '&fileIds=' + fileIds + '&extraFuncClass=' + extraFuncClass;

  CUF_executeServletGetResponse(extraFuncServlet, extraFuncArgs);
}

// deaktywacja przyciskow po odlaczeniu/usunieciu wszystkich dokumentow
function CUF_deactivateButtons(buttonIdsArray) {
  for (var i = 0; i < buttonIdsArray.length; i++) {
    var buttonId = buttonIdsArray[i];
    var button = Ext.getCmp(buttonId);
    if (button) {
      button.hide();
    }
  }
}

// ////////////wyswietlenie dokumentu z archiwum ////////////////
function CUF_displayDocument(fileId) {
  var url = CUF_getUrl();

  if (!fileId || fileId == '') {
    Ext.Msg.show({
      title: CUF.t('error'),
      msg: CUF.t('file-error-no-id'),
      buttons: Ext.Msg.OK,
      icon: Ext.MessageBox.ERROR,
    });
    return;
  }

  if (url != '') {
    url += '/ShowFile.do?fileName=' + CUF_base64.encode(fileId);
    window.open(url, 'Display');
  }
}

// //////////// pobieranie dokumentu z archiwum ////////////////
function CUF_downloadDocument(fileId) {
  var url = CUF_getUrl();

  if (!fileId || fileId == '') {
    Ext.Msg.show({
      title: CUF.t('error'),
      msg: CUF.t('file-error-no-id'),
      buttons: Ext.Msg.YES,
      icon: Ext.MessageBox.ERROR,
    });
    return;
  }

  if (url != '') {
    url += '/CUF.DownloadDocument.customServlet?fileId=' + fileId;
    window.open(url, 'Download');
  }
}

// //////////// dodawanie komentarza dla dokumentu ///////////////
function CUF_saveCommentForDoc(fileId) {
  var askBeforeAddComment = false;
  var currentDocComment = '';
  var getDocCommentServlet = 'CUF.GetDocumentComment.customServlet';
  var getDocCommentArgs = 'fileId=' + fileId;

  var docCommentResponse = CUF_executeServletGetResponse(getDocCommentServlet, getDocCommentArgs);
  var docCommentResponseTxt = docCommentResponse.responseText;
  var docCommentJsonData = Ext.util.JSON.decode(docCommentResponseTxt);
  var docCommentSuccess = docCommentJsonData.success;

  if (docCommentSuccess) {
    currentDocComment = docCommentJsonData.message;

    if (currentDocComment != null && currentDocComment != undefined && currentDocComment != '') {
      askBeforeAddComment = true;
    }
  }

  if (askBeforeAddComment) {
    Ext.Msg.show({
      title: CUF.t('comment-adding'),
      msg: CUF.t('document-replace-comment'),
      buttons: Ext.Msg.YESNO,
      fn: function (buttonId) {
        if (buttonId == 'yes') {
          CUF_executeSaveCommentForDoc(fileId, currentDocComment);
        }
      },
      icon: Ext.MessageBox.QUESTION,
    });
  } else {
    CUF_executeSaveCommentForDoc(fileId, '');
  }
}

function CUF_executeSaveCommentForDoc(fileId, currentDocComment) {
  CUF_importCssForFrame('processdocuments');
  var data_window = null;
  var width = 400;

  var commentItem = new Ext.form.TextArea({
    id: 'CUF_doc_comment_item',
    inputType: 'text',
    fieldLabel: CUF.t('comment'),
    width: width - 30,
    value: currentDocComment,
    maxLength: 255,
    maxLengthText: CUF.t('comment-error-length'),
  });

  var form = new Ext.FormPanel({
    id: 'CUF_doc_comment_panel',
    width: width + 10,
    frame: true,
    bodyStyle: 'padding: 10px 10px 0 10px;',
    labelAlign: 'top',
    items: [commentItem],
  });

  var addCommentButton = new Ext.Button({
    id: 'CUF_accept_add_comment_button',
    text: CUF.t('add'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/comment_add.png',
    handler: function () {
      var comment = commentItem.getValue();

      if (comment.length > 255) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('comment-error-length'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      data_window.close();

      var saveCommentServlet = 'CUF.SaveDocumentComment.customServlet';
      var saveCommentArgs = 'fileId=' + fileId + '&comment=' + comment;

      var addCommentResponse = CUF_executeServletGetResponse(saveCommentServlet, saveCommentArgs);
      var addCommentResponseTxt = addCommentResponse.responseText;
      var addCommentJsonData = Ext.util.JSON.decode(addCommentResponseTxt);
      var addCommentSuccess = addCommentJsonData.success;

      if (!addCommentSuccess) {
        Ext.Msg.show({
          title: CUF.t('error'),
          msg: CUF.t('document-error-comment-add'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.ERROR,
        });
        return;
      }

      Ext.Msg.show({
        title: CUF.t('comment-added'),
        msg: CUF.t('document-comment-added'),
        buttons: Ext.Msg.OK,
        icon: Ext.Msg.INFO,
      });
    },
  });

  form.addButton(addCommentButton);

  data_window = new Ext.Window({
    id: 'CUF_add_comment_window',
    width: width + 20,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    title: CUF.t('comment-add'),
    iconCls: 'x-CUFwindow-adddoccomment',
    items: form,
  });

  data_window.show();
}

// //////// zmiana opisu dokumentu //////////////////////////
function CUF_changeDescriptionForDoc(fileId, record) {
  CUF_importCssForFrame('processdocuments');
  var data_window = null;
  var width = 400;
  var labelWidth = 120;

  var getDocumentInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
  var getDocumentInfoArgs = 'fileIds=' + fileId;
  var DocInfoResponse = CUF_executeServletGetResponse(getDocumentInfoServlet, getDocumentInfoArgs);
  var DocInfoResponseTxt = DocInfoResponse.responseText;
  var DocInfoJsonData = Ext.util.JSON.decode(DocInfoResponseTxt);
  var DocInfoSuccess = DocInfoJsonData.success;

  if (!DocInfoSuccess) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: DocInfoJsonData.message,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.ERROR,
    });
    return;
  }

  var DocInfoRecord = DocInfoJsonData.records[0];
  var descr = DocInfoRecord.description;
  if (descr == null) {
    descr = '';
  }

  var descriptionItem = new Ext.form.TextField({
    id: 'CUF_description',
    inputType: 'text',
    fieldLabel: CUF.t('document-description'),
    width: width - 30,
    value: descr,
    maxLength: 255,
    maxLengthText: CUF.t('document-error-description-length'),
  });

  var form = new Ext.FormPanel({
    id: 'CUF_change_descr_panel',
    width: width + 10,
    frame: true,
    bodyStyle: 'padding: 10px 10px 0 10px;',
    labelWidth: labelWidth,
    labelAlign: 'top',
    items: [descriptionItem],
  });

  var changeDescriptionButton = new Ext.Button({
    text: CUF.t('change'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/arrow_refresh.png',
    handler: function () {
      var description = descriptionItem.getValue();

      if (description.length > 255) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('document-error-description-length'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }

      var changeDescriptionServlet = 'CUF.ChangeDocumentDescription.customServlet';
      var changeDescriptionArgs = 'fileId=' + fileId + '&description=' + description;

      var changeDescriptionResponse = CUF_executeServletGetResponse(changeDescriptionServlet, changeDescriptionArgs);
      var changeDescriptionResponseTxt = changeDescriptionResponse.responseText;
      var changeDescriptionJsonData = Ext.util.JSON.decode(changeDescriptionResponseTxt);
      var changeDescriptionSuccess = changeDescriptionJsonData.success;

      data_window.close();

      if (!changeDescriptionSuccess) {
        Ext.Msg.show({
          title: CUF.t('error'),
          msg: CUF.t('document-error-description-update'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.ERROR,
        });
        return;
      }

      if (record) {
        record.beginEdit();
        record.set('description', description);
        record.commit();
        record.endEdit();
      }

      Ext.Msg.show({
        title: CUF.t('description-updated'),
        msg: CUF.t('document-comment-updated'),
        buttons: Ext.Msg.OK,
        icon: Ext.Msg.INFO,
      });

      if (!(formPanel.layout || formPanel.layout.east)) {
        return;
      }

      var docViewPanel = formPanel.layout.east.panel;
      if (docViewPanel == null || docViewPanel == undefined) {
        return;
      }

      if (!docViewPanel.hidden) {
        var docViewTabPanel = docViewPanel.items.first();
        var tabs = docViewTabPanel.find('fileId', fileId);

        if (tabs.length > 0) {
          var tab = tabs[0];

          if (description.length > 13) {
            description = description.substring(0, 10) + '...';
          } else if (description.length == 0) {
            description = 'Doc';
          }

          tab.setTitle(description);
        }
      }
    },
  });

  form.addButton(changeDescriptionButton);

  data_window = new Ext.Window({
    id: 'CUF_change_descr_window',
    width: width + 20,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    title: CUF.t('description-change'),
    iconCls: 'x-CUFwindow-changedescr',
    items: form,
  });

  data_window.show();
}

// //////////// pobierane obecnej lokacji ///////////////////
function CUF_getUrl() {
  var url = [document.location.protocol, '//', document.location.host, document.location.pathname].join('');
  var lastSymbol = url.lastIndexOf('/');
  if (lastSymbol != 0 && lastSymbol != -1) {
    url = url.substring(0, lastSymbol);
  }

  return url;
}

// //////// odswiezanie zadania z mozliwoscia zapisania formularza //////
function CUF_refreshActivity(activityId, processId, save) {
  if (save) {
    CUF_saveAllVariables();
  }

  var link = CUF_getUrl();
  link = link + '/ProcesActivity.do?activityId=' + activityId + '&processKey=' + processId;

  document.location.href = link;
}

// ///// zarzadzanie wyswtetlaniem dokumentow w procesie
// ///// dodawanie zakladki
function CUF_addDocumentsToView(fileIds) {
  try {
    var processId = document.getElementsByName('processId')[0].value;
    var activityId = document.getElementsByName('activityId')[0].value;

    if (!(formPanel.layout || formPanel.layout.east)) {
      return;
    }

    var docViewPanel = formPanel.layout.east.panel;

    if (!docViewPanel) {
      return;
    }

    var docViewTabPanel = docViewPanel.items.first();
    var currentTabDocs = new Array();

    Ext.each(docViewTabPanel.items.items, function (item, index, allItems) {
      currentTabDocs.push(item.fileId);
    });

    var verifyDocViewServlet = 'CUF.VerifyDocumentView.customServlet';
    var verifyDocViewArgs =
      'userId=' +
      sTaskUserName +
      '&processId=' +
      processId +
      '&activityId=' +
      activityId +
      '&oldDocIds=' +
      currentTabDocs.join(',') +
      '&newDocIds=' +
      fileIds;

    var docViewResponse = CUF_executeServletGetResponse(verifyDocViewServlet, verifyDocViewArgs);
    var docViewResponseTxt = docViewResponse.responseText;
    var docViewJsonData = Ext.util.JSON.decode(docViewResponseTxt);
    var docViewSuccess = docViewJsonData.success;

    if (!docViewSuccess) {
      return;
    }

    fileIds = docViewJsonData.addFiles;
    var toRemoveFileIds = docViewJsonData.removeFiles;

    if (fileIds == '') {
      return;
    }

    if (docViewPanel.hidden) {
      docViewPanel.show();
      CUF_refreshTaskPanel();
    }

    var lastAddedTab = '';

    var activityFormPanelFiles = formPanel.files;
    var frameId = activityFormPanelFiles.length;

    var height = Ext.getBody().getHeight() + 'px';
    var url = CUF_getUrl();

    var getDocumentsInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
    var getDocumentsInfoArgs = 'fileIds=' + fileIds;
    var docInfoResponse = CUF_executeServletGetResponse(getDocumentsInfoServlet, getDocumentsInfoArgs);
    var docInfoResponseTxt = docInfoResponse.responseText;
    var docInfoJsonData = Ext.util.JSON.decode(docInfoResponseTxt);
    var docInfoSuccess = docInfoJsonData.success;

    if (!docInfoSuccess) {
      return;
    }

    var docInfoRecords = docInfoJsonData.records;

    for (var i = 0; i < docInfoRecords.length; i++) {
      var docInfo = docInfoRecords[i];
      var fileId = docInfo.fileId;

      var checkTab = docViewTabPanel.find('fileId', fileId);

      if (checkTab.length > 0) {
        lastAddedTab = checkTab[0].id;
        continue;
      }

      var file = new Object();
      var fileDescr = docInfo.description;

      if (fileDescr.length > 13) {
        fileDescr = fileDescr.substring(0, 10) + '...';
      }

      file.description = fileDescr;
      file.id = fileId;
      file.name = docInfo.name;
      file.org_name = docInfo.orgName;
      file.link = url + '/com.plusmpm.servlet.ShowFileServlet.customServlet?fileId=' + fileId;

      if (Ext.isGecko3) {
        file.html =
          '<object id="iframe_id_' +
          frameId +
          '" width="100%" height="' +
          height +
          '" scrolling="auto" marginwidth="0" marginheight="0"><p>Brak obsługi object.</p></object>';
      } else {
        file.html =
          '<iframe id="iframe_id_' +
          frameId +
          '" width="100%" height="' +
          height +
          '" scrolling="auto" marginwidth="0" marginheight="0"><p>Brak obsługi iframe.</p></iframe>';
      }

      docViewTabPanel.add({
        id: 'tab_id_' + frameId,
        title: file.description,
        html: file.html,
        tabTip: CUF.t('file-name') + ': ' + file.org_name,
        closable: true,
        cls: 'x-panel-body-PlusMPM',
        unstyled: true,
        fileId: file.id,
      });

      formPanel.files.push(file);
      lastAddedTab = 'tab_id_' + frameId;
      frameId++;
    }

    if (toRemoveFileIds != '') {
      var filesToRemoveTab = toRemoveFileIds.split(',');

      for (var i = 0; i < filesToRemoveTab.length; i++) {
        try {
          docViewPanel.deleteDocument(filesToRemoveTab[i]);
        } catch (e) {}
      }
    }

    var activeTab = docViewTabPanel.getActiveTab();

    if (activeTab != null && activeTab.id != lastAddedTab) {
      docViewTabPanel.setActiveTab(lastAddedTab);
    } else if (activeTab == null) {
      docViewTabPanel.setActiveTab(lastAddedTab);
    }
  } catch (e) {}
}

// usuwanie zakladki podgladu dokumentu
function CUF_deleteDocumentFromView(fileId) {
  try {
    if (!fileId || fileId == '') {
      return;
    }

    var activityFormPanel = formPanel;
    if (!activityFormPanel.layout.east) {
      return;
    }

    var docViewPanel = activityFormPanel.layout.east.panel;
    if (!docViewPanel) {
      return;
    }

    try {
      docViewPanel.deleteDocument(fileId);
    } catch (e) {}

    var docViewTabPanel = docViewPanel.items.first();
    if (docViewTabPanel.items.length == 0) {
      docViewPanel.hide();
      CUF_refreshTaskPanel();
    }
  } catch (e) {}
}

// odswiezenie widoku zadania (tabel dynamicznych)
function CUF_refreshTaskPanel() {
  formPanel.doLayout();

  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_refreshGridColumnHeaders(table);
    } catch (e) {}
  });
}

// dodawanie miniatury dokumentu do rekordu w gridzie
function CUF_showDocToolTip(grid, config) {
  var url = CUF_getUrl();
  var gridView = grid.getView();
  var store = grid.getStore();
  var width = 0;
  var height = 200;
  var paging = false;
  var fileIdVar = 'fileId';

  if (config) {
    height = config.docToolTipHeight;
    paging = config.docToolTipPaging;
    fileIdVar = config.fileIdVar;

    if (!height) {
      height = 200;
    }

    if (!fileIdVar) {
      fileIdVar = 'fileId';
    }
  }

  width = Math.ceil(0.707 * height);

  grid.tip = new Ext.ToolTip({
    id: 'CUF_show_doc_tooltip_' + Ext.id(),
    target: gridView.mainBody,
    delegate: '.x-grid3-row',
    trackMouse: true,
    renderTo: config.renderToolTipTo,
    dismissDelay: 0,
    width: width + 12,
    height: height + 12,
    hideBorders: true,
    shadow: false,
    listeners: {
      beforeshow: function (tip) {
        var onRowIndex = gridView.findRowIndex(tip.triggerElement);

        var fileId = store.getAt(onRowIndex).get(fileIdVar);

        if (!fileId) {
          return;
        }

        tip.pageNum = 1;
        tip.fileId = fileId;

        if (Ext.isGecko3) {
          tip.body.dom.innerHTML =
            '<object id="iframe_id_' +
            fileId +
            '" width="' +
            width +
            'px" height="' +
            tempHeight +
            'px" scrolling="no" marginwidth="0" marginheight="0" data="' +
            url +
            '/CUF.ShowDocToolTip.customServlet?fileId=' +
            fileId +
            '&pageNum=' +
            tip.pageNum +
            '&width=' +
            width +
            '&height=' +
            height +
            '"><p>Brak obsługi object.</p></object>';
        } else {
          tip.body.dom.innerHTML =
            '<iframe id="iframe_id_' +
            fileId +
            '" width="' +
            width +
            'px" height="' +
            height +
            'px" scrolling="no" frameborder="0" marginwidth="0" marginheight="0" src="' +
            url +
            '/CUF.ShowDocToolTip.customServlet?fileId=' +
            fileId +
            '&pageNum=' +
            tip.pageNum +
            '&width=' +
            width +
            '&height=' +
            height +
            '"><p>Brak obsługi iframe.</p></iframe>';
        }

        tip.setSize({
          width: 'auto',
          height: 'auto',
        });

        tip.setPosition(tip.pageX, tip.pageY);
      },
      move: function (tip, x, y) {
        var newX = x;
        var maxWidth = screen.width;
        var changePos = false;

        if (x + tip.getWidth() + 20 > maxWidth) {
          changePos = true;
          newX = maxWidth - tip.getWidth() - 20;
        }

        if (changePos) {
          tip.setPosition(newX, y);
        }
      },
      beforerender: function (container, component, index) {
        if (!paging) {
          return;
        }

        Ext.EventManager.addListener(document, 'keyup', function (e) {
          var key = e.getKey();

          if (key == '107' || key == '109') {
            var tip = grid.tip;
            var currentPageNum = tip.pageNum;

            if (key == '107') {
              currentPageNum++;
            } else if (key == '109') {
              currentPageNum--;

              if (currentPageNum < 1) {
                currentPageNum = 1;
              }
            }

            tip.pageNum = currentPageNum;
            var fileId = tip.fileId;

            if (!fileId || fileId == -1) {
              return;
            }

            if (Ext.isGecko3) {
              tip.body.dom.innerHTML =
                '<object id="iframe_id_' +
                fileId +
                '" width="' +
                width +
                'px" height="' +
                height +
                'px" scrolling="no" marginwidth="0" marginheight="0" data="' +
                url +
                '/CUF.ShowDocToolTip.customServlet?fileId=' +
                fileId +
                '&pageNum=' +
                tip.pageNum +
                '&width=' +
                width +
                '&height=' +
                height +
                '"><p>Brak obsługi object.</p></object>';
            } else {
              tip.body.dom.innerHTML =
                '<iframe id="iframe_id_' +
                fileId +
                '" width="' +
                width +
                'px" height="' +
                height +
                'px" scrolling="no" frameborder="0" marginwidth="0" marginheight="0" src="' +
                url +
                '/CUF.ShowDocToolTip.customServlet?fileId=' +
                fileId +
                '&pageNum=' +
                tip.pageNum +
                '&width=' +
                width +
                '&height=' +
                height +
                '"><p>Brak obsługi iframe.</p></iframe>';
            }
          }
        });
      },
    },
  });
}

// dodanie akcji po dwukliku na wiersz w tabeli dynamicznej //
function CUF_handleRowDblClick(grid, config) {
  try {
    if (!(grid && config)) {
      return;
    }

    var downloadDocOnDblClick = config.downloadDocOnDblClick;
    var showDocOnDblClick = config.showDocOnDblClick;
    var fileIdVar = config.fileIdVar;

    if (!fileIdVar) {
      fileIdVar = 'fileId';
    }

    if (downloadDocOnDblClick || showDocOnDblClick) {
      grid.on('rowdblclick', function (grid, rowIndex, e) {
        var store = grid.getStore();

        if (downloadDocOnDblClick) {
          var fileId = store.getAt(rowIndex).get(fileIdVar);
          CUF_downloadDocument(fileId);
        } else if (showDocOnDblClick) {
          var fileId = store.getAt(rowIndex).get(fileIdVar);
          CUF_displayDocument(fileId);
        }
      });
    }
  } catch (e) {}
}

// dodanie akcji po dwukliku na komorke w tabeli dynamicznej //
function CUF_handleCellDblClick(grid, config) {
  try {
    if (!(grid && config)) {
      return;
    }

    var downloadDocOnDblClick = config.downloadDocOnDblClick;
    var showDocOnDblClick = config.showDocOnDblClick;
    var fileIdVar = config.fileIdVar;
    var callDblClickActionOn = config.callDblClickActionOn;

    if (!fileIdVar) {
      fileIdVar = 'fileId';
    }

    if (downloadDocOnDblClick || showDocOnDblClick) {
      grid.on('beforeedit', function (object) {
        try {
          var callAction = false;
          var store = grid.getStore();

          if (callDblClickActionOn) {
            var clickedKeys = callDblClickActionOn.split(';');
            var columnModel = grid.getColumnModel();
            var clickedDataIdx = columnModel.getDataIndex(object.column);

            var idx = clickedKeys.indexOf(clickedDataIdx);

            if (idx != -1) {
              object.cancel = true;
              callAction = true;
            }
          } else {
            callAction = true;
          }

          if (callAction && downloadDocOnDblClick) {
            var fileId = store.getAt(object.row).get(fileIdVar);
            CUF_downloadDocument(fileId);
          } else if (callAction && showDocOnDblClick) {
            var fileId = store.getAt(object.row).get(fileIdVar);
            CUF_displayDocument(fileId);
          }
        } catch (e) {}
      });
    }
  } catch (e) {}
}

// wczytanie dokumentow podpietych do procesu do tabeli dynamicznej //
function CUF_loadProcessDocsToDT(config) {
  window.setTimeout(function () {
    CUF_executeLoadProcessDocsToDT(config);
  }, 0);
}

function CUF_executeLoadProcessDocsToDT(config) {
  try {
    var gridNumber = config.gridNumber;

    if (!gridNumber) {
      gridNumber = 1;
    }

    var grid = CUF_getActivityGrid(gridNumber);
    var store = grid.getStore();

    var DTIndexes = config.DTIndexes;
    var callDblClickActionOn = config.callDblClickActionOn;
    var fileIdVar = config.fileIdVar;
    var showRowContextMenu = config.showRowContextMenu;
    var contextMenuDef = config.contextMenuDef;

    var processId = document.getElementsByName('processId')[0].value;
    var activityId = document.getElementsByName('activityId')[0].value;

    if (callDblClickActionOn) {
      CUF_handleCellDblClick(grid, config);
    } else {
      CUF_handleRowDblClick(grid, config);
    }

    if (config.showDocToolTip) {
      Ext.apply(config, {
        renderToolTipTo: formPanel.id,
      });

      CUF_showDocToolTip(grid, config);
    }

    if (showRowContextMenu && contextMenuDef && contextMenuDef.length > 0) {
      var ctxMenu = new Ext.menu.Menu({
        id: 'CUF_context_menu_' + Ext.id(),
        renderTo: formPanel.id,
        docView: false,
      });

      CUF_createContextMenu(ctxMenu, contextMenuDef);

      grid.on('rowcontextmenu', function (thisGrid, rowIndex, e) {
        e.stopEvent();
        var clickPos = e.getXY();
        var thisStore = thisGrid.getStore();
        e.rowIndex = rowIndex;
        e.tab = undefined;
        e.store = thisStore;
        e.fileId = thisStore.getAt(rowIndex).get(fileIdVar);
        e.clickPos = clickPos;
        ctxMenu.showAt(clickPos);
      });
    }

    if (!DTIndexes) {
      return;
    }

    var attachedDocIdsServlet = 'CUF.GetDocumentsAttachedToProcess.customServlet';
    var attachedDocIdsArgs = 'sProcessId=' + processId + '&sActivityId=' + activityId;

    var DocIdsResponse = CUF_executeServletGetResponse(attachedDocIdsServlet, attachedDocIdsArgs);
    var DocIdsResponseTxt = DocIdsResponse.responseText;
    var DocIdsJsonData = Ext.util.JSON.decode(DocIdsResponseTxt);
    var DocIdsSuccess = DocIdsJsonData.success;

    if (!DocIdsSuccess) {
      return;
    }

    var docIdsForProcess = DocIdsJsonData.message;

    if (docIdsForProcess == '') {
      try {
        store.autoSave = false;
        store.removeAll();
      } catch (e) {
      } finally {
        store.autoSave = true;
      }

      return;
    }

    var getDocumentsInfoServlet = 'CUF.GetDocumentsInfo.customServlet';
    var getDocumentsInfoArgs = 'fileIds=' + docIdsForProcess;
    var DocInfoResponse = CUF_executeServletGetResponse(getDocumentsInfoServlet, getDocumentsInfoArgs);
    var DocInfoResponseTxt = DocInfoResponse.responseText;
    var DocInfoJsonData = Ext.util.JSON.decode(DocInfoResponseTxt);
    var DocInfoSuccess = DocInfoJsonData.success;

    if (!DocInfoSuccess) {
      return;
    }

    var DocInfoRecords = DocInfoJsonData.records;
    var showIndexesServlet = 'CUF.GetDocumentIndexesInfo.customServlet';
    var docIdsTab = docIdsForProcess.split(',');

    if (store.getCount() > 0) {
      var recordsToRemove = new Array();

      store.each(function (record) {
        if (docIdsTab.indexOf(record.get(fileIdVar)) == -1) {
          recordsToRemove.push(record);
        }
      });

      try {
        store.autoSave = false;

        store.remove(recordsToRemove);
        store.save();
      } catch (e) {
      } finally {
        store.autoSave = true;
      }
    }

    var recordsToAdd = new Array();

    for (var i = 0; i < DocInfoRecords.length; i++) {
      var newRecord = new store.recordType({});
      var DocInfoRecord = DocInfoRecords[i];
      var fileId = DocInfoRecord.fileId;

      if (store.find(fileIdVar, fileId) != -1) {
        continue;
      }

      var tempIndexes = new Array();

      var showIndexesArgs = 'fileId=' + fileId;
      var showIndexesResponse = CUF_executeServletGetResponse(showIndexesServlet, showIndexesArgs);
      var showIndexesResponseTxt = showIndexesResponse.responseText;
      var showIndexesJsonData = Ext.util.JSON.decode(showIndexesResponseTxt);
      var showIndexesSuccess = showIndexesJsonData.success;

      if (showIndexesSuccess) {
        tempIndexes = showIndexesJsonData.records;
      }

      for (var idx in DTIndexes) {
        if (DTIndexes.hasOwnProperty(idx)) {
          var tempIndexName = DTIndexes[idx];
          var tempIndexValue = '';
          var field = newRecord.fields.get(idx);
          var fieldType = field.type.type;

          if (tempIndexName == 'fileId') {
            tempIndexValue = fileId;
          } else if (tempIndexName == 'processId') {
            tempIndexName = processId;
          } else if (tempIndexName == 'activityId') {
            tempIndexName = activityId;
          } else if (tempIndexName == 'orgFileName') {
            tempIndexValue = DocInfoRecord.orgName;
          } else if (tempIndexName == 'description') {
            tempIndexName = DocInfoRecord.description;
          } else if (tempIndexName == 'docClassName') {
            tempIndexName = DocInfoRecord.docClassName;
          } else if (tempIndexName == 'owner') {
            tempIndexValue = DocInfoRecord.owner;
          } else if (tempIndexName == 'version') {
            tempIndexValue = DocInfoRecord.version;
          } else if (tempIndexName == 'comment') {
            tempIndexValue = DocInfoRecord.comment;
          } else {
            for (var j = 0; j < tempIndexes.length; j++) {
              var thisIndexName = tempIndexes[j].name;

              if (thisIndexName == tempIndexName) {
                tempIndexValue = tempIndexes[j].value;
                break;
              }
            }
          }

          if (!tempIndexValue) {
            tempIndexValue = '';
          }

          if (fieldType == 'float' || fieldType == 'double') {
            tempIndexValue = tempIndexValue.toString();
            tempIndexValue = tempIndexValue.replace(',', '.');
          }

          newRecord.set(idx, tempIndexValue);
        }
      }

      recordsToAdd.push(newRecord);
    }

    try {
      store.autoSave = false;

      store.add(recordsToAdd);
      store.save();
    } catch (e) {
    } finally {
      store.autoSave = true;
    }
  } catch (e) {}
}

// tworzenie contextmenu dla dokumentow w dowolnym obiekcie
function CUF_createContextMenu(ctxMenu, contextMenuDef) {
  if (!ctxMenu || !contextMenuDef) {
    return;
  }

  var docView = ctxMenu.docView;
  var processId = document.getElementsByName('processId')[0].value;
  var activityId = document.getElementsByName('activityId')[0].value;
  var tempDef = new Array();

  for (var i = 0; i < contextMenuDef.length; i++) {
    var action = contextMenuDef[i].action;
    var text = contextMenuDef[i].text;
    var icon = contextMenuDef[i].icon;
    var itemId = 'CUF_ctx_menu_item_' + Ext.id();

    if (action == 'preview' && docView) {
      continue;
    }

    tempDef[itemId] = contextMenuDef[i];

    var ctxMenuItem = new Ext.menu.Item({
      id: itemId,
      cls: 'x-btn-text-icon',
    });

    if (action == 'move') {
      if (!icon) {
        icon = 'style/img/fam/arrow_turn_right.png';
      }
      if (!text) {
        text = CUF.t('move');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        var moveDocConfig = new Object();
        var documentsToMove = new Array();
        documentsToMove.push(fileId);
        moveDocConfig.documentsToMove = documentsToMove;
        moveDocConfig.moveDocDocClasses = tempDef[item.id].moveDocDocClasses;
        moveDocConfig.moveDocCheckProtection = tempDef[item.id].moveDocCheckProtection;
        moveDocConfig.moveDocButtonLabel = tempDef[item.id].moveDocButtonLabel;
        moveDocConfig.moveDocCopyIndexes = tempDef[item.id].moveDocCopyIndexes;
        moveDocConfig.funcNameAfterDocAction = tempDef[item.id].funcNameAfterDocAction;

        CUF_moveDocumentBetweenDocClasses(moveDocConfig);
      });
    } else if (action == 'display') {
      if (!icon) {
        icon = 'style/img/fam/zoom.png';
      }
      if (!text) {
        text = CUF.t('view');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        CUF_displayDocument(fileId);
      });
    } else if (action == 'download') {
      if (!icon) {
        icon = 'style/img/fam/page_white_go.png';
      }
      if (!text) {
        text = CUF.t('download');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        CUF_downloadDocument(fileId);
      });
    } else if (action == 'detach') {
      if (!icon) {
        icon = 'style/img/fam/delete.png';
      }
      if (!text) {
        text = CUF.t('detach');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;

        Ext.Msg.show({
          title: CUF.t('document-detaching'),
          msg: CUF.t('document-process-detach-question'),
          buttons: Ext.Msg.YESNO,
          fn: function (buttonId) {
            if (buttonId != 'yes') {
              return;
            }

            var detachSuccess = CUF_detachDocumentFromProcess(processId, activityId, fileId);

            if (!detachSuccess) {
              Ext.Msg.show({
                title: CUF.t('error'),
                msg: CUF.t('document-process-detach-error'),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR,
              });
              return;
            }

            CUF_deleteDocumentFromView(fileId);

            try {
              var rowIndex = e.rowIndex;
              var store = e.store;
              var record = store.getAt(rowIndex);

              store.autoSave = false;

              store.remove(record);
              store.save();
            } catch (e) {
            } finally {
              try {
                store.autoSave = true;
              } catch (e) {}
            }
          },
          icon: Ext.Msg.QUESTION,
        });
      });
    } else if (action == 'delete') {
      if (!icon) {
        icon = 'style/img/fam/cancel.png';
      }
      if (!text) {
        text = CUF.t('delete');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;

        Ext.Msg.show({
          title: CUF.t('document-deleting'),
          msg: CUF.t('document-process-detach-delete-question'),
          buttons: Ext.Msg.YESNO,
          fn: function (buttonId) {
            if (buttonId != 'yes') {
              return;
            }
            var detachSuccess = CUF_detachDocumentFromProcess(processId, activityId, fileId);

            if (!detachSuccess) {
              Ext.Msg.show({
                title: CUF.t('error'),
                msg: CUF.t('document-process-detach-error'),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR,
              });
              return;
            }

            CUF_deleteDocumentFromView(fileId);
            CUF_deleteDocumentFromArchive(fileId);

            try {
              var rowIndex = e.rowIndex;
              var store = e.store;
              var record = store.getAt(rowIndex);

              store.autoSave = false;

              store.remove(record);
              store.save();
            } catch (e) {
            } finally {
              try {
                store.autoSave = true;
              } catch (e) {}
            }
          },
          icon: Ext.Msg.QUESTION,
        });
      });
    } else if (action == 'indexes') {
      if (!icon) {
        icon = 'style/img/fam/page_white_wrench.png';
      }
      if (!text) {
        text = CUF.t('indexes');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        var updateIndexes = tempDef[item.id].update;
        CUF_showDocIndexes(fileId, updateIndexes);
      });
    } else if (action == 'preview') {
      if (!icon) {
        icon = 'style/img/fam/page_white_magnify.png';
      }
      if (!text) {
        text = CUF.t('preview');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var tooltip = null;
        var fileId = e.fileId;
        var paging = tempDef[item.id].paging;
        var height = tempDef[item.id].height;

        if (!height) {
          height = 200;
        }

        var width = Math.ceil(height * 0.707);
        var url = CUF_getUrl();

        tooltip = new Ext.ToolTip({
          id: 'CUF_tooltip_' + Ext.id(),
          dismissDelay: 0,
          trackMouse: true,
          renderTo: formPanel.id,
          hideBorders: true,
          shadow: false,
          listeners: {
            beforerender: function (container, component, index) {
              if (!paging) {
                return;
              }

              Ext.EventManager.addListener(document, 'keyup', function (e) {
                var key = e.getKey();

                if (key == '107' || key == '109') {
                  var currentPageNum = tooltip.pageNum;

                  if (key == '107') {
                    currentPageNum++;
                  } else if (key == '109') {
                    currentPageNum--;

                    if (currentPageNum < 1) {
                      currentPageNum = 1;
                    }
                  }

                  tooltip.pageNum = currentPageNum;

                  if (Ext.isGecko3) {
                    tooltip.body.dom.innerHTML =
                      '<object id="iframe_id_' +
                      fileId +
                      '" width="' +
                      width +
                      'px" height="' +
                      height +
                      'px" scrolling="no" marginwidth="0" marginheight="0" data="' +
                      url +
                      '/CUF.ShowDocToolTip.customServlet?fileId=' +
                      fileId +
                      '&pageNum=' +
                      tooltip.pageNum +
                      '&width=' +
                      width +
                      '&height=' +
                      height +
                      '"><p>Brak obsługi object.</p></object>';
                  } else {
                    tooltip.body.dom.innerHTML =
                      '<iframe id="iframe_id_' +
                      fileId +
                      '" width="' +
                      width +
                      'px" height="' +
                      height +
                      'px" scrolling="no" frameborder="0" marginwidth="0" marginheight="0" src="' +
                      url +
                      '/CUF.ShowDocToolTip.customServlet?fileId=' +
                      fileId +
                      '&pageNum=' +
                      tooltip.pageNum +
                      '&width=' +
                      width +
                      '&height=' +
                      height +
                      '"><p>Brak obsługi iframe.</p></iframe>';
                  }
                }
              });
            },
          },
        });

        tooltip.pageNum = 1;

        if (Ext.isGecko3) {
          tooltip.body.dom.innerHTML =
            '<object id="iframe_id_' +
            fileId +
            '" width="' +
            width +
            'px" height="' +
            height +
            'px" scrolling="no" marginwidth="0" marginheight="0" data="' +
            url +
            '/CUF.ShowDocToolTip.customServlet?fileId=' +
            fileId +
            '&pageNum=' +
            tooltip.pageNum +
            '&width=' +
            width +
            '&height=' +
            height +
            '"><p>Brak obsługi object.</p></object>';
        } else {
          tooltip.body.dom.innerHTML =
            '<iframe id="iframe_id_' +
            fileId +
            '" width="' +
            width +
            'px" height="' +
            height +
            'px" scrolling="no" frameborder="0" marginwidth="0" marginheight="0" src="' +
            url +
            '/CUF.ShowDocToolTip.customServlet?fileId=' +
            fileId +
            '&pageNum=' +
            tooltip.pageNum +
            '&width=' +
            width +
            '&height=' +
            height +
            '"><p>Brak obsługi iframe.</p></iframe>';
        }

        tooltip.showAt(e.clickPos);
      });
    } else if (action == 'closeTab') {
      if (!icon) {
        icon = 'style/img/fam/cross.png';
      }
      if (!text) {
        text = CUF.t('close');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        try {
          var activityFormPanel = formPanel;
          var documentViewTabPanel = activityFormPanel.layout.east.panel.items.first();
          documentViewTabPanel.remove(e.tab);
        } catch (e) {}
      });
    } else if (action == 'showInfo') {
      if (!icon) {
        icon = 'style/img/fam/information.png';
      }
      if (!text) {
        text = CUF.t('informations');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        CUF_moreInfoForDocument(fileId);
      });
    } else if (action == 'addComment') {
      if (!icon) {
        icon = 'style/img/fam/comment.png';
      }
      if (!text) {
        text = CUF.t('comment-add');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        CUF_saveCommentForDoc(fileId);
      });
    } else if (action == 'changeDescription') {
      if (!icon) {
        icon = 'style/img/fam/arrow_refresh.png';
      }
      if (!text) {
        text = CUF.t('description-change');
      }

      ctxMenuItem.icon = icon;
      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        CUF_changeDescriptionForDoc(fileId);
      });
    } else {
      if (!icon) {
        icon = 'style/img/fam/link.png';
      }

      ctxMenuItem.icon = icon;

      if (!text) {
        text = CUF.t('use');
      }

      ctxMenuItem.text = text;

      ctxMenuItem.on('click', function (item, e) {
        ctxMenu.hide();
        var fileId = e.fileId;
        var func = tempDef[item.id].func;
        var thisAction = tempDef[item.id].action;
        var obj = new Object();

        try {
          var rowIndex = e.rowIndex;

          var record = e.store.getAt(rowIndex);
          obj.record = record;
        } catch (e) {}

        obj.fileId = fileId;
        obj.action = thisAction;
        obj.tab = e.tab;

        try {
          window[func](obj);
        } catch (e) {}
      });
    }

    ctxMenu.addMenuItem(ctxMenuItem);
  }
}

// ///////////////// zapisywanie zawartosci tabeli dynamicznej ////////////
function CUF_saveThisTable(oButton) {
  var excludedCols = new Array();
  excludedCols.push('ilosc');

  CUF_saveDynamicTableContent(oButton, CUF.t('variableset-save'), true, true, excludedCols);
}

function CUF_saveDynamicTableContent(oButton, windowTitle, saveHidden, enableSetAsDefault, excludedColumns, optionalTabTitle) {
  var config = new Object();
  config.windowTitle = windowTitle;
  config.saveHidden = saveHidden;
  config.enableSetAsDefault = enableSetAsDefault;
  config.excludedColumns = excludedColumns.join(';');
  config.optionalTabTitle = optionalTabTitle;
  CUF_executeSaveDynamicTableContent(oButton, config);
}

function CUF_executeSaveDynamicTableContent(oButton, config) {
  var vss = ServiceFactory.getVariableSetService(),
    variableSet = oButton.up('variableset');
  if (!variableSet) {
    // TODO: log
    return;
  }

  var title = vss.getInfo(variableSet).originalName;

  if (!Ext.isEmpty(title)) {
    CUF.FormTemplateService.openTemplateCreator({
      category: 'STORAGE[' + title + ']',
      setDefault: false,
      allowFieldsSelection: false,
      fields: [
        {
          type: 'variableset',
          id: title,
        },
      ],
    });
  }
}

function CUF_dynamicTableContentManagement(
  oButton,
  windowTitle,
  limitByUser,
  hidden,
  enableFiltering,
  enableLoad,
  enableLoadButtonLabel,
  askBeforeAdd,
  deleteWhenAdd,
  enableShow,
  enableShowButtonLabel,
  enableSetDefault,
  enableSetDefaultButtonLabel,
  enableDelete,
  enableDeleteButtonLabel,
  optionalTabTitle
) {
  var config = new Object();
  config.windowTitle = windowTitle;
  config.limitByUser = limitByUser;
  config.hidden = hidden;
  config.enableFiltering = enableFiltering;
  config.enableLoad = enableLoad;
  config.enableLoadButtonLabel = enableLoadButtonLabel;
  config.askBeforeAdd = askBeforeAdd;
  config.deleteWhenAdd = deleteWhenAdd;
  config.enableShow = enableShow;
  config.enableShowButtonLabel = enableShowButtonLabel;
  config.enableSetDefault = enableSetDefault;
  config.enableSetDefaultButtonLabel = enableSetDefaultButtonLabel;
  config.enableDelete = enableDelete;
  config.enableDeleteButtonLabel = enableDeleteButtonLabel;
  config.optionalTabTitle = optionalTabTitle;
  CUF_executeDynamicTableContentManagement(oButton, config);
}

function CUF_executeDynamicTableContentManagement(oButton, config) {
  var vss = ServiceFactory.getVariableSetService(),
    variableSet = oButton.up('variableset');
  if (!variableSet) {
    // TODO: log/msg
    return;
  }

  var title = vss.getInfo(variableSet).originalName;

  if (!Ext.isEmpty(title)) {
    CUF.FormTemplateService.openTemplateLoader({
      category: 'STORAGE[' + title + ']',
      userScope: config.limitByUser == 'true',
      activityScope: false,
    });
  }
}

// ////// ladowanie zawartosci tabeli dynamicznej domyslnym zestawem danych
function CUF_loadDefaultDynamicTableContentOnStart(oStore, optionalTabTitle) {
  try {
    var userId = sTaskUserName;
    var tabTitle = '';
    var storeId = oStore.storeId;

    if (!optionalTabTitle || optionalTabTitle == '') {
      Ext.ComponentMgr.all.each(function (item) {
        var itemTitle = item.title;
        if (itemTitle != null && itemTitle != undefined) {
          var storeForThisGrid = item.store;
          if (storeForThisGrid != null && storeForThisGrid != undefined) {
            var thisStoreId = storeForThisGrid.storeId;
            if (thisStoreId == storeId) {
              tabTitle = itemTitle;
              return;
            }
          }
        }
      });
    } else {
      tabTitle = optionalTabTitle;
    }

    var rows = oStore.getCount();

    if (tabTitle != null && tabTitle != undefined && tabTitle != '' && tabTitle != ' ' && rows == 0) {
      var getDefaultNameServlet = 'CUF.DynamicTableStorageGetDefaultName.customServlet';
      var getDefaultNameArgs = 'username=' + userId + '&tabTitle=' + tabTitle;
      var responseDefaultName = CUF_executeServletGetResponse(getDefaultNameServlet, getDefaultNameArgs);
      var responseDefaultNameTxt = responseDefaultName.responseText;
      var defaultNameJsonData = Ext.util.JSON.decode(responseDefaultNameTxt);
      var defaultNameSuccess = defaultNameJsonData.success;

      if (defaultNameSuccess) {
        var defaultNameRecords = defaultNameJsonData.records;

        if (defaultNameRecords != null && defaultNameRecords != undefined && defaultNameRecords.length == 1) {
          var storageName = defaultNameRecords[0].storageName;

          if (storageName != null && storageName != undefined && storageName != '') {
            var getColumnIdsServlet = 'CUF.DynamicTableStorageGetColumnIds.customServlet';
            var getColumnIdsArgs = 'storageName=' + storageName + '&owner=' + userId + '&tabTitle=' + tabTitle;
            var columnIdsResponse = CUF_executeServletGetResponse(getColumnIdsServlet, getColumnIdsArgs);
            var columnIdsResponseTxt = columnIdsResponse.responseText;
            var columnIdsJsonData = Ext.util.JSON.decode(columnIdsResponseTxt);
            var columnIdsSuccess = columnIdsJsonData.success;

            if (columnIdsSuccess) {
              var columnIdsRecords = columnIdsJsonData.records;

              if (columnIdsRecords != null && columnIdsRecords != undefined && columnIdsRecords.length > 0) {
                var colIds = new Array();

                for (var z = 0; z < columnIdsRecords.length; z++) {
                  colIds[colIds.length] = columnIdsRecords[z].columnId;
                }

                CUF_loadDynamicTableContent(storageName, userId, tabTitle, colIds, false, oStore);
              }
            }
          }
        }
      }
    }

    var loadEventFunc = null;

    var loadListeners = oStore.events.load.listeners;
    loadEventFunc = loadListeners[0].fn;

    oStore.un('load', loadEventFunc);
  } catch (e) {
    return;
  }
}

// docClassTab-tablica nazw klas
// checkRights-true jezeli ma sprawdzic uprawnienia do klas
// zwraca tablice obiektow, z ktorych pierwszy to tablica nazw dostepnych klas a
// drugi to tablica obiektow klas dokumentow
function CUF_getAvailableDocClasses(docClassTab, checkRights) {
  var checkDocClassServlet = 'CUF.CheckDocClass.customServlet';
  var checkDocClassProtectionServlet = 'CUF.CheckDocClassProtection.customServlet';
  var checkDocClassArgs = '';
  var checkDocClassProtectionArgs = '';
  var acceptedDocClasses = [];
  var docClassData = {};
  for (var k = 0; k < docClassTab.length; k++) {
    checkDocClassArgs = 'docClassName=' + docClassTab[k] + '&processId=' + processId + '&activityId=' + activityId;
    var docClassResponse = CUF_executeServletGetResponse(checkDocClassServlet, checkDocClassArgs);
    var docClassResponseTxt = docClassResponse.responseText;
    var docClassJsonData = Ext.util.JSON.decode(docClassResponseTxt);
    var docClassId = docClassJsonData.message;

    if (docClassId != '-1') {
      if (checkRights != null && (checkRights == 'true' || checkRights == true)) {
        checkDocClassProtectionArgs = 'userId=' + sTaskUserName + '&docClassName=' + docClassTab[k];
        var protectionResponse = CUF_executeServletGetResponse(checkDocClassProtectionServlet, checkDocClassProtectionArgs);
        var protectionResponseTxt = protectionResponse.responseText;
        var protectionJsonData = Ext.util.JSON.decode(protectionResponseTxt);
        var protectionSuccess = protectionJsonData.success;
        if (protectionSuccess) {
          acceptedDocClasses.push(docClassTab[k]);
          docClassData[docClassTab[k]] = docClassJsonData.records;
        }
      } else {
        acceptedDocClasses.push(docClassTab[k]);
        docClassData[docClassTab[k]] = docClassJsonData.records;
      }
    }
  }
  return [acceptedDocClasses, docClassData];
}
function CUF_addFileResult(state, newIdx, accepted, files) {
  if (state == 'ok') {
    var win = Ext.getCmp('addClipboardWindow');
    try {
      CUF_appletLog(CUFAttachClipboardAppletConf.error);
      if (CUFAttachClipboardAppletConf.error == true) {
        newIdx = -2;
      } else {
        CUF_appletLog('ada');
      }
      if (newIdx >= 0) {
        // jezeli jest nastepny plik

        CUF_attachClipboardResults('', '', '', null);
        // window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds,"added");
        win.action = 'added';
        win.last = false;
        win.close();
        CUF_showNewFileWindow(files, newIdx, accepted);
      } else if (newIdx == -2) {
        // blad
        CUF_attachClipboardResults('', '', '', null);
        // window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds,"error");
        win.action = 'error';
        win.last = false;
        Ext.Msg.alert(CUF.t('error'), CUF.t('file-error-add'));
        CUF_showNewFileWindow(files, index + 1, accepted);
      } else {
        // ostatni plik

        CUF_attachClipboardResults('', '', '', null);
        // window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds,"added");
        // window[CUFAttachClipboardAppletConf.sJsFunction](CUFAttachClipboardAppletConf.fileIds);
        win.action = 'added';
        win.last = true;
        win.close();
      }
    } catch (e) {
      CUF_attachClipboardResults('warning', CUF.t('option-inactive'), '', null);
    }
  } else if (state == 'error') {
    CUF_attachClipboardResults('', '', '', null);
    window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds, 'error');
    win.action = 'error';
    win.last = false;
    Ext.Msg.alert(CUF.t('error'), CUF.t('file-error-add'));
  }
}
function CUF_showNewFileWindow(files, index, accepted) {
  CUF_importCssForFrame('processdocuments');
  CUF_saveAllVariables();
  var filePath = '';
  if (files.split(';').length > index) filePath = files.split(';')[index];
  else return;
  var acceptedDocClasses = []; // nazwy dostepnych klas
  var docClassData = []; // dane o klasach
  var checkProtection = CUFAttachClipboardAppletConf.checkProtection; // czy
  // sprawdzic
  // uprawnienia
  // do
  // klas
  var docClassTab = CUFAttachClipboardAppletConf.sAllDocClassName.split(';'); // nazwy
  // podanych
  // klas
  var data = CUF_getAvailableDocClasses(docClassTab, checkProtection);
  acceptedDocClasses = data[0];
  docClassData = data[1];
  var processId = processId;
  var activityId = activityId;
  var setIndexes = true;
  if (acceptedDocClasses.length == 0) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('docClass-no-available'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
  } else {
    var fileName = filePath.substr(filePath.lastIndexOf('\\') + 1, filePath.length);
    var width = 500;
    var indexesForm = null;
    var panel = null;
    var addClipboardWindow = null;
    var store = new Ext.data.SimpleStore();
    var docClass = Ext.data.Record.create([
      {
        name: 'id',
        type: 'string',
      },
      {
        name: 'docClassName',
        type: 'string',
      },
    ]);

    for (var i = 0; i < acceptedDocClasses.size(); i++) {
      var record = new docClass({
        id: i,
        docClassName: acceptedDocClasses[i],
      });
      store.add(record);
    }

    store.commitChanges();
    var adjustIndexesPanel = function (indexesForm) {
      var idxPanel = indexesForm.find('id', 'CUF_indexes_panel')[0];
      if (!Ext.isEmpty(idxPanel)) {
        var idxLabel = indexesForm.find('id', 'CUF_index_label_item')[0];
        idxLabel.labelStyle = labelStyle;
        idxPanel.setHeight(250);
        idxPanel.labelAlign = 'left';
        idxPanel.labelWidth = 120;
        for (var i = 0; i < idxPanel.items.length; i++) {
          // idxPanel.items.get(i).width=200;
          idxPanel.items.get(i).anchor = '90%';
          idxPanel.items.get(i).style = fieldStyle;
          idxPanel.items.get(i).labelStyle = labelStyle;
        }
      }
    };
    var labelStyle = 'margin:3px 0px 0px 0px !important;padding:0px 0px 0px 0px !important;';
    var fieldStyle = 'margin-top:0px;margin-bottom:0px';
    var docClassItem = new Ext.form.ComboBox({
      id: 'CUF_doc_class_name_item',
      anchor: '100%',
      displayField: 'docClassName',
      valueField: 'docClassName',
      store: store,
      labelStyle: labelStyle,
      style: fieldStyle,
      resizable: true,
      triggerAction: 'all',
      // value : acceptedDocClasses[0],
      fieldLabel: CUF.t('docClass'),
      mode: 'local',
      forceSelection: true,
      listeners: {
        select: function (combo, record, index) {
          if (setIndexes) {
            var comboValue = combo.getValue();

            if (combo.isValid() && comboValue != '') {
              indexesForm.remove('CUF_index_label_item');
              indexesForm.remove('CUF_indexes_panel');

              CUF_createIndexesItem(indexesForm, docClassData, comboValue, width, true, false);
              adjustIndexesPanel(indexesForm);
              indexesForm.doLayout();
            }
          }
        },
      },
    });

    var filePathLabel = new Ext.form.Label({
      id: 'CUF_file_path',
      fieldLabel: CUF.t('file-path'),
      labelStyle: labelStyle,
      style: fieldStyle,
      text: filePath,
    });

    var nameLabel = new Ext.form.Label({
      id: 'CUF_name_label',
      fieldLabel: CUF.t('object-name'),
      text: fileName,
      labelStyle: labelStyle,
      style: fieldStyle,
    });
    var status = '';
    if (accepted == true) status = CUF.t('accepted');
    else status = 'Niezaakceptowany';
    var statusLabel = new Ext.form.Label({
      id: 'CUF_status_label',
      fieldLabel: CUF.t('status'),
      text: status,
      labelStyle: labelStyle,
      style: fieldStyle,
    });

    panel = new Ext.FormPanel({
      id: 'CUF_clipboard_indexes_form',
      autoEl: 'div',
      width: width - 20,
      frame: true,
      autoHeight: true,
      bodyStyle: 'padding: 5px 5px 5px 5px;',
      labelWidth: 120,
    });

    indexesForm = new Ext.Panel({
      id: 'CUF_internal_indexes_panel',
      anchor: '100%',
      autoScroll: true,
      layout: 'form',
    });

    var descriptionItem = new Ext.form.TextField({
      id: 'CUF_description_item',
      inputType: 'text',
      fieldLabel: CUF.t('file-description'),
      anchor: '100%',
      labelStyle: labelStyle,
      style: fieldStyle,
    });

    var newVersionItem = new Ext.form.Checkbox({
      id: 'CUF_new_version_item',
      fieldLabel: CUF.t('save-new-version'),
      checked: false,
      labelStyle: labelStyle,
      style: fieldStyle,
    });
    var addDocButton = new Ext.Button({
      id: 'CUF_add_doc_button',
      text: CUF.t('add'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/add.png',
      handler: function () {
        var choosenDocClassName = docClassItem.getValue();

        if (docClassItem.isValid() && choosenDocClassName != '') {
          description = descriptionItem.getValue();
          newVersion = newVersionItem.getValue();

          hmMapping = new Array();

          if (setIndexes) {
            var counter = 1;
            var indexesItem = Ext.get('CUF_indexes_panel');

            if (indexesItem != null && indexesItem != undefined) {
              var docClassJsonRecs = docClassData[choosenDocClassName];

              var indexItem = Ext.get('index' + counter);

              while (indexItem != null && indexItem != undefined) {
                var label = docClassJsonRecs[counter - 1].indexName;

                hmMapping.push(label + '=' + indexItem.getValue());
                counter++;
                indexItem = Ext.get('index' + counter);
              }
            }
          }
          var form = this.findParentByType('form');
          var appletObj = document.getElementById('com.plusmpm.applets.CUFAttachClipboard');
          var win = addClipboardWindow;
          if (appletObj) {
            try {
              CUF_showLoadingMask('CUF_attach_clipboard_mask', CUF.t('clipboard-attaching'));
              appletObj.hidden = false;
              CUFAttachClipboardAppletConf.bNewVersion = newVersion;
              CUFAttachClipboardAppletConf.sActivityId = activityId;
              CUFAttachClipboardAppletConf.sProcessId = processId;
              CUFAttachClipboardAppletConf.sDescription = description;
              CUFAttachClipboardAppletConf.sDocClassName = choosenDocClassName;
              CUFAttachClipboardAppletConf.sFileName = fileName;
              CUFAttachClipboardAppletConf.sHmMapping = hmMapping.join(';');
              CUFAttachClipboardAppletConf.sUploader = sTaskUserName;
              CUFAttachClipboardAppletConf.loading = true;
              CUFAttachClipboardAppletConf.error = false;
              appletObj.addFile(index);
            } catch (e) {
              CUF_attachClipboardResults('warning', CUF.t('option-inactive'), '', null);
            }
          }
        } else {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('docClass-choose'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.WARNING,
          });
        }
      },
    });

    var skipButton = new Ext.Button({
      id: 'CUF_skip_button',
      text: CUF.t('skip'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/arrow_right.png',
      handler: function () {
        var form = this.findParentByType('form');
        var win = form.findParentByType(Ext.Window);
        action = 'skip';

        var f = files.split(';');
        if (f.length > index + 1) {
          addClipboardWindow.last = false;
          win.close();
          // window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds,"skip");
          var appletObj = document.getElementById('com.plusmpm.applets.CUFAttachClipboard');
          if (appletObj) {
            try {
              appletObj.hidden = false;
              var accepted = appletObj.isFileAccepted(index + 1);

              CUF_attachClipboardResults('', '', '', null);
              CUF_showNewFileWindow(files, index + 1, accepted);
            } catch (e) {
              CUF_attachClipboardResults('warning', CUF.t('error-unexpected'), '', null);
            }
          }
        } else {
          addClipboardWindow.last = true;
          win.close();
          // window[CUFAttachClipboardAppletConf.sJsFunction](CUFAttachClipboardAppletConf.fileIds,"skip");
        }
      },
    });

    var interruptButton = new Ext.Button({
      id: 'CUF_cancel_button',
      text: CUF.t('abort'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/cancel.png',
      handler: function () {
        // window[CUFAttachClipboardAppletConf.sJsFunction](CUFAttachClipboardAppletConf.fileIds,"break");
        addClipboardWindow.action = 'break';
        addClipboardWindow.last = true;
        Ext.EventManager.removeListener(document, 'keyup');
        addClipboardWindow.close();
      },
    });
    var previewButton = new Ext.Button({
      id: 'CUF_preview_button',
      text: CUF.t('preview'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/page_white_magnify.png',
      handler: function () {
        var appletObj = document.getElementById('com.plusmpm.applets.CUFAttachClipboard');
        if (appletObj) {
          try {
            appletObj.hidden = false;
            appletObj.showPreview(index);
          } catch (e) {
            CUF_attachClipboardResults('warning', CUF.t('file-format-unsupported'), '', null);
          }
        }
      },
    });
    panel.add(filePathLabel);
    panel.add(nameLabel);
    panel.add(statusLabel);
    indexesForm.add(descriptionItem);
    indexesForm.add(newVersionItem);
    indexesForm.add(docClassItem);

    panel.add(indexesForm);
    panel.addButton(addDocButton);
    panel.addButton(skipButton);
    panel.addButton(interruptButton);
    panel.addButton(previewButton);

    if (accepted != true) {
      addDocButton.hide();
      indexesForm.hide();
    }

    addClipboardWindow = new Ext.Window({
      width: width + 20,
      id: 'addClipboardWindow',
      y: 0,
      autoHeight: true,
      modal: true,
      constrain: true,
      shadow: false,
      listeners: {
        close: function (p) {
          try {
            window[CUFAttachClipboardAppletConf.sJsCheckFunction](CUFAttachClipboardAppletConf.fileIds, p.action);
          } catch (e) {}
          if (p.last == true) {
            try {
              window[CUFAttachClipboardAppletConf.sJsFunction](CUFAttachClipboardAppletConf.fileIds);
            } catch (e) {}
          }
        },
        show: function () {
          try {
            CUF_removeLoadingMask('CUF_attach_clipboard_mask');
          } catch (e) {}
        },
      },
      resizable: false,
      layout: 'fit',
      title: CUF.t('document-add-clipboard'),
      iconCls: 'x-CUFwindow-adddoc',
      items: panel,
    });

    addClipboardWindow.show();

    Ext.EventManager.addListener(document, 'keyup', function (e) {
      if (e.getKey() == '107') {
        if (!skipButton.hidden) {
          skipButton.handler.call(skipButton, Ext.EventObject);
        }
      } else if (e.getKey() == '27') {
        interruptButton.handler.call(interruptButton, Ext.EventObject);
      } else if (e.getKey() == '13') {
        addDocButton.handler.call(addDocButton, Ext.EventObject);
      }
    });
  }
}
// ///// dodawanie tekstu do toolbaru DT /////////////////////
function CUF_addTextToDTBottomToolbar(config) {
  if (!config) {
    return;
  }

  var gridNumber = config.gridNumber;
  var text = config.text;
  var position = config.position;
  var bold = config.bold;

  var grid = CUF_getActivityGrid(gridNumber);

  if (!grid) {
    return;
  }

  var gridToolbar = grid.bottomToolbar;

  if (position && position == 'right') {
    gridToolbar.addFill();
  }

  if (bold) {
    gridToolbar.add('<b>' + text + '</b>');
  } else {
    gridToolbar.add(text);
  }

  grid.doLayout();
}

// pobieranie obiektu grid na podstawie jego numeru w zadaniu
function CUF_getActivityGrid(gridNumber) {
  var searchedStoreId = 'data_dynamic_tabel_div_' + (gridNumber - 1) + '_JsonStore';
  var grid = undefined;

  Ext.ComponentMgr.all.each(function (item) {
    try {
      var store = item.getStore();
      var thisStoreId = store.storeId;

      if (thisStoreId == searchedStoreId) {
        grid = item;
      }
    } catch (e) {}
  });

  return grid;
}

// konfigurowalny weryfikator podlaczania dokumentow do procesu
function CUF_processDocumentsVerifier(config) {
  try {
    var verify = config.verify;
    var documents = config.documents;
    var addHere = config.addHere;
    var addInActivity = config.addInActivity;

    if (!verify) {
      return true;
    }

    var processId = ServiceFactory.getActivityInfoService().getProcessId();
    var activityId = ServiceFactory.getActivityInfoService().getActivityId();
    var toVerify = Ext.util.JSON.encode(documents);
    var verifyServlet = 'CUF.VerifyProcessDocuments.customServlet';
    var verifyArgs =
      'processId=' +
      processId +
      '&activityId=' +
      activityId +
      '&addHere=' +
      addHere +
      '&addInActivity=' +
      addInActivity +
      '&toVerify=' +
      toVerify;

    var verifyResponse = CUF_executeServletGetResponse(verifyServlet, verifyArgs);
    var verifyResponseTxt = verifyResponse.responseText;
    var verifyJsonData = Ext.util.JSON.decode(verifyResponseTxt);
    var verifySuccess = verifyJsonData.success;

    if (verifySuccess) {
      return true;
    }

    var message = verifyJsonData.message;

    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: message,
      maxWidth: 400,
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return false;
  } catch (e) {
    throw e;
    return false;
  }
}

// aktywowanie menu dla podgladu dokumentow
function CUF_reconfigureDocView(config) {
  window.setTimeout(function () {
    CUF_executeReconfigureDocView(config);
  }, 0);
}

function CUF_executeReconfigureDocView(config) {
  try {
    var contextMenuDef = config.contextMenuDef;
    var disableTabClosing = config.disableTabClosing;
    var activateAddDocTool = config.activateAddDocTool;

    if (!(formPanel.layout || formPanel.layout.east)) {
      return;
    }

    var docViewPanel = formPanel.layout.east.panel;

    if (!docViewPanel) {
      return;
    }

    var docViewTabPanel = docViewPanel.items.first();

    if (!docViewTabPanel) {
      return;
    }

    if (contextMenuDef && contextMenuDef.length > 0) {
      var ctxMenu = new Ext.menu.Menu({
        id: 'CUF_context_menu_' + Ext.id(),
        renderTo: formPanel.id,
        docView: true,
      });

      CUF_createContextMenu(ctxMenu, contextMenuDef);

      docViewTabPanel.on('contextmenu', function (tabPanel, tab, e) {
        e.stopEvent();
        var clickPos = e.getXY();
        e.fileId = tab.fileId;
        e.clickPos = clickPos;
        e.tab = tab;
        e.rowIndex = undefined;
        clickPos[1] = clickPos[1] - ctxMenu.getHeight();

        if (clickPos[1] < 0) {
          clickPos[1] = 0;
        }

        var currentActiveTab = tabPanel.getActiveTab();

        if (currentActiveTab != null && currentActiveTab.id != tab.id) {
          tabPanel.setActiveTab(tab.id);
        }

        ctxMenu.showAt(clickPos);
      });
    }

    if (disableTabClosing) {
      Ext.each(docViewTabPanel.items.items, function (item, index, allItems) {
        if (Ext.isIE) {
          item.tabEl.removeAttribute('class', 'x-tab-strip-closable');
        } else {
          item.tabEl.removeClassName('x-tab-strip-closable');
        }
      });

      docViewTabPanel.on('beforeadd', function (container, component, index) {
        component.closable = false;
      });
    }

    if (activateAddDocTool) {
      var addDocToolQtip = config.addDocToolQtip;

      if (!addDocToolQtip) {
        addDocToolQtip = CUF.t('document-add');
      }

      docViewPanel.addTool({
        id: 'plus',
        handler: function (e, toolEl, panel, tc) {
          CUF_attachDocument(config);
        },
        qtip: addDocToolQtip,
      });
    }
  } catch (e) {}
}

// maksymalizowanie tabeli dynamicznej
function CUF_enableDTMaximize(grid, config) {
  if (!grid) {
    return;
  }

  var documentsViewPanel = undefined;

  if (formPanel.layout && formPanel.layout.east) {
    documentsViewPanel = formPanel.layout.east.panel;
  }

  var maximizeQtip = CUF.t('maximize');
  var minimizeQtip = CUF.t('minimize');
  var hideDocView = true;

  if (config) {
    maximizeQtip = config.maximizeQtip;
    minimizeQtip = config.minimizeQtip;
    hideDocView = config.hideDocView;

    if (!maximizeQtip) {
      maximizeQtip = CUF.t('maximize');
    }

    if (!minimizeQtip) {
      minimizeQtip = CUF.t('minimize');
    }

    if (!hideDocView) {
      hideDocView = true;
    }
  }

  var gridWindow = undefined;
  var gridLocation = grid.container.id;

  grid.addTool({
    id: 'maximize',
    handler: function (e, toolEl, panel, tc) {
      var activityTaskPanel = formPanel.items.first();

      Ext.apply(panel, {
        lastBoxMaxHeight: panel.boxMaxHeight,
        lastWidth: panel.getWidth(),
        lastHeight: panel.getHeight(),
        lastWindowOnscroll: window.onscroll,
        clientMaximized: true,
      });

      delete panel.boxMaxHeight;

      panel.tools.maximize.hide();
      panel.tools.minimize.show();

      var newGridWidth = activityTaskPanel.getWidth() + activityTaskPanel.getPosition()[0];
      var newGridHeight = Ext.getBody().getViewSize().height;

      if (documentsViewPanel) {
        var lastDocViewOpened = false;

        if (!documentsViewPanel.hidden) {
          if (!documentsViewPanel.collapsed) {
            lastDocViewOpened = true;

            if (hideDocView) {
              var toggleToolId = documentsViewPanel.tools.toggle.id;
              document.getElementById(toggleToolId).click();

              newGridWidth = formPanel.getWidth() + formPanel.getPosition()[0] - 5;
            }
          }
        } else {
          newGridWidth = Ext.getBody().getViewSize().width;

          if (!Ext.isIE) {
            newGridWidth = newGridWidth - 17;
          }
        }

        Ext.apply(panel, {
          lastDocViewOpened: lastDocViewOpened,
        });
      }

      panel.setWidth(newGridWidth);
      panel.setHeight(newGridHeight);
      panel.on('resize', CUF_maximizedGridResized);

      if (!gridWindow) {
        gridWindow = new Ext.Window({
          id: 'CUF_maximized_grid_window',
          frame: false,
          closable: false,
          modal: false,
          constrain: true,
          shadow: false,
          resizable: false,
          border: false,
          header: false,
          items: panel,
        });
      }

      gridWindow.show();
      var lastScrollPos = CUF_setMaximizedWindowPosition(gridWindow);

      Ext.apply(panel, {
        lastScrollPos: lastScrollPos,
      });

      CUF_refreshGridColumnHeaders(grid);

      window.onscroll = function () {
        CUF_setMaximizedWindowPosition(gridWindow);
      };
    },
    qtip: maximizeQtip,
    hidden: false,
  });

  grid.addTool({
    id: 'minimize',
    handler: function (e, toolEl, panel, tc) {
      if (documentsViewPanel) {
        if (!documentsViewPanel.hidden) {
          if ((documentsViewPanel.collapsed && panel.lastDocViewOpened) || (!documentsViewPanel.collapsed && !panel.lastDocViewOpened)) {
            var toggleToolId = documentsViewPanel.tools.toggle.id;
            document.getElementById(toggleToolId).click();
          }
        }
      }

      gridWindow.hide();

      panel.un('resize', CUF_maximizedGridResized);
      panel.tools.maximize.show();
      panel.tools.minimize.hide();
      panel.setWidth(panel.lastWidth);
      panel.setHeight(panel.lastHeight);

      window.onscroll = panel.lastWindowOnscroll;

      Ext.apply(panel, {
        boxMaxHeight: panel.lastBoxMaxHeight,
      });

      var lastScrollPos = panel.lastScrollPos;
      delete panel.lastBoxMaxHeight;
      delete panel.lastWidth;
      delete panel.lastHeight;
      delete panel.lastScrollPos;
      delete panel.lastDocViewOpened;
      delete panel.lastWindowOnscroll;
      delete panel.clientMaximized;

      var panelPlace = Ext.fly(gridLocation);
      panelPlace.insertFirst(panel.id);

      window.scroll(0, lastScrollPos);

      formPanel.recalculateHeights();

      window.setTimeout(function () {
        CUF_refreshGridColumnHeaders(grid);
      }, 100);
    },
    qtip: minimizeQtip,
    hidden: true,
  });
}

// ustawianie pozycji zmaksymalizowanego okna
function CUF_setMaximizedWindowPosition(maxWindow) {
  try {
    var pageYOffset = 0;

    if (Ext.isIE && Ext.isStrict) {
      pageYOffset = document.documentElement.scrollTop;
    } else {
      pageYOffset = window.pageYOffset;
    }

    maxWindow.setPosition(0, pageYOffset);

    return pageYOffset;
  } catch (e) {
    return 0;
  }
}

// zmiana szerokosci zmaksymalizowanego grida
function CUF_maximizedGridResized(panel, adjWidth, adjHeight, rawWidth, rawHeight) {
  var documentsViewPanel = undefined;

  if (formPanel.layout.east) {
    documentsViewPanel = formPanel.layout.east.panel;
  }

  if (documentsViewPanel) {
    var newGridWidth = 0;

    if (!documentsViewPanel.hidden) {
      if (!documentsViewPanel.maximized) {
        var activityTaskPanel = formPanel.items.first();

        newGridWidth = activityTaskPanel.getWidth() + activityTaskPanel.getPosition()[0];
      }
    } else {
      newGridWidth = Ext.getBody().getViewSize().width;

      if (!Ext.isIE) {
        newGridWidth = newGridWidth - 17;
      }
    }

    panel.setWidth(newGridWidth);
  }

  CUF_refreshGridColumnHeaders(panel);
}

// blokowanie Data Choosera dla danej zmiennej
function CUF_blockDataChooser(dataChooserId) {
  try {
    var variable = document.getElementById(dataChooserId);
    variable.readOnly = true;
    var parent = variable.parentNode;
    var cbox = Ext.fly(parent.id);
    cbox.purgeAllListeners();
  } catch (e) {}
}

// blokowanie wszystkich Data Chooserow
function CUF_blockAllDataChoosers() {
  try {
    var extCmp = Ext.ComponentMgr.all;

    for (var i = 0; i < extCmp.length; i++) {
      var item = extCmp.items[i];

      if (
        !item.store ||
        item.store == null ||
        !item.store.baseParams ||
        item.store.baseParams == null ||
        !item.store.baseParams.sDataChooserId ||
        item.store.baseParams.sDataChooserId == null
      ) {
        continue;
      }

      var dataChooserId = item.store.baseParams.sDataChooserId;
      CUF_blockDataChooser(dataChooserId);
    }
  } catch (e) {}
}

// dodawanie naglowkow kolumn we wszystkich Data Chooser na formularzu
function CUF_addHeadersToDataChoosers() {
  window.setTimeout(CUF_executeAddHeadersToDataChoosers, 0);
}

function CUF_executeAddHeadersToDataChoosers() {
  try {
    var extCmp = Ext.ComponentMgr.all;

    for (var i = 0; i < extCmp.length; i++) {
      var item = extCmp.items[i];

      if (
        !item.store ||
        item.store == null ||
        !item.store.baseParams ||
        item.store.baseParams == null ||
        !item.store.baseParams.sDataChooserId ||
        item.store.baseParams.sDataChooserId == null
      ) {
        continue;
      }

      var chooserConfigId = item.store.baseParams.sDataChooserId;
      var dcConf = window['oChooserConfig_chooser_' + chooserConfigId];

      if (!dcConf || dcConf == null || !item.tpl || item.tpl == null || !item.tpl.html || item.tpl.html == null) {
        continue;
      }

      var htmlTpl = item.tpl.html;
      var newInfo = htmlTpl.replace('<tpl for="."><div class="search-item">', '');
      newInfo = newInfo.replace('</div></tpl>', '');

      var sCss = 'text-align: center;' + 'background-color: #F0F0F0;' + 'font-weight: bold;' + 'border-bottom: 2px solid #DDDDDD';

      newInfo = newInfo.replace('<table width = "100%">', '<table width = "100%" style="' + sCss + '">');
      newInfo = newInfo.replace(/{/g, '');
      newInfo = newInfo.replace(/}/g, '');

      for (var j = 0; j < dcConf.sMappings.length; j++) {
        var map = dcConf.sMappings[j];
        var mapTab = map.split(';');
        var dcId = mapTab[0];
        var dcName = mapTab[2];
        newInfo = newInfo.replace(dcId, dcName);
      }

      var newTmpl = newInfo + htmlTpl;
      item.tpl = newTmpl;
    }
  } catch (e) {}
}

// dodawanie menu do naglowka wszystkich tabel dynamicznych
function CUF_addMenuToAllDTHeader() {
  window.setTimeout(CUF_executeAddMenuToAllDTHeader, 0);
}

function CUF_executeAddMenuToAllDTHeader() {
  try {
    var gridNo = 1;
    var grid = CUF_getActivityGrid(gridNo);

    while (grid) {
      var columns = window['oDataDynamicTableDefinition_' + (gridNo - 1)];

      if (columns) {
        columns = columns.jsonData.columns;
        CUF_executeAddMenuToDTHeader(grid, columns);
      }

      gridNo++;
      grid = CUF_getActivityGrid(gridNo);
    }
  } catch (e) {}
}

// dodawanie menu do konkretnego naglowka tabeli dynamicznej
function CUF_addMenuToDTHeader(gridNo) {
  window.setTimeout(function () {
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    var columns = window['oDataDynamicTableDefinition_' + (gridNo - 1)];

    if (!columns) {
      return;
    }

    columns = columns.jsonData.columns;
    CUF_executeAddMenuToDTHeader(grid, columns);
  }, 0);
}

function CUF_executeAddMenuToDTHeader(grid, columns) {
  try {
    var ctxMenu = new Ext.menu.Menu({
      id: 'CUF_context_menu_' + Ext.id(),
      renderTo: formPanel.id,
    });

    var addColItems = new Array();
    var colModel = grid.getColumnModel();

    for (var i = 0; i < columns.length; i++) {
      var key = columns[i].dataIndex;
      var colIndex = colModel.findColumnIndex(key);

      if (columns[i].hidden || key == 'row_number_column_id') {
        continue;
      }

      var colHeader = colModel.getColumnHeader(colIndex).unescapeHTML();

      var addColItem = new Ext.menu.Item({
        id: 'CUF_show_col_' + key + '_' + Ext.id(),
        text: colHeader,
        cls: 'x-btn-text',
        thisColIdx: colIndex,
        handler: function (item, e) {
          ctxMenu.hide();
          colModel.setHidden(this.thisColIdx, false);
          CUF_refreshGridColumnHeaders(grid);
        },
      });

      addColItems.push(addColItem);
    }

    ctxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_hide_column_' + Ext.id(),
        text: CUF.t('column-hide'),
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/delete.png',
        handler: function (item, e) {
          ctxMenu.hide();
          if (colModel.getColumnCount(true) > 2) {
            var columnIdx = e.columnIndex;
            colModel.setHidden(columnIdx, true);
            CUF_refreshGridColumnHeaders(grid);
          }
        },
      })
    );
    ctxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_show_column_' + Ext.id(),
        text: CUF.t('column-show'),
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/add.png',
        hideOnClick: false,
        menu: {
          items: addColItems,
        },
      })
    );
    ctxMenu.addMenuItem(
      new Ext.menu.Item({
        id: 'CUF_ctx_menu_restore_' + Ext.id(),
        text: CUF.t('restore'),
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/application_home.png',
        handler: function (item, e) {
          ctxMenu.hide();
          var refreshId = grid.tools.refresh.id;
          document.getElementById(refreshId).click();
        },
      })
    );

    grid.on('headercontextmenu', function (grid, columnIndex, e) {
      e.stopEvent();
      var clickPos = e.getXY();
      e.columnIndex = columnIndex;
      ctxMenu.showAt(clickPos);
    });
  } catch (e) {}
}

// odswiezanie naglowka okreslonej tabeli dynamicznej
function CUF_refreshGridColumnHeaders(grid) {
  try {
    grid.suspendEvents(false);
    grid.getView().forceFit = true;
    grid.stopEditing();
    grid.getView().refresh(true);
  } catch (e) {
  } finally {
    grid.getView().forceFit = false;
    grid.mpm_clearState();
    grid.resumeEvents();
  }
}

// dostosowanie szerokosci formularza po zmniejszeniu/zwiekszeniu czcionki
function CUF_fixFormPanelOnZoom(extraFunc) {
  Ext.EventManager.addListener(document, 'keyup', function (e) {
    var key = e.getKey();
    if (!e.ctrlKey || !(key == e.NUM_PLUS || key == e.NUM_MINUS)) {
      return;
    }

    var container = formPanel.container;
    var containerWidth = container.getWidth();
    var formPanelItems = formPanel.items;
    var taskPanel = formPanelItems.first();

    if (formPanelItems.length == 1) {
      taskPanel.setWidth(containerWidth);
    } else {
      var docPanel = formPanelItems.items[1];

      if (!docPanel.hidden && !docPanel.collapsed) {
        taskPanel.setWidth(containerWidth * 0.7);
        docPanel.setWidth(containerWidth * 0.3);
      } else {
        taskPanel.setWidth(containerWidth);
      }
    }

    CUF_refreshTaskPanel();

    if (extraFunc != undefined && extraFunc != null && extraFunc != '') {
      try {
        window[extraFunc](e);
      } catch (e) {}
    }
  });
}

// dopasowanie wysokosci wszystkich tabel dynamicznych do ilosci rekordow
function CUF_fixAllGridsToRecords(config) {
  window.onload = new (function () {
    window.setTimeout(function () {
      CUF_executeFixAllGridsToRecords(config);
    }, 0);
  })();
}

function CUF_executeFixAllGridsToRecords(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeFixGridToRecords(config, table);
    } catch (e) {}
  });
}

// dopasowanie wysokosci okreslonej tabeli dynamicznej do ilosci rekordow
function CUF_fixGridToRecords(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeFixGridToRecords(config, grid);
  }, 0);
}

function CUF_executeFixGridToRecords(config, grid) {
  var store = grid.getStore();
  var gridView = grid.getView();
  var gridBody = grid.body;
  var gridDOM = Ext.getDom(grid.id);
  var headerObj = Ext.query('*[class=x-grid3-header-offset]', gridDOM);
  var header = headerObj[0];
  var gridContainer = grid.container;
  var maxSize = 0;

  if (config && config.maxSize) {
    maxSize = config.maxSize;
  }

  CUF_fixGridToRecordsCalculate(grid, store, gridView, gridBody, header, gridContainer, maxSize, true);

  grid.on('collapse', function (grid, animate) {
    gridContainer.setHeight(grid.header.getHeight());

    try {
      formPanel.recalculateHeights();
    } catch (e) {}
  });

  grid.on('expand', function (grid, animate) {
    var expandHeight = grid.getHeight();
    grid.queuedBodySize.height = expandHeight;
    grid.lastSize.height = expandHeight;
    gridContainer.setHeight(expandHeight);

    try {
      formPanel.recalculateHeights();
    } catch (e) {}
  });

  store.on('add', function (store, records, index) {
    CUF_fixGridToRecordsCalculate(grid, store, gridView, gridBody, header, gridContainer, maxSize, true);
  });

  store.on('remove', function (store, records, index) {
    CUF_fixGridToRecordsCalculate(grid, store, gridView, gridBody, header, gridContainer, maxSize, false);
  });
}

function CUF_fixGridToRecordsCalculate(grid, store, gridView, gridBody, header, gridContainer, maxSize, adding) {
  var recordsHeight = 0;
  var headerHeight = 0;
  var storeCount = store.getCount();

  if (storeCount > 0) {
    var firstRow = gridView.getRow(0);
    var rowsContainer = firstRow.parentNode;
    recordsHeight = Ext.get(rowsContainer).getHeight();
  }

  if (recordsHeight == 0) {
    return;
  }

  if (maxSize > 0 && maxSize < storeCount) {
    recordsHeight = (recordsHeight / storeCount) * maxSize;
  }

  recordsHeight += 4;
  var gridBodyHeight = gridBody.getHeight();
  var gridHeight = grid.getHeight();
  if (Ext.isIE) {
    headerHeight = header.clientHeight;
  } else {
    headerHeight = header.getHeight();
  }

  if (adding && gridBodyHeight - headerHeight >= recordsHeight) {
    return;
  }

  if (!adding && store.getCount() < 7) {
    CUF_restoreGridInitialHeight(grid);
    return;
  }

  var newHeight = gridHeight + recordsHeight + headerHeight - gridBodyHeight;
  var initialHeight = grid.initialConfig.boxMaxHeight;

  if (newHeight < initialHeight) {
    newHeight = initialHeight;
  }

  delete grid.boxMaxHeight;
  gridContainer.setHeight(newHeight);
  grid.setHeight(newHeight);
  grid.boxMaxHeight = newHeight;

  try {
    formPanel.recalculateHeights();
  } catch (e) {}
}

// dodanie przycisku usun wszystkie rekordy do wszystkich tabel dynamicznych
function CUF_addRemoveAllButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddRemoveAllButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddRemoveAllButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddRemoveAllButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku usun wszystkie rekordy do okreslonej tabeli dynamicznej
function CUF_addRemoveAllButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_addRemoveAllButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddRemoveAllButtonToGrid(config, grid) {
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_delete_all_button_' + Ext.id();
  var buttonText = CUF.t('clear');
  var buttonTooltip = CUF.t('record-delete-all');
  var buttonIcon = 'style/img/fam/bin_empty.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var deleteAllButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_removeAllFromDT(this);
    },
  });

  bbar.addButton(deleteAllButton);
  bbar.doLayout();
}

function CUF_removeAllFromDT(oButton) {
  var grid = oButton.up('variableset');
  var table = VariableSetService.getVariableSet(grid.getId());
  table.clear();
}

// dodanie przycisku usun zaznaczone rekordy do wszystkich tabel dynamicznych
function CUF_addRemoveSelectedButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddRemoveSelectedButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddRemoveSelectedButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddRemoveSelectedButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku usun zaznaczone rekordy do okreslonej tabeli dynamicznej
function CUF_addRemoveSelectedButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_addRemoveSelectedButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddRemoveSelectedButtonToGrid(config, grid) {
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_delete_selected_button_' + Ext.id();
  var buttonText = CUF.t('delete-selected');
  var buttonTooltip = CUF.t('record-delete-selected');
  var buttonIcon = 'style/img/fam/bin_empty.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var deleteSelectedButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_removeSelectedFromDT(this);
    },
  });

  bbar.addButton(deleteSelectedButton);
  bbar.doLayout();
}

function CUF_removeSelectedFromDT(oButton) {
  var grid = oButton.up('variableset');
  var store = grid.getStore();
  var gridView = grid.getView();
  var selectedRows = new Array();

  for (var i = 0; i < store.getCount(); i++) {
    var row = gridView.getRow(i);
    var rowObj = Ext.get(row);
    if (rowObj.hasClass('x-CUFgrid-marked-row')) {
      selectedRows.push(store.getAt(i));
      rowObj.removeClass('x-CUFgrid-marked-row');
    }
  }

  if (selectedRows.length == 0) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('record-select'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return;
  }

  try {
    store.autoSave = false;
    store.remove(selectedRows);
    store.save();
  } catch (e) {
  } finally {
    store.autoSave = true;
  }

  if (grid.clientMaximized) {
    return;
  }

  CUF_restoreGridInitialHeight(grid);
}

// przywracanie domyslnej wysokosci tabeli dynamicznej
function CUF_restoreGridInitialHeight(grid) {
  delete grid.boxMaxHeight;
  var gridContainer = grid.container;
  var initialHeight = grid.initialConfig.boxMaxHeight;
  gridContainer.setHeight(initialHeight);
  grid.setHeight(initialHeight);
  grid.boxMaxHeight = initialHeight;

  try {
    formPanel.recalculateHeights();
  } catch (e) {}
}

// dodanie przycisku dopasuj do rekordow do wszystkich tabel dynamicznych
function CUF_addFitButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddFitButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddFitButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddFitButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku dopasuj do rekordow do okreslonej tabeli dynamicznej
function CUF_addFitButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddFitButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddFitButtonToGrid(config, grid) {
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_fit_button_' + Ext.id();
  var buttonText = CUF.t('match');
  var buttonTooltip = CUF.t('record-match-height');
  var buttonIcon = 'style/img/fam/arrow_inout.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var fitButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {},
  });

  bbar.addButton(fitButton);
  bbar.doLayout();
}

function CUF_addFitButtonToDT(oButton, config) {
  var grid = oButton.up('variableset');
  CUF_executeFixGridToRecords(config, grid);
}

// dodanie przycisku powielania wszystkich rekordow do wszystkich tabel
// dynamicznych
function CUF_addDuplicateAllButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddDuplicateAllButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddDuplicateAllButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddDuplicateAllButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku powielania wszystkich rekordow do okreslonej tabeli
// dynamicznej
function CUF_addDuplicateAllButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddDuplicateAllButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddDuplicateAllButtonToGrid(config, grid) {
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_duplicate_all_button_' + Ext.id();
  var buttonText = CUF.t('duplicate');
  var buttonTooltip = CUF.t('record-copy-all');
  var buttonIcon = 'style/img/fam/table_multiple.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var duplicateAllButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_duplicateAllFromDT(this);
    },
  });

  bbar.addButton(duplicateAllButton);
  bbar.doLayout();
}

function CUF_duplicateAllFromDT(oButton) {
  var grid = oButton.up('variableset');
  grid.stopEditing();
  var store = grid.getStore();

  try {
    store.autoSave = false;
    var storeCount = store.getCount();
    var recordsToAdd = new Array();

    if (storeCount == 0) {
      return;
    }

    for (var i = 0; i < storeCount; i++) {
      var oldRecord = store.getAt(i);
      var newRecord = oldRecord.copy(-1);
      newRecord.id = Ext.data.Record.id(newRecord);
      // newRecord.id = (storeCount + i).toString();
      // newRecord.set('row_number_column_id', (storeCount +
      // i).toString());
      recordsToAdd.push(newRecord);
    }

    store.add(recordsToAdd);
    store.save();
  } catch (e) {
  } finally {
    store.autoSave = true;
  }
}

// dodanie przycisku powielania wybranych rekordow do wszystkich tabel
// dynamicznych
function CUF_addDuplicateSelectedButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddDuplicateSelectedButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddDuplicateSelectedButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddDuplicateSelectedButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku powielania wybranych rekordow do okreslonej tabeli
// dynamicznej
function CUF_addDuplicateSelectedButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddDuplicateSelectedButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddDuplicateSelectedButtonToGrid(config, grid) {
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_duplicate_selected_button_' + Ext.id();
  var buttonText = CUF.t('duplicate-selected');
  var buttonTooltip = CUF.t('record-copy-selected');
  var buttonIcon = 'style/img/fam/table_multiple.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var duplicateSelectedButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_duplicateSelectedFromDT(this);
    },
  });

  bbar.addButton(duplicateSelectedButton);
  bbar.doLayout();
}

function CUF_duplicateSelectedFromDT(oButton) {
  var grid = oButton.up('variableset');
  grid.stopEditing();
  var store = grid.getStore();
  var storeCount = store.getCount();
  var gridView = grid.getView();
  var selectedRows = new Array();

  for (var i = 0; i < storeCount; i++) {
    var row = gridView.getRow(i);
    var rowObj = Ext.get(row);
    if (rowObj.hasClass('x-CUFgrid-marked-row')) {
      var oldRecord = store.getAt(i);
      var newRecord = oldRecord.copy();
      newRecord.id = Ext.data.Record.id(newRecord);
      // var newId = (storeCount + selectedRows.length).toString();
      // newRecord.id = newId;
      // newRecord.set('row_number_column_id', newId);
      selectedRows.push(newRecord);
      rowObj.removeClass('x-CUFgrid-marked-row');
    }
  }

  if (selectedRows.length == 0) {
    Ext.Msg.show({
      title: CUF.t('warning'),
      msg: CUF.t('record-select'),
      buttons: Ext.Msg.OK,
      icon: Ext.Msg.WARNING,
    });
    return;
  }

  try {
    store.autoSave = false;
    store.add(selectedRows);
    store.save();
  } catch (e) {
  } finally {
    store.autoSave = true;
  }
}

// mozliwosc wskazania wybranych rekordow we wszystkich tabelach dynamicznych
function CUF_enableAllGridsRowMarking() {
  window.setTimeout(CUF_executeEnableAllGridsRowMarking, 0);
}

function CUF_executeEnableAllGridsRowMarking() {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeEnableGridRowMarking(table);
    } catch (e) {}
  });
}

// mozliwosc wskazania wybranych rekordow w okreslonej tabeli dynamicznej
function CUF_enableGridRowMarking(gridNo) {
  window.setTimeout(function () {
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeEnableGridRowMarking(grid);
  }, 0);
}

function CUF_executeEnableGridRowMarking(grid) {
  CUF_importCssForFrame('gridadditionals');
  grid.on('rowcontextmenu', function (grid, rowIndex, e) {
    e.stopEvent();
    var gridView = grid.getView();
    var row = gridView.getRow(rowIndex);
    CUF_handleGridRecordMark(grid, row);
  });
}

function CUF_handleGridRecordMark(grid, row) {
  var rowObj = Ext.get(row);
  if (rowObj.hasClass('x-CUFgrid-marked-row')) {
    rowObj.removeClass('x-CUFgrid-marked-row');
  } else {
    rowObj.addClass('x-CUFgrid-marked-row');
  }
}

// dodanie przycisku zaznaczania wszystkich rekordow we wszystkich tabelach
// dynamicznych
function CUF_addSelectAllButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddSelectAllButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddSelectAllButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddSelectAllButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku zaznaczania wszystkich rekordow w okreslonej tabeli
// dynamicznej
function CUF_addSelectAllButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddSelectAllButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddSelectAllButtonToGrid(config, grid) {
  CUF_importCssForFrame('gridadditionals');
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_select_all_button_' + Ext.id();
  var buttonText = CUF.t('select-all');
  var buttonTooltip = CUF.t('record-select-all');
  var buttonIcon = 'style/img/fam/mouse_add.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var selectAllButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_selectAllFromDT(this);
    },
  });

  bbar.addButton(selectAllButton);
  bbar.doLayout();
}

function CUF_selectAllFromDT(oButton) {
  var grid = oButton.up('variableset');
  CUF_markAllRecordsInGrid(grid);
}

function CUF_markAllRecordsInGrid(grid) {
  var gridView = grid.getView();
  var store = grid.getStore();
  var storeCount = store.getCount();

  if (storeCount == 0) {
    return;
  }

  for (var i = 0; i < storeCount; i++) {
    var row = gridView.getRow(i);
    var rowObj = Ext.get(row);
    if (!rowObj.hasClass('x-CUFgrid-marked-row')) {
      rowObj.addClass('x-CUFgrid-marked-row');
    }
  }
}

// dodanie przycisku odznaczania wszystkich rekordow we wszystkich tabelach
// dynamicznych
function CUF_addDeselectAllButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddDeselectAllButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddDeselectAllButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddDeselectAllButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie przycisku odznaczania wszystkich rekordow w okreslonej tabeli
// dynamicznej
function CUF_addDeselectAllButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddDeselectAllButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddDeselectAllButtonToGrid(config, grid) {
  CUF_importCssForFrame('gridadditionals');
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_deselect_all_button_' + Ext.id();
  var buttonText = CUF.t('deselect-all');
  var buttonTooltip = CUF.t('record-deselect-all');
  var buttonIcon = 'style/img/fam/mouse_delete.png';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
  }

  var deselectAllButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      CUF_deselectAllFromDT(this);
    },
  });

  bbar.addButton(deselectAllButton);
  bbar.doLayout();
}

function CUF_deselectAllFromDT(oButton) {
  var grid = oButton.up('variableset');
  CUF_unmarkAllRecordsInGrid(grid);
}

function CUF_unmarkAllRecordsInGrid(grid) {
  var gridView = grid.getView();
  var store = grid.getStore();
  var storeCount = store.getCount();

  if (storeCount == 0) {
    return;
  }

  for (var i = 0; i < storeCount; i++) {
    var row = gridView.getRow(i);
    var rowObj = Ext.get(row);
    if (rowObj.hasClass('x-CUFgrid-marked-row')) {
      rowObj.removeClass('x-CUFgrid-marked-row');
    }
  }
}

// dodanie dowolnego przycisku we wszystkich tabelach dynamicznych
function CUF_addCustomButtonToAllGrids(config) {
  window.setTimeout(function () {
    CUF_executeAddCustomButtonToAllGrids(config);
  }, 0);
}

function CUF_executeAddCustomButtonToAllGrids(config) {
  Ext.each(formPanel.tables, function (table, index, allTables) {
    try {
      CUF_executeAddCustomButtonToGrid(config, table);
    } catch (e) {}
  });
}

// dodanie dowolnego przycisku w okreslonej tabeli dynamicznej
function CUF_addCustomButtonToGrid(config) {
  window.setTimeout(function () {
    if (!config) {
      return;
    }

    var gridNo = config.gridNo;
    var grid = CUF_getActivityGrid(gridNo);

    if (!grid) {
      return;
    }

    CUF_executeAddCustomButtonToGrid(config, grid);
  }, 0);
}

function CUF_executeAddCustomButtonToGrid(config, grid) {
  CUF_importCssForFrame('gridadditionals');
  var bbar = grid.getBottomToolbar();
  var buttonId = 'CUF_custom_button_' + Ext.id();
  var buttonText = CUF.t('execute');
  var buttonTooltip = CUF.t('operation-executing');
  var buttonIcon = 'style/img/fam/table_gear.png';
  var buttonFunction = '';

  if (config) {
    if (config.buttonId && config.buttonId != '') {
      buttonId = config.buttonId;
    }
    if (config.buttonText && config.buttonText != '') {
      buttonText = config.buttonText;
    }
    if (config.buttonTooltip && config.buttonTooltip != '') {
      buttonTooltip = config.buttonTooltip;
    }
    if (config.buttonIcon && config.buttonIcon != '') {
      buttonIcon = config.buttonIcon;
    }
    if (config.buttonFunction && config.buttonFunction != '') {
      buttonFunction = config.buttonFunction;
    }
  }

  var customButton = new Ext.Button({
    id: buttonId,
    text: buttonText,
    cls: 'x-btn-text-icon',
    icon: buttonIcon,
    tooltip: buttonTooltip,
    handler: function () {
      if (!buttonFunction || buttonFunction == '') {
        Ext.Msg.show({
          title: CUF.t('error'),
          msg: CUF.t('operation-no-definition'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.ERROR,
        });
        return;
      }

      try {
        window[buttonFunction](grid);
      } catch (e) {
        Ext.Msg.show({
          title: CUF.t('error'),
          msg: CUF.t('operation-error-execute') + e,
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.ERROR,
        });
      }
    },
  });

  bbar.addButton(customButton);
  bbar.doLayout();
}

// wyszukiwanie rekordow w tabeli dynamicznej
function CUF_searchRecordsInGrid(oButton) {
  var grid = oButton.up('variableset');
  CUF_executeSearchRecordsInGrid(grid);
}

function CUF_executeSearchRecordsInGrid(grid) {
  CUF_importCssForFrame('gridadditionals');
  var gridView = grid.getView();
  var store = grid.getStore();
  var columns = store.fields.keys;
  var searchWindow = Ext.getCmp('CUF_grid_search_window');
  var history = new Array();
  var winTitle = CUF.t('record-searching');
  var searchPanel = Ext.getCmp('CUF_grid_search_fields_panel');
  var wrappedSearchItem = Ext.getCmp('CUF_grid_search_wrapped_search_item');
  var caseSensitiveItem = Ext.getCmp('CUF_grid_search_case_sensitive_item');
  var wholeWordsItem = Ext.getCmp('CUF_grid_search_whole_words_item');
  var foundRow = null;
  var lastScrollPos = 0;
  var startIndex = -1;
  var selectIndex = 0;
  var clientCellClick = false;
  var cellClickFunction = function (gridPanel, rowIndex, columnIndex, e) {
    clientCellClick = true;
    startIndex = rowIndex;
  };

  if (!searchWindow) {
    var selModel = grid.getSelectionModel();
    var columnModel = grid.getColumnModel();
    var visibleColumnsIds = new Array();
    var visibleColumnsNames = new Array();
    var visibleColumnsTypes = new Array();
    var visibleColumnsWidths = new Array();
    var oneRecord = new Array();

    for (var k = 0; k < columns.length; k++) {
      var key = columns[k];
      var colIndex = columnModel.findColumnIndex(key);

      if (columnModel.isHidden(colIndex)) {
        continue;
      }

      if (key != 'row_number_column_id') {
        var columnName = columnModel.getColumnHeader(colIndex).unescapeHTML();
        var field = store.fields.get(key);
        var fieldType = field.type.type;
        var columnWidth = columnModel.getColumnWidth(colIndex);
        visibleColumnsIds.push(key);
        visibleColumnsNames.push(columnName);
        visibleColumnsTypes.push(fieldType);
        visibleColumnsWidths.push(columnWidth);

        var oneCell = new Ext.data.Field({
          name: key,
          type: fieldType,
          dateFormat: 'Y-m-d',
        });
        oneRecord.push(oneCell);
      }
    }
    var historyRecord = Ext.data.Record.create(oneRecord);

    if (visibleColumnsIds.length == 0) {
      Ext.Msg.show({
        title: CUF.t('warning'),
        msg: CUF.t('variableset-no-columns'),
        buttons: Ext.Msg.OK,
        icon: Ext.Msg.WARNING,
      });
      return;
    }
    selectIndex = columnModel.findColumnIndex(visibleColumnsIds[0]);

    var searchPanelToolbar = new Ext.Toolbar({
      id: 'CUF_search_grid_fields_panel_toolbar',
    });

    searchPanel = new Ext.FormPanel({
      id: 'CUF_grid_search_fields_panel',
      layout: 'table',
      region: 'north',
      header: false,
      hideBorders: true,
      autoScroll: true,
      layoutConfig: {
        columns: visibleColumnsIds.length,
      },
      labelWidth: 200,
      labelAlign: 'top',
      buttonAlign: 'left',
      height: 80,
      frame: true,
    });

    var mainPanel = new Ext.Panel({
      id: 'CUF_search_grid_main_panel',
      items: searchPanel,
      bbar: searchPanelToolbar,
    });

    for (var i = 0; i < visibleColumnsIds.length; i++) {
      var filterId = visibleColumnsIds[i];
      var filterName = visibleColumnsNames[i];
      var filterType = visibleColumnsTypes[i];
      var filterWidth = visibleColumnsWidths[i];

      var filter = new Ext.Container({
        id: 'filter_container_' + filterId,
        autoEl: 'div',
        layout: 'form',
        width: filterWidth,
      });

      if (filterType == 'int') {
        var filterField = new Ext.form.NumberField({
          id: filterId + '_filter',
          allowDecimals: false,
          fieldLabel: filterName,
          width: filterWidth,
          selectOnFocus: true,
        });

        filter.add(filterField);
      } else if (filterType == 'float') {
        var filterField = new Ext.form.NumberField({
          id: filterId + '_filter',
          allowDecimals: true,
          decimalSeparator: ',',
          fieldLabel: filterName,
          width: filterWidth,
          selectOnFocus: true,
        });

        filter.add(filterField);
      } else if (filterType == 'date') {
        var filterField = new Ext.form.DateField({
          id: filterId + '_filter',
          format: 'Y-m-d',
          fieldLabel: filterName,
          width: filterWidth,
          selectOnFocus: true,
        });

        filter.add(filterField);
      } else {
        var filterField = new Ext.form.TextField({
          id: filterId + '_filter',
          fieldLabel: filterName,
          width: filterWidth,
          selectOnFocus: true,
        });

        filter.add(filterField);
      }

      searchPanel.add(filter);
    }

    var nextRecordButton = new Ext.Button({
      id: 'CUF_grid_search_next_button',
      cls: 'x-btn-icon',
      icon: 'style/img/fam/arrow_down.png',
      handler: function () {
        var filterDef = searchWindow.readFilters();
        if (!filterDef) {
          return;
        }
        var filterIds = filterDef.filterIds;
        var filterValues = filterDef.filterValues;
        searchWindow.handleLastFoundRow();

        var foundRowIndex = this.findRecord(filterIds, filterValues);
        if (foundRowIndex == -1) {
          foundRow = null;
          Ext.Msg.show({
            title: CUF.t('no-results'),
            msg: CUF.t('search-no-results'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        searchWindow.goToFoundRecord(foundRowIndex);
      },
      findRecord: function (filterIds, filterValues) {
        var foundRowIndex = -1;
        var storeSize = store.getCount();
        var wrappedSearch = false;
        var enableWrappedSearch = wrappedSearchItem.getValue();
        var caseSensitive = caseSensitiveItem.getValue();
        var anyMatch = !wholeWordsItem.getValue();

        do {
          var foundCounter = 0;
          foundRowIndex = store.find(filterIds[0], filterValues[0], startIndex + 1, anyMatch, caseSensitive);

          if (foundRowIndex == -1) {
            if (wrappedSearch || !enableWrappedSearch) {
              return -1;
            }
            startIndex = -1;
            wrappedSearch = true;
            continue;
          }

          foundCounter++;
          for (var i = 1; i < filterIds.length; i++) {
            var tempIndex = store.find(filterIds[i], filterValues[i], startIndex + 1, anyMatch, caseSensitive);
            if (tempIndex == foundRowIndex) {
              foundCounter++;
            } else {
              break;
            }
          }

          if (foundCounter == filterIds.length) {
            return foundRowIndex;
          } else {
            startIndex = foundRowIndex;
          }
        } while (startIndex + 1 < storeSize);

        return -1;
      },
    });
    searchPanelToolbar.addButton(nextRecordButton);
    searchPanelToolbar.addSeparator();
    var previousRecordButton = new Ext.Button({
      id: 'CUF_grid_search_previous_button',
      cls: 'x-btn-icon',
      icon: 'style/img/fam/arrow_up.png',
      handler: function () {
        var filterDef = searchWindow.readFilters();
        if (!filterDef) {
          return;
        }
        var filterIds = filterDef.filterIds;
        var filterValues = filterDef.filterValues;
        searchWindow.handleLastFoundRow();

        var foundRowIndex = this.findRecord(filterIds, filterValues, startIndex);
        if (foundRowIndex == -1) {
          foundRow = null;
          Ext.Msg.show({
            title: CUF.t('no-results'),
            msg: CUF.t('search-no-results'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.INFO,
          });
          return;
        }

        searchWindow.goToFoundRecord(foundRowIndex);
      },
      findRecord: function (filterIds, filterValues, limit) {
        var storeSize = store.getCount();
        var enableWrappedSearch = wrappedSearchItem.getValue();

        if (limit <= 0) {
          if (!enableWrappedSearch) {
            return -1;
          } else {
            limit = storeSize;
          }
        }

        var foundIndexes = this.searchFromTo(filterIds, filterValues, -1, limit);

        if (foundIndexes.length == 0) {
          if (!enableWrappedSearch) {
            return -1;
          }

          foundIndexes = this.searchFromTo(filterIds, filterValues, limit, storeSize);

          if (foundIndexes.length == 0) {
            return -1;
          }
        }

        return foundIndexes.last();
      },
      searchFromTo: function (filterIds, filterValues, from, to) {
        var foundIndexes = new Array();
        var caseSensitive = caseSensitiveItem.getValue();
        var anyMatch = !wholeWordsItem.getValue();
        var foundRowIndex = from;

        do {
          var foundCounter = 0;
          foundRowIndex = store.find(filterIds[0], filterValues[0], foundRowIndex + 1, anyMatch, caseSensitive);

          if (foundRowIndex == -1) {
            return foundIndexes;
          }

          foundCounter++;
          for (var i = 1; i < filterIds.length; i++) {
            var tempIndex = store.find(filterIds[i], filterValues[i], foundRowIndex + 1, anyMatch, caseSensitive);
            if (tempIndex == foundRowIndex) {
              foundCounter++;
            } else {
              break;
            }
          }

          if (foundCounter != filterIds.length) {
            foundRowIndex++;
            continue;
          }

          if (foundRowIndex != to) {
            foundIndexes.push(foundRowIndex);
          }
        } while (foundRowIndex + 1 < to);

        return foundIndexes;
      },
    });
    searchPanelToolbar.addButton(previousRecordButton);
    searchPanelToolbar.addSeparator();
    var clearButton = new Ext.Button({
      id: 'CUF_grid_search_clear_button',
      text: CUF.t('clear'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/bin.png',
      handler: function () {
        startIndex = -1;
        if (foundRow) {
          CUF_handleGridRecordMark(grid, foundRow);
          foundRow = null;
        }

        for (var i = 0; i < visibleColumnsIds.length; i++) {
          var filterId = visibleColumnsIds[i];
          var filterField = Ext.getCmp(filterId + '_filter');
          if (filterField) {
            filterField.setValue('');
          }
        }
        searchPanel.items.first().items.first().focus(true, 100);
      },
    });
    searchPanelToolbar.addButton(clearButton);
    searchPanelToolbar.addSeparator();
    var historyButton = new Ext.Button({
      id: 'CUF_grid_search_history_button',
      text: CUF.t('history'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/table.png',
      handler: function () {
        var config = new Object();
        config.history = history;
        config.columnKeys = visibleColumnsIds;
        config.columnNames = visibleColumnsNames;
        config.columnTypes = visibleColumnsTypes;
        config.historyRecord = historyRecord;
        CUF_showSearchGridHistory(config);
      },
    });
    searchPanelToolbar.addButton(historyButton);
    searchPanelToolbar.addSeparator();
    var closeButton = new Ext.Button({
      id: 'CUF_grid_search_close_button',
      text: CUF.t('close'),
      cls: 'x-btn-text-icon',
      icon: 'style/img/fam/cross.png',
      handler: function () {
        searchWindow.hide();
      },
    });
    searchPanelToolbar.addButton(closeButton);
    searchPanelToolbar.addSeparator();
    wrappedSearchItem = new Ext.form.Checkbox({
      id: 'CUF_grid_search_wrapped_search_item',
      fieldLabel: '',
      boxLabel: '',
      inputValue: 'true',
      checked: true,
    });
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(CUF.t('search-cyclic'));
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(wrappedSearchItem);
    caseSensitiveItem = new Ext.form.Checkbox({
      id: 'CUF_grid_search_case_sensitive_item',
      fieldLabel: '',
      boxLabel: '',
      inputValue: 'true',
      checked: false,
    });
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(CUF.t('casesensitive-use'));
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(caseSensitiveItem);
    wholeWordsItem = new Ext.form.Checkbox({
      id: 'CUF_grid_search_whole_words_item',
      fieldLabel: '',
      boxLabel: '',
      inputValue: 'true',
      checked: false,
    });
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(CUF.t('whole-words'));
    searchPanelToolbar.addSpacer();
    searchPanelToolbar.add(wholeWordsItem);

    searchWindow = new Ext.Window({
      id: 'CUF_grid_search_window',
      iconCls: 'x-CUFgrid-searchWindow',
      width: grid.getWidth(),
      autoHeight: true,
      modal: false,
      constrain: true,
      shadow: false,
      resizable: false,
      closable: false,
      title: winTitle,
      items: mainPanel,
      gridColumnModel: columnModel,
      columnIds: columns,
      history: new Object(),
      listeners: {
        hide: function (win) {
          if (foundRow) {
            CUF_handleGridRecordMark(grid, foundRow);
            foundRow = null;
          }
          startIndex = -1;
          window.scroll(0, lastScrollPos);
          grid.un('columnresize', CUF_fitSearchPanelToColumns);
          grid.un('cellclick', cellClickFunction);
        },
      },
      readFilters: function () {
        var filterDef = null;
        var filterIds = new Array();
        var filterValues = new Array();

        for (var i = 0; i < visibleColumnsIds.length; i++) {
          var filterId = visibleColumnsIds[i];
          var filterValue = Ext.getCmp(filterId + '_filter').getValue();

          if (filterValue != '') {
            filterIds.push(filterId);
            filterValues.push(filterValue);
          }
        }

        if (filterIds.length == 0) {
          Ext.Msg.show({
            title: CUF.t('warning'),
            msg: CUF.t('filters-no-specified'),
            buttons: Ext.Msg.OK,
            icon: Ext.Msg.WARNING,
          });
          return filterDef;
        }

        filterDef = new Object();
        filterDef.filterIds = filterIds;
        filterDef.filterValues = filterValues;
        this.addToHistory(filterDef);
        return filterDef;
      },
      handleLastFoundRow: function () {
        if (foundRow) {
          if (!clientCellClick) {
            startIndex = gridView.findRowIndex(foundRow);
          }
          CUF_handleGridRecordMark(grid, foundRow);
        }
        clientCellClick = false;
      },
      goToFoundRecord: function (foundRowIndex) {
        foundRow = gridView.getRow(foundRowIndex);
        foundRow.scrollIntoView(grid.id, false);
        window.scroll(0, searchWindow.getPosition()[1]);
        selModel.select(foundRowIndex, selectIndex);
        CUF_handleGridRecordMark(grid, foundRow);
      },
      addToHistory: function (filterDef) {
        if (this.isInHistory(filterDef)) {
          return;
        }
        var rec = new historyRecord();
        var filterIds = filterDef.filterIds;
        var filterValues = filterDef.filterValues;

        for (var i = 0; i < filterIds.length; i++) {
          var filterId = filterIds[i];
          var filterValue = filterValues[i];
          rec.set(filterId, filterValue);
        }
        history.push(rec);
        this.history = history;
      },
      isInHistory: function (filterDef) {
        var filterIds = filterDef.filterIds;
        var filterValues = filterDef.filterValues;
        for (var i = 0; i < history.length; i++) {
          var rec = history[i];
          var foundCounter = 0;
          for (var j = 0; j < visibleColumnsIds.length; j++) {
            var key = visibleColumnsIds[j];
            var idx = filterIds.indexOf(key);
            if (idx != -1) {
              if (rec.get(key) == filterValues[idx]) {
                foundCounter++;
              }
            } else {
              if (rec.get(key) == '') {
                foundCounter++;
              }
            }
          }
          if (foundCounter == filterIds.length) {
            return true;
          }
        }
        return false;
      },
    });
  } else {
    history = searchWindow.history;
  }

  searchWindow.show();
  if (Ext.isIE && Ext.isStrict) {
    lastScrollPos = document.documentElement.scrollTop;
  } else {
    lastScrollPos = window.pageYOffset;
  }
  var gridPos = grid.getPosition();
  searchWindow.setPosition(gridPos[0], gridPos[1] - searchWindow.getHeight());
  window.scroll(0, searchWindow.getPosition()[1]);
  grid.on('columnresize', CUF_fitSearchPanelToColumns);
  grid.on('cellclick', cellClickFunction);
  searchPanel.items.first().items.first().focus(true, 100);
}

function CUF_fitSearchPanelToColumns(colIdx, newSize) {
  var searchWindow = Ext.getCmp('CUF_grid_search_window');
  var colModel = searchWindow.gridColumnModel;
  var columnIds = searchWindow.columnIds;

  for (var i = 0; i < columnIds.length; i++) {
    var colId = columnIds[i];
    var colIndex = colModel.findColumnIndex(colId);

    if (colModel.isHidden(colIndex)) {
      continue;
    }

    if (colId != 'row_number_column_id') {
      var colWidth = colModel.getColumnWidth(colIndex);

      var filterContainer = Ext.getCmp('filter_container_' + colId);
      filterContainer.setWidth(colWidth);

      var filterField = Ext.getCmp(colId + '_filter');
      filterField.setWidth(colWidth);
    }
  }
}

function CUF_showSearchGridHistory(config) {
  var history = config.history;
  var columnKeys = config.columnKeys;
  var columnNames = config.columnNames;
  var columnTypes = config.columnTypes;
  var historyRecord = config.historyRecord;
  var historyWindow = null;
  var historyWinTitle = CUF.t('search-history');
  var historyGrid = null;
  var width = Ext.getBody().getViewSize().width * 0.8;
  var height = Ext.getBody().getViewSize().height * 0.5;

  var historyColumns = new Array();

  for (var k = 0; k < columnKeys.length; k++) {
    var oneColumnId = columnKeys[k];
    var oneColumnName = columnNames[k];
    var oneColumnType = columnTypes[k];
    var oneColumnAlign = 'center';

    if (oneColumnType == 'int' || oneColumnType == 'double' || oneColumnType == 'float') {
      oneColumnAlign = 'right';
    }

    var oneColDef = new Ext.grid.Column({
      id: oneColumnId,
      header: '<center>' + oneColumnName + '</center>',
      dataIndex: oneColumnId,
      align: oneColumnAlign,
    });

    historyColumns.push(oneColDef);
  }

  var historyStoreReader = new Ext.data.JsonReader({
    id: 'CUF_grid_search_history_store_reader',
    fields: historyRecord,
  });
  var historyStore = new Ext.data.Store({
    storeId: 'CUF_grid_search_history_store',
    reader: historyStoreReader,
    fields: historyRecord,
  });

  for (var i = 0; i < history.length; i++) {
    historyStore.add(history[i]);
  }
  historyStore.commitChanges();

  var historyToolbar = new Ext.Toolbar({
    id: 'CUF_grid_search_history_toolbar',
  });

  var useFilterButton = new Ext.Button({
    id: 'CUF_grid_search_use_filter_button',
    text: CUF.t('use'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/cog.png',
    handler: function () {
      var selModel = historyGrid.getSelectionModel();
      var rows = selModel.getSelections();
      if (rows.length == 0) {
        Ext.Msg.show({
          title: CUF.t('warning'),
          msg: CUF.t('filter-select'),
          buttons: Ext.Msg.OK,
          icon: Ext.Msg.WARNING,
        });
        return;
      }
      var filter = rows[0];
      for (var i = 0; i < columnKeys.length; i++) {
        var fieldId = columnKeys[i];
        var filterValue = filter.get(fieldId);
        var filterField = Ext.getCmp(fieldId + '_filter');
        filterField.setValue(filterValue);
      }
      historyWindow.close();
    },
  });
  historyToolbar.addButton(useFilterButton);
  historyToolbar.addSeparator();
  var closeButton = new Ext.Button({
    id: 'CUF_grid_search_close_history_button',
    text: CUF.t('close'),
    cls: 'x-btn-text-icon',
    icon: 'style/img/fam/cross.png',
    handler: function () {
      historyWindow.close();
    },
  });
  historyToolbar.addButton(closeButton);

  historyGrid = new Ext.grid.GridPanel({
    id: 'CUF_storage_preview_grid_item',
    width: width,
    height: height,
    store: historyStore,
    enableColumnResize: true,
    stripeRows: true,
    columnLines: true,
    autoScroll: true,
    colModel: new Ext.grid.ColumnModel({
      defaults: {
        sortable: true,
        menuDisabled: false,
        css: 'vertical-align:middle;',
      },
      columns: historyColumns,
    }),
    viewConfig: {
      forceFit: true,
    },
    sm: new Ext.grid.RowSelectionModel({
      singleSelect: true,
    }),
    bbar: historyToolbar,
  });

  historyWindow = new Ext.Window({
    id: 'CUF_grid_search_history_window',
    iconCls: 'x-CUFgrid-searchHistoryWindow',
    autoHeight: true,
    modal: true,
    constrain: true,
    shadow: false,
    resizable: false,
    closable: true,
    title: historyWinTitle,
    items: historyGrid,
  });

  historyWindow.show();
}

// dodawanie dowolnego przycisku na formularzu obok zmiennej
function CUF_addCustomButtonOnForm(config) {
  if (!config) {
    return;
  }

  var varId = config.varId;
  var defaultFunction = config.defaultFunction;
  var pressedFunction = config.pressedFunction;
  var defaultText = config.defaultText;
  var pressedText = config.pressedText;
  var defaultIcon = config.defaultIcon;
  var pressedIcon = config.pressedIcon;
  var defaultTooltip = config.defaultTooltip;
  var pressedTooltip = config.pressedTooltip;
  var location = config.location;
  var buttonCls = 'x-btn-text';

  if (!varId) {
    return;
  }
  var varField = document.getElementById(varId);
  if (!varField) {
    return;
  }
  var varFieldContainer = varField.parentNode;
  if (!varFieldContainer) {
    return;
  }
  if (!pressedText) {
    pressedText = defaultText;
  }
  if (!pressedIcon) {
    pressedIcon = defaultIcon;
  }
  if (!pressedTooltip) {
    pressedTooltip = defaultTooltip;
  }
  if (!pressedFunction) {
    pressedFunction = defaultFunction;
  }
  if (defaultIcon && defaultText) {
    buttonCls = 'x-btn-text-icon';
  } else if (defaultIcon && !defaultText) {
    buttonCls = 'x-btn-icon';
  }
  if (!location) {
    location = 'bottom';
  }

  var customButton = new Ext.Button({
    id: 'CUF_custom_form_button_' + varId,
    cls: buttonCls,
    tooltipType: 'title',
    buttonPressed: false,
    handler: function () {
      try {
        if (this.buttonPressed) {
          this.buttonPressed = false;
          if (defaultText) {
            this.setText(defaultText);
          }
          if (defaultIcon) {
            this.setIcon(defaultIcon);
          }
          if (defaultTooltip) {
            this.setTooltip(defaultTooltip);
          }
          if (defaultFunction) {
            window[defaultFunction]();
          }
        } else {
          this.buttonPressed = true;
          if (pressedText) {
            this.setText(pressedText);
          }
          if (pressedIcon) {
            this.setIcon(pressedIcon);
          }
          if (pressedTooltip) {
            this.setTooltip(pressedTooltip);
          }
          if (pressedFunction) {
            window[pressedFunction]();
          }
        }
      } catch (e) {}
    },
  });

  if (defaultText) {
    customButton.setText(defaultText);
  }
  if (defaultIcon) {
    customButton.setIcon(defaultIcon);
  }
  if (defaultTooltip) {
    customButton.setTooltip(defaultTooltip);
  }

  new Ext.Container({
    id: customButton.id + '_container',
    layout: 'fit',
    items: customButton,
    renderTo: varFieldContainer,
    width: customButton.width,
    autoEl: {
      tag: 'div',
    },
  });
}

// dodawanie separatora pomiedzy zmiennymi na formularzu
function CUF_addFormSeparator(config) {
  if (!config) {
    return;
  }

  var varId = config.varId;
  var varDescr = config.varDescr;
  var separator = config.separator;
  var iterations = config.iterations;
  var center = config.center;
  var big = config.big;
  var color = config.color;

  if (!varId) {
    return;
  }

  var varField = document.getElementById(varId);
  if (!varField) {
    return;
  }
  if (!varDescr) {
    varDescr = '';
  }
  if (!separator) {
    separator = '-';
  }
  try {
    iterations = parseInt(iterations);
    if (iterations <= 0) {
      iterations = 150;
    }
  } catch (e) {
    iterations = 150;
  }
  if (center == undefined) {
    center = false;
  }
  if (big == undefined) {
    big = false;
  }
  if (color == undefined) {
    color = false;
  }

  var tblBodyObj = document.getElementById('taskpanel').tBodies[0];
  var rowNo = -1;
  for (var i = 0; i < tblBodyObj.rows.length; i++) {
    var row = tblBodyObj.rows[i];
    var variableDescr = null;
    if (Ext.isIE) {
      variableDescr = row.firstChild;
    } else {
      variableDescr = row.firstElementChild;
    }
    if (variableDescr && variableDescr.id == 'variabledesc_' + varId) {
      rowNo = i;
    }
  }
  if (rowNo == -1) {
    return;
  }
  var newRow = tblBodyObj.insertRow(rowNo + 1);
  var newCellDescr = newRow.insertCell(-1);
  newCellDescr.innerHTML = varDescr;
  Ext.fly(newCellDescr).addClass('field');
  var newCellSeparator = newRow.insertCell(-1);
  var extNewCellSeparator = Ext.fly(newCellSeparator);
  var separatorValue = '';
  for (var i = 0; i < iterations; i++) {
    separatorValue += separator;
  }
  newCellSeparator.innerHTML = separatorValue;
  if (!(center || big || color)) {
    extNewCellSeparator.addClass('field-value');
    return;
  }
  CUF_importCssForFrame('formadditionals');
  if (center && big && color) {
    extNewCellSeparator.addClass('x-CUFform-separator-big-center-color');
  } else if (center && big) {
    extNewCellSeparator.addClass('x-CUFform-separator-big-center');
  } else if (center && color) {
    extNewCellSeparator.addClass('x-CUFform-separator-center-color');
  } else if (big && color) {
    extNewCellSeparator.addClass('x-CUFform-separator-big-color');
  } else if (center) {
    extNewCellSeparator.addClass('x-CUFform-separator-center');
  } else if (big) {
    extNewCellSeparator.addClass('x-CUFform-separator-big');
  } else if (color) {
    extNewCellSeparator.addClass('x-CUFform-separator-color');
  }
}

// //////////////////////////////////////////////////////////
// //////////////////////////////////////////////////////////
// ///////////////// testowe ////////////////////////////////
// //////////////////////////////////////////////////////////
// //////////////////////////////////////////////////////////
function CUF_executeServletToGetOrder(servlet, arg) {
  var req = servlet;
  try {
    request = new XMLHttpRequest();
    request.overrideMimeType('text/xml');
  } catch (e) {
    try {
      request = new ActiveXObject('Microsoft.XMLHTTP');
    } catch (e) {
      alert('XMLHttpRequest error: ' + e);
    }
  }

  bbb = CUF_addrequest(req, req, arg);
  var xmlDoc = bbb.responseXML;

  return xmlDoc;
}

function CUF_formatAmountString(amount) {
  if ((amount != null && amount != '') || amount == 0) {
    if (isNaN(amount)) {
    } else {
      amount = parseFloat(amount);
      if (!isNaN(amount)) {
        amount = amount.numberFormat('### ### ###.##');
        amount = amount.trim();
      } else return 0;
    }
  }
  return amount;
}

function CUF_ChooseSearchResult(order) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  if (oStore.getCount() > 0) {
    alert(CUF.t('order-error-add-single'));
  } else {
    if (order != null) {
      var tab = order.split(';');
      if (CUF_CheckIfOrderExistInDynamicTable(tab[0]) != 'FALSE') {
        CUF_addOrderToTab0(tab[0], tab[2], tab[9]);
        var supplier = unescape(tab[3].replace(/\+/g, ' '));
        CUF_setSupplierInInvoiceProcess(supplier, tab[1], tab[8]);
        var ans = CUF_getOrderValueServlet(tab[0], 'false', '');
        CUF_addAnswerFromServerToTab(ans, '');
        CUF_uploadInvoiceTable(
          Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore'),
          Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore'),
          tab[0],
          'TAK'
        );
        var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore');
        CUF_setWartoscZamowienia(oStore);
      }
    }
  }
}

function CUF_addOrderToTab0(order, amount, condition) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  oStore.add(
    new oStore.recordType({
      nr_zamowienia: order,
      wartosc_zamowienia_tab: amount,
      warunek_platnosci: condition,
    })
  );
  document.getElementById('nr_zamowienia_formularz').value = order;
}

function CUF_CheckIfOrderExistInDynamicTable(order) {
  var record = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  var rows = CUF_checkTableSize_extjs('0');
  for (var i = 0; i < rows; i = i + 1) {
    var nNrZamowienia = record.getAt(i).get('nr_zamowienia');
    if (nNrZamowienia == order) {
      return 'FALSE';
    }
  }
  return 'TRUE';
}

function CUF_checkTableSize_extjs(tableId) {
  return Ext.StoreMgr.get('data_dynamic_tabel_div_' + tableId + '_JsonStore').data.length;
}

function CUF_setSupplierInInvoiceProcess(sDostawca, sKod, sMiasto) {
  var dostawcaObj = document.getElementById('nazwa');
  var dostawca = dostawcaObj.value;
  if (dostawca != null && dostawca == '') {
    dostawcaObj.value = sDostawca;
  }
  var kodObj = document.getElementById('kod_dostawcy');
  var kod = kodObj.value;
  if (kod != null && kod == '') {
    kodObj.value = sKod;
  }
  var miastoObj = document.getElementById('miasto');
  var miasto = miastoObj.value;
  if (miasto != null && miasto == '') {
    miastoObj.value = sMiasto;
  }
}

function CUF_getOrderValueServlet(order, option, record) {
  var sServletName = 'CUF.DynamicTable.customServlet';
  var x = option;
  var arg = '&param=' + order + '&param1=' + x + '&param2=' + processId + '&param3=' + record;
  ans = CUF_executeServletToGetOrder(sServletName, arg);
  return ans;
}

function CUF_addAnswerFromServerToTab(ans, option) {
  r1 = '';
  r2 = '';
  r3 = '';
  r4 = '';
  r5 = '';
  r6 = '';
  ttt = ans.getElementsByTagName('value');
  if (ans.getElementsByTagName('value')[0] != null) {
    if (ans.getElementsByTagName('value')[0] != null && ans.getElementsByTagName('value')[0].firstChild.data != 'null')
      r1 = ans.getElementsByTagName('value')[0].firstChild.data;
    if (ans.getElementsByTagName('value')[1] != null && ans.getElementsByTagName('value')[1].firstChild.data != 'null')
      r2 = ans.getElementsByTagName('value')[1].firstChild.data;
    if (ans.getElementsByTagName('value')[2] != null && ans.getElementsByTagName('value')[2].firstChild.data != 'null')
      r3 = ans.getElementsByTagName('value')[2].firstChild.data;
    if (ans.getElementsByTagName('value')[3] != null && ans.getElementsByTagName('value')[3].firstChild.data != 'null')
      r4 = ans.getElementsByTagName('value')[3].firstChild.data;
    if (ans.getElementsByTagName('value')[4] != null && ans.getElementsByTagName('value')[4].firstChild.data != 'null')
      r5 = ans.getElementsByTagName('value')[4].firstChild.data;
    if (ans.getElementsByTagName('value')[5] != null && ans.getElementsByTagName('value')[5].firstChild.data != 'null')
      r6 = ans.getElementsByTagName('value')[5].firstChild.data;
    if (option == 'search') {
      CUF_addOrderValueToTab3(CUF_toTable(r1), CUF_toTable(r2), CUF_toTable(r3), CUF_toTable(r4), CUF_toTable(r5), CUF_toTable(r6));
    } else {
      CUF_addOrderValueToTab2(CUF_toTable(r1), CUF_toTable(r2), CUF_toTable(r3), CUF_toTable(r4), CUF_toTable(r5), CUF_toTable(r6));
    }
  }
}

function CUF_addOrderValueToTab2(t1, t2, t3, t4, t5, t6) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore');
  var tSize = t1.length;

  try {
    oStore.autoSave = false;

    for (var nRow = 0; nRow < tSize; nRow = nRow + 1) {
      var oNewRecord = new oStore.recordType({});
      oNewRecord.set('nazwa_tab', t1[nRow]);
      oNewRecord.set('kod_dostawcy_tab', t2[nRow]);
      oNewRecord.set('towar', t3[nRow]);
      oNewRecord.set('ilosc', t4[nRow]);
      oNewRecord.set('cena_jednostkowa', t5[nRow]);
      oNewRecord.set('wartosc_pozycji', t6[nRow]);
      if (t6[nRow] != '') {
        oStore.add(oNewRecord);
      }
    }

    oStore.save();
  } catch (oException) {
    alert(oException);
  } finally {
    oStore.autoSave = true;
  }
}

function CUF_addOrderValueToTab3(t1, t2, t3, t4, t5, t6) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_2_JsonStore');
  var tSize = t1.length;

  try {
    oStore.autoSave = false;

    for (var nRow = 0; nRow < tSize; nRow = nRow + 1) {
      var oNewRecord = new oStore.recordType({});
      oNewRecord.set('nazwa_tab_search', t1[nRow]);
      oNewRecord.set('kod_dostawcy_tab_search', t2[nRow]);
      oNewRecord.set('towar_search', t3[nRow]);
      oNewRecord.set('ilosc_search', t4[nRow]);
      oNewRecord.set('cena_jednostkowa_search', t5[nRow]);
      oNewRecord.set('wartosc_pozycji_search', t6[nRow]);
      if (t6[nRow] != '') {
        oStore.add(oNewRecord);
      }
    }

    oStore.save();
  } catch (oException) {
    alert(oException);
  } finally {
    oStore.autoSave = true;
  }
}

function CUF_toTable(val) {
  var tab = '';
  if (val != null) {
    tab = val.split(';');
  }
  return tab;
}

function CUF_uploadInvoiceTable(oJsonSore1, oJsonSore2, numer_zamowienia, status_podpiecia) {
  var nr_zamowienia = '';
  var wartosc_zamowienia = '';
  var nazwa_tab = '';
  var kod_dostawcy_tab = '';
  var towar = '';
  var ilosc = '';
  var cena_jednostkowa = '';
  var wartosc_pozycji = '';

  for (var i = 0; i < oJsonSore1.getCount(); i = i + 1) {
    var row = oJsonSore1.getAt(i);
    nr_zamowienia = nr_zamowienia + row.get('nr_zamowienia') + ';';
    wartosc_zamowienia = wartosc_zamowienia + row.get('wartosc_zamowienia_tab') + ';';
  }
  for (var i = 0; i < oJsonSore2.getCount(); i = i + 1) {
    var row = oJsonSore2.getAt(i);
    nazwa_tab = nazwa_tab + row.get('nazwa_tab') + ';';
    kod_dostawcy_tab = kod_dostawcy_tab + row.get('kod_dostawcy_tab') + ';';
    towar = towar + row.get('towar') + ';';
    ilosc = ilosc + row.get('ilosc') + ';';
    cena_jednostkowa = cena_jednostkowa + row.get('cena_jednostkowa') + ';';
    wartosc_pozycji = wartosc_pozycji + row.get('wartosc_pozycji') + ';';
  }

  return CUF_changeOrderStatusInOrderProcess(
    numer_zamowienia,
    status_podpiecia,
    nr_zamowienia.substring(0, nr_zamowienia.length - 1),
    wartosc_zamowienia.substring(0, wartosc_zamowienia.length - 1),
    nazwa_tab.substring(0, nazwa_tab.length - 1),
    kod_dostawcy_tab.substring(0, kod_dostawcy_tab.length - 1),
    towar.substring(0, towar.length - 1),
    ilosc.substring(0, ilosc.length - 1),
    cena_jednostkowa.substring(0, cena_jednostkowa.length - 1),
    wartosc_pozycji.substring(0, wartosc_pozycji.length - 1)
  );
}

function CUF_changeOrderStatusInOrderProcess(
  sOrderId,
  sOrderStatus,
  sNrZamowienia,
  sWartoscZam,
  sNazwa,
  sKod,
  sTowar,
  sIlosc,
  sCenaJedn,
  sWartoscPoz
) {
  var sServletName = 'CUF.ChangeOrderStatusNotCommon.customServlet';

  var arg =
    '&param=' +
    sOrderId +
    '&param2=' +
    sOrderStatus +
    '&param3=' +
    activityId +
    '&param4=' +
    processId +
    '&param5=' +
    sNrZamowienia +
    '&param6=' +
    sWartoscZam +
    '&param7=' +
    sNazwa +
    '&param8=' +
    sKod +
    '&param9=' +
    sTowar +
    '&param10=' +
    sIlosc +
    '&param11=' +
    sCenaJedn +
    '&param12=' +
    sWartoscPoz;

  var ans = CUF_executeServlet(sServletName, arg);
  return ans;
}

function CUF_removeFromInvoiceTable(oJsonSore, record, index) {
  try {
    var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore');
    var numer_zamowienia = record.get('nr_zamowienia');
    var status = CUF_uploadInvoiceTable(oJsonSore, Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore'), numer_zamowienia, 'NIE');

    if (status != 'true') {
      document.getElementById('nr_zamowienia_formularz').value = '';
      document.getElementById('wartosc_zamowienia').value = '';
      oStore.autoSave = false;
      while (oStore.getCount() > 0) {
        var x = oStore.getAt(oStore.getCount() - 1);
        oStore.remove(x);
      }
      oStore.save();
      oStore.autoSave = true;
      return false;
    } else {
      return true;
    }
  } catch (oError) {
    alert(oError);
    return false;
  }
}

function CUF_setData(zmienna1) {
  var oStore = null;
  if (zmienna1 == 'search') {
    oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_2_JsonStore');
  } else {
    oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore');
  }

  if (oStore.getCount() == 0 && zmienna1 == 'search') {
    CUF_setTable('search');
  } else {
    oStore.autoSave = false;
    while (oStore.getCount() > 0) {
      var x = oStore.getAt(oStore.getCount() - 1);
      oStore.remove(x);
    }
    oStore.save();
    oStore.autoSave = true;
  }

  return true;
}

function CUF_erase() {
  if (document.getElementById('nazwa_tab_dodaj') != null) {
    document.getElementById('nazwa_tab_dodaj').value = '';
  }
  if (document.getElementById('kod_dostawcy_tab_dodaj') != null) {
    document.getElementById('kod_dostawcy_tab_dodaj').value = '';
  }
  if (document.getElementById('towar_dodaj') != null) {
    document.getElementById('towar_dodaj').value = '';
  }
  if (document.getElementById('ilosc_dodaj') != null) {
    document.getElementById('ilosc_dodaj').value = '';
  }
  if (document.getElementById('cena_dodaj') != null) {
    document.getElementById('cena_dodaj').value = '';
  }
  if (document.getElementById('wartosc_dodaj') != null) {
    document.getElementById('wartosc_dodaj').value = '';
  }
}

function CUF_writeInTable(oJsonStore, sAction, oRs, tRes, rs) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  if (sAction == 'destroy' && tRes.success == true && oStore.getCount() > 0) {
    CUF_setTable('supplier');
    oJsonStore.reload();
  }
  if (document.getElementById('wartosc_zamowienia') != null && (sAction == 'create' || sAction == 'update') && oJsonStore.getCount() > 0) {
    CUF_setWartoscZamowienia(oJsonStore);
  }
}

function CUF_writeInTableRecord(oJsonStore, sAction, oRs, tRes, rs) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  if (sAction == 'destroy' && tRes.success == true && oStore.getCount() > 0) {
    CUF_setTable('record');
    oJsonStore.reload();
  }
}

function CUF_writeInTableSearch(oJsonStore, sAction, oRs, tRes, rs) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  if (sAction == 'destroy' && tRes.success == true && oStore.getCount() > 0) {
    CUF_setTable('search');
    oJsonStore.reload();
  }
}

function CUF_setWartoscZamowienia(oJsonStore) {
  if (oJsonStore.getCount() > 0) {
    document.getElementById('wartosc_zamowienia').value = CUF_formatAmountString(oJsonStore.sum('wartosc_pozycji'));
  } else {
    document.getElementById('wartosc_zamowienia').value = '';
  }
}

function CUF_setTable(zmienna1) {
  var nr_zam = document.getElementById('nr_zamowienia_formularz').value;
  var nazwa = '';
  var kod = '';
  var towar = '';
  var ilosc = '';
  var cena = '';
  var wartosc = '';

  if (document.getElementById('nazwa_tab_dodaj') != null) {
    towar = document.getElementById('nazwa_tab_dodaj').value;
  }
  if (document.getElementById('kod_dostawcy_tab_dodaj') != null) {
    towar = document.getElementById('kod_dostawcy_tab_dodaj').value;
  }
  if (document.getElementById('towar_dodaj') != null) {
    towar = document.getElementById('towar_dodaj').value;
  }
  if (document.getElementById('ilosc_dodaj') != null) {
    ilosc = document.getElementById('ilosc_dodaj').value;
  }
  if (document.getElementById('cena_dodaj') != null) {
    cena = document.getElementById('cena_dodaj').value;
  }
  if (document.getElementById('wartosc_dodaj') != null) {
    wartosc = document.getElementById('wartosc_dodaj').value;
  }

  var rekord = nazwa + ';' + kod + ';' + towar + ';' + ilosc + ';' + cena + ';' + wartosc;

  var ans = CUF_getOrderValueServlet(nr_zam, zmienna1, rekord);
  CUF_addAnswerFromServerToTab(ans, zmienna1);

  if (zmienna1 == 'record') {
    CUF_erase();
  }
  if (zmienna1 != 'search') {
    CUF_uploadInvoiceTable(
      Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore'),
      Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore'),
      nr_zam,
      'TAK'
    );
  }
}

function CUF_SetDynamicTableGrid(oEditorGridPanel) {
  oEditorGridPanel.bottomToolbar.remove(oEditorGridPanel.bottomToolbar[0]);
  oEditorGridPanel.bottomToolbar.remove(oEditorGridPanel.bottomToolbar[1]);
  oEditorGridPanel.keys = null;
}

function CUF_aktualizujWartZam(tabela) {
  var oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore');
  oStore.autoSave = false;
  var x = oStore.getAt(oStore.getCount() - 1);
  oStore.remove(x);
  oStore.save();
  oStore.autoSave = true;
  var order = document.getElementById('nr_zamowienia_formularz').value;
  var platnosc = document.getElementById('warunek_platnosci').value;
  oStore = Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore');
  var amount = CUF_formatAmountString(oStore.sum('wartosc_pozycji'));
  document.getElementById('wartosc_zamowienia').value = amount;

  if (CUF_CheckIfOrderExistInDynamicTable(order) != 'FALSE') {
    CUF_addOrderToTab0(order, amount, platnosc);
    CUF_uploadInvoiceTable(
      Ext.StoreMgr.get('data_dynamic_tabel_div_0_JsonStore'),
      Ext.StoreMgr.get('data_dynamic_tabel_div_1_JsonStore'),
      order,
      'TAK'
    );
  }

  var sServletName = 'CUF.UpdateWartZamNotCommon.customServlet';
  var arg = '&param=' + activityId + '&param1=' + processId + '&param2=' + order + '&param3=' + amount + '&param4=' + tabela;
  CUF_executeServlet(sServletName, arg);
}

function CUF_saveVarsAndShowStatus() {
  var ans = CUF_saveAllVariablesDisplayStatus();
  document.getElementById('save_variables_status').value = ans;
}

function CUF_exec(nazwa, arg) {
  try {
    xmlhttp = new XMLHttpRequest();
    xmlhttp.overrideMimeType('text/xml');
  } catch (e) {
    try {
      xmlhttp = new ActiveXObject('Microsoft.XMLHTTP');
    } catch (e) {
      alert('XMLHttpRequest error: ' + e);
    }
  }
  xmlhttp.open('POST', nazwa, false);
  xmlhttp.setRequestHeader('charset', 'utf-8');
  xmlhttp.setRequestHeader('Content-type', 'application/x-www-form-urlencoded');
  xmlhttp.send(arg);
  try {
    if (xmlhttp.responseText) {
      return eval('(' + xmlhttp.responseText + ')');
    }
  } catch (err) {}
}

function CUF_showMsg(msg) {
  Ext.Msg.show({
    title: CUF.t('message'),
    msg: msg,
    maxWidth: 400,
    buttons: Ext.Msg.OK,
    icon: Ext.Msg.WARNING,
  });
}

function CUF_checkCommentWindow(cfg) {
  CUF.CommentService.checkCommentWindow(cfg);
}
function CUF_executeFormAction(actionName, checkForm) {
  bSaveForm = true;
  iAdditionalSubmitCheck = false;
  var sValidationLoadinMaskId = 'validationMaskId';
  CUF_showLoadingMask(sValidationLoadinMaskId, CUF.t('data-validating'));
  if (authorizeUser(false)) {
    document.taskForm.action = 'AcceptActivity.do?actionName=' + actionName + '&checkForm=' + checkForm;
    submitTaskForm();
  } else {
    removeLoadingMask(sValidationLoadinMaskId);
  }
}

function CUF_enableSortConfig(grid, config) {
  grid.addTool({
    id: 'gear',
    handler: function () {
      CUF_showSortConfigWindow(grid, config);
    },
  });
  var sortInfo = CUF_getSortInfo(config.tableId, processId, activityId);
  if (!Ext.isEmpty(sortInfo)) {
    grid.getView().updateHeaderSortState = CUF_updateHeaderSortState;
    CUF_mutliSortGrid(grid, sortInfo.sortData);
  }
}

function CUF_updateHeaderSortState() {
  var st = this.ds;
  if (st.hasMultiSort == true) {
    var sorters = st.multiSortInfo.sorters;
    for (var i = 0; i < sorters.length; i++) {
      var sortColumn = this.cm.findColumnIndex(sorters[i].field);
      if (sortColumn != -1) {
        var sortDir = sorters[i].direction;
        CUF_markSort(sortColumn, sortDir, this);
      }
    }
  } else {
    var state = this.ds.getSortState();
    if (!state) {
      return;
    }

    if (!this.sortState || this.sortState.field != state.field || this.sortState.direction != state.direction) {
      this.grid.fireEvent('sortchange', this.grid, state);
    }

    this.sortState = state;

    var sortColumn = this.cm.findColumnIndex(state.field);
    if (sortColumn != -1) {
      var sortDir = state.direction;
      this.updateSortIcon(sortColumn, sortDir);
    }
  }
}
function CUF_getSortInfo(tableId, processId, activityId, showAlert) {
  var msg = CUF_exec(
    'com.plusmpm.servlet.extension.CUF.GetSortInfo.customServlet',
    'processId=' + processId + '&activityId=' + activityId + '&tableId=' + tableId
  );
  if (showAlert == false) {
    return msg;
  } else {
    if (msg.status == -1) {
      Ext.Msg.alert(CUF.t('error'), msg.errorMsg);
      return null;
    } else {
      if (!Ext.isEmpty(msg.data)) {
        msg.data.sortData = Ext.util.JSON.decode(msg.data.sortData);
      }
      return msg.data;
    }
  }
}
function CUF_showSortConfigWindow(grid, config) {
  var tableId = config.tableId;
  var cols = [];
  var win = null;
  var cm = grid.getColumnModel();
  var colCount = cm.getColumnCount();
  var sorters = [];
  if (grid.getStore().hasMultiSort == true) {
    sorters = grid.getStore().multiSortInfo.sorters;
  } else if (!Ext.isEmpty(grid.getStore().getSortState())) {
    sorters.push(grid.getStore().getSortState());
  }
  for (var i = 0; i < colCount; i++) {
    var col = cm.getColumnAt(i);
    col.direction = undefined;
    if (col.id != 'numberer' && col.hidden == false) {
      if (sorters.length > 0) {
        for (var j = 0; j < sorters.length; j++) {
          if (sorters[j].field != col.dataIndex) {
            cols.push(col);
          } else {
            col.direction = sorters[j].direction;
            cols.push(col);
          }
        }
      } else {
        cols.push(col);
      }
    }
  }
  for (var i = sorters.length - 1; i >= 0; i--) {
    for (var j = 0; j < cols.length; j++) {
      if (sorters[i].field == cols[j].dataIndex) {
        var tmp = cols[j];
        cols.splice(j, 1);
        cols.unshift(tmp);
        break;
      }
    }
  }
  var store = new Ext.data.JsonStore({
    fields: ['header', 'dataIndex', 'direction', 'dirDivId'],
    data: cols,
  });
  var colGrid = new Ext.grid.GridPanel({
    id: 'CUF_sortConfigGrid',
    flex: 1,
    ddGroup: 'GridDDGroup',
    ddScope: this,
    enableDragDrop: true,
    store: store,
    viewConfig: {
      markDirty: false,
      forceFit: true,
    },
    listeners: {
      rowclick: function (grid, rowIdx, e) {
        var selRec = grid.selectedRowIdx;
        if (rowIdx == selRec) {
          CUF_reverseSortDir(grid.getStore().getAt(rowIdx).get('dirDivId'));
        }
        grid.selectedRowIdx = rowIdx;
      },
      render: function (grid) {
        new Ext.dd.DropTarget(grid.getEl(), {
          ddGroup: 'GridDDGroup',
          copy: false,
          notifyDrop: function (dd, e, data) {
            var store = grid.store;
            var sm = grid.getSelectionModel();
            var rows = sm.getSelections();
            var dragData = dd.getDragData(e);
            if (dragData) {
              var dindex = dragData.rowIndex;
              if (typeof dindex != 'undefined') {
                store.remove(store.getById(rows[0].id));
                store.insert(dindex, data.selections);
                sm.clearSelections();
                grid.getView().refresh(false);
              }
            }
          },
        });
      },
    },
    sm: new Ext.grid.RowSelectionModel({
      singleSelect: true,
      listeners: {
        beforerowselect: function (sm, rowIdx) {
          sm.grid.selectedRowIdx = sm.grid.getStore().indexOf(sm.getSelected());
        },
      },
    }),
    columns: [
      new Ext.grid.RowNumberer({ width: 20 }),
      { header: CUF.t('column-header'), dataIndex: 'header' },
      {
        header: CUF.t('sort-direction'),
        dataIndex: 'direction',
        width: 130,
        fixed: true,
        renderer: function (v, metaData, record, rowIndex, colIndex, store) {
          var id = 'row-' + rowIndex + '-' + Ext.id();
          record.data['dirDivId'] = id;
          if (Ext.isEmpty(v)) {
            return (
              '<div style="width:100%;" title="' +
              CUF.t('sort-direction-set-click') +
              '" align="center"><div onclick="CUF_reverseSortDir(this)" id="' +
              id +
              '" class="x-CUFgrid-sortDirColNone">brak</div></div>'
            );
          } else {
            var cls;
            if (v == 'ASC') cls = 'x-CUFgrid-sortDirColAsc';
            else cls = 'x-CUFgrid-sortDirColDsc';
            return (
              '<div style="width:100%;" align="center" title="' +
              CUF.t('sort-direction-set-click') +
              '"><div onclick="CUF_reverseSortDir(this)" id="' +
              id +
              '" class="' +
              cls +
              '">&nbsp;</div></div>'
            );
          }
        },
      },
    ],
  });
  CUF_importCssForFrame('gridadditionals');
  var moveFunc = function (dir) {
    var rec = colGrid.getSelectionModel().getSelected();
    var idx = store.indexOf(rec);
    if (idx + dir < 0 || idx + dir >= store.getCount()) return;
    store.remove(rec, true);
    store.insert(idx + dir, rec);
    colGrid.getView().refreshRow(store.getAt(idx));
    colGrid.getSelectionModel().selectRow(idx + dir);
  };
  var arrows = new Ext.Container({
    layout: 'vbox',
    width: 50,
    defaults: {
      width: 40,
    },
    items: [
      new Ext.Container({ flex: 1 }),
      new Ext.Button({
        margins: { top: 5, right: 5, bottom: 0, left: 5 },
        icon: 'scripts/ext-3.2.2/resources/images/access/grid/col-move-bottom.gif',
        handler: function () {
          moveFunc(-1);
        },
      }),
      new Ext.Button({
        margins: { top: 5, right: 5, bottom: 5, left: 5 },
        icon: 'scripts/ext-3.2.2/resources/images/access/grid/col-move-top.gif',
        handler: function () {
          moveFunc(1);
        },
      }),
      new Ext.Container({ flex: 1 }),
    ],
  });
  var hpanel = new Ext.Container({
    layout: 'hbox',
    flex: 1,
    layoutConfig: {
      align: 'stretch',
      pack: 'start',
    },
    items: [colGrid, arrows],
  });

  var buttons = new Ext.Container({
    height: 50,
    layout: 'column',
    defaults: {
      width: 100,
    },
    items: [
      new Ext.Button({
        text: CUF.t('save'),
        cls: 'x-btn-text-icon',
        icon: 'style/img/fam/table_save.png',
        style: { padding: '5px' },
        handler: function () {
          var st = colGrid.getStore();
          var sortData = [];
          for (var i = 0; i < store.getCount(); i++) {
            var r = st.getAt(i);
            if (Ext.isEmpty(r.get('direction'))) continue;
            var sd = {
              field: r.get('dataIndex'),
              direction: r.get('direction'),
            };
            sortData.push(sd);
          }
          CUF_mutliSortGrid(grid, sortData);

          CUF_saveSortInfo(sortData, tableId, activityId, processId);
          win.close();
        },
      }),
      new Ext.Button({
        text: CUF.t('cancel'),
        style: { padding: '5px' },
        handler: function () {
          win.close();
        },
      }),
    ],
  });
  var vpanel = new Ext.Container({
    layout: 'vbox',

    layoutConfig: {
      align: 'stretch',
      pack: 'start',
    },
    items: [hpanel, buttons],
  });
  win = new Ext.Window({
    title: CUF.t('sort-config'),
    width: 600,
    padding: 5,
    height: 400,
    modal: true,
    layout: 'fit',
    items: [vpanel],
  });
  win.show();
}
function CUF_mutliSortGrid(grid, sortData) {
  grid.getStore().multiSort(sortData);
  CUF_clearSortMarks(grid.getView());
  for (var i = 0; i < sortData.length; i++) {
    CUF_markSort(grid.getColumnModel().findColumnIndex(sortData[i].field), sortData[i].direction, grid.getView());
  }
}
function CUF_saveSortInfo(sortData, tableId, activityId, processId, showAlert) {
  var jSortData = Ext.util.JSON.encode(sortData);
  var msg = CUF_exec(
    'com.plusmpm.servlet.extension.CUF.SaveSortInfo.customServlet',
    'processId=' + processId + '&activityId=' + activityId + '&tableId=' + tableId + '&sortData=' + jSortData
  );
  if (msg.status == -1 && showAlert != false) {
    Ext.Msg.alert(CUF.t('error'), msg.errorMsg);
  }
  return msg;
}
function CUF_reverseSortDir(div) {
  var el = Ext.get(div);
  var rowIdx = el.dom.id.split('-')[1];
  var st = Ext.getCmp('CUF_sortConfigGrid').getStore();
  var rec = st.getAt(rowIdx);
  if (el.hasClass('x-CUFgrid-sortDirColDsc')) {
    el.removeClass('x-CUFgrid-sortDirColDsc');
    el.addClass('x-CUFgrid-sortDirColNone');
    el.dom.innerHTML = 'brak';
    rec.set('direction', '');
  } else {
    if (el.hasClass('x-CUFgrid-sortDirColNone')) {
      el.removeClass('x-CUFgrid-sortDirColNone');
      el.addClass('x-CUFgrid-sortDirColAsc');
      el.dom.innerHTML = '&nbsp;';
      rec.set('direction', 'ASC');
    } else {
      el.removeClass('x-CUFgrid-sortDirColAsc');
      el.addClass('x-CUFgrid-sortDirColDsc');
      rec.set('direction', 'DESC');
    }
  }
}

function CUF_markSort(col, dir, gv) {
  var sc = gv.sortClasses;
  var hds = gv.mainHd.select('td');
  hds.item(col).addClass(sc[dir == 'DESC' ? 1 : 0]);
}
function CUF_clearSortMarks(gv) {
  var sc = gv.sortClasses;
  gv.mainHd.select('td').removeClass(sc);
}
function CUF_mytest(grid) {
  CUF_enableSortConfig(grid, { tableId: 'tbl1' });
}

CUF_base64 = {
  base64s: 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/',

  encode: function (decStr) {
    if (typeof btoa === 'function') {
      return btoa(decStr);
    }
    var base64s = this.base64s;
    var bits;
    var dual;
    var i = 0;
    var encOut = '';
    while (decStr.length >= i + 3) {
      bits = ((decStr.charCodeAt(i++) & 0xff) << 16) | ((decStr.charCodeAt(i++) & 0xff) << 8) | (decStr.charCodeAt(i++) & 0xff);
      encOut +=
        base64s.charAt((bits & 0x00fc0000) >> 18) +
        base64s.charAt((bits & 0x0003f000) >> 12) +
        base64s.charAt((bits & 0x00000fc0) >> 6) +
        base64s.charAt(bits & 0x0000003f);
    }
    if (decStr.length - i > 0 && decStr.length - i < 3) {
      dual = Boolean(decStr.length - i - 1);
      bits = ((decStr.charCodeAt(i++) & 0xff) << 16) | (dual ? (decStr.charCodeAt(i) & 0xff) << 8 : 0);
      encOut +=
        base64s.charAt((bits & 0x00fc0000) >> 18) +
        base64s.charAt((bits & 0x0003f000) >> 12) +
        (dual ? base64s.charAt((bits & 0x00000fc0) >> 6) : '=') +
        '=';
    }
    return encOut;
  },

  decode: function (encStr) {
    if (typeof atob === 'function') {
      return atob(encStr);
    }
    var base64s = this.base64s;
    var bits;
    var decOut = '';
    var i = 0;
    for (; i < encStr.length; i += 4) {
      bits =
        ((base64s.indexOf(encStr.charAt(i)) & 0xff) << 18) |
        ((base64s.indexOf(encStr.charAt(i + 1)) & 0xff) << 12) |
        ((base64s.indexOf(encStr.charAt(i + 2)) & 0xff) << 6) |
        (base64s.indexOf(encStr.charAt(i + 3)) & 0xff);
      decOut += String.fromCharCode((bits & 0xff0000) >> 16, (bits & 0xff00) >> 8, bits & 0xff);
    }
    if (encStr.charCodeAt(i - 2) == 61) {
      return decOut.substring(0, decOut.length - 2);
    } else if (encStr.charCodeAt(i - 1) == 61) {
      return decOut.substring(0, decOut.length - 1);
    } else {
      return decOut;
    }
  },
};
//------------Szablony formularza

/**
 * Wyswietala okno ladowania szablonu
 * @param cfg
 * -templatesFromAllUsers-widoczonść wszystkich użytkowników(true),lub tylko aktualnego
 * DEPRECATED
 */
function CUF_loadFormTemplateWindow(cfg) {
  config = Ext.applyIf(cfg || {}, {
    templatesFromAllUsers: false,
  });

  CUF.FormTemplateService.openTemplateLoader({
    userScope: !config.templatesFromAllUsers,
  });
}
/**
 *
 * @param cfg
 * -checkSaveTables-domyslna wartosc checkboxa 1
 * -checkDefault-domyslna wartosc checkboxa2
 * DEPRECATED
 */
function CUF_saveFormTemplateWindow(cfg) {
  config = Ext.applyIf(cfg || {}, {
    checkDefault: true,
  });

  CUF.FormTemplateService.openTemplateCreator({
    setDefault: config.checkDefault,
  });
}
function CUF_loadDefaultFormTemplate() {
  CUF.FormTemplateService.loadDefaultTemplate();
}

//----------------------------
//Pobieranie dokumentow z procesu
function CUF_getDocumentsForProcess() {
  var resp = CUF_exec(
    'com.plusmpm.servlet.extension.CUF.GetDocumentsForProcess.customServlet',
    'processId=' + processId + '&activityId=' + activityId
  );
  if (resp.status == 0) return resp.data;
  else {
    Ext.Msg.alert(CUF.t('error'), CUF.t('document-error-process-loading'));
    return -1;
  }
}
/*
 * Sprawdza czy w procesie jest jakis dokument z podanej klasy
 */
function CUF_checkProcessHasDocumentFromClass(docClassName) {
  var has = false;
  var docs = CUF_getDocumentsForProcess();

  if (docs != -1) {
    for (var i = 0; i < docs.length; i++) {
      var doc = docs[i];
      if (doc.docClassName == docClassName) {
        has = true;
        break;
      }
    }
  }
  return has;
}

//----------------Pobranie danych DC
//funkcja tworzyc argumenty zapytania na podstawie obiektu JS
function CUF_createArgs(obj) {
  var args = '';
  for (var i in obj) {
    if (obj.hasOwnProperty(i) && (typeof obj[i] == 'string' || typeof obj[i] == 'number')) {
      args += i + '=' + encodeURIComponent(obj[i]) + '&';
    }
  }
  return args.substring(0, args.length - 1);
}
//Pobiera dane z datachoosera
function CUF_getDataChooserResults(cfg) {
  if (!Ext.isEmpty(cfg.className)) {
    var args = CUF_createArgs(cfg);
    if (!Ext.isEmpty(cfg.criteria)) {
      if (!Ext.isEmpty(cfg.criteria.DCKEY)) {
        cfg.criteria.GetDataChooserContentServlet_parameter_query = cfg.criteria.DCKEY;
      }
      args += '&criteria=' + Ext.util.JSON.encode(cfg.criteria);
    }
    if (!Ext.isEmpty(cfg.customKeys)) {
      args += '&customKeys=' + Ext.util.JSON.encode(cfg.customKeys);
    }
    return CUF_exec('com.plusmpm.servlet.extension.CUF.GetDataChooserResults.customServlet', args);
  } else {
    return CUF.t('class-error-no-name');
  }
}

//---------------- WALIDACJA
function CUF_validatePESEL(pesel) {
  var resp = CUF_exec('com.plusmpm.servlet.extension.CUF.Validator.customServlet', 'value=' + pesel + '&type=pesel');
  if (resp.status == 0) {
    return resp.data;
  } else {
    Ext.Msg.alert(CUF.t('error'), CUF.t('error-validation') + ' ' + CUF.t('details') + ': ' + resp.errorMsg);
  }
}

function CUF_validateNIP(nip) {
  var resp = CUF_exec('com.plusmpm.servlet.extension.CUF.Validator.customServlet', 'value=' + nip + '&type=nip');
  if (resp.status == 0) {
    return resp.data;
  } else {
    Ext.Msg.alert(CUF.t('error'), CUF.t('error-validation') + ' ' + CUF.t('details') + ': ' + resp.errorMsg);
  }
}

function CUF_validateNRB(nrb) {
  var resp = CUF_exec('com.plusmpm.servlet.extension.CUF.Validator.customServlet', 'value=' + nrb + '&type=nrb');
  if (resp.status == 0) {
    return resp.data;
  } else {
    Ext.Msg.alert(CUF.t('error'), CUF.t('error-validation') + ' ' + CUF.t('details') + ': ' + resp.errorMsg);
  }
}

function CUF_validateIBAN(iban) {
  var resp = CUF_exec('com.plusmpm.servlet.extension.CUF.Validator.customServlet', 'value=' + iban + '&type=iban');
  if (resp.status == 0) {
    return resp.data;
  } else {
    Ext.Msg.alert(CUF.t('error'), CUF.t('error-validation') + ' ' + CUF.t('details') + ': ' + resp.errorMsg);
  }
}

//--------------------
//-------------------- Kopiowanie Pola formularza ------------------
function CUF_completeRequiredFieldsWindowTest() {
  var cfg = {
    fields: ['dc1', 'calkowita'],
    afterSave: function (allCompleted, values) {
      if (allCompleted == false) {
        CUF_completeRequiredFieldsWindowTest();
        Ext.Msg.alert(CUF.t('message'), CUF.t('fields-fill-all'));
      }
    },
  };
  CUF_completeRequiredFieldsWindow(cfg);
}
function CUF_completeRequiredFieldsWindow(cfg) {
  var items = [];
  for (var i = 0; i < cfg.fields.length; i++) {
    var varId = cfg.fields[i];
    var f = CUF_cloneField(varId);
    if (Ext.isEmpty(f.originalField.getValue())) {
      f.name = varId;
      f.fieldLabel = Ext.get('tooltip_' + varId).dom.children[0].innerHTML.unescapeHTML();

      items.push(f);
    }
  }
  if (items.length == 0) return;
  var win = new Ext.Window({
    title: CUF.t('fields-fill-required'),
    width: 500,
    closeAction: 'hide',
    height: 300,
    layout: 'fit',
    items: [
      {
        xtype: 'form',
        id: 'ee',
        labelAlign: 'top',
        frame: true,
        items: items,
      },
    ],
    buttons: [
      {
        text: CUF.t('save'),
        handler: function (btn) {
          var form = win.getComponent(0).getForm();
          var values = form.getValues();
          var allCompleted = true;
          for (var i = 0; i < cfg.fields.length; i++) {
            var varId = cfg.fields[i];
            var f = form.findField(varId);
            if (Ext.isEmpty(f)) continue;
            var val = values[varId];
            if (Ext.isEmpty(val)) {
              allCompleted = false;
              continue;
            }
            if (f.isDc == true) {
              f.originalField.setValue(val);
              var st = f.originalField.getStore();
              var idx = st.find(f.originalField.valueField, val);
              f.originalField.fireEvent('select', f.originalField.valueField, st.getAt(idx), idx);
            } else {
              f.originalField.dom.value = val;
            }
          }
          win.close();
          if (!Ext.isEmpty(cfg.afterSave)) cfg.afterSave(allCompleted, values);
        },
      },
      {
        text: CUF.t('cancel'),
        handler: function () {
          win.hide();
        },
      },
    ],
  });
  win.show();
}
function CUF_cloneField(varId) {
  var el = Ext.get(varId);
  var isDc = CUF_isDc(varId);
  if (isDc == true) {
    var cmp = CUF_getDcCmp(varId);
    var clone = cmp.cloneConfig({ applyTo: null });
    clone.originalField = cmp;
    clone.isDc = true;
    return clone;
  } else {
    return {
      value: el.getValue(),
      width: 205,
      xtype: 'textfield',
      originalField: el,
      getValue: function () {
        return this.value;
      },
    };
  }
}
function CUF_isDc(varId) {
  var el = Ext.get(varId);
  var isDc = el.parent().hasClass('x-form-field-wrap');
  return isDc;
}
function CUF_getDcCmp(varId) {
  for (var i = 0; i < Ext.ComponentMgr.all.items.length; i++) {
    try {
      var cmp = Ext.ComponentMgr.all.items[i];
      var e = cmp.el;
      if (e.id == varId) {
        return cmp;
      }
    } catch (err) {}
  }
  return null;
}
//-----------------------

//------ Zmiana rozmiaru pól formularza------

function CUF_setFieldsWidth(width) {
  var fd = CUF_getFieldsData();
  for (var i = 0; i < fd.length; i++) {
    if (!Ext.isEmpty(fd[i].type)) {
      if (fd[i].type == 'CHECKBOX' || fd[i].type == 'RADIOBUTTON') {
        continue;
      } else if (fd[i].type == 'TEXTAREA') {
        field = Ext.get(fd[i].id);
        var parent = field.parent();
        parent.setWidth(width + 5);
      }
    }
    var id = fd[i].id;
    var field;
    if (CUF_isDc(id) == true) {
      field = CUF_getDcCmp(id);
    } else {
      field = Ext.get(id);
    }
    field.setWidth(width);
  }
}
//--------------------------------------------

//---- EKSPORT IMPORT TABELI DYNAMICZNEJ Z OBSLUGA XLSX i XLS------------------
function CUF_getTableDefinition(grid, showHidden) {
  return CUF.table.core.TableService.getTableDefinition(grid, showHidden);
}
function CUF_exportDTSchemeToExcel(btn, config) {
  CUF.table.core.TableService.exportSchemeToExcel(btn.up('variableset'), config);
}

function CUF_exportDTToExcel(btn, config) {
  config.includeHiddenColumns = config.showHidden;
  CUF.table.core.TableService.exportDataToExcel(btn.up('variableset'), config);
}
function CUF_importDTFromExcel(btn, config) {
  config.includeHiddenColumns = config.showHidden;
  CUF.table.core.TableService.importDataFromExcel(btn.up('variableset'), config);
  return;
  try {
    //CUF_importCssForFrame( 'fileuploadfield' );
    var showHidden = config.showHidden;

    var fileButtonLabel = config.fileButtonLabel;
    var loadButtonLabel = config.loadButtonLabel;
    var windowTitle = config.windowTitle;
    var functionName = config.functionName;
    var notSaveStoreAfterLoad = config.notSaveStoreAfterLoad;
    var fs = null;
    var oGrid = oButton.up('grid');
    var oStore = oGrid.getStore();
    var colsId = new Array();
    var colsMapping = new Array();
    var index = 0;

    if (showHidden == undefined) {
      showHidden = false;
    }
    if (!fileButtonLabel || fileButtonLabel == '') {
      fileButtonLabel = CUF.t('choose-file');
    }
    if (!loadButtonLabel || loadButtonLabel == '') {
      loadButtonLabel = CUF.t('enter');
    }
    if (!windowTitle || windowTitle == '') {
      windowTitle = CUF.t('file-xls-loading');
    }

    var tableDefinition = CUF_getTableDefinition(oGrid, showHidden);

    var fs = {
      xtype: 'form',
      frame: true,
      bodyStyle: 'padding: 10px 10px 0 10px;',
      items: [
        {
          xtype: 'filefield',
          emptyText: CUF.t('file-excel-choose'),
          fieldLabel: CUF.t('file-excel'),
          name: 'file',
          buttonText: fileButtonLabel,
          width: 400,
          labelWidth: 150,
          alowBlank: false,
          padding: '0px 1px 0px 0px',
          id: 'Xls_panel_label',
        },
        {
          id: 'Xls_panel_checkbox',
          xtype: 'checkbox',
          labelWidth: 150,
          name: 'override',
          fieldLabel: CUF.t('clean-current-data'),
          checked: false,
        },
      ],
      buttons: [
        {
          id: 'CUF_import_from_xls_to_dt_button',
          text: loadButtonLabel,
          cls: 'x-btn-text-icon',
          icon: 'style/img/fam/table_edit.png',
          handler: function (btn) {
            var form = btn.up('form').getForm();
            var overrideStore = form.getFieldValues().override;

            var servletArgs = colsMapping.join(';');
            servletArgs = encodeURI(servletArgs);

            var headers = CUF_utils.encode(tableDefinition.headers);
            headers = Ext.util.JSON.encode(headers);
            var params = CUF_utils.buildParams({
              processId: processId,
              headers: headers,
            });

            form.submit({
              url: 'CUF.ImportDTFromExcel.customServlet?' + params,
              waitMsg: CUF.t('file-processing'),
              waitTitle: CUF.t('please-wait'),
              success: function (form, action) {
                win.close();
                var data = action.result.data;
                if (Ext.isEmpty(data)) return;
                var mappings = {};
                for (var i in data[0]) {
                  if (data[0].hasOwnProperty(i)) {
                    var h = i;

                    for (var j in tableDefinition.headerMappings) {
                      if (tableDefinition.headerMappings.hasOwnProperty(j) && h == tableDefinition.headerMappings[j]) {
                        mappings[h] = j;
                        break;
                      }
                    }
                  }
                }
                CUF_log(data);
                CUF_log(mappings);
                CUF_setStoreData(oStore, data, mappings, overrideStore, notSaveStoreAfterLoad, oGrid);
                if (functionName) {
                  try {
                    window.setTimeout(function () {
                      window[functionName](oStore);
                    }, 500);
                  } catch (e) {}
                }
              },
              failure: function (form, action) {
                var msg = action.result.errorMsg;
                if (/^Niezn/.test(msg)) {
                  msg = CUF.t('file-error-processing') + ' \n' + CUF.t('check-file-format-columns');
                }
                Ext.Msg.alert(CUF.t('error'), msg);
              },
            });
          },
        },
      ],
    };

    win = Ext4.widget('window', {
      layout: 'fit',
      height: 160,
      width: 530,
      shadow: false,
      closable: true,
      constrain: true,
      title: windowTitle,
      draggable: true,
      plain: true,
      resizable: false,
      modal: true,
      items: [fs],
    });

    win.show();
  } catch (e) {
    CUF_log(e);
  }
}
function CUF_setStoreData(store, data, mappings, override, notSaveStoreAfterLoad, grid) {
  if (override == true) {
    ServiceFactory.getVariableSetService().clear(grid);
  }
  var records = [];
  for (var i = 0; i < data.length; i++) {
    var r = {};
    var d = data[i];
    for (var j in d) {
      if (d.hasOwnProperty(j)) {
        r[mappings[j]] = d[j];
      }
    }
    records.push(r);
  }
  ServiceFactory.getVariableSetService().addRow(grid, false, records);
}
function CUF_importTest(oButton) {
  var config = new Object();
  config.showHidden = true;
  config.fileButtonLabel = 'Choose';
  config.loadButtonLabel = 'Load';
  config.windowTitle = 'Load from xls';
  config.functionName = 'CUF_message';
  config.notSaveStoreAfterLoad = false;
  CUF_showExcelImportWin(oButton, config);
}
//------------------------------------------------------------------------------

//------------PRZYDATNE FUNKCJE----------------
CUF_utils = {
  buildParams: function (params) {
    var sParams = '';
    for (var i in params) {
      if (params.hasOwnProperty(i)) {
        sParams += '&' + i + '=' + params[i];
      }
    }
    return sParams;
  },
  encode: function (data) {
    if (data instanceof Array) {
      var ret = [];
      for (var i = 0; i < data.length; i++) {
        ret[i] = this.encode(data[i]);
      }
      return ret;
    } else if (data instanceof Date) {
      return data.format('Y-m-d h:i:s');
    } else if (data instanceof Object) {
      var ret = {};
      for (var i in data) {
        if (data.hasOwnProperty(i)) {
          ret[i] = this.encode(data[i]);
        }
      }
      return ret;
    } else {
      return encodeURIComponent(data);
    }
  },
  getContextPath: function () {
    var link = document.location.href;
    var lastSymbol = link.lastIndexOf('/');
    if (lastSymbol != 0 && lastSymbol != -1) {
      link = link.substring(0, lastSymbol);
    }
    return link;
  },
  getStoreData: function (store) {
    var data = [];
    for (var i = 0; i < store.getCount(); i++) {
      data.push(this.clone(store.getAt(i).data));
    }
    return data;
  },
  clone: function (obj) {
    var cl = {};
    for (var i in obj) {
      if (obj.hasOwnProperty(i)) {
        cl[i] = obj[i];
      }
    }
    return cl;
  },
};
//-------------------------------------

//-----------Drzewko histori zadania----------------
function CUF_showHistoryTreeTest() {
  var params = {
    activityId: activityId,
    processId: processId,
    className: 'com.plusmpm.CUF.util.extension.historyTree.test.HistoryImpl',
  };
  CUF_showHistoryTree(params);
}
function CUF_showHistoryTree(params) {
  var sParams = CUF_utils.buildParams(params);
  Viewer(CUF_utils.getContextPath() + '/pages/CUF/historyTree/historyTree.jsp?' + sParams, false);
}
//--------------------------------------------------
