String.prototype.endsWith || (String.prototype.endsWith = function (t, n) {
    return (void 0 === n || n > this.length) && (n = this.length), this.substring(n - t.length, n) === t
}); //string.endsWith polyfill

window.TableStore = {
    create: function (parameter) {
        try {
            var tableStore = Array.isArray(parameter) ? createFromVariables(parameter) : createFromTableId(parameter);
            return JSON.stringify(tableStore);
        } catch (error) {
            console.error(error);
        }
    },
    createWithNewKeyMapping: function (parameter, keyMapping) {
        try {
            var tableStore = Array.isArray(parameter) ? createFromVariablesWithNewKeyMapping(parameter, keyMapping) : createFromTableIdWithNewKeyMapping(parameter);
            return JSON.stringify(tableStore);
        } catch (error) {
            console.error(error);
        }
    },
    length: function (tableStoreStr) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            if (Array.isArray(tableStore.data)) return tableStore.data.length;
            throw new Error("TableStore object is not an array!");
        } catch (err) {
            console.error("Cannot calculate length (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    keys: function (tableStoreStr) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            if (!Array.isArray(tableStore.data) || tableStore.data.length <= 0) return JSON.stringify({});
            var record = tableStore.data.shift();
            var keys = [];
            for (var key in record) {
                if (Object.prototype.hasOwnProperty.call(record, key)) {
                    keys.push(key);
                }
            }
            return keys;
        } catch (err) {
            console.error("Cannot obtain keys (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    getFieldValues: function (tableStoreStr, fieldId) {
        var tableStore = JSON.parse(tableStoreStr);
        if (!Array.isArray(tableStore.data) || tableStore.data.length <= 0) return "";
        var values = [];
        for (var i = 0; i < tableStore.data.length; i++) {
            if (tableStore.data[i][fieldId] === undefined) {
                throw new Error("Record " + i + " of TableStore does not contain a field with id: " + fieldId);
            }
            values.push(tableStore.data[i][fieldId]);
        }
        return values;
    },

    joinField: function (tableStoreStr, fieldId, separator) {
        try {
            var valuesToJoin = TableStore.getFieldValues(tableStoreStr, fieldId);
            return valuesToJoin.join(separator);
        } catch (err) {
            console.error("Cannot obtain fields (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    addRecord: function (tableStoreStr, fieldIds, values) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            if (!Array.isArray(tableStore.data)) throw new Error("TableStore is not an array!");
            if (fieldIds.every(function (id) {
                return Object.keys(tableStore.variableType).indexOf(id) > -1;
            })) {
                var record = {};
                for (var id in tableStore.variableType) {
                    if (tableStore.variableType.hasOwnProperty(id)) {
                        var i = fieldIds.indexOf(id);
                        switch (tableStore.variableType[id]) {
                            case "integer":
                            case "float":
                                record[id] = i > -1 ? Number(values[i]) : 0;
                                break;
                            case "string":
                                record[id] = i > -1 ? values[i] : "";
                                break;
                            case "date":
                                record[id] = i > -1 ? values[i] : null;
                                break;
                            default:
                                throw new Error("Invalid type " + tableStore.variableType[id] + " of variable " + id);
                        }
                    }
                }
                tableStore.data.push(record);
                return JSON.stringify(tableStore);
            }
            throw new Error("Keys of record to add ([" + fieldIds + "]) are not applicable with TableStore keys ([" + Object.keys(tableStore.data[0]) + "])!");
        } catch (err) {
            console.error("Cannot add record (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    addRecords: function (tableStoreStr, tableStoreToAddStr) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            var tableStoreToAdd = JSON.parse(tableStoreToAddStr);
            if (!Array.isArray(tableStore.data) || !Array.isArray(tableStoreToAdd.data)) throw new Error("Both objects must be arrays!");
            if (Object.keys(tableStoreToAdd.variableType).some(function (id) {
                return tableStore.variableType[id] === undefined || tableStore.variableType[id] !== tableStoreToAdd.variableType[id];
            })) {
                throw new Error("TableStores variable types mismatch!");
            }
            var getDefaultValue = function (type) {
                switch (type) {
                    case "integer":
                    case "float":
                        return 0;
                    case "date":
                        return null;
                    case "string":
                        return "";
                }
            };
            tableStoreToAdd.data.forEach(function (recordToAdd) {
                var record = {};
                Object.keys(tableStore.variableType).forEach(function (id) {
                    record[id] = recordToAdd[id] ? recordToAdd[id] : getDefaultValue(tableStore.variableType[id]);
                });
                tableStore.data.push(record);
            });
            return JSON.stringify(tableStore);
        } catch (err) {
            console.error("Cannot add records (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    filter: function (tableStoreStr, logicalOperator, fieldIds, operators, values) {
        try {
            if (!fieldIds.length || !operators.length || !values.length) return tableStoreStr;
            var tableStore = JSON.parse(tableStoreStr);
            if (!Array.isArray(tableStore.data)) throw new Error("Cannot filter non-array object!");
            tableStore.data = tableStore.data.filter(function (record) {
                var conditionalResults = [];
                for (var i = 0; i < fieldIds.length; i++) {
                    var id = fieldIds[i];
                    var value = values[i];
                    var valueFromTableStoreToCompare = record[id];
                    if (value.length === 0 && tableStore.variableType[id] === "date") {
                        value = null;
                    } else if (tableStore.variableType[id] === "date") {
                        value = new Date(value);
                        valueFromTableStoreToCompare = new Date(valueFromTableStoreToCompare);
                    }
                    if (tableStore.variableType[id] === "integer" || tableStore.variableType[id] === "float") {
                        value = Number(value);
                    }
                    switch (operators[i]) {
                        case "=":
                            conditionalResults.push(valueFromTableStoreToCompare === value);
                            break;
                        case "!=":
                            conditionalResults.push(valueFromTableStoreToCompare !== value);
                            break;
                        case "<":
                            conditionalResults.push(valueFromTableStoreToCompare < value);
                            break;
                        case "<=":
                            conditionalResults.push(valueFromTableStoreToCompare <= value);
                            break;
                        case ">":
                            conditionalResults.push(valueFromTableStoreToCompare > value);
                            break;
                        case ">=":
                            conditionalResults.push(valueFromTableStoreToCompare >= value);
                            break;
                        case "regex":
                            conditionalResults.push(new RegExp(value).test(valueFromTableStoreToCompare));
                            break;
                        default:
                            throw new Error("Invalid operator in row: #" + i + ".");
                    }
                }
                if (logicalOperator.toLowerCase() === "and") {
                    return conditionalResults.every(function (item) {
                        return item
                    });
                } else {
                    return conditionalResults.some(function (item) {
                        return item
                    });
                }
            });
            return JSON.stringify(tableStore);
        } catch (err) {
            console.error("Cannot filter (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    getItem: function (tableStoreStr, fieldId, elementNr, type) {
        var tableStore = JSON.parse(tableStoreStr);
        if (!Array.isArray(tableStore.data)) throw new Error("TableStore is not an array!");
        if (tableStore.data[elementNr] === undefined) {
            throw new Error("TableStore contains " + tableStore.data.length + " rows. Cannot get element of #" + (elementNr + 1) + " row!");
        }
        var item = tableStore.data[elementNr][fieldId];
        if (item !== null && type === "date") {
            var date = new Date(item);
            date.setHours(0);
            date.setMinutes(0);
            date.setSeconds(0);
            return date;
        }
        return item;
    },

    setItem: function (tableStoreStr, fieldId, elementNr, value) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            if (!Array.isArray(tableStore.data)) throw new Error("TableStore is not an array!");
            if (tableStore.data[elementNr] === undefined) {
                throw new Error("TableStore contains " + tableStore.data.length + " rows. Cannot modify #" + (elementNr + 1) + " row!");
            }
            if (value instanceof Date) {
                value.setHours(12);
                tableStore.data[elementNr][fieldId] = value.toISOString().slice(0, 10);
            } else {
                tableStore.data[elementNr][fieldId] = value;
            }
            return JSON.stringify(tableStore);
        } catch (err) {
            console.error("Cannot set item (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    setItems: function (tableStoreStr, elementNr, fieldIds, values) {
        try {
            var tableStore = JSON.parse(tableStoreStr);
            if (!Array.isArray(tableStore.data)) throw new Error("TableStore is not an array!");
            var recordToChange = tableStore.data[elementNr];
            if (!recordToChange) {
                throw new Error("TableStore contains " + tableStore.data.length + " rows. Cannot modify #" + (elementNr + 1) + " row!");
            }
            var getProperTypedValue = function (type, value) {
                switch (type) {
                    case "string":
                        return value;
                    case "date":
                        return new Date(value).toISOString().slice(0, 10);
                    case "integer":
                    case"float":
                        return Number(value);
                    default:
                        throw new Error("Invalid type '" + type + "' of value: " + value);
                }
            }
            for (var i = 0; i < fieldIds.length; i++) {
                var id = fieldIds[i];
                if (recordToChange[id] === undefined) throw new Error("Invalid TableStore key '" + id + "'!");
                recordToChange[id] = getProperTypedValue(tableStore.variableType[id], values[i]);
            }
            return JSON.stringify(tableStore);
        } catch (err) {
            console.error("Cannot set item (invalid TableStore object provided)!");
            console.error(err);
        }
    },

    compareKeysOfTwoObjects: function (obj1, obj2) {
        var obj1Keys = Object.keys(obj1).sort();
        var obj2Keys = Object.keys(obj2).sort();
        return JSON.stringify(obj1Keys) === JSON.stringify(obj2Keys);
    },

    getVariableTypes: function (tableStoreStr) {
        var tableStore = JSON.parse(tableStoreStr);
        return tableStore.variableType;
    },

    addColumn: function (tableStoreStr, columnId, dataType, initialValues) {
        var tableStore = JSON.parse(tableStoreStr);
        if (!columnId) {
            console.error("Column ID must be specified!");
            throw new Error("Column ID must be specified!");
        }
        if (Object.keys(tableStore.variableType).includes(columnId)) {
            console.error("TableStore already contains column with id: '" + columnId + "'!"); //konieczne, bo treść poniższego błędu nie jest wyświetlana w konsoli przeglądarki
            throw new Error("TableStore already contains column with id: '" + columnId + "'!");
        }
        tableStore.variableType[columnId] = dataType.toLowerCase();
        if (initialValues !== undefined) {
            insertInitialValuesToNewColumn(tableStore.data, initialValues, columnId);
        } else {
            insertDefaultValuesToNewColumn(tableStore.data, dataType, columnId);
        }
        return JSON.stringify(tableStore);
    },

    sort: function (tableStoreStr, columnId, sortDirection) {
        if (!columnId) {
            console.error("Column ID must be specified!");
            throw new Error("Column ID must be specified!");
        }
        let tableStore = JSON.parse(tableStoreStr);
        switch (sortDirection) {
            case "ASC":
                tableStore.data.sort((o1, o2) => (o1[columnId] > o2[columnId]) ? 1 : -1);
                break;
            case "DESC":
                tableStore.data.sort((o1, o2) => (o1[columnId] < o2[columnId]) ? 1 : -1);
                break;
            default:
                throw new Error("Invalid sort direction '" + sortDirection + "'!");
        }
        return JSON.stringify(tableStore);
    },

    deleteRecord: function (tableStoreStr, elementNumber) {
        let tableStore = JSON.parse(tableStoreStr);
        if (tableStore.data.length <= elementNumber) {
            console.error("Incorrect row number!");
            throw new Error("Incorrect row number!");
        }
        tableStore.data.splice(elementNumber, 1);
        return JSON.stringify(tableStore);
    },
}

var createFromTableId = function (tableId) {
    var variableSet = VariableSetService.getVariableSet(tableId);
    if (variableSet) {
        var variables = variableSet.getVariables();
        return createFromVariables(variables);
    } else {
        console.error('Invalid VariableSet id: ' + tableId);
        alert(CUFCommon.t('action.invoking-action-error-occured'));
    }
}

var createFromTableIdWithNewKeyMapping = function (tableId, keyMapping) {
    var variableSet = VariableSetService.getVariableSet(tableId);
    if (variableSet) {
        var variables = variableSet.getVariables();
        return createFromVariablesWithNewKeyMapping(variables, keyMapping);
    } else {
        console.error('Invalid VariableSet id: ' + tableId);
        alert(CUFCommon.t('action.invoking-action-error-occured'));
    }
}
var createFromVariables = function (variables) {
    if (!variables.every(function (variable) {
        return Array.isArray(variable.getValue());
    })) {
        throw new Error("Variables passed into TableStore.create function must be arrays!");
    }
    if (variables.length <= 0) return JSON.stringify({});
    var tableStore = {
        variableType: {},
        data: []
    };
    variables.forEach(function (variable) {
        if (variable.getValue().length !== variables[0].getValue().length) {
            throw new Error("Variables passed into create function are not equal in size! [" + variables[0].getId() + ".length=" + variables[0].getValue().length + " | " + variable.getId() + ".length=" + variable.getValue().length + "]");
        }
        var variableType = variable.getType().name;
        tableStore.variableType[variable.getId()] = variableType.endsWith("[]") ? variableType.substr(0, variableType.length - 2) : variableType;
    });
    var colLength = variables[0].getValue().length;
    for (var i = 0; i < colLength; i++) {
        var record = {};
        variables.forEach(function (variable) {
            var variableType = variable.getType().name;
            if (variableType !== "date[]") {
                record[variable.getId()] = variable.getValueAt(i);
                return;
            }
            var date = variable.getValueAt(i);
            if (date !== null) {
                date.setHours(12);
                record[variable.getId()] = date.toISOString().slice(0, 10);
            } else {
                record[variable.getId()] = date;
            }
        });
        tableStore.data.push(record);
    }
    if (colLength === 0) tableStore.data = [];
    return tableStore;
}
var createFromVariablesWithNewKeyMapping = function (variables, keyMapping) {
    if (!variables.every(function (variable) {
        return Array.isArray(variable.getValue());
    })) {
        throw new Error("Variables passed into TableStore.create function must be arrays!");
    }
    if (variables.length <= 0) return JSON.stringify({});
    var tableStore = {
        variableType: {},
        data: []
    };
    variables.forEach(function (variable) {
        if (variable.getValue().length !== variables[0].getValue().length) {
            throw new Error("Variables passed into create function are not equal in size! [" + variables[0].getId() + ".length=" + variables[0].getValue().length + " | " + variable.getId() + ".length=" + variable.getValue().length + "]");
        }
        var variableType = variable.getType().name;
        tableStore.variableType[keyMapping.get(variable.getId())] = variableType.endsWith("[]") ? variableType.substr(0, variableType.length - 2) : variableType;
    });
    var colLength = variables[0].getValue().length;
    for (var i = 0; i < colLength; i++) {
        var record = {};
        variables.forEach(function (variable) {
            var variableType = variable.getType().name;
            if (variableType !== "date[]") {
                record[keyMapping.get(variable.getId())] = variable.getValueAt(i);
                return;
            }
            var date = variable.getValueAt(i);
            if (date !== null) {
                date.setHours(12);
                record[keyMapping.get(variable.getId())] = date.toISOString().slice(0, 10);
            } else {
                record[keyMapping.get(variable.getId())] = date;
            }
        });
        tableStore.data.push(record);
    }
    if (colLength === 0) tableStore.data = [];
    return tableStore;
}
var insertInitialValuesToNewColumn = function (existingData, arrayOfInitialValues, columnId) {
    if (existingData.length !== arrayOfInitialValues.length) {
        console.error("Array of initial values has different size than existing data in TableStore!"); //konieczne, bo treść poniższego błędu nie jest wyświetlana w konsoli przeglądarki
        throw new Error("Array of initial values has different size than existing data in TableStore!");
    }
    for (var i = 0; i < existingData.length; i++) {
        var value = arrayOfInitialValues[i];
        if (value instanceof Date) {
            value.setHours(12);
            existingData[i][columnId] = value.toISOString().slice(0, 10);
        } else {
            existingData[i][columnId] = value;
        }
    }
}

var insertDefaultValuesToNewColumn = function (existingData, dataType, columnId) {
    switch (dataType) {
        case "DATE":
            insertDefaultValue(existingData, columnId, null);
            break;
        case "FLOAT":
        case "INTEGER":
            insertDefaultValue(existingData, columnId, 0);
            break;
        case "STRING":
            insertDefaultValue(existingData, columnId, "");
            break;
        default:
            throw new Error("Added column has wrong type: '" + dataType + "'!");
    }
}

var insertDefaultValue = function (existingData, columnId, value) {
    for (var i = 0; i < existingData.length; i++) {
        existingData[i][columnId] = value;
    }
}

PW.Functions.register("TableStore.create", "string", ["variable[]"], TableStore.create);

PW.Functions.register("TableStore.create", "string", ["string"], TableStore.create);

PW.Functions.register("TableStore.length", "integer", ["string"], TableStore.length);

PW.Functions.register("TableStore.keys", "string[]", ["string"], TableStore.keys);

PW.Functions.register("TableStore.joinField", "string", ["string", "string", "string"], TableStore.joinField);

PW.Functions.register("TableStore.addRecord", "string", ["string", "string[]", "string[]"], TableStore.addRecord);

PW.Functions.register("TableStore.addRecords", "string", ["string", "string"], TableStore.addRecords);

PW.Functions.register("TableStore.filter", "string", ["string", "string", "string[]", "string[]", "string[]"], TableStore.filter);

PW.Functions.register("TableStore.getItem", "string", ["string", "string", "integer"], function (tableStoreStr, fieldId, elementNr) {
    return TableStore.getItem(tableStoreStr, fieldId, elementNr, "string");
});

PW.Functions.register("TableStore.getItem", "integer", ["string", "string", "integer"], function (tableStoreStr, fieldId, elementNr) {
    return TableStore.getItem(tableStoreStr, fieldId, elementNr, "integer");
});

PW.Functions.register("TableStore.getItem", "float", ["string", "string", "integer"], function (tableStoreStr, fieldId, elementNr) {
    return TableStore.getItem(tableStoreStr, fieldId, elementNr, "float");
});

PW.Functions.register("TableStore.getItem", "boolean", ["string", "string", "integer"], function (tableStoreStr, fieldId, elementNr) {
    return TableStore.getItem(tableStoreStr, fieldId, elementNr, "boolean");
});

PW.Functions.register("TableStore.getItem", "date", ["string", "string", "integer"], function (tableStoreStr, fieldId, elementNr) {
    return TableStore.getItem(tableStoreStr, fieldId, elementNr, "date");
});

PW.Functions.register("TableStore.setItems", "string", ["string", "integer", "string[]", "string[]"], TableStore.setItems);

PW.Functions.register("TableStore.addColumn", "string", ["string", "string", "string"], TableStore.addColumn);

PW.Functions.register("TableStore.sort", "string", ["string", "string", "string"], TableStore.sort);

PW.Functions.register("TableStore.deleteRecord", "string", ["string", "integer"], TableStore.deleteRecord);