/**
 * Calculates the difference between two dates counted in days.
 */
PW.Functions.register("calculateDaysDifference", "integer", ["date", "date", "boolean"], function (firstDate, secondDate, includeFreeDays) {
    var earlierDate = getEarlierDate(firstDate, secondDate),
        laterDate = getLaterDate(firstDate, secondDate);

    if (!firstDate || !secondDate) {
        return 0;
    }

    return calculateDaysDifference(earlierDate, laterDate, includeFreeDays);
});


/**
 * Calculates the time difference between two dates .
 */
PW.Functions.register("calculateTimeDifference", "integer", ["datetime", "datetime", "boolean","boolean", "string"], function (firstDateTime, secondDateTime, includeFreeDays,timeShift, returnType) {
    if (!firstDateTime || !secondDateTime) {
        return 0;
    }
    var earlierDate = getEarlierDate(firstDateTime, secondDateTime),
        laterDate = getLaterDate(firstDateTime, secondDateTime);

    return calculateTimeDifference(earlierDate, laterDate, includeFreeDays,timeShift, returnType);
});

PW.Functions.register("calcDateWorkingDays", "date", ["date", "integer"], function(date, workingDays) {
	return calcDateWorkingDays(date, workingDays);
});

PW.Functions.register("calcDateWorkingDays", "datetime", ["datetime", "integer"], function(dateTime, workingDays) {
	return calcDateWorkingDays(dateTime, workingDays);
});

PW.Functions.register("nextWorkingDay", "date", ["date"], function(date) {
    return nextWorkingDay(date);
});

PW.Functions.register("nextWorkingDay", "datetime", ["datetime"], function(date) {
    return nextWorkingDay(date);
});

PW.Functions.register("previousWorkingDay", "date", ["date"], function(date) {
    return previousWorkingDay(date);
});

PW.Functions.register("previousWorkingDay", "datetime", ["datetime"], function(date) {
    return previousWorkingDay(date);
});

function getEarlierDate(firstDate, secondDate) {
    if (firstDate < secondDate) {
        return firstDate;
    }
    return secondDate;
};

function getLaterDate(firstDate, secondDate) {
    if (firstDate > secondDate) {
        return firstDate;
    }
    return secondDate;
};

function calculateTimeDifference(earlierDate, laterDate, includeFreeDays,timeShift, returnType) {
    var nonWorkingDays = 0;
    var differenceDay = Math.round((laterDate.getTime() - earlierDate.getTime()) / CUFCommon.DateTimeConstants.MILLIS_PER_DAY);
    var firstDayOff;
    var lastDayOff;
    if (!includeFreeDays) {
        for (var i = 0; i <= differenceDay; i++) {
            var dayInCheck = new Date(earlierDate.getFullYear(), earlierDate.getMonth(), earlierDate.getDate() + i);
            if (isHolidayOrWeekend(dayInCheck)) {
                if (i == 0) { firstDayOff = true; }
                else if (i == differenceDay) { lastDayOff = true; }
                else { nonWorkingDays++; }
            }
        }

    }
    var timeShiftMilis=0;
    if(!timeShift || !includeFreeDays){
    	
    	timeShiftMilis-=((laterDate.getTimezoneOffset()-earlierDate.getTimezoneOffset())*CUFCommon.DateTimeConstants.MILLIS_PER_MINUTE);
    }
    laterDate.setTime(laterDate.getTime() - calcNonWorkingTime(nonWorkingDays, earlierDate, laterDate, firstDayOff, lastDayOff));

    var duration = laterDate.getTime() - earlierDate.getTime()+timeShiftMilis;

    if (duration < 0) {
        return 0;
    }
    var yearDifference = laterDate.getFullYear() - earlierDate.getFullYear();
    var result = 0;
    switch (returnType) {
        case "SECOND":
            result = duration / CUFCommon.DateTimeConstants.MILLIS_PER_SECOND;
            break;
        case "MINUTE":
            result = duration / CUFCommon.DateTimeConstants.MILLIS_PER_MINUTE;
            break;
        case "HOUR":
            result = duration / CUFCommon.DateTimeConstants.MILLIS_PER_HOUR;
            break;
        case "DAY":
            result = duration / CUFCommon.DateTimeConstants.MILLIS_PER_DAY;
            break;
        case "WEEK":
            result = duration / CUFCommon.DateTimeConstants.MILLIS_PER_WEEK;
            break;
        case "MONTH":
            var monthDifference = laterDate.getMonth() - earlierDate.getMonth();
            result = yearDifference * 12 + monthDifference;
            result += getMonthCorrection(laterDate, earlierDate);
            break;
        case "YEAR":
            result = yearDifference + getYearCorrection(laterDate, earlierDate);
            break;
        default:
            result = 0;
    }
    return Math.floor(result);
};

function calcDateWorkingDays(date, workingDays) {
    if (!date) {
        return null;
    }
    if (workingDays === 0) {
        return date;
    }
    let subtractionDaysFromDate = workingDays < 0;

    if (!subtractionDaysFromDate) {
        if (isHolidayOrWeekend(date)){
            date = setDateAtFirstWorkingDay(date);
        }
        let nextDay = 1;
        let totalWorkingDays = workingDays, holidaysAndWorkingDays = workingDays;
        while (totalWorkingDays > 0) {
            let checkDayInDate = new Date(date.getFullYear(), date.getMonth(), date.getDate() + nextDay);
            if (isHolidayOrWeekend(checkDayInDate)) {
                holidaysAndWorkingDays++;
                nextDay++;
            } else {
                totalWorkingDays--;
                nextDay++;
            }
        }
        return Ext4.Date.add(date, Ext4.Date.DAY, holidaysAndWorkingDays);
    } else {
        if (isHolidayOrWeekend(date)){
            date = setDateAtFirstPastWorkingDay(date);
        }
        let previousDay = -1;
        let totalWorkingDays = workingDays, holidaysAndWorkingDays = workingDays;
        while (totalWorkingDays < 0) {
            let checkDayInDate = new Date(date.getFullYear(), date.getMonth(), date.getDate() + previousDay);
            if (isHolidayOrWeekend(checkDayInDate)) {
                holidaysAndWorkingDays--;
                previousDay--;
            } else {
                totalWorkingDays++;
                previousDay--;
            }
        }
        return Ext4.Date.add(date, Ext4.Date.DAY, holidaysAndWorkingDays);
    }
}

function nextWorkingDay(date) {
    let startingDate = Ext4.Date.add(date, Ext4.Date.DAY, 1)
    return setDateAtFirstWorkingDay(startingDate);
}

function previousWorkingDay(date) {
    let startingDate = Ext4.Date.add(date, Ext4.Date.DAY, -1)
    return setDateAtFirstPastWorkingDay(startingDate);
}

function setDateAtFirstWorkingDay(date){
	var checkDayInDate = date;
	while(isHolidayOrWeekend(checkDayInDate)){
		checkDayInDate=Ext4.Date.add(checkDayInDate, Ext4.Date.DAY, 1);
	}
	return checkDayInDate;
}

function setDateAtFirstPastWorkingDay(date){
	var checkDayInDate = date;
	while(isHolidayOrWeekend(checkDayInDate)){
		checkDayInDate=Ext4.Date.add(checkDayInDate, Ext4.Date.DAY, -1);
	}
	return checkDayInDate;
}

function calcNonWorkingTime(nonWorkingDays, earlierDate, laterDate, firstDayOff, lastDayOff) {
    var nonWorkingTime = nonWorkingDays * CUFCommon.DateTimeConstants.MILLIS_PER_DAY;
    if (firstDayOff) { nonWorkingTime += CUFCommon.DateTimeConstants.MILLIS_PER_DAY - milsOfTime(earlierDate); }
    if (lastDayOff) { nonWorkingTime += milsOfTime(laterDate); }
    return nonWorkingTime;
}

function getYearCorrection(laterDate, earlierDate) {
    if (laterDate.getMonth() == earlierDate.getMonth()) { return getMonthCorrection(laterDate, earlierDate); }
    else if (laterDate.getMonth() > earlierDate.getMonth()) { return 0 }
    else { return -1; }
}

function getMonthCorrection(laterDate, earlierDate) {
    if (laterDate.getDate() == earlierDate.getDate()) { return getTimeCorrection(laterDate, earlierDate); }
    else if (laterDate.getDate() > earlierDate.getDate()) { return 0 }
    else { return -1; }
}

function getTimeCorrection(laterDate, earlierDate) {
    return milsOfTime(laterDate) < milsOfTime(earlierDate) ? -1 : 0;
}

function milsOfTime(date) {
    return date.getHours() * CUFCommon.DateTimeConstants.MILLIS_PER_HOUR
        + date.getMinutes() * CUFCommon.DateTimeConstants.MILLIS_PER_MINUTE
        + date.getSeconds() * CUFCommon.DateTimeConstants.MILLIS_PER_SECOND;
}

function calculateDaysDifference(earlierDate, laterDate, includeFreeDays) {

    var workingDayCounter = 0;
    // 86400000 = miliseconds in a day
    var difference = Math.round((laterDate.getTime() - earlierDate.getTime()) / 86400000);

    if (!includeFreeDays) {

        for (var i = 0; i <= difference; i++) {
            var dayInCheck = new Date(earlierDate.getFullYear(), earlierDate.getMonth(), earlierDate.getDate() + i);
            if (!isHolidayOrWeekend(dayInCheck)) {
                workingDayCounter++;
            }
        }
    }
    else {
        workingDayCounter = difference + 1;
    }
    return workingDayCounter;
};

function isHolidayOrWeekend(dt) {

    if (belongsToWeekend(dt) || isStaticHoliday(dt) || belongToCustomHolidays(dt)) {
        return true;
    }

    // Easter calculating algorithm
    if (dt.getMonth() >= 3 || dt.getMonth() <= 6) {
        var a = dt.getFullYear() % 19,
            b = Math.floor(dt.getFullYear() / 100),
            c = dt.getFullYear() % 100;
        var d = Math.floor(b / 4),
            e = b % 4,
            f = Math.floor((b + 8) / 25);
        var g = Math.floor((b - f + 1) / 3);
        var h = (19 * a + b - d - g + 15) % 30,
            i = Math.floor(c / 4),
            k = c % 4;
        var l = (32 + 2 * e + 2 * i - h - k) % 7,
            m = Math.floor((a + 11 * h + 22 * l) / 451);
        var p = (h + l - 7 * m + 114) % 31;

        var day = p + 1,
            month = Math.floor((h + l - 7 * m + 114) / 31);

        month--;// months in JS start at 0

        var easterFirst = new Date(dt.getFullYear(), month, day),
            easterSecond = new Date(dt.getFullYear(), month, day + 1);
        // end of Easter calculating algorithm

        // holidays that rely on Easter
        var ZDS = new Date(dt.getFullYear(), month, day + 49),
            BC = new Date(dt.getFullYear(), month, day + 60);

        if (dt.valueOf() == easterFirst.valueOf() || dt.valueOf() == easterSecond.valueOf() || dt.valueOf() == ZDS.valueOf() || dt.valueOf() == BC.valueOf()) {
            return true;
        } else {
            return false;
        }
    }
};

function isStaticHoliday(dt) {

    switch (dt.getMonth()) {
        case 0: {
            if (dt.getDate() == 1 || dt.getDate() == 6) {
                return true;
            }
            break;
        }
        case 4: {
            if (dt.getDate() == 1 || dt.getDate() == 3) {
                return true;
            }
            break;
        }
        case 7: {
            if (dt.getDate() == 15) {
                return true;
            }
            break;
        }
        case 10: {
            if (dt.getDate() == 1 || dt.getDate() == 11) {
                return true;
            }
            break;
        }
        case 11: {
            if (dt.getDate() == 25 || dt.getDate() == 26) {
                return true;
            }
            break;
        }
            return false;
    }
};

function belongsToWeekend(dt) {
    if (dt.getDay() === 6 || dt.getDay() === 0) {
        return true;
    } else {
        return false;
    }
};

function belongToCustomHolidays(dt) {
    var rawDate = SystemParameterService.getRawValue("CustomHolidays");
    var returnValue = false;
    if (rawDate) {
        rawDateArray = rawDate.split(";");
        rawDateArray.forEach(function (item) {
            partDate = item.split("-");
            if (partDate.length == 2) {
                if (dt.getFullYear() == (new Date()).getFullYear() && (dt.getMonth() + 1) == partDate[0] && dt.getDate() == partDate[1]) {
                    returnValue = true;
                }
            } else if
                (partDate.length == 3) {
                if (dt.getFullYear() == partDate[0] && (dt.getMonth() + 1) == partDate[1] && dt.getDate() == partDate[2]) {
                    returnValue = true;
                }
            } else {
                Logger.error("Wrong Format " + item);
            }
        });
    }
    return returnValue;
};