/*
 * Decompiled with CFR 0.152.
 */
package com.suncode.cuf.common.documents.libreoffice.services;

import com.sun.star.bridge.XUnoUrlResolver;
import com.sun.star.comp.helper.Bootstrap;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import com.suncode.cuf.common.documents.libreoffice.exceptions.LibreOfficeConnectionException;
import com.suncode.cuf.common.documents.libreoffice.exceptions.LibreOfficeConnectionTimeoutException;
import com.suncode.cuf.common.documents.libreoffice.exceptions.LibreOfficeInitializationException;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.ServerSocket;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LibreOfficeManager {
    private static final Logger log = LoggerFactory.getLogger(LibreOfficeManager.class);
    private static final String HOST = "localhost";
    private static final List<String> CONNECTION_OPTIONS = Arrays.asList("--headless", "--nofirststartwizard", "--nologo", "--nodefault", "--norestore", "--nocrashreport", "--nolockcheck");

    public static void startLibreOffice(LibreOfficeConnection libreOfficeConnection) throws LibreOfficeConnectionException, LibreOfficeInitializationException, LibreOfficeConnectionTimeoutException {
        try {
            int port = LibreOfficeConnectionParams.getUniqueFreePort();
            libreOfficeConnection.setPort(port);
            File userDir = LibreOfficeConnectionParams.getUniqueUserDir();
            libreOfficeConnection.setUserDir(userDir);
            String acceptOption = "--accept=socket,host=localhost,port=" + port + ";urp;";
            String unoConnect = "uno:socket,host=localhost,port=" + port + ";urp;StarOffice.ComponentContext";
            String userInstallOption = "-env:UserInstallation=file:///" + userDir.getCanonicalPath().replace(File.separator, "/").replace(" ", "+");
            Process process = LibreOfficeManager.startLibreOfficeProcess(libreOfficeConnection.executablePath, acceptOption, userInstallOption);
            libreOfficeConnection.setProcess(process);
            LibreOfficeManager.startLoggingThread(process);
            XComponentContext context = LibreOfficeManager.waitForConnection(unoConnect);
            libreOfficeConnection.setContext(context);
        }
        catch (IOException e) {
            throw new LibreOfficeConnectionException("LibreOffice socket port error!", e);
        }
        catch (ExecutionException e) {
            throw new LibreOfficeInitializationException("LibreOffice initialization error!", e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new LibreOfficeInitializationException("LibreOffice initialization interrupted!", e);
        }
        catch (TimeoutException e) {
            throw new LibreOfficeConnectionTimeoutException("LibreOffice connection timeout!");
        }
    }

    public static void stopLibreOffice(LibreOfficeConnection libreOfficeConnection) {
        Process process = libreOfficeConnection.getProcess();
        if (process != null && process.isAlive()) {
            process.destroy();
            try {
                if (!process.waitFor(60L, TimeUnit.SECONDS)) {
                    process.destroyForcibly();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        LibreOfficeConnectionParams.freePort(libreOfficeConnection.getPort());
        LibreOfficeConnectionParams.cleanUpUserDir(libreOfficeConnection.getUserDir());
    }

    private static Process startLibreOfficeProcess(String libreOfficePath, String acceptOption, String userInstallOptions) throws IOException {
        List<String> command = Stream.concat(Stream.of(libreOfficePath), Stream.concat(CONNECTION_OPTIONS.stream(), Stream.of(acceptOption, userInstallOptions))).collect(Collectors.toList());
        ProcessBuilder builder = new ProcessBuilder(command);
        builder.redirectErrorStream(true);
        return builder.start();
    }

    private static void startLoggingThread(Process process) {
        new Thread(() -> {
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    log.debug("LibreOffice: {}", (Object)line);
                }
            }
            catch (IOException e) {
                log.error("Failed to read LibreOffice output", (Throwable)e);
            }
        }).start();
    }

    private static XComponentContext waitForConnection(String unoConnect) throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<XComponentContext> future = CompletableFuture.supplyAsync(() -> {
            long deadline = System.currentTimeMillis() + 60000L;
            Exception lastException = null;
            while (System.currentTimeMillis() < deadline) {
                try {
                    log.info("Trying to connect to LibreOffice service. Connection string: {}", (Object)unoConnect);
                    XComponentContext localContext = Bootstrap.createInitialComponentContext(null);
                    Object urlResolver = localContext.getServiceManager().createInstanceWithContext("com.sun.star.bridge.UnoUrlResolver", localContext);
                    Object initialObject = ((XUnoUrlResolver)UnoRuntime.queryInterface(XUnoUrlResolver.class, (Object)urlResolver)).resolve(unoConnect);
                    return (XComponentContext)UnoRuntime.queryInterface(XComponentContext.class, (Object)initialObject);
                }
                catch (Exception e) {
                    lastException = e;
                    try {
                        log.info("Retry connection in 2s...");
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        throw new RuntimeException("Interrupted while connecting to LibreOffice", ie);
                    }
                }
            }
            throw new RuntimeException("Timed out waiting for LibreOffice connection", lastException);
        });
        Thread.sleep(2000L);
        return future.get(60L, TimeUnit.SECONDS);
    }

    private LibreOfficeManager() {
    }

    public static class LibreOfficeConnectionParams {
        private static final Set<Integer> usedPorts = new HashSet<Integer>();
        private static final Set<File> userDirs = new HashSet<File>();

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public static int getUniqueFreePort() {
            int maxAttempts = 100;
            int attempt = 0;
            while (attempt < maxAttempts) {
                try (ServerSocket socket = new ServerSocket(0);){
                    int port = socket.getLocalPort();
                    if (usedPorts.add(port)) {
                        int n = port;
                        return n;
                    }
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                ++attempt;
            }
            throw new RuntimeException("Failed to acquire a unique free port after " + maxAttempts + " attempts.");
        }

        public static synchronized File getUniqueUserDir() throws IOException {
            File userDir;
            File baseDir = new File(System.getProperty("java.io.tmpdir"), "lo-temp-user-dir");
            Files.createDirectories(baseDir.toPath(), new FileAttribute[0]);
            while ((userDir = new File(baseDir, UUID.randomUUID().toString())).exists()) {
            }
            Files.createDirectories(userDir.toPath(), new FileAttribute[0]);
            userDirs.add(userDir);
            log.info("Created temp user dir: {}", (Object)userDir.getCanonicalPath());
            return userDir;
        }

        public static synchronized void freePort(Integer port) {
            usedPorts.remove(port);
        }

        public static synchronized void cleanUpUserDir(File userDir) {
            LibreOfficeConnectionParams.deleteDirectoryRecursively(userDir);
            userDirs.remove(userDir);
        }

        private static void deleteDirectoryRecursively(File dir) {
            if (dir != null && dir.exists()) {
                File[] contents = dir.listFiles();
                if (contents != null) {
                    for (File file : contents) {
                        LibreOfficeConnectionParams.deleteDirectoryRecursively(file);
                    }
                }
                dir.delete();
            }
        }

        private LibreOfficeConnectionParams() {
        }
    }

    public static class LibreOfficeConnection {
        private String executablePath;
        private XComponentContext context;
        private Process process;
        private int port;
        private File userDir;

        public LibreOfficeConnection(String executablePath) {
            this.executablePath = executablePath;
        }

        public String getExecutablePath() {
            return this.executablePath;
        }

        public XComponentContext getContext() {
            return this.context;
        }

        public Process getProcess() {
            return this.process;
        }

        public int getPort() {
            return this.port;
        }

        public File getUserDir() {
            return this.userDir;
        }

        public void setExecutablePath(String executablePath) {
            this.executablePath = executablePath;
        }

        public void setContext(XComponentContext context) {
            this.context = context;
        }

        public void setProcess(Process process) {
            this.process = process;
        }

        public void setPort(int port) {
            this.port = port;
        }

        public void setUserDir(File userDir) {
            this.userDir = userDir;
        }
    }
}

