const variableSetService = ServiceFactory.getVariableSetService();
const messageService = ServiceFactory.getMessageService();

PW.FormActions.create('data-table-from-datasource', {
    init: function () {
        this.variables = this.readVariables();
    },

    defaultActions: {
        button: function () {
            if (MobileService.isMobile()) {
                messageService.showFailure(CUFCommon.t("action.mobileUnsupported"));
                return;
            }

            this.prepareColumnParameters();
            this.showDynamicTableWindow();
        },

        dtButton: function () {
            if (MobileService.isMobile()) {
                messageService.showFailure(CUFCommon.t("action.mobileUnsupported"));
                return;
            }

            this.prepareColumnParameters();
            this.showDynamicTableWindow();
        }
    },

    prepareColumnParameters: function () {
        this.columnIndices = this.get("outputParameterId");
        this.columnIndices.push("internalId");
        this.columnParameters = [];

        this.columnParameters = PW.map(this.get("columnName"), function (columnName, i) {
            return {
                flex: 1,
                text: columnName,
                dataIndex: this.columnIndices[i],
                filter: this.getFilter(this.get("type")[i]),
                menuDisabled: !this.get("columnContextMenu"),
                hidden: this.get("hidden") ? this.get("hidden")[i] : false
            };
        }, this);
    },

    getFilter: function (type) {
        return CUF.DataFilters.like();

        //czekamy na ewentualną zmianę w systemie
        /*switch (type) {
            case "integer":
                return CUF.DataFilters.betweenInt();
            case "float":
                return CUF.DataFilters.betweenDouble();
            case "date":
                return CUF.DataFilters.betweenDate();
            case "string":
            default:
                return CUF.DataFilters.like();
        }*/
    },

    showDynamicTableWindow: function () {
        this.cufWindow = CUF.DataService.showDataWin({
            query: this.getParams(),
            type: "webservice",
            winCfg: {
                title: this.get("windowTitle") ?? CUFCommon.t("action.data-table-from-datasource.defaultWindowTitle")
            },
            columns: this.columnParameters,
            fields: this.columnIndices,
            url: "plugin/com.suncode-cuf-components/datasources/get-data",
            gridCfg: {
                showRowCount: true,
                rememberSelection: true,
                numbered: this.get("rowNumber"),
                pageSize: this.get("pageSize") || 50,
                checkboxSelection: (this.get("transfer") && !window.isHistory),
                selectionType: this.get("multipleRowsSelection") ? "MULTI" : "SINGLE",
                tbar: (this.get("transfer") && !window.isHistory) ? this.getButtons() : []
            }
        });
    },

    getButtons: function () {
        const buttonsArray = [this.createTransferButton()];
        if (this.get("multipleRowsSelection")) {
            buttonsArray.push(this.createSelectAllRowsButton(), this.createDeselectAllRowsButton());
        }
        return buttonsArray;
    },

    createSelectAllRowsButton: function () {
        return {
            text: CUFCommon.t("action.data-table-from-datasource.selectAllButton"),
            xtype: "button",
            style: {
                marginBottom:"10px",
                border:"1px solid #cacccf"
            },
            handler: function () {
                this.up("grid").getSelectionModel().selectAll();
            }
        }
    },

    createDeselectAllRowsButton: function () {
        return {
            text: CUFCommon.t("action.data-table-from-datasource.deselectAllButton"),
            xtype: "button",
            style: {
                marginBottom:"10px",
                border:"1px solid #cacccf"
            },
            handler: function () {
                this.up("grid").getSelectionModel().clearSelections();
                this.up("grid").getSelectionModel().deselectAll();
            }
        }
    },

    createTransferButton: function () {
        const me = this;
        return {
            text: CUFCommon.t("action.data-table-from-datasource.transferButton"),
            xtype: "button",
            style: {
                marginBottom:"10px",
                border:"1px solid #cacccf"
            },
            handler: function () {
                const selected = this.up("grid").getSelectionModel().getSelection();
                if (selected && selected.length > 0 && me.get("outputParameterValue")) {
                    FormService.save({
                        scope: me,
                        success: me.moveSelected,
                        successParams: [selected]
                    });
                }
            }
        }
    },

    getParams: function () {
        return {
            datasourceId: this.get("datasource"),
            datasourceInputParametersId: this.get("inputParameterId"),
            datasourceInputParametersValue: this.get("inputParameterValue").map(param => param.getValue()),
            sorters: [{
                property: this.columnIndices[0],
                direction: "ASC"
            }]
        };
    },

    moveSelected: function (selected) {
        this.cufWindow.mask(CUFCommon.t("action-data-table-from-datasource.transferMessage"));

        this.setColumnData(selected);
        this.writeDataToVariables();

        if (this.get("saveForm")) {
            FormService.save();
        }

        this.cufWindow.unmask();
        this.cufWindow.close();
    },

    writeDataToVariables: function () {
        const columnsToSet = this.get("outputParameterValue");
        const columnData = this.columnData;
        const currentValues = this.getCurrentValues(columnsToSet);

        try {
            if (!this.get("executeAutomapping")) {
                this.suspendAutoMappings(columnsToSet);
            }

            if (this.get("overwrite")) {
                const uniqueVariableSets = [...new Set(columnsToSet.map(column => column.variableSet).filter(element => element != null))];
                uniqueVariableSets.forEach(variableSet => variableSet.clear())
            }

            PW.each(columnsToSet, function (column, index) {
                if (column) {
                    const newValue = columnData[column.getId()];
                    if (column.isStandaloneVariable) {
                        if (newValue.length < 2) {
                            column.setValue(this.convertValue(newValue[0], column.getType()));
                        } else if (column.getType().is("string")) {
                            switch (this.get("multipleValuesSupport")) {
                                case "UNIQUE" : {
                                    this.handleUniqueValues(column, newValue);
                                    break;
                                }
                                case "FIRST" : {
                                    column.setValue(newValue[0]);
                                    break;
                                }
                                default :
                                    Logger.error("Cannot add multiple values to a non-array variable with multiple value handling undefined or set to BLOCK!");
                                    break;
                            }
                        } else {
                            Logger.error("Cannot add multiple values to a non-array variable which is also of a different type than string!");
                        }
                    } else {
                        const tempValues = [];

                        for (let i = 0; i < newValue.length; i++) {
                            tempValues.push(this.convertValue(newValue[i], column.getType()));
                        }

                        if (this.get("overwrite")) {
                            column.setValue(tempValues);
                        } else {
                            column.setValue(currentValues[column.getId()].concat(tempValues));
                        }
                    }
                }
            }, this);
        } finally {
            if (!this.get("executeAutomapping")) {
                this.resumeAutoMappings(columnsToSet);
            }
        }
    },

    handleUniqueValues: function (variable, data) {
        const uniqueValues = [];
        for (let i = 0; i < data.length; i++) {
            var value = data[i];
            if (!uniqueValues.includes(value)) {
                uniqueValues.push(value);
            }
        }
        variable.setValue(uniqueValues.join(";"));
    },

    convertValue: function (value, type) {
        if (type.is("float") || type.is("integer") || type.is("float[]") || type.is("integer[]")) {
            if (value && typeof value === 'string') {
                value = value.replace(/ /g, '').replace(/,/g, '.');
            }
            return Number(value);
        } else if (type.is("date") || type.is("datetime") || type.is("date[]") || type.is("datetime[]")) {
            try {
                return value ? new Date(value) : null;
            } catch (e) {
                Logger.error('Could not create Date from value: ' + value);
                return null;
            }
        } else if (type.is("boolean") || type.is("boolean[]")) {
            if (!value) {
                return false;
            } else if (typeof value === "boolean") {
                return value;
            } else if (typeof value === "string") {
                return (value === 'true');
            } else {
                return false;
            }
        } else {
            return String(value);
        }
    },

    setColumnData: function (selected) {
        this.resetColumnData();

        PW.each(selected, function (param, index) {
            this.addRow(param.data, index);
        }, this);
    },

    getCurrentValues: function (variables) {
        var values = {};
        PW.each(variables, function (variable) {
            if (variable) {
                values[variable.getId()] = variable.getValue();
            }
        }, this);
        return values;
    },

    resetColumnData: function () {
        this.columnData = {};
        PW.each(this.columnIndices, function (name, index) {
            if (name !== "internalId") {
                this.columnData[this.variables[index].id] = [];
            }
        }, this);
    },

    readVariables: function () {
        if (!this.get("outputParameterValue")) {
            return [];
        }

        return this.get("outputParameterValue")
            .map(element => ({id: element?.getId() ?? ""}));
    },

    addRow: function (row, rowNumber) {
        PW.each(this.columnIndices, function (name, index) {
            if (name !== "internalId") {
                this.columnData[this.variables[index].id][rowNumber] = row[name];
            }
        }, this);
    },

    suspendAutoMappings: function (columnsToSet) {
        const variablesId = [];
        const variables = columnsToSet.concat(this.table ? this.table.getVariables() : []);
        for (let i = 0; i < variables.length; i++) {
            if (variables[i]) {
                variablesId.push(variables[i].getId());
            }
        }
        VariableService.suspendAutoMapping(variablesId);
    },

    resumeAutoMappings: function (columnsToSet) {
        const variablesId = [];
        const variables = columnsToSet.concat(this.table ? this.table.getVariables() : []);
        for (let i = 0; i < variables.length; i++) {
            if (variables[i]) {
                variablesId.push(variables[i].getId());
            }
        }
        VariableService.resumeAutoMapping(variablesId);
    },
});