Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

/**
 * Obiekt zadania - rozwijany panel z szczegółami oraz sterowaniem.
 */
Ext.ux.plusmpm.scheduledtasks.TaskPanel = function (config) {
  var status = {
    processing: config.task.processing,
    active: config.task.active,
    progress: config.task.progress,
    changed: function (newStatus) {
      return this.processing !== newStatus.processing || this.active !== newStatus.active || this.progress !== newStatus.progress;
    },
  };

  config = Ext.apply(
    {
      isDuringExpandEvent: false,

      status: status,
      executor: new Ext.ux.plusmpm.scheduledtasks.TaskActionExecutor({
        taskId: config.task.id,
        listeners: {
          scope: this,
          success: this.onActionSuccess,
          failure: this.onActionFailure,
          error: this.onActionError,
        },
      }),

      bbar: {
        xtype: 'st_taskcontrol',
        taskRecord: config.record,
        listeners: {
          scope: this,
          activechange: this.onActiveChange,
          execute: this.onExecuteAction,
          cancel: this.onCancelAction,
          remove: this.onRemoveAction,
          history: this.onHistoryClickHandler,
          beforeparametersrender: this.onBeforeParametersRender,
        },
      },

      items: [
        {
          xtype: 'panel',
          ref: 'innerPanel',
          header: true,
          frame: false,
          border: false,
          collapsible: true,
          collapsed: true,
          titleCollapse: true,
        },
      ],
      cls: 'x-st-task',
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskPanel.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskPanel constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskPanel, Ext.Panel, {
  /**
   * @Override
   * @param title
   */
  setTitle: function () {
    var id = this.record.get('id');
    var name = this.record.get('name');
    var lastRun = this.record.get('lastRun');
    var deprecated = this.record.json.deprecated;

    if (deprecated) {
      name = '<s>' + name + '</s>';
    }

    var newTitle = `[${id}] ${name}`;

    if (lastRun) {
      newTitle += ` [${lastRun.format('Y-m-d H:i:s')}]`;
    }

    this.items.first().setTitle(newTitle);
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskPanel.superclass.initComponent.call(this);
    this.setTitle();
    this.record.store.on('update', this.onStoreUpdate, this);
    this.on(
      'destory',
      function () {
        Ext.destroy(executeMask, cancelMask);
      },
      this
    );
    this.innerPanel.on(
      'expand',
      function () {
        this.isDuringExpandEvent = true;
        this.expandParametersFieldSet();
        this.isDuringExpandEvent = false;

        this.innerPanel.getEl().dom.scrollIntoView(true);
      },
      this
    );
  },

  expandParametersFieldSet: function () {
    this.fireEvent('beforeparametersrender');

    this.basicTaskParametersStep = Ext.create({
      xtype: 'st_edit_wizard_step2',
      taskRecord: this.record,
      editable: false,
    });
    this.innerPanel.add(this.basicTaskParametersStep);

    this.paramsFieldSet = this.generateParametersFieldSet(this.task);
    if (this.paramsFieldSet.parametersContainer.items?.getCount()) {
      this.innerPanel.add(this.paramsFieldSet);
    }

    this.doLayout();
  },

  propagateBeforeParametersRenderEvent: function () {
    if (!this.isDuringExpandEvent) {
      this.innerPanel.collapse();
    }

    this.clearBasicParametersStep();
    this.clearParametersFieldSet();
  },

  clearBasicParametersStep: function () {
    if (this.basicTaskParametersStep !== undefined) {
      this.innerPanel.remove(this.basicTaskParametersStep);
      this.basicTaskParametersStep = undefined;
    }
  },

  clearParametersFieldSet: function () {
    if (this.paramsFieldSet !== undefined) {
      this.innerPanel.remove(this.paramsFieldSet);
      this.paramsFieldSet = undefined;
    }
  },

  generateParametersFieldSet: function (data) {
    var parametersDefinition = new Ext.util.MixedCollection();
    Ext.each(
      data.parameters,
      function (parameter) {
        var definition = {
          componentParamId: parameter.componentParameterId,
          name: parameter.annotationName,
          description: parameter.annotationDesc,
          optional: parameter.isOptional,
          position: parameter.position,
          type: parameter.type,
          fieldType: parameter.fieldType,
          value: parameter.value,
          valueMetadata: parameter.valueMetadata,
        };
        parametersDefinition.add(parameter.componentParameterId || parameter.position, definition);
      },
      this
    );

    var parametersBuilder = new Ext.ux.plusmpm.scheduledtasks.ParametersBuilder({
      parametersDefinition: parametersDefinition,
    });

    var parametersContainer = {
      xtype: 'st_parameterscontainer',
      id: 'st-parameterscontainer',
      style: 'padding-top: 10px;',
      ref: 'parametersContainer',

      window: this,
      parametersBuilder: parametersBuilder,
      isInnerContainer: false,
      editable: false,
    };

    var paramsFieldSet = Ext.create({
      xtype: 'fieldset',
      title: PW.t('parameters'),
      collapsible: true,
      items: [parametersContainer],
    });

    paramsFieldSet.parametersContainer.buildParameters(data.componentTaskId);

    return paramsFieldSet;
  },

  onHistoryClickHandler: function (toolbar, showLogs) {
    var window = new Ext.ux.plusmpm.scheduledtasks.HistoryWindow({
      taskId: this.record.id,
      taskName: this.record.data.name,
      showLogs: showLogs,
    });
    window.show();
  },

  onBeforeParametersRender: function () {
    this.fireEvent('beforeparametersrender');
  },

  /**
   * Odświeżenie danych zadania.
   * @param record
   */
  updateTask: function (record) {
    this.record = record;
    this.setTitle();
    this.getBottomToolbar().onTaskUpdate(record);
  },

  /**
   * Zmiana jakiegoś rekordu tego zadania.
   */
  onStoreUpdate: function (store, record) {
    if (record === this.record) {
      this.setTitle();
    }
  },

  /**
   * Wykonywanie działań na zadaniu (uruchomienie anulowanie usuniecie)
   */
  onActionSuccess: function (action, result) {
    switch (action) {
      case 'execute':
        this.executeMask.hide();
        this.record.set('processing', true);
        this.record.set('progress', result.progress);
        this.executor.execute('awaitcompletion');
        break;
      case 'cancel':
        this.cancelMask.hide();
        this.record.set('processing', false);
        this.record.set('progress', -1);
        break;
      case 'remove':
        this.removeMask.hide();
        this.ownerCt.ownerCt.store.remove(this.record);
        this.ownerCt.remove(this);
        Ext.StoreMgr.get('tasksCategoryStore').load();
        return;
        break;
      case 'awaitcompletion':
        this.record.set('processing', false);
        this.record.set('progress', -1);
        break;
    }

    this.record.commit();
    this.getBottomToolbar().onTaskUpdate(this.record);
  },

  onActionFailure: function (action) {
    if (this.executeMask) this.executeMask.hide();

    if (this.cancelMask) this.cancelMask.hide();

    if (this.removeMask) this.removeMask.hide();

    this.record.store.reload();
  },

  onActionError: function (action) {
    if (this.executeMask) this.executeMask.hide();

    if (this.cancelMask) this.cancelMask.hide();

    if (this.removeMask) this.removeMask.hide();

    this.record.store.reload();
  },

  onExecuteAction: function (bar) {
    if (!this.executeMask) {
      this.executeMask = new Ext.LoadMask(this.el, { msg: PW.t('attemptToStart') + ' ' + PW.t('pleaseWait') });
    }

    this.executeMask.show();
    this.executor.execute('execute');
  },

  onCancelAction: function (bar) {
    if (!this.cancelMask) {
      this.cancelMask = new Ext.LoadMask(this.el, { msg: PW.t('attemptToCancel') + ' ' + PW.t('pleaseWait') });
    }

    this.cancelMask.show();
    this.executor.execute('cancel');
  },

  onRemoveAction: function (bar) {
    if (!this.removeMask) {
      this.removeMask = new Ext.LoadMask(this.el, { msg: PW.t('attemptToDelete') + ' ' + PW.t('pleaseWait') });
    }

    this.removeMask.show();
    this.executor.execute('remove');
  },

  /**
   * Obsługa toolbaru sterującego
   */

  onActiveChange: function (active) {
    this.record.set('active', active);
    this.record.commit(true);
  },
}); // end of Ext.extend
Ext.reg('st_taskpanel', Ext.ux.plusmpm.scheduledtasks.TaskPanel);

/**
 * Obiekt wspomagający wykonywanie działan na zadaniu - uruchamianie anulowanie etc.
 *
 * eventy:
 * success - akcja powiodła się
 * failure - akcja nie jest dozwolona
 * error - nieznany błąd
 */
Ext.ux.plusmpm.scheduledtasks.TaskActionExecutor = Ext.extend(Ext.util.Observable, {
  execute: function (action) {
    this.sendRequest(action);
  },

  sendRequest: function (action) {
    Ext.Ajax.request({
      url: 'api/scheduledtasks/invoke',
      method: 'POST',
      params: { id: this.taskId, action: action },
      scope: this,
      callback: function (options, success, response) {
        try {
          if (success) {
            var responseDecoded = Ext.decode(response.responseText);
            if (responseDecoded.success === true) {
              this.fireEvent('success', action, responseDecoded);
            } else {
              this.fireEvent('failure', action, response);
            }
          } else {
            this.fireEvent('error', action, response);
          }
        } catch (error) {
          Logger.error(error);
          this.fireEvent('error', action, response);
        }
      },
    });
  },

  constructor: function (config) {
    this.taskId = config.taskId;
    this.listeners = config.listeners;

    this.addEvents(
      /**
       * Akcja powiodła się
       */
      'success',
      /**
       * Akcja nie powiodła się
       */
      'failure',
      /**
       * Nieoczekiwany błąd
       */
      'error',
      'beforeparametersrender'
    );

    Ext.ux.plusmpm.scheduledtasks.TaskActionExecutor.superclass.constructor.call(this, config);
  },
}); // end of Ext.extend
