Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

/**
 * Okno tworzenia nowego zadania
 */
Ext.ux.plusmpm.scheduledtasks.NewTaskWindow = function (config) {
  config = Ext.apply(
    {
      width: 1000,
      y: 50,
      autoHeight: true,
      shadow: false,
      modal: true,
      layout: 'fit',
      title: PW.t('scheduledTaskWizard'),
      items: [
        {
          xtype: 'form',
          autoHeight: true,
          border: false,
          ref: 'wizard',
          layout: 'card',
          activeItem: 0,
          bbar: [
            '->',
            {
              text: '&laquo; ' + PW.t('back'),
              handler: this.onPreviousHandler,
              disabled: true,
              scope: this,
              ref: '../../prevButton',
            },
            {
              text: PW.t('next') + ' &raquo;',
              handler: this.onNextHandler,
              disabled: true,
              scope: this,
              ref: '../../nextButton',
            },
            {
              text: PW.t('add'),
              handler: this.onAddHandler,
              hidden: true,
              scope: this,
              iconCls: 'dvnt-icon-plus-symbol',
              ref: '../../addButton',
            },
          ],

          items: [
            {
              xtype: 'st_wizard_step1',
              ref: 'step1',
              listeners: {
                scope: this,
                ready: function (record) {
                  this.nextButton.setDisabled(false);
                  this.choosenMethod = record;
                },
              },
            },
            {
              xtype: 'st_wizard_step2',
              defaultCategory: config.defaultCategory,
            },
          ],
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.NewTaskWindow.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.NewTaskWindow constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.NewTaskWindow, Ext.Window, {
  /**
   * Confnięcie na poprzednią kartę
   */
  onPreviousHandler: function () {
    this.wizard.getLayout().setActiveItem(0);
    this.prevButton.setDisabled(true);
    this.nextButton.setVisible(true);
    this.addButton.setVisible(false);
  },

  /**
   * Przełączenie na następną kartę
   */
  onNextHandler: function (record) {
    if (this.validateParameters()) {
      this.wizard.getLayout().setActiveItem(1);
      this.prevButton.setDisabled(false);
      this.nextButton.setVisible(false);
      this.addButton.setVisible(true);
    }
  },

  validateParameters: function () {
    var hasErrors = false;

    var allFields = this.getAllParameterFields();
    allFields.forEach(function (item) {
      if (!item.validate()) {
        hasErrors = true;
      }
    });

    return !hasErrors;
  },

  getAllParameterFields: function () {
    var paramsFieldSet = this.wizard.step1.paramsFieldSet;
    if (!paramsFieldSet) {
      return [];
    }

    return paramsFieldSet.parametersContainer.findBy(function (cmp) {
      return cmp.isFormField;
    }, this);
  },

  onAddHandler: async function () {
    var form = this.wizard.getForm();
    if (form.isValid()) {
      var categoryName = form.findField('categoryName').getValue();
      var categoryStore = Ext.StoreMgr.get('tasksCategoryStore');
      var catIndex = categoryStore.findExact('name', categoryName);

      if (catIndex > -1) {
        var record = categoryStore.getAt(catIndex);
        form.findField('categoryId').setValue(record.data.id);
      } else {
        form.findField('categoryId').setValue('');
      }

      var params = await this._buildParams();

      form.submit({
        url: 'api/scheduledtasks/new',
        method: 'POST',
        params: {
          params: Ext.encode(params),
        },
        scope: this,
        waitMsg: PW.t('addingScheduledTask'),
        success: function (form, action) {
          Ext.StoreMgr.get('tasksCategoryStore').load();
          this.close();
        },
        failure: function (form, action) {
          Ext.Msg.show({
            title: PW.t('error'),
            msg: action?.result?.error ?? PW.t('failedToEditTask'),
            buttons: Ext.Msg.OK,
            icon: Ext.MessageBox.ERROR,
          });
        },
      });
    }
  },

  _buildParams: async function () {
    var definitions = this.choosenMethod.data.parameters;
    var fields = this.wizard.findBy(function (comp) {
      return comp.isFormField && !comp.disabled && comp.isScheduledTaskParameter;
    }, this);

    var params = [];

    for (var definition of definitions) {
      var arrayValue = [];
      var isArray = definition.type.indexOf('[') !== -1;

      for (var field of fields) {
        var fieldMatchesDefinition = this._hasMatchingComponentParameterIdOrPosition(field, definition);

        if (fieldMatchesDefinition) {
          var singleValue;
          if (field.xtype === 'checkbox') {
            singleValue = {
              value: String(field.checked),
            };
          } else if (field.xtype === 'fileuploadfield') {
            var files = field.fileInput.dom.files;
            var file = files[0];
            if (files.length > 0) {
              singleValue = {
                value: await this._toBase64(file),
                metadata: {
                  fileName: file.name,
                },
              };
            } else {
              singleValue = {
                value: '',
                metadata: {
                  fileName: 'empty.txt',
                },
              };
            }

            if (field.fileId) {
              Ext.apply(singleValue.metadata, {
                fileId: field.fileId,
              });
            }
          } else if (field.xtype === 'combo') {
            singleValue = {
              value: field.getValue(),
            };
          } else {
            singleValue = {
              value: field.getRawValue(),
            };
          }

          singleValue = Ext.apply(
            {
              valueType: 'single',
            },
            singleValue
          );
          arrayValue.push(singleValue);
        }
      }

      var value;
      if (isArray) {
        value = {
          valueType: 'array',
          value: arrayValue,
        };
      } else if (arrayValue.length > 0) {
        value = arrayValue[0];
      } else {
        value = {
          valueType: 'single',
        };
      }

      params.push({
        position: definition.position,
        value: value,
        type: definition.type,
        componentParameterId: definition.componentParameterId,
      });
    }
    return params;
  },

  _toBase64: function (file) {
    return new Promise(resolve => {
      var reader = new FileReader();
      reader.readAsDataURL(file);
      reader.onload = function () {
        resolve(reader.result);
      };
      reader.onerror = function (error) {
        resolve('');
      };
    });
  },

  _hasMatchingComponentParameterIdOrPosition: function (field, definition) {
    if (definition.componentParameterId) {
      return definition.componentParameterId == field.componentParameterId;
    }

    return definition.position == field.parameterPosition;
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.NewTaskWindow.superclass.initComponent.call(this);
  },
}); // end of Ext.extend

/**
 * Krok 1 - wybranie klasy zadania, wyszukiwanie zadań
 */
Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1 = function (config) {
  this.store = new Ext.data.JsonStore({
    proxy: new Ext.data.HttpProxy({
      method: 'GET',
      url: 'api/scheduledtasks/available',
    }),
    root: 'data',
    fields: [
      { name: 'isComponent' },
      { name: 'componentTaskId' },
      { name: 'deprecated' },
      { name: 'name', sortType: Ext.data.SortTypes.asUCString },
      { name: 'className', sortType: Ext.data.SortTypes.asUCString },
      { name: 'annotationName', sortType: Ext.data.SortTypes.asUCString },
      { name: 'annotationDesc', sortType: Ext.data.SortTypes.asUCString },
      { name: 'category', sortType: Ext.data.SortTypes.asUCString },
      { name: 'returns', sortType: Ext.data.SortTypes.asUCString },
      {
        name: 'parameters',
        defaultValue: [],
        convert: function (v, record) {
          var params = [];
          Ext.each(v, function (item, index) {
            var param = {
              componentParameterId: item.componentParameterId ? item.componentParameterId : '',
              type: item.type ? item.type : '',
              fieldType: item.fieldType ? item.fieldType : 'string',
              name: item.annotationName ? item.annotationName : '',
              description: item.annotationDesc ? item.annotationDesc : '',
              position: item.position,
              isOptional: item.isOptional,
              defaultValue: item.defaultValue,
            };
            params.push(param);
          });
          return params;
        },
      },
    ],
  });

  this.initTemplates();

  config = Ext.apply(
    {
      autoHeight: true,
      layout: 'form',
      style: 'padding: 10px',
      items: [
        {
          xtype: 'box',
          autoEl: {
            children: [
              {
                tag: 'div',
                style: 'color: gray; margin-bottom: 5px;',
                cn: [
                  { tag: 'span', html: PW.t('taskCasAlsoBeFound') + ' ' },
                  { tag: 'a', href: '#', html: PW.t('searchEngine') },
                ],
              },
              { tag: 'p', html: PW.t('specilyFullPath') },
            ],
          },
          style: 'padding-bottom: 5px',
          listeners: {
            scope: this,
            afterrender: function (comp) {
              comp.el.select('a').on('click', this.onShowSearchWindow, this, { preventDefault: true });
            },
          },
        },
        {
          xtype: 'compositefield',
          fieldLabel: PW.t('className'),
          anchor: '95%',
          msgTarget: 'side',
          ref: 'classNameField',
          buildCombinedErrorMessage: function (errors) {
            return errors[0].error;
          },
          items: [
            {
              xtype: 'textfield',
              name: 'className',
              allowBlank: false,
              flex: 1,
              listeners: {
                scope: this,
                change: this.onSearchClass,
              },
            },
            {
              xtype: 'button',
              iconCls: 'dvnt-icon-search',
              handler: this.onSearchClass,
              scope: this,
              flex: 0,
              style: 'margin-top: 9px;',
            },
          ],
        },
        {
          xtype: 'combo',
          name: 'method',
          fieldLabel: PW.t('selectTheMethod'),
          anchor: '95%',
          emptyText: PW.t('selectTheMethod'),
          mode: 'local',
          triggerAction: 'all',
          forceSelection: true,
          store: this.store,
          valueField: 'name',
          displayField: 'name',
          tpl: this.methodComboTpl,
          itemSelector: 'div.method-combo-item',
          listeners: {
            scope: this,
            select: this.onMethodComboSelect,
          },
          ref: 'methodComboBox',
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1 constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1, Ext.Container, {
  descImg: '<img style="padding-left: 5px" src="pages/images/tooltip.png"/>',

  onSearchClass: function () {
    var classField = this.ownerCt.getForm().findField('className');
    if (classField.isValid()) {
      this.getClassMethods(classField.getValue());
    }
  },

  onShowSearchWindow: function () {
    this.store.removeAll();

    var window = new Ext.ux.plusmpm.scheduledtasks.SearchWindow({
      store: this.store,
      modal: true,
      listeners: {
        scope: this,
        methodchoosen: this.onSearchMethodChoosen,
        close: function () {
          this.ownerCt.ownerCt.focus();
        },
      },
    });
    window.show();
  },

  onSearchMethodChoosen: function (record) {
    if (record.data.isComponent) {
      this.methodComboBox.setDisabled(true);
    } else {
      this.methodComboBox.setDisabled(false);
    }

    this.methodComboBox.setValue(record.data.name);
    this.methodComboBox.onSelect(record, 0);

    this.classNameField.items.first().setValue(record.data.className);
    this.getClassMethods(record.data.className);
  },

  /**
   * Przejscie do następnego kroku, walidacja danych...
   * @param handler zakonczono walidację
   */
  onStepAccept: function (handler) {},

  getClassMethods: function (className) {
    this.store.load({
      params: { className: className },
      scope: this,
      callback: function (records, opts, success) {
        if (success === false) {
          var classField = this.ownerCt.getForm().findField('className');
          classField.markInvalid(PW.t('classNotExist'));
        }
      },
    });
  },

  onMethodComboSelect: function (combo, record, index) {
    this.addParametersFieldSet(record.data);
    this.fireEvent('ready', record);
  },

  addParametersFieldSet: function (data) {
    if (this.paramsFieldSet !== undefined) {
      this.remove(this.paramsFieldSet);
      this.paramsFieldSet = undefined;
    }

    if (data.parameters.length > 0) {
      var parametersDefinition = new Ext.util.MixedCollection();
      Ext.each(
        data.parameters,
        function (parameter) {
          var definition = {
            itemId: parameter.componentParameterId,
            componentParamId: parameter.componentParameterId,
            name: parameter.name,
            description: parameter.description,
            optional: parameter.isOptional,
            position: parameter.position,
            type: parameter.type,
            fieldType: parameter.fieldType,
            value: parameter.defaultValue,
          };
          parametersDefinition.add(parameter.componentParameterId || parameter.position, definition);
        },
        this
      );

      var parametersBuilder = new Ext.ux.plusmpm.scheduledtasks.ParametersBuilder({
        parametersDefinition: parametersDefinition,
      });

      this.paramsFieldSet = Ext.create({
        xtype: 'fieldset',
        title: PW.t('parameters'),
        collapsible: true,
        items: [
          {
            xtype: 'st_parameterscontainer',
            id: 'st-parameterscontainer',
            style: 'padding: 5px;',
            ref: 'parametersContainer',

            window: this,
            parametersBuilder: parametersBuilder,
            isInnerContainer: false,
            editable: true,
          },
        ],
      });

      this.paramsFieldSet.parametersContainer.buildParameters(data.componentTaskId);

      if (this.paramsFieldSet.parametersContainer.items?.getCount()) {
        this.add(this.paramsFieldSet);
      }
    }

    this.doLayout();
    this.ownerCt.setHeight(this.getHeight() + 20);
  },

  initTemplates: function () {
    this.methodComboTpl = new Ext.XTemplate(
      '<tpl for=".">',
      '<div class="method-combo-item {[xindex % 2 == 0 ? "even" : "odd"]}">',
      '<tpl if="deprecated == true">',
      '<div><b>' + PW.t('method') + ':</b><s><span>{name} : {returns}</span></s></div>',
      '<tpl if="Ext.isEmpty(parameters) == false">',
      '<div><b>' + PW.t('parameters') + ':</b><s><span><tpl for="parameters"> {type}, </tpl></span></s></div>',
      '</tpl>',
      '<tpl if="Ext.isEmpty(annotationName) == false">',
      '<div><b>' + PW.t('taskName') + ':</b><s><span>{annotationName}</span></s></div>',
      '<div><s><span>(' + PW.t('deprecated') + ')</span></s></div>',
      '</tpl>',
      '</tpl>',
      '<tpl if="deprecated == false">',
      '<div><b>' + PW.t('method') + ':</b><span>{name} : {returns}</span></div>',
      '<tpl if="Ext.isEmpty(parameters) == false">',
      '<div><b>' + PW.t('parameters') + ':</b><span><tpl for="parameters"> {type}, </tpl></span></div>',
      '</tpl>',
      '<tpl if="Ext.isEmpty(annotationName) == false">',
      '<div><b>' + PW.t('taskName') + ':</b><span>{annotationName}</span></div>',
      '</tpl>',
      '</tpl>',
      '</div>',
      '</tpl>'
    );
    this.methodComboTpl.compile();
  },

  onActivate: function () {
    this.ownerCt.setHeight(this.getHeight() + 20);
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1.superclass.initComponent.call(this);
    this.on('activate', this.onActivate, this);
    this.on(
      'afterrender',
      function () {
        this.classSearchMask = new Ext.LoadMask(this.el, { store: this.store });
      },
      this
    );
    this.on(
      'destroy',
      function () {
        Ext.destroy(this.classSearchMask, this.store);
      },
      this
    );
  },
}); // end of Ext.extend
Ext.reg('st_wizard_step1', Ext.ux.plusmpm.scheduledtasks.TaskCreateStep1);

/**
 * Krok 1 - wybranie klasy zadania, wyszukiwanie zadań
 */
Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2 = function (config) {
  var store = Ext.StoreMgr.get('tasksCategoryStore');
  var categoryNames = [];

  store.each(function (item) {
    categoryNames.push([item.data.name]);
  }, this);

  var categoryStore = new Ext.data.ArrayStore({
    autoDestroy: true,
    data: categoryNames,
    fields: ['name'],
  });

  config = Ext.apply(
    {
      layout: 'form',
      style: 'padding: 10px',
      autoHeight: true,
      listeners: {
        show: function () {
          //robimy to, bo na IE nie do końca dobrze renderują się wszystkie pola
          this.doLayout();
        },
      },
      items: [
        {
          xtype: 'fieldset',
          title: PW.t('taskInformation'),
          labelWidth: 150,
          collapsible: true,
          collapsed: false,
          defaults: {
            xtype: 'textfield',
            anchor: '95%',
          },
          items: [
            {
              name: 'name',
              fieldLabel: PW.t('taskName'),
              allowBlank: false,
              ref: '../nameField',
            },
            {
              name: 'description',
              fieldLabel: PW.t('taskDescription'),
              ref: '../descField',
            },
            {
              xtype: 'datefield',
              fieldLabel: PW.t('firstRunDate'),
              id: 'first_run_field',
              name: 'firstRun',

              format: 'Y-m-d H:i:s',
              minValue: new Date(),
              allowBlank: false,

              triggerClass: 'dvnt-icon-time',
              onTriggerClick: function () {},
              listeners: {
                afterrender: function (field) {
                  field.trigger.on('mouseover', function () {
                    var x = field.trigger.getX();
                    var y = field.trigger.getY();

                    Calendar.setup({
                      inputField: 'first_run_field',
                      ifFormat: '%Y-%m-%d %H:%M:%S',
                      button: field.trigger.id,
                      singleClick: true,
                      showsTime: true,
                      timeFormat: '24',
                      position: [x + 70, y],
                      onUpdate: function (cal) {
                        field.validate();
                      },
                    });
                  });
                },
              },
            },
            {
              xtype: 'combo',
              name: 'categoryName',
              mode: 'local',
              fieldLabel: PW.t('category'),
              displayField: 'name',
              triggerAction: 'all',
              ref: '../categoryField',
              store: categoryStore,
            },
            {
              xtype: 'hidden',
              name: 'categoryId',
            },
            {
              xtype: 'checkbox',
              fieldLabel: PW.t('executeTaskOnce'),
              name: 'executeOnce',
              checked: false,
              listeners: {
                check: function (checkBox) {
                  var periodCompositeField = Ext.getCmp('periodCompositeField');
                  var workingHoursCompositeField = Ext.getCmp('workingHours');

                  if (checkBox.getValue()) {
                    periodCompositeField.disable();
                    periodCompositeField.validate();

                    workingHoursCompositeField.disable();
                    workingHoursCompositeField.validate();
                  } else {
                    periodCompositeField.enable();

                    workingHoursCompositeField.enable();
                  }
                },
              },
            },
            {
              xtype: 'compositefield',
              fieldLabel: PW.t('period'),
              id: 'periodCompositeField',
              items: [
                {
                  xtype: 'numberfield',
                  name: 'periodValue',
                  width: 80,
                  allowNegative: false,
                  allowDecimals: false,
                  allowBlank: false,
                  minValue: 1,
                  value: 1,
                },
                {
                  xtype: 'combo',
                  name: 'periodUnit',
                  editable: false,
                  forceSelection: true,
                  mode: 'local',
                  displayField: 'text',
                  valueField: 'unit',
                  hiddenName: 'periodUnit',
                  flex: 1,
                  triggerAction: 'all',
                  store: new Ext.data.ArrayStore({
                    fields: ['text', 'unit'],
                    data: [
                      [PW.t('minutes'), 'MINUTES'],
                      [PW.t('hours'), 'HOURS'],
                      [PW.t('days'), 'DAYS'],
                      [PW.t('months'), 'MONTHS'],
                      [PW.t('years'), 'YEARS'],
                    ],
                  }),
                  listeners: {
                    afterrender: function () {
                      this.setValue('MINUTES');
                    },
                  },
                },
              ],
            },
            {
              xtype: "compositefield",
              id: "workingHours",
              fieldLabel: PW.t("workingHours"),

              items: [{
                xtype: "st_bound_numberfield",
                boundComponentIds: ["startingHour", "endingHour"],

                id: "startingHour",
                name: "startingHour",
                width: 50,
                allowNegative: false,
                allowDecimals: false,
                allowBlank: true,
                minValue: 0,
                maxValue: 23,
              }, {
                xtype: "label",
                text: "-",
                margins: "13"
              }, {
                xtype: "st_bound_numberfield",
                boundComponentIds: ["startingHour", "endingHour"],

                id: "endingHour",
                name: "endingHour",
                width: 50,
                allowNegative: false,
                allowDecimals: false,
                allowBlank: true,
                minValue: 0,
                maxValue: 23,
              }]
            },
            {
              xtype: "st_bound_numberfield",
              boundComponentIds: ["startingHour", "endingHour"],

              name: "maxWorkingMinutes",
              fieldLabel: PW.t('maxWorkingMinutes'),
              width: 100,
              allowNegative: false,
              allowDecimals: false,
              allowBlank: true,
              minValue: 1
            },
            {
              xtype: 'checkbox',
              name: 'active',
              fieldLabel: PW.t('activeTask'),
              checked: false,
            },
            {
              xtype: 'checkbox',
              name: 'saveHistory',
              fieldLabel: PW.t('saveHistoryExecution'),
              checked: false,
            },
            {
              xtype: 'arrayfield',
              fieldLabel: PW.t('userNameNotifications'),

              fieldConfig: {
                xtype: 'combo',

                name: 'userNameNotifications',
                hiddenName: 'userNameNotifications',
                allowBlank: false,
                height: 40,

                mode: 'remote',
                pageSize: 25,
                valueField: 'userName',
                displayField: 'displayName',
                store: {
                  xtype: 'jsonstore',
                  fields: [
                    {
                      name: 'userName',
                      type: 'string',
                    },
                    {
                      name: 'displayName',
                      type: 'string',
                    },
                  ],
                  url: Suncode.getAbsolutePath('api/scheduledtasks/notifications/users'),
                  restful: true,
                  root: 'data',
                  totalProperty: 'total',
                  remoteSort: true,
                  sortInfo: {
                    field: "userName",
                    direction: "ASC",
                  },
                  autoLoad: true
                },

                forceSelection: true,
                triggerAction: 'all',
                minChars: 3,
                invalidClass: 'avoid-border-non-existent-class',
                listeners: {
                  invalid: function (field) {
                    field.wrap.addClass('x-form-invalid');
                  },
                  valid: function (field) {
                    field.wrap.removeClass('x-form-invalid');
                  },
                },
              },
            },
            {
              xtype: 'arrayfield',
              fieldLabel: PW.t('emailNotifications'),

              fieldConfig: {
                xtype: 'textfield',

                name: 'emailNotifications',
                allowBlank: false,
                height: 40,
              }
            },
          ],
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2 constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2, Ext.Container, {
  onActivate: function () {
    this.ownerCt.setHeight(this.getHeight() + 20);
    this.nameField.focus();

    //sprawia, że będziemy poruszac się tabulatorem tylko po polach wewnątrz okna
    var firstElement = this.nameField;
    var lastElement = this.ownerCt.ownerCt.addButton;

    Ext.EventManager.addListener(firstElement.el, 'keydown', function (e) {
      var key = e.getKey();
      var shiftKey = e.shiftKey;
      if (shiftKey && key == e.TAB) {
        e.stopEvent();
        lastElement.focus(false, 100);
      }
    });

    Ext.EventManager.addListener(lastElement.el, 'keydown', function (e) {
      var key = e.getKey();
      var shiftKey = e.shiftKey;
      if (!shiftKey && key == e.TAB) {
        e.stopEvent();
        firstElement.focus(false, 100);
      }
    });

    var method = this.ownerCt.ownerCt.choosenMethod;
    if (method) {
      if (method.data.annotationName) this.nameField.setValue(method.data.annotationName);

      if (method.data.annotationDesc) this.descField.setValue(method.data.annotationDesc);

      if (method.data.category) this.categoryField.setValue(method.data.category);
      else {
        this.categoryField.setValue(this.defaultCategory);
      }
    }
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2.superclass.initComponent.call(this);
    this.on('activate', this.onActivate, this);
  },
}); // end of Ext.extend
Ext.reg('st_wizard_step2', Ext.ux.plusmpm.scheduledtasks.TaskCreateStep2);

/**
 * Okno wyszukiwania zadań zaplanowanych
 */
Ext.ux.plusmpm.scheduledtasks.SearchWindow = function (config) {
  config = Ext.apply(
    {
      width: 1000,
      height: 600,
      title: PW.t('scheduledTasksFinder'),
      layout: 'fit',
      id: 'task-finder',
      bbar: [
        '->',
        {
          xtype: 'button',
          text: PW.t('select'),
          iconCls: 'dvnt-icon-plus-circle',
          handler: this.onChooseHandler,
          scope: this,
        },
      ],
      tbar: {
        xtype: 'panel',
        layout: 'vbox',
        height: 100,
        border: false,
        items: [
          {
            xtype: 'toolbar',
            defaults: {
              xtype: 'checkbox',
              enableKeyEvents: true,
              listeners: {
                scope: this,
                check: this.onFilterFieldChange,
              },
            },
            items: [
              {
                field: 'deprecated',
                boxLabel: PW.t('show-deprecated'),
              },
            ],
          },
          {
            xtype: 'toolbar',
            defaults: {
              xtype: 'textfield',
              enableKeyEvents: true,
              listeners: {
                scope: this,
                keyup: this.onFilterFieldChange,
              },
            },
            items: [
              {
                field: 'annotationName',
              },
              {
                field: 'annotationDesc',
                cls: 'no-left-border',
              },
              {
                field: 'name',
                cls: 'no-left-border',
              },
              {
                field: 'className',
                cls: 'no-left-border',
              },
              {
                field: 'category',
                cls: 'no-left-border',
              },
            ],
          },
        ],
      },
      items: [
        {
          xtype: 'grid',
          ref: 'methodsGrid',
          stripeRows: true,
          enableColumnHide: false,
          enableColumnMove: false,
          store: config.store,
          cls: 'st-task-finder-grid',
          colModel: new Ext.grid.ColumnModel({
            defaults: {
              renderer: function (value, p, record) {
                if (record.data.deprecated) {
                  if (!Ext.isEmpty(value)) {
                    p.attr += 'style="text-decoration:line-through;';
                  }
                  p.attr += 'style=white-space:normal;"';

                  if (p.id == 0) value += '</br>(' + PW.t('deprecated') + ')';
                } else {
                  p.attr = 'style="white-space:normal;"';
                }
                return value;
              },
            },
            columns: [
              { header: PW.t('taskName'), sortable: true, dataIndex: 'annotationName' },
              { header: PW.t('taskDescription'), sortable: true, dataIndex: 'annotationDesc' },
              { header: PW.t('methodName'), sortable: true, dataIndex: 'name' },
              { header: PW.t('className'), sortable: true, dataIndex: 'className' },
              { header: PW.t('category'), sortable: true, dataIndex: 'category' },
            ],
          }),
          listeners: {
            scope: this,
            columnResize: this.columnResizeHandler,
            rowdblclick: this.onChooseHandler,
          },
          viewConfig: { forceFit: true },
          sm: new Ext.grid.RowSelectionModel({ singleSelect: true }),
          iconCls: 'icon-grid',
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.SearchWindow.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.SearchWindow constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.SearchWindow, Ext.Window, {
  onChooseHandler: function () {
    var record = this.methodsGrid.getSelectionModel().getSelected();
    if (record !== undefined) {
      var newRecord = record.copy();
      this.fireEvent('methodchoosen', newRecord);
      this.close();
    }
  },

  onAfterRender: function () {
    this.loadingMask = new Ext.LoadMask(this.body, { msg: PW.t('searchingForTasks'), store: this.store });
    this.loadTasks();
    this.columnResizeHandler();
    this.methodsGrid.on('resize', this.columnResizeHandler, this);
  },

  onStoreLoad: function () {},

  loadTasks: function () {
    this.store.load({
      scope: this,
      callback: function () {
        this.onFilterFieldChange();
      },
    });
  },

  columnResizeHandler: function () {
    var grid = this.methodsGrid;
    if (grid.rendered == false) return;

    var colModel = grid.getColumnModel();
    var noOfCols = colModel.getColumnCount(false);
    var textFields = this.getTopToolbar().items.get(1).items;

    for (var i = 0; i < noOfCols; i++) {
      var colWidth = colModel.getColumnWidth(i);

      var fromField = textFields.get(i);

      fromField.setWidth(colWidth + 2);
    }
  },

  onFilterFieldChange: function () {
    var filterArray = [];
    this.getTopToolbar()
      .items.get(0)
      .items.each(function (item) {
        var value = item.checked;
        if (value == false) {
          var filter = {
            property: item.field,
            value: value,
          };
          filterArray.push(filter);
        }
      }, this);

    this.getTopToolbar()
      .items.get(1)
      .items.each(function (item) {
        var value = item.el.getValue();
        if (Ext.isEmpty(value) == false) {
          var filter = {
            property: item.field,
            value: value,
            caseSensitive: false,
            anyMatch: true,
          };
          filterArray.push(filter);
        }
      }, this);

    if (filterArray.length > 0) this.store.filter(filterArray);
    else this.store.clearFilter();
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.SearchWindow.superclass.initComponent.call(this);

    this.on('afterrender', this.onAfterRender, this);
    this.on(
      'destroy',
      function () {
        Ext.destroy(this.loadingMask, this.detailsTemplate);
      },
      this
    );
    this.store.on('load', this.onStoreLoad, this);

    this.addEvents('methodchoosen');
  },
}); // end of Ext.extend
